;; gorilla-repl.fileformat = 1

;; **
;;; # Huri examples
;; **

;; @@
(ns huri.examples
  (:use (huri core plot etl))
  (:require (clj-time [format :as t.format]
                      [core :as t])
            [clojure.string :as s]))
;; @@

;; @@
(def df (:stationBeanList (slurp-json "http://www.bayareabikeshare.com/stations/json")))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;huri.examples/df</span>","value":"#'huri.examples/df"}
;; <=

;; @@
(cols df)
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>(:availableBikes :statusKey :renting :testStation :stAddress1 :landMark :city :longitude :lastCommunicationTime :id :stationName :postalCode :stAddress2 :latitude :totalDocks :statusValue :availableDocks :location :is_renting :altitude)</span>","value":"(:availableBikes :statusKey :renting :testStation :stAddress1 :landMark :city :longitude :lastCommunicationTime :id :stationName :postalCode :stAddress2 :latitude :totalDocks :statusValue :availableDocks :location :is_renting :altitude)"}
;; <=

;; @@
(size df)
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-unkown'>70</span>","value":"70"},{"type":"html","content":"<span class='clj-unkown'>20</span>","value":"20"}],"value":"[70 20]"}
;; <=

;; @@
(rollup :city count df)
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-map'>{</span>","close":"<span class='clj-map'>}</span>","separator":", ","items":[{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;Mountain View&quot;</span>","value":"\"Mountain View\""},{"type":"html","content":"<span class='clj-unkown'>7</span>","value":"7"}],"value":"[\"Mountain View\" 7]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;Palo Alto&quot;</span>","value":"\"Palo Alto\""},{"type":"html","content":"<span class='clj-unkown'>5</span>","value":"5"}],"value":"[\"Palo Alto\" 5]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;Redwood City&quot;</span>","value":"\"Redwood City\""},{"type":"html","content":"<span class='clj-unkown'>7</span>","value":"7"}],"value":"[\"Redwood City\" 7]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;San Francisco&quot;</span>","value":"\"San Francisco\""},{"type":"html","content":"<span class='clj-unkown'>35</span>","value":"35"}],"value":"[\"San Francisco\" 35]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;San Jose&quot;</span>","value":"\"San Jose\""},{"type":"html","content":"<span class='clj-unkown'>16</span>","value":"16"}],"value":"[\"San Jose\" 16]"}],"value":"{\"Mountain View\" 7, \"Palo Alto\" 5, \"Redwood City\" 7, \"San Francisco\" 35, \"San Jose\" 16}"}
;; <=

;; @@
(rollup :city mean :totalDocks df)
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-map'>{</span>","close":"<span class='clj-map'>}</span>","separator":", ","items":[{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;Mountain View&quot;</span>","value":"\"Mountain View\""},{"type":"html","content":"<span class='clj-double'>16.71428571428571</span>","value":"16.71428571428571"}],"value":"[\"Mountain View\" 16.71428571428571]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;Palo Alto&quot;</span>","value":"\"Palo Alto\""},{"type":"html","content":"<span class='clj-double'>15.0</span>","value":"15.0"}],"value":"[\"Palo Alto\" 15.0]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;Redwood City&quot;</span>","value":"\"Redwood City\""},{"type":"html","content":"<span class='clj-double'>16.71428571428571</span>","value":"16.71428571428571"}],"value":"[\"Redwood City\" 16.71428571428571]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;San Francisco&quot;</span>","value":"\"San Francisco\""},{"type":"html","content":"<span class='clj-double'>19.11428571428571</span>","value":"19.11428571428571"}],"value":"[\"San Francisco\" 19.11428571428571]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;San Jose&quot;</span>","value":"\"San Jose\""},{"type":"html","content":"<span class='clj-double'>16.5</span>","value":"16.5"}],"value":"[\"San Jose\" 16.5]"}],"value":"{\"Mountain View\" 16.71428571428571, \"Palo Alto\" 15.0, \"Redwood City\" 16.71428571428571, \"San Francisco\" 19.11428571428571, \"San Jose\" 16.5}"}
;; <=

;; @@
(where {:availableDocks [< 2]} df)
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-map'>{</span>","close":"<span class='clj-map'>}</span>","separator":", ","items":[{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:availableBikes</span>","value":":availableBikes"},{"type":"html","content":"<span class='clj-unkown'>10</span>","value":"10"}],"value":"[:availableBikes 10]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:statusKey</span>","value":":statusKey"},{"type":"html","content":"<span class='clj-unkown'>1</span>","value":"1"}],"value":"[:statusKey 1]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:renting</span>","value":":renting"},{"type":"html","content":"<span class='clj-unkown'>true</span>","value":"true"}],"value":"[:renting true]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:testStation</span>","value":":testStation"},{"type":"html","content":"<span class='clj-unkown'>false</span>","value":"false"}],"value":"[:testStation false]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:stAddress1</span>","value":":stAddress1"},{"type":"html","content":"<span class='clj-string'>&quot;Santa Clara at Almaden&quot;</span>","value":"\"Santa Clara at Almaden\""}],"value":"[:stAddress1 \"Santa Clara at Almaden\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:landMark</span>","value":":landMark"},{"type":"html","content":"<span class='clj-string'>&quot;San Jose&quot;</span>","value":"\"San Jose\""}],"value":"[:landMark \"San Jose\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:city</span>","value":":city"},{"type":"html","content":"<span class='clj-string'>&quot;San Jose&quot;</span>","value":"\"San Jose\""}],"value":"[:city \"San Jose\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:longitude</span>","value":":longitude"},{"type":"html","content":"<span class='clj-double'>-121.894902</span>","value":"-121.894902"}],"value":"[:longitude -121.894902]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:lastCommunicationTime</span>","value":":lastCommunicationTime"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"[:lastCommunicationTime nil]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:id</span>","value":":id"},{"type":"html","content":"<span class='clj-unkown'>4</span>","value":"4"}],"value":"[:id 4]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:stationName</span>","value":":stationName"},{"type":"html","content":"<span class='clj-string'>&quot;Santa Clara at Almaden&quot;</span>","value":"\"Santa Clara at Almaden\""}],"value":"[:stationName \"Santa Clara at Almaden\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:postalCode</span>","value":":postalCode"},{"type":"html","content":"<span class='clj-string'>&quot;&quot;</span>","value":"\"\""}],"value":"[:postalCode \"\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:stAddress2</span>","value":":stAddress2"},{"type":"html","content":"<span class='clj-string'>&quot;&quot;</span>","value":"\"\""}],"value":"[:stAddress2 \"\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:latitude</span>","value":":latitude"},{"type":"html","content":"<span class='clj-double'>37.333988</span>","value":"37.333988"}],"value":"[:latitude 37.333988]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:totalDocks</span>","value":":totalDocks"},{"type":"html","content":"<span class='clj-unkown'>11</span>","value":"11"}],"value":"[:totalDocks 11]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:statusValue</span>","value":":statusValue"},{"type":"html","content":"<span class='clj-string'>&quot;In Service&quot;</span>","value":"\"In Service\""}],"value":"[:statusValue \"In Service\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:availableDocks</span>","value":":availableDocks"},{"type":"html","content":"<span class='clj-unkown'>1</span>","value":"1"}],"value":"[:availableDocks 1]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:location</span>","value":":location"},{"type":"html","content":"<span class='clj-string'>&quot;W Santa Clara Street&quot;</span>","value":"\"W Santa Clara Street\""}],"value":"[:location \"W Santa Clara Street\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:is_renting</span>","value":":is_renting"},{"type":"html","content":"<span class='clj-unkown'>true</span>","value":"true"}],"value":"[:is_renting true]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:altitude</span>","value":":altitude"},{"type":"html","content":"<span class='clj-string'>&quot;&quot;</span>","value":"\"\""}],"value":"[:altitude \"\"]"}],"value":"{:availableBikes 10, :statusKey 1, :renting true, :testStation false, :stAddress1 \"Santa Clara at Almaden\", :landMark \"San Jose\", :city \"San Jose\", :longitude -121.894902, :lastCommunicationTime nil, :id 4, :stationName \"Santa Clara at Almaden\", :postalCode \"\", :stAddress2 \"\", :latitude 37.333988, :totalDocks 11, :statusValue \"In Service\", :availableDocks 1, :location \"W Santa Clara Street\", :is_renting true, :altitude \"\"}"}],"value":"[{:availableBikes 10, :statusKey 1, :renting true, :testStation false, :stAddress1 \"Santa Clara at Almaden\", :landMark \"San Jose\", :city \"San Jose\", :longitude -121.894902, :lastCommunicationTime nil, :id 4, :stationName \"Santa Clara at Almaden\", :postalCode \"\", :stAddress2 \"\", :latitude 37.333988, :totalDocks 11, :statusValue \"In Service\", :availableDocks 1, :location \"W Santa Clara Street\", :is_renting true, :altitude \"\"}]"}
;; <=

;; @@
(count-where {(any-of :availableDocks :availableBikes) pos?} df)
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>70</span>","value":"70"}
;; <=

;; @@
(count-where {(every-of :availableDocks :availableBikes) pos?} df)
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>69</span>","value":"69"}
;; <=

;; @@
(count-where {:city [s/starts-with? "San"]} df)
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>51</span>","value":"51"}
;; <=

;; @@
(rollup :city {:mean-available [mean :availableDocks ]
               :n-stations count} 
        df)
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-map'>{</span>","close":"<span class='clj-map'>}</span>","separator":", ","items":[{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;Mountain View&quot;</span>","value":"\"Mountain View\""},{"type":"list-like","open":"<span class='clj-map'>{</span>","close":"<span class='clj-map'>}</span>","separator":", ","items":[{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:mean-available</span>","value":":mean-available"},{"type":"html","content":"<span class='clj-double'>8.571428571428571</span>","value":"8.571428571428571"}],"value":"[:mean-available 8.571428571428571]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:n-stations</span>","value":":n-stations"},{"type":"html","content":"<span class='clj-unkown'>7</span>","value":"7"}],"value":"[:n-stations 7]"}],"value":"{:mean-available 8.571428571428571, :n-stations 7}"}],"value":"[\"Mountain View\" {:mean-available 8.571428571428571, :n-stations 7}]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;Palo Alto&quot;</span>","value":"\"Palo Alto\""},{"type":"list-like","open":"<span class='clj-map'>{</span>","close":"<span class='clj-map'>}</span>","separator":", ","items":[{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:mean-available</span>","value":":mean-available"},{"type":"html","content":"<span class='clj-double'>8.2</span>","value":"8.2"}],"value":"[:mean-available 8.2]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:n-stations</span>","value":":n-stations"},{"type":"html","content":"<span class='clj-unkown'>5</span>","value":"5"}],"value":"[:n-stations 5]"}],"value":"{:mean-available 8.2, :n-stations 5}"}],"value":"[\"Palo Alto\" {:mean-available 8.2, :n-stations 5}]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;Redwood City&quot;</span>","value":"\"Redwood City\""},{"type":"list-like","open":"<span class='clj-map'>{</span>","close":"<span class='clj-map'>}</span>","separator":", ","items":[{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:mean-available</span>","value":":mean-available"},{"type":"html","content":"<span class='clj-double'>8.857142857142858</span>","value":"8.857142857142858"}],"value":"[:mean-available 8.857142857142858]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:n-stations</span>","value":":n-stations"},{"type":"html","content":"<span class='clj-unkown'>7</span>","value":"7"}],"value":"[:n-stations 7]"}],"value":"{:mean-available 8.857142857142858, :n-stations 7}"}],"value":"[\"Redwood City\" {:mean-available 8.857142857142858, :n-stations 7}]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;San Francisco&quot;</span>","value":"\"San Francisco\""},{"type":"list-like","open":"<span class='clj-map'>{</span>","close":"<span class='clj-map'>}</span>","separator":", ","items":[{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:mean-available</span>","value":":mean-available"},{"type":"html","content":"<span class='clj-double'>10.62857142857143</span>","value":"10.62857142857143"}],"value":"[:mean-available 10.62857142857143]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:n-stations</span>","value":":n-stations"},{"type":"html","content":"<span class='clj-unkown'>35</span>","value":"35"}],"value":"[:n-stations 35]"}],"value":"{:mean-available 10.62857142857143, :n-stations 35}"}],"value":"[\"San Francisco\" {:mean-available 10.62857142857143, :n-stations 35}]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;San Jose&quot;</span>","value":"\"San Jose\""},{"type":"list-like","open":"<span class='clj-map'>{</span>","close":"<span class='clj-map'>}</span>","separator":", ","items":[{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:mean-available</span>","value":":mean-available"},{"type":"html","content":"<span class='clj-double'>9.875</span>","value":"9.875"}],"value":"[:mean-available 9.875]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:n-stations</span>","value":":n-stations"},{"type":"html","content":"<span class='clj-unkown'>16</span>","value":"16"}],"value":"[:n-stations 16]"}],"value":"{:mean-available 9.875, :n-stations 16}"}],"value":"[\"San Jose\" {:mean-available 9.875, :n-stations 16}]"}],"value":"{\"Mountain View\" {:mean-available 8.571428571428571, :n-stations 7}, \"Palo Alto\" {:mean-available 8.2, :n-stations 5}, \"Redwood City\" {:mean-available 8.857142857142858, :n-stations 7}, \"San Francisco\" {:mean-available 10.62857142857143, :n-stations 35}, \"San Jose\" {:mean-available 9.875, :n-stations 16}}"}
;; <=

;; @@
(rollup-fuse :city {:mean-available [mean :availableDocks]
                    :n-stations count} 
        df)
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>({:mean-available 8.571428571428571, :n-stations 7, :city &quot;Mountain View&quot;} {:mean-available 8.2, :n-stations 5, :city &quot;Palo Alto&quot;} {:mean-available 8.857142857142858, :n-stations 7, :city &quot;Redwood City&quot;} {:mean-available 10.62857142857143, :n-stations 35, :city &quot;San Francisco&quot;} {:mean-available 9.875, :n-stations 16, :city &quot;San Jose&quot;})</span>","value":"({:mean-available 8.571428571428571, :n-stations 7, :city \"Mountain View\"} {:mean-available 8.2, :n-stations 5, :city \"Palo Alto\"} {:mean-available 8.857142857142858, :n-stations 7, :city \"Redwood City\"} {:mean-available 10.62857142857143, :n-stations 35, :city \"San Francisco\"} {:mean-available 9.875, :n-stations 16, :city \"San Jose\"})"}
;; <=

;; @@
(count-distinct :city df)
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>5</span>","value":"5"}
;; <=

;; @@
(distribution :city df)
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-map'>{</span>","close":"<span class='clj-map'>}</span>","separator":", ","items":[{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;San Francisco&quot;</span>","value":"\"San Francisco\""},{"type":"html","content":"<span class='clj-double'>0.5000000000000002</span>","value":"0.5000000000000002"}],"value":"[\"San Francisco\" 0.5000000000000002]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;San Jose&quot;</span>","value":"\"San Jose\""},{"type":"html","content":"<span class='clj-double'>0.22857142857142865</span>","value":"0.22857142857142865"}],"value":"[\"San Jose\" 0.22857142857142865]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;Redwood City&quot;</span>","value":"\"Redwood City\""},{"type":"html","content":"<span class='clj-double'>0.10000000000000003</span>","value":"0.10000000000000003"}],"value":"[\"Redwood City\" 0.10000000000000003]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;Mountain View&quot;</span>","value":"\"Mountain View\""},{"type":"html","content":"<span class='clj-double'>0.10000000000000003</span>","value":"0.10000000000000003"}],"value":"[\"Mountain View\" 0.10000000000000003]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;Palo Alto&quot;</span>","value":"\"Palo Alto\""},{"type":"html","content":"<span class='clj-double'>0.07142857142857145</span>","value":"0.07142857142857145"}],"value":"[\"Palo Alto\" 0.07142857142857145]"}],"value":"{\"San Francisco\" 0.5000000000000002, \"San Jose\" 0.22857142857142865, \"Redwood City\" 0.10000000000000003, \"Mountain View\" 0.10000000000000003, \"Palo Alto\" 0.07142857142857145}"}
;; <=

;; @@
(distribution :city {:weightfn :totalDocks} df)
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-map'>{</span>","close":"<span class='clj-map'>}</span>","separator":", ","items":[{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;San Francisco&quot;</span>","value":"\"San Francisco\""},{"type":"html","content":"<span class='clj-double'>0.538647342995169</span>","value":"0.538647342995169"}],"value":"[\"San Francisco\" 0.538647342995169]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;San Jose&quot;</span>","value":"\"San Jose\""},{"type":"html","content":"<span class='clj-double'>0.21256038647342995</span>","value":"0.21256038647342995"}],"value":"[\"San Jose\" 0.21256038647342995]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;Redwood City&quot;</span>","value":"\"Redwood City\""},{"type":"html","content":"<span class='clj-double'>0.09420289855072464</span>","value":"0.09420289855072464"}],"value":"[\"Redwood City\" 0.09420289855072464]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;Mountain View&quot;</span>","value":"\"Mountain View\""},{"type":"html","content":"<span class='clj-double'>0.09420289855072464</span>","value":"0.09420289855072464"}],"value":"[\"Mountain View\" 0.09420289855072464]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-string'>&quot;Palo Alto&quot;</span>","value":"\"Palo Alto\""},{"type":"html","content":"<span class='clj-double'>0.06038647342995169</span>","value":"0.06038647342995169"}],"value":"[\"Palo Alto\" 0.06038647342995169]"}],"value":"{\"San Francisco\" 0.538647342995169, \"San Jose\" 0.21256038647342995, \"Redwood City\" 0.09420289855072464, \"Mountain View\" 0.09420289855072464, \"Palo Alto\" 0.06038647342995169}"}
;; <=

;; @@
(extent :totalDocks df)
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-vector'>[</span>","close":"<span class='clj-vector'>]</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-unkown'>11</span>","value":"11"},{"type":"html","content":"<span class='clj-unkown'>27</span>","value":"27"}],"value":"[11 27]"}
;; <=

;; @@
(take 2 (select-cols [:stationName :statusValue] df))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-map'>{</span>","close":"<span class='clj-map'>}</span>","separator":", ","items":[{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:stationName</span>","value":":stationName"},{"type":"html","content":"<span class='clj-string'>&quot;San Jose Diridon Caltrain Station&quot;</span>","value":"\"San Jose Diridon Caltrain Station\""}],"value":"[:stationName \"San Jose Diridon Caltrain Station\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:statusValue</span>","value":":statusValue"},{"type":"html","content":"<span class='clj-string'>&quot;In Service&quot;</span>","value":"\"In Service\""}],"value":"[:statusValue \"In Service\"]"}],"value":"{:stationName \"San Jose Diridon Caltrain Station\", :statusValue \"In Service\"}"},{"type":"list-like","open":"<span class='clj-map'>{</span>","close":"<span class='clj-map'>}</span>","separator":", ","items":[{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:stationName</span>","value":":stationName"},{"type":"html","content":"<span class='clj-string'>&quot;San Jose Civic Center&quot;</span>","value":"\"San Jose Civic Center\""}],"value":"[:stationName \"San Jose Civic Center\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:statusValue</span>","value":":statusValue"},{"type":"html","content":"<span class='clj-string'>&quot;In Service&quot;</span>","value":"\"In Service\""}],"value":"[:statusValue \"In Service\"]"}],"value":"{:stationName \"San Jose Civic Center\", :statusValue \"In Service\"}"}],"value":"({:stationName \"San Jose Diridon Caltrain Station\", :statusValue \"In Service\"} {:stationName \"San Jose Civic Center\", :statusValue \"In Service\"})"}
;; <=

;; @@
(take 2 (derive-cols {:occupancy [/ :availableBikes  :totalDocks]} df))
;; @@
;; =>
;;; {"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"list-like","open":"<span class='clj-map'>{</span>","close":"<span class='clj-map'>}</span>","separator":", ","items":[{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:availableBikes</span>","value":":availableBikes"},{"type":"html","content":"<span class='clj-unkown'>8</span>","value":"8"}],"value":"[:availableBikes 8]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:statusKey</span>","value":":statusKey"},{"type":"html","content":"<span class='clj-unkown'>1</span>","value":"1"}],"value":"[:statusKey 1]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:renting</span>","value":":renting"},{"type":"html","content":"<span class='clj-unkown'>true</span>","value":"true"}],"value":"[:renting true]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:testStation</span>","value":":testStation"},{"type":"html","content":"<span class='clj-unkown'>false</span>","value":"false"}],"value":"[:testStation false]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:occupancy</span>","value":":occupancy"},{"type":"html","content":"<span class='clj-ratio'>8/27</span>","value":"8/27"}],"value":"[:occupancy 8/27]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:stAddress1</span>","value":":stAddress1"},{"type":"html","content":"<span class='clj-string'>&quot;San Jose Diridon Caltrain Station&quot;</span>","value":"\"San Jose Diridon Caltrain Station\""}],"value":"[:stAddress1 \"San Jose Diridon Caltrain Station\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:landMark</span>","value":":landMark"},{"type":"html","content":"<span class='clj-string'>&quot;San Jose&quot;</span>","value":"\"San Jose\""}],"value":"[:landMark \"San Jose\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:city</span>","value":":city"},{"type":"html","content":"<span class='clj-string'>&quot;San Jose&quot;</span>","value":"\"San Jose\""}],"value":"[:city \"San Jose\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:longitude</span>","value":":longitude"},{"type":"html","content":"<span class='clj-double'>-121.901782</span>","value":"-121.901782"}],"value":"[:longitude -121.901782]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:lastCommunicationTime</span>","value":":lastCommunicationTime"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"[:lastCommunicationTime nil]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:id</span>","value":":id"},{"type":"html","content":"<span class='clj-unkown'>2</span>","value":"2"}],"value":"[:id 2]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:stationName</span>","value":":stationName"},{"type":"html","content":"<span class='clj-string'>&quot;San Jose Diridon Caltrain Station&quot;</span>","value":"\"San Jose Diridon Caltrain Station\""}],"value":"[:stationName \"San Jose Diridon Caltrain Station\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:postalCode</span>","value":":postalCode"},{"type":"html","content":"<span class='clj-string'>&quot;&quot;</span>","value":"\"\""}],"value":"[:postalCode \"\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:stAddress2</span>","value":":stAddress2"},{"type":"html","content":"<span class='clj-string'>&quot;&quot;</span>","value":"\"\""}],"value":"[:stAddress2 \"\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:latitude</span>","value":":latitude"},{"type":"html","content":"<span class='clj-double'>37.329732</span>","value":"37.329732"}],"value":"[:latitude 37.329732]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:totalDocks</span>","value":":totalDocks"},{"type":"html","content":"<span class='clj-unkown'>27</span>","value":"27"}],"value":"[:totalDocks 27]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:statusValue</span>","value":":statusValue"},{"type":"html","content":"<span class='clj-string'>&quot;In Service&quot;</span>","value":"\"In Service\""}],"value":"[:statusValue \"In Service\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:availableDocks</span>","value":":availableDocks"},{"type":"html","content":"<span class='clj-unkown'>19</span>","value":"19"}],"value":"[:availableDocks 19]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:location</span>","value":":location"},{"type":"html","content":"<span class='clj-string'>&quot;Crandall Street&quot;</span>","value":"\"Crandall Street\""}],"value":"[:location \"Crandall Street\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:is_renting</span>","value":":is_renting"},{"type":"html","content":"<span class='clj-unkown'>true</span>","value":"true"}],"value":"[:is_renting true]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:altitude</span>","value":":altitude"},{"type":"html","content":"<span class='clj-string'>&quot;&quot;</span>","value":"\"\""}],"value":"[:altitude \"\"]"}],"value":"{:availableBikes 8, :statusKey 1, :renting true, :testStation false, :occupancy 8/27, :stAddress1 \"San Jose Diridon Caltrain Station\", :landMark \"San Jose\", :city \"San Jose\", :longitude -121.901782, :lastCommunicationTime nil, :id 2, :stationName \"San Jose Diridon Caltrain Station\", :postalCode \"\", :stAddress2 \"\", :latitude 37.329732, :totalDocks 27, :statusValue \"In Service\", :availableDocks 19, :location \"Crandall Street\", :is_renting true, :altitude \"\"}"},{"type":"list-like","open":"<span class='clj-map'>{</span>","close":"<span class='clj-map'>}</span>","separator":", ","items":[{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:availableBikes</span>","value":":availableBikes"},{"type":"html","content":"<span class='clj-unkown'>7</span>","value":"7"}],"value":"[:availableBikes 7]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:statusKey</span>","value":":statusKey"},{"type":"html","content":"<span class='clj-unkown'>1</span>","value":"1"}],"value":"[:statusKey 1]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:renting</span>","value":":renting"},{"type":"html","content":"<span class='clj-unkown'>true</span>","value":"true"}],"value":"[:renting true]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:testStation</span>","value":":testStation"},{"type":"html","content":"<span class='clj-unkown'>false</span>","value":"false"}],"value":"[:testStation false]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:occupancy</span>","value":":occupancy"},{"type":"html","content":"<span class='clj-ratio'>7/15</span>","value":"7/15"}],"value":"[:occupancy 7/15]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:stAddress1</span>","value":":stAddress1"},{"type":"html","content":"<span class='clj-string'>&quot;San Jose Civic Center&quot;</span>","value":"\"San Jose Civic Center\""}],"value":"[:stAddress1 \"San Jose Civic Center\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:landMark</span>","value":":landMark"},{"type":"html","content":"<span class='clj-string'>&quot;San Jose&quot;</span>","value":"\"San Jose\""}],"value":"[:landMark \"San Jose\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:city</span>","value":":city"},{"type":"html","content":"<span class='clj-string'>&quot;San Jose&quot;</span>","value":"\"San Jose\""}],"value":"[:city \"San Jose\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:longitude</span>","value":":longitude"},{"type":"html","content":"<span class='clj-double'>-121.888979</span>","value":"-121.888979"}],"value":"[:longitude -121.888979]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:lastCommunicationTime</span>","value":":lastCommunicationTime"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"[:lastCommunicationTime nil]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:id</span>","value":":id"},{"type":"html","content":"<span class='clj-unkown'>3</span>","value":"3"}],"value":"[:id 3]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:stationName</span>","value":":stationName"},{"type":"html","content":"<span class='clj-string'>&quot;San Jose Civic Center&quot;</span>","value":"\"San Jose Civic Center\""}],"value":"[:stationName \"San Jose Civic Center\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:postalCode</span>","value":":postalCode"},{"type":"html","content":"<span class='clj-string'>&quot;&quot;</span>","value":"\"\""}],"value":"[:postalCode \"\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:stAddress2</span>","value":":stAddress2"},{"type":"html","content":"<span class='clj-string'>&quot;&quot;</span>","value":"\"\""}],"value":"[:stAddress2 \"\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:latitude</span>","value":":latitude"},{"type":"html","content":"<span class='clj-double'>37.330698</span>","value":"37.330698"}],"value":"[:latitude 37.330698]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:totalDocks</span>","value":":totalDocks"},{"type":"html","content":"<span class='clj-unkown'>15</span>","value":"15"}],"value":"[:totalDocks 15]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:statusValue</span>","value":":statusValue"},{"type":"html","content":"<span class='clj-string'>&quot;In Service&quot;</span>","value":"\"In Service\""}],"value":"[:statusValue \"In Service\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:availableDocks</span>","value":":availableDocks"},{"type":"html","content":"<span class='clj-unkown'>8</span>","value":"8"}],"value":"[:availableDocks 8]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:location</span>","value":":location"},{"type":"html","content":"<span class='clj-string'>&quot;W San Carlos Street&quot;</span>","value":"\"W San Carlos Street\""}],"value":"[:location \"W San Carlos Street\"]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:is_renting</span>","value":":is_renting"},{"type":"html","content":"<span class='clj-unkown'>true</span>","value":"true"}],"value":"[:is_renting true]"},{"type":"list-like","open":"","close":"","separator":" ","items":[{"type":"html","content":"<span class='clj-keyword'>:altitude</span>","value":":altitude"},{"type":"html","content":"<span class='clj-string'>&quot;&quot;</span>","value":"\"\""}],"value":"[:altitude \"\"]"}],"value":"{:availableBikes 7, :statusKey 1, :renting true, :testStation false, :occupancy 7/15, :stAddress1 \"San Jose Civic Center\", :landMark \"San Jose\", :city \"San Jose\", :longitude -121.888979, :lastCommunicationTime nil, :id 3, :stationName \"San Jose Civic Center\", :postalCode \"\", :stAddress2 \"\", :latitude 37.330698, :totalDocks 15, :statusValue \"In Service\", :availableDocks 8, :location \"W San Carlos Street\", :is_renting true, :altitude \"\"}"}],"value":"({:availableBikes 8, :statusKey 1, :renting true, :testStation false, :occupancy 8/27, :stAddress1 \"San Jose Diridon Caltrain Station\", :landMark \"San Jose\", :city \"San Jose\", :longitude -121.901782, :lastCommunicationTime nil, :id 2, :stationName \"San Jose Diridon Caltrain Station\", :postalCode \"\", :stAddress2 \"\", :latitude 37.329732, :totalDocks 27, :statusValue \"In Service\", :availableDocks 19, :location \"Crandall Street\", :is_renting true, :altitude \"\"} {:availableBikes 7, :statusKey 1, :renting true, :testStation false, :occupancy 7/15, :stAddress1 \"San Jose Civic Center\", :landMark \"San Jose\", :city \"San Jose\", :longitude -121.888979, :lastCommunicationTime nil, :id 3, :stationName \"San Jose Civic Center\", :postalCode \"\", :stAddress2 \"\", :latitude 37.330698, :totalDocks 15, :statusValue \"In Service\", :availableDocks 8, :location \"W San Carlos Street\", :is_renting true, :altitude \"\"})"}
;; <=

;; **
;;; ## Data and time
;; **

;; @@
(week (t/now))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>#object[org.joda.time.DateTime 0x45598f34 &quot;2016-03-14T00:00:00.000Z&quot;]</span>","value":"#object[org.joda.time.DateTime 0x45598f34 \"2016-03-14T00:00:00.000Z\"]"}
;; <=

;; @@
(week-of-year (t/now))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>11</span>","value":"11"}
;; <=

;; @@
(year-month (t/now))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>#object[org.joda.time.DateTime 0x6152be47 &quot;2016-03-01T00:00:00.000Z&quot;]</span>","value":"#object[org.joda.time.DateTime 0x6152be47 \"2016-03-01T00:00:00.000Z\"]"}
;; <=

;; @@
(date (t/now))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>#object[org.joda.time.DateTime 0x64331dce &quot;2016-03-19T00:00:00.000Z&quot;]</span>","value":"#object[org.joda.time.DateTime 0x64331dce \"2016-03-19T00:00:00.000Z\"]"}
;; <=

;; @@
(day-of-year (t/now))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-long'>79</span>","value":"79"}
;; <=

;; @@
(in? (t/now) 2016)
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>true</span>","value":"true"}
;; <=

;; @@
(in? (t/now) 2016 3)
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>true</span>","value":"true"}
;; <=

;; @@
(before-now? (t/date-time 2011))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>true</span>","value":"true"}
;; <=

;; @@
(after-now? (t/date-time 2111))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>true</span>","value":"true"}
;; <=

;; **
;;; ## Plotting
;; **

;; @@
(histogram :availableBikes {:bins 10} df)
;; @@
;; =>
;;; {"type":"html","content":"<?xml version='1.0' encoding='UTF-8'?>\n<svg version='1.1' viewBox='0 0 648 360' height='360pt' width='648pt' xmlns:xlink='http://www.w3.org/1999/xlink' xmlns='http://www.w3.org/2000/svg'>\n<defs>\n<g>\n<symbol id='9ac48d51-05d5-4cee-a961-5578526b22b0' overflow='visible'>\n<path d='M 0.21875 0 L 0.21875 -5.015625 L 4.21875 -5.015625 L 4.21875 0 Z M 3.578125 -0.625 L 3.578125 -4.390625 L 0.859375 -4.390625 L 0.859375 -0.625 Z M 3.578125 -0.625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='dcad652e-1793-4162-a8f7-cc7a4e8bab1a' overflow='visible'>\n<path d='M 1.890625 -4.890625 C 2.523438 -4.890625 2.984375 -4.628906 3.265625 -4.109375 C 3.484375 -3.703125 3.59375 -3.148438 3.59375 -2.453125 C 3.59375 -1.785156 3.492188 -1.234375 3.296875 -0.796875 C 3.015625 -0.171875 2.546875 0.140625 1.890625 0.140625 C 1.304688 0.140625 0.867188 -0.113281 0.578125 -0.625 C 0.335938 -1.050781 0.21875 -1.625 0.21875 -2.34375 C 0.21875 -2.90625 0.289062 -3.382812 0.4375 -3.78125 C 0.707031 -4.519531 1.191406 -4.890625 1.890625 -4.890625 Z M 1.890625 -0.421875 C 2.203125 -0.421875 2.453125 -0.5625 2.640625 -0.84375 C 2.835938 -1.132812 2.9375 -1.664062 2.9375 -2.4375 C 2.9375 -2.988281 2.863281 -3.441406 2.71875 -3.796875 C 2.582031 -4.160156 2.320312 -4.34375 1.9375 -4.34375 C 1.570312 -4.34375 1.304688 -4.171875 1.140625 -3.828125 C 0.972656 -3.484375 0.890625 -2.984375 0.890625 -2.328125 C 0.890625 -1.828125 0.941406 -1.425781 1.046875 -1.125 C 1.210938 -0.65625 1.492188 -0.421875 1.890625 -0.421875 Z M 1.890625 -0.421875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='6f8e18d4-863a-4651-bc7c-cdc4f2010343' overflow='visible'>\n<path d='M 0.859375 -1.25 C 0.898438 -0.894531 1.066406 -0.648438 1.359375 -0.515625 C 1.503906 -0.453125 1.671875 -0.421875 1.859375 -0.421875 C 2.222656 -0.421875 2.492188 -0.535156 2.671875 -0.765625 C 2.847656 -0.992188 2.9375 -1.25 2.9375 -1.53125 C 2.9375 -1.875 2.832031 -2.140625 2.625 -2.328125 C 2.414062 -2.515625 2.164062 -2.609375 1.875 -2.609375 C 1.65625 -2.609375 1.46875 -2.566406 1.3125 -2.484375 C 1.164062 -2.398438 1.039062 -2.285156 0.9375 -2.140625 L 0.40625 -2.171875 L 0.78125 -4.8125 L 3.3125 -4.8125 L 3.3125 -4.21875 L 1.234375 -4.21875 L 1.03125 -2.859375 C 1.144531 -2.941406 1.253906 -3.003906 1.359375 -3.046875 C 1.535156 -3.128906 1.742188 -3.171875 1.984375 -3.171875 C 2.429688 -3.171875 2.8125 -3.023438 3.125 -2.734375 C 3.4375 -2.441406 3.59375 -2.070312 3.59375 -1.625 C 3.59375 -1.164062 3.445312 -0.757812 3.15625 -0.40625 C 2.875 -0.0507812 2.421875 0.125 1.796875 0.125 C 1.398438 0.125 1.046875 0.015625 0.734375 -0.203125 C 0.429688 -0.429688 0.257812 -0.78125 0.21875 -1.25 Z M 0.859375 -1.25 ' style='stroke:none;'/>\n</symbol>\n<symbol id='2f89e33b-46b8-4b97-acde-3f9bec4e57e2' overflow='visible'>\n<path d='M 0.671875 -3.46875 L 0.671875 -3.9375 C 1.117188 -3.976562 1.425781 -4.050781 1.59375 -4.15625 C 1.769531 -4.257812 1.90625 -4.5 2 -4.875 L 2.484375 -4.875 L 2.484375 0 L 1.828125 0 L 1.828125 -3.46875 Z M 0.671875 -3.46875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='f7a36802-bdfb-46f0-8c5c-feb1bcd4cc13' overflow='visible'>\n<path d='M 0.21875 0 C 0.238281 -0.414062 0.320312 -0.78125 0.46875 -1.09375 C 0.625 -1.40625 0.925781 -1.691406 1.375 -1.953125 L 2.03125 -2.328125 C 2.320312 -2.503906 2.523438 -2.648438 2.640625 -2.765625 C 2.828125 -2.953125 2.921875 -3.171875 2.921875 -3.421875 C 2.921875 -3.703125 2.832031 -3.925781 2.65625 -4.09375 C 2.488281 -4.257812 2.265625 -4.34375 1.984375 -4.34375 C 1.554688 -4.34375 1.265625 -4.1875 1.109375 -3.875 C 1.023438 -3.695312 0.976562 -3.457031 0.96875 -3.15625 L 0.34375 -3.15625 C 0.34375 -3.582031 0.421875 -3.929688 0.578125 -4.203125 C 0.835938 -4.671875 1.304688 -4.90625 1.984375 -4.90625 C 2.546875 -4.90625 2.957031 -4.753906 3.21875 -4.453125 C 3.476562 -4.148438 3.609375 -3.816406 3.609375 -3.453125 C 3.609375 -3.054688 3.46875 -2.71875 3.1875 -2.4375 C 3.03125 -2.28125 2.742188 -2.085938 2.328125 -1.859375 L 1.859375 -1.59375 C 1.640625 -1.46875 1.46875 -1.347656 1.34375 -1.234375 C 1.113281 -1.035156 0.96875 -0.816406 0.90625 -0.578125 L 3.578125 -0.578125 L 3.578125 0 Z M 0.21875 0 ' style='stroke:none;'/>\n</symbol>\n<symbol id='f52d78ce-fae6-4ef8-ba04-f5c548cd5b3a' overflow='visible'>\n<path d='M 0.265625 0 L 0.265625 -5.734375 L 4.8125 -5.734375 L 4.8125 0 Z M 4.09375 -0.71875 L 4.09375 -5.015625 L 0.984375 -5.015625 L 0.984375 -0.71875 Z M 4.09375 -0.71875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='cf02e886-39c1-499e-a942-733b8f141a75' overflow='visible'>\n<path d='M 1.0625 -1.109375 C 1.0625 -0.910156 1.132812 -0.75 1.28125 -0.625 C 1.425781 -0.507812 1.601562 -0.453125 1.8125 -0.453125 C 2.050781 -0.453125 2.289062 -0.507812 2.53125 -0.625 C 2.914062 -0.820312 3.109375 -1.132812 3.109375 -1.5625 L 3.109375 -2.125 C 3.023438 -2.070312 2.914062 -2.023438 2.78125 -1.984375 C 2.644531 -1.953125 2.507812 -1.929688 2.375 -1.921875 L 1.953125 -1.859375 C 1.703125 -1.828125 1.507812 -1.773438 1.375 -1.703125 C 1.164062 -1.578125 1.0625 -1.378906 1.0625 -1.109375 Z M 2.765625 -2.53125 C 2.921875 -2.550781 3.023438 -2.617188 3.078125 -2.734375 C 3.109375 -2.796875 3.125 -2.890625 3.125 -3.015625 C 3.125 -3.253906 3.039062 -3.425781 2.875 -3.53125 C 2.707031 -3.632812 2.460938 -3.6875 2.140625 -3.6875 C 1.773438 -3.6875 1.515625 -3.585938 1.359375 -3.390625 C 1.273438 -3.285156 1.21875 -3.125 1.1875 -2.90625 L 0.53125 -2.90625 C 0.550781 -3.425781 0.71875 -3.785156 1.03125 -3.984375 C 1.351562 -4.179688 1.726562 -4.28125 2.15625 -4.28125 C 2.644531 -4.28125 3.046875 -4.1875 3.359375 -4 C 3.660156 -3.8125 3.8125 -3.519531 3.8125 -3.125 L 3.8125 -0.71875 C 3.8125 -0.644531 3.828125 -0.585938 3.859375 -0.546875 C 3.890625 -0.503906 3.953125 -0.484375 4.046875 -0.484375 C 4.078125 -0.484375 4.113281 -0.484375 4.15625 -0.484375 C 4.195312 -0.484375 4.238281 -0.488281 4.28125 -0.5 L 4.28125 0.015625 C 4.164062 0.046875 4.078125 0.0664062 4.015625 0.078125 C 3.960938 0.0859375 3.890625 0.09375 3.796875 0.09375 C 3.554688 0.09375 3.378906 0.00390625 3.265625 -0.171875 C 3.210938 -0.265625 3.171875 -0.394531 3.140625 -0.5625 C 3.003906 -0.375 2.800781 -0.207031 2.53125 -0.0625 C 2.257812 0.0703125 1.960938 0.140625 1.640625 0.140625 C 1.253906 0.140625 0.9375 0.0234375 0.6875 -0.203125 C 0.445312 -0.441406 0.328125 -0.738281 0.328125 -1.09375 C 0.328125 -1.488281 0.445312 -1.789062 0.6875 -2 C 0.925781 -2.21875 1.242188 -2.347656 1.640625 -2.390625 Z M 2.171875 -4.28125 Z M 2.171875 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='697b2392-7ba9-4bae-b0d6-558241e34ac0' overflow='visible'>\n<path d='M 0.859375 -4.1875 L 1.984375 -0.78125 L 3.140625 -4.1875 L 3.921875 -4.1875 L 2.34375 0 L 1.59375 0 L 0.046875 -4.1875 Z M 0.859375 -4.1875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='9ab7684f-97e9-41d3-8629-23259e00ef9c' overflow='visible'>\n<path d='M 0.515625 -4.171875 L 1.234375 -4.171875 L 1.234375 0 L 0.515625 0 Z M 0.515625 -5.734375 L 1.234375 -5.734375 L 1.234375 -4.9375 L 0.515625 -4.9375 Z M 0.515625 -5.734375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='d431d41e-7dbd-41b6-9281-c859236d2e74' overflow='visible'>\n<path d='M 0.53125 -5.734375 L 1.234375 -5.734375 L 1.234375 0 L 0.53125 0 Z M 0.53125 -5.734375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='1c1596df-497c-4b93-bc97-54230f9b0ab0' overflow='visible'>\n<path d='M 0.46875 -5.765625 L 1.140625 -5.765625 L 1.140625 -3.671875 C 1.296875 -3.878906 1.476562 -4.035156 1.6875 -4.140625 C 1.90625 -4.242188 2.140625 -4.296875 2.390625 -4.296875 C 2.910156 -4.296875 3.332031 -4.113281 3.65625 -3.75 C 3.976562 -3.394531 4.140625 -2.867188 4.140625 -2.171875 C 4.140625 -1.515625 3.976562 -0.960938 3.65625 -0.515625 C 3.34375 -0.078125 2.898438 0.140625 2.328125 0.140625 C 2.015625 0.140625 1.75 0.0625 1.53125 -0.09375 C 1.394531 -0.1875 1.253906 -0.332031 1.109375 -0.53125 L 1.109375 0 L 0.46875 0 Z M 2.296875 -0.484375 C 2.671875 -0.484375 2.953125 -0.632812 3.140625 -0.9375 C 3.328125 -1.238281 3.421875 -1.640625 3.421875 -2.140625 C 3.421875 -2.578125 3.328125 -2.941406 3.140625 -3.234375 C 2.953125 -3.523438 2.675781 -3.671875 2.3125 -3.671875 C 1.988281 -3.671875 1.707031 -3.550781 1.46875 -3.3125 C 1.226562 -3.070312 1.109375 -2.679688 1.109375 -2.140625 C 1.109375 -1.742188 1.15625 -1.425781 1.25 -1.1875 C 1.4375 -0.71875 1.785156 -0.484375 2.296875 -0.484375 Z M 2.296875 -0.484375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='737da28f-6997-4e0b-b914-592bf03a4bdf' overflow='visible'>\n<path d='M 2.265625 -4.28125 C 2.554688 -4.28125 2.84375 -4.207031 3.125 -4.0625 C 3.40625 -3.925781 3.617188 -3.75 3.765625 -3.53125 C 3.898438 -3.320312 3.992188 -3.078125 4.046875 -2.796875 C 4.085938 -2.597656 4.109375 -2.289062 4.109375 -1.875 L 1.03125 -1.875 C 1.050781 -1.445312 1.15625 -1.101562 1.34375 -0.84375 C 1.53125 -0.59375 1.816406 -0.46875 2.203125 -0.46875 C 2.578125 -0.46875 2.875 -0.585938 3.09375 -0.828125 C 3.21875 -0.960938 3.304688 -1.125 3.359375 -1.3125 L 4.046875 -1.3125 C 4.023438 -1.15625 3.960938 -0.984375 3.859375 -0.796875 C 3.753906 -0.609375 3.640625 -0.457031 3.515625 -0.34375 C 3.304688 -0.132812 3.039062 0.0078125 2.71875 0.09375 C 2.550781 0.132812 2.363281 0.15625 2.15625 0.15625 C 1.632812 0.15625 1.191406 -0.03125 0.828125 -0.40625 C 0.460938 -0.789062 0.28125 -1.328125 0.28125 -2.015625 C 0.28125 -2.679688 0.460938 -3.222656 0.828125 -3.640625 C 1.191406 -4.066406 1.671875 -4.28125 2.265625 -4.28125 Z M 3.375 -2.421875 C 3.351562 -2.734375 3.289062 -2.976562 3.1875 -3.15625 C 2.988281 -3.5 2.664062 -3.671875 2.21875 -3.671875 C 1.894531 -3.671875 1.625 -3.550781 1.40625 -3.3125 C 1.1875 -3.082031 1.070312 -2.785156 1.0625 -2.421875 Z M 2.1875 -4.28125 Z M 2.1875 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='13624127-21b3-4c96-a874-86c356b316d6' overflow='visible'>\n<path d='M 2.765625 -3.3125 C 3.097656 -3.3125 3.351562 -3.359375 3.53125 -3.453125 C 3.820312 -3.597656 3.96875 -3.851562 3.96875 -4.21875 C 3.96875 -4.59375 3.8125 -4.847656 3.5 -4.984375 C 3.332031 -5.054688 3.078125 -5.09375 2.734375 -5.09375 L 1.359375 -5.09375 L 1.359375 -3.3125 Z M 3.03125 -0.671875 C 3.507812 -0.671875 3.847656 -0.804688 4.046875 -1.078125 C 4.171875 -1.253906 4.234375 -1.46875 4.234375 -1.71875 C 4.234375 -2.132812 4.050781 -2.414062 3.6875 -2.5625 C 3.488281 -2.644531 3.222656 -2.6875 2.890625 -2.6875 L 1.359375 -2.6875 L 1.359375 -0.671875 Z M 0.59375 -5.734375 L 3.0625 -5.734375 C 3.726562 -5.734375 4.203125 -5.535156 4.484375 -5.140625 C 4.648438 -4.898438 4.734375 -4.625 4.734375 -4.3125 C 4.734375 -3.957031 4.632812 -3.660156 4.4375 -3.421875 C 4.320312 -3.304688 4.164062 -3.195312 3.96875 -3.09375 C 4.257812 -2.976562 4.476562 -2.851562 4.625 -2.71875 C 4.882812 -2.46875 5.015625 -2.117188 5.015625 -1.671875 C 5.015625 -1.304688 4.898438 -0.972656 4.671875 -0.671875 C 4.328125 -0.222656 3.773438 0 3.015625 0 L 0.59375 0 Z M 0.59375 -5.734375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='689c0526-fc39-4e82-a3fc-0242eb89922a' overflow='visible'>\n<path d='M 0.5 -5.734375 L 1.171875 -5.734375 L 1.171875 -2.40625 L 2.984375 -4.1875 L 3.875 -4.1875 L 2.28125 -2.625 L 3.96875 0 L 3.078125 0 L 1.765625 -2.109375 L 1.171875 -1.578125 L 1.171875 0 L 0.5 0 Z M 0.5 -5.734375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='247980f7-11e7-4929-ac75-228d3e79ddd0' overflow='visible'>\n<path d='M 0.9375 -1.3125 C 0.957031 -1.082031 1.015625 -0.90625 1.109375 -0.78125 C 1.285156 -0.550781 1.59375 -0.4375 2.03125 -0.4375 C 2.289062 -0.4375 2.519531 -0.492188 2.71875 -0.609375 C 2.914062 -0.722656 3.015625 -0.894531 3.015625 -1.125 C 3.015625 -1.3125 2.9375 -1.453125 2.78125 -1.546875 C 2.675781 -1.597656 2.472656 -1.660156 2.171875 -1.734375 L 1.625 -1.875 C 1.257812 -1.96875 0.992188 -2.066406 0.828125 -2.171875 C 0.523438 -2.367188 0.375 -2.632812 0.375 -2.96875 C 0.375 -3.363281 0.515625 -3.679688 0.796875 -3.921875 C 1.085938 -4.171875 1.472656 -4.296875 1.953125 -4.296875 C 2.585938 -4.296875 3.039062 -4.109375 3.3125 -3.734375 C 3.488281 -3.503906 3.578125 -3.253906 3.578125 -2.984375 L 2.90625 -2.984375 C 2.894531 -3.140625 2.835938 -3.285156 2.734375 -3.421875 C 2.578125 -3.597656 2.300781 -3.6875 1.90625 -3.6875 C 1.632812 -3.6875 1.429688 -3.632812 1.296875 -3.53125 C 1.160156 -3.4375 1.09375 -3.304688 1.09375 -3.140625 C 1.09375 -2.953125 1.179688 -2.804688 1.359375 -2.703125 C 1.460938 -2.640625 1.617188 -2.582031 1.828125 -2.53125 L 2.28125 -2.421875 C 2.789062 -2.296875 3.128906 -2.175781 3.296875 -2.0625 C 3.578125 -1.882812 3.71875 -1.601562 3.71875 -1.21875 C 3.71875 -0.84375 3.570312 -0.519531 3.28125 -0.25 C 3 0.0195312 2.570312 0.15625 2 0.15625 C 1.375 0.15625 0.929688 0.015625 0.671875 -0.265625 C 0.410156 -0.546875 0.273438 -0.894531 0.265625 -1.3125 Z M 1.96875 -4.28125 Z M 1.96875 -4.28125 ' style='stroke:none;'/>\n</symbol>\n</g>\n<clipPath id='c7ed374c-b5a6-419e-8c42-db7a19c70e2d'>\n<path d='M 32.855469 22.472656 L 643 22.472656 L 643 332 L 32.855469 332 Z M 32.855469 22.472656 '/>\n</clipPath>\n<clipPath id='887d787c-76f4-4dfd-9bc4-7b499fe3c7db'>\n<path d='M 32.855469 317 L 643 317 L 643 318 L 32.855469 318 Z M 32.855469 317 '/>\n</clipPath>\n<clipPath id='f98e4922-2e04-44fb-9367-886c10095cd6'>\n<path d='M 32.855469 223 L 643 223 L 643 224 L 32.855469 224 Z M 32.855469 223 '/>\n</clipPath>\n<clipPath id='28e43d9c-11ae-4d30-95ee-7f041fa33bcc'>\n<path d='M 32.855469 129 L 643 129 L 643 131 L 32.855469 131 Z M 32.855469 129 '/>\n</clipPath>\n<clipPath id='43ffbca3-81ec-4557-ad11-7bcefb0f15a5'>\n<path d='M 32.855469 36 L 643 36 L 643 37 L 32.855469 37 Z M 32.855469 36 '/>\n</clipPath>\n<clipPath id='f752d5fc-310b-4d75-91cf-a671c192ec5b'>\n<path d='M 106 22.472656 L 107 22.472656 L 107 332 L 106 332 Z M 106 22.472656 '/>\n</clipPath>\n<clipPath id='55191c71-5e17-4fb1-8a00-7da922b97507'>\n<path d='M 227 22.472656 L 229 22.472656 L 229 332 L 227 332 Z M 227 22.472656 '/>\n</clipPath>\n<clipPath id='76917aeb-68d4-4d30-881b-9b67ffa74569'>\n<path d='M 349 22.472656 L 351 22.472656 L 351 332 L 349 332 Z M 349 22.472656 '/>\n</clipPath>\n<clipPath id='e0febc37-6232-4ac9-aa70-fec12ab197c0'>\n<path d='M 470 22.472656 L 472 22.472656 L 472 332 L 470 332 Z M 470 22.472656 '/>\n</clipPath>\n<clipPath id='f057888b-a6cc-4b64-a5cf-84eae9065699'>\n<path d='M 592 22.472656 L 594 22.472656 L 594 332 L 592 332 Z M 592 22.472656 '/>\n</clipPath>\n<clipPath id='f0ea71be-ad03-48f6-a1eb-a9d6b0e358e6'>\n<path d='M 295 22.472656 L 297 22.472656 L 297 332 L 295 332 Z M 295 22.472656 '/>\n</clipPath>\n<clipPath id='d3fc2a3c-28fd-494a-a54b-90dbd8e4e7ad'>\n<path d='M 32.855469 316 L 643 316 L 643 318 L 32.855469 318 Z M 32.855469 316 '/>\n</clipPath>\n<clipPath id='ec9e715b-6c01-476b-a8c0-aa3b1ae6568b'>\n<path d='M 32.855469 22.472656 L 643 22.472656 L 643 332 L 32.855469 332 Z M 32.855469 22.472656 '/>\n</clipPath>\n</defs>\n<g id='31563d90-4de3-4f8d-a305-ff55da91246a'>\n<rect style='fill:rgb(100%,100%,100%);fill-opacity:1;stroke:none;' height='360' width='648' y='0' x='0'/>\n<rect style='fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke:none;' height='360' width='648' y='0' x='0'/>\n<path d='M 0 360 L 648 360 L 648 0 L 0 0 Z M 0 360 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g clip-rule='nonzero' clip-path='url(#c7ed374c-b5a6-419e-8c42-db7a19c70e2d)'>\n<path d='M 32.855469 331.332031 L 642.328125 331.332031 L 642.328125 22.472656 L 32.855469 22.472656 Z M 32.855469 331.332031 ' style='fill-rule:nonzero;fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#887d787c-76f4-4dfd-9bc4-7b499fe3c7db)'>\n<path d='M 32.855469 317.292969 L 642.332031 317.292969 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#f98e4922-2e04-44fb-9367-886c10095cd6)'>\n<path d='M 32.855469 223.699219 L 642.332031 223.699219 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#28e43d9c-11ae-4d30-95ee-7f041fa33bcc)'>\n<path d='M 32.855469 130.105469 L 642.332031 130.105469 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#43ffbca3-81ec-4557-ad11-7bcefb0f15a5)'>\n<path d='M 32.855469 36.511719 L 642.332031 36.511719 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#f752d5fc-310b-4d75-91cf-a671c192ec5b)'>\n<path d='M 106.730469 331.332031 L 106.730469 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#55191c71-5e17-4fb1-8a00-7da922b97507)'>\n<path d='M 228.238281 331.332031 L 228.238281 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#76917aeb-68d4-4d30-881b-9b67ffa74569)'>\n<path d='M 349.746094 331.332031 L 349.746094 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#e0febc37-6232-4ac9-aa70-fec12ab197c0)'>\n<path d='M 471.25 331.332031 L 471.25 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#f057888b-a6cc-4b64-a5cf-84eae9065699)'>\n<path d='M 592.757812 331.332031 L 592.757812 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<path d='M 106.730469 317.292969 L 152.902344 317.292969 L 152.902344 298.574219 L 106.730469 298.574219 Z M 106.730469 317.292969 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.74902;'/>\n<path d='M 152.90625 317.292969 L 199.078125 317.292969 L 199.078125 148.824219 L 152.90625 148.824219 Z M 152.90625 317.292969 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.74902;'/>\n<path d='M 199.078125 317.292969 L 245.25 317.292969 L 245.25 36.511719 L 199.078125 36.511719 Z M 199.078125 317.292969 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.74902;'/>\n<path d='M 245.25 317.292969 L 291.421875 317.292969 L 291.421875 111.386719 L 245.25 111.386719 Z M 245.25 317.292969 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.74902;'/>\n<path d='M 291.421875 317.292969 L 337.59375 317.292969 L 337.59375 55.230469 L 291.421875 55.230469 Z M 291.421875 317.292969 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.74902;'/>\n<path d='M 337.59375 317.292969 L 383.765625 317.292969 L 383.765625 148.824219 L 337.59375 148.824219 Z M 337.59375 317.292969 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.74902;'/>\n<path d='M 383.765625 317.292969 L 429.9375 317.292969 L 429.9375 242.417969 L 383.765625 242.417969 Z M 383.765625 317.292969 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.74902;'/>\n<path d='M 429.9375 317.292969 L 476.109375 317.292969 L 476.109375 279.855469 L 429.9375 279.855469 Z M 429.9375 317.292969 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.74902;'/>\n<path d='M 476.109375 317.292969 L 522.28125 317.292969 L 522.28125 279.855469 L 476.109375 279.855469 Z M 476.109375 317.292969 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.74902;'/>\n<path d='M 522.28125 317.292969 L 568.453125 317.292969 L 568.453125 279.855469 L 522.28125 279.855469 Z M 522.28125 317.292969 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.74902;'/>\n<path d='M 568.453125 317.292969 L 614.625 317.292969 L 614.625 298.574219 L 568.453125 298.574219 Z M 568.453125 317.292969 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.74902;'/>\n<g clip-rule='nonzero' clip-path='url(#f0ea71be-ad03-48f6-a1eb-a9d6b0e358e6)'>\n<path d='M 295.933594 331.332031 L 295.933594 22.472656 ' style='fill:none;stroke-width:1.066978;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:1;stroke-dasharray:4.267913,4.267913;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#d3fc2a3c-28fd-494a-a54b-90dbd8e4e7ad)'>\n<path d='M 32.855469 317.292969 L 642.332031 317.292969 ' style='fill:none;stroke-width:0.853583;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#ec9e715b-6c01-476b-a8c0-aa3b1ae6568b)'>\n<path d='M 32.855469 331.332031 L 642.328125 331.332031 L 642.328125 22.472656 L 32.855469 22.472656 Z M 32.855469 331.332031 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='319.800781' x='24.925781' xlink:href='#dcad652e-1793-4162-a8f7-cc7a4e8bab1a'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='226.207031' x='24.925781' xlink:href='#6f8e18d4-863a-4651-bc7c-cdc4f2010343'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='132.613281' x='21.035156' xlink:href='#2f89e33b-46b8-4b97-acde-3f9bec4e57e2'/>\n<use y='132.613281' x='24.928223' xlink:href='#dcad652e-1793-4162-a8f7-cc7a4e8bab1a'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='39.019531' x='21.035156' xlink:href='#2f89e33b-46b8-4b97-acde-3f9bec4e57e2'/>\n<use y='39.019531' x='24.928223' xlink:href='#6f8e18d4-863a-4651-bc7c-cdc4f2010343'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='104.785156' xlink:href='#dcad652e-1793-4162-a8f7-cc7a4e8bab1a'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='226.292969' xlink:href='#6f8e18d4-863a-4651-bc7c-cdc4f2010343'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='345.851562' xlink:href='#2f89e33b-46b8-4b97-acde-3f9bec4e57e2'/>\n<use y='340.382812' x='349.744629' xlink:href='#dcad652e-1793-4162-a8f7-cc7a4e8bab1a'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='467.355469' xlink:href='#2f89e33b-46b8-4b97-acde-3f9bec4e57e2'/>\n<use y='340.382812' x='471.248535' xlink:href='#6f8e18d4-863a-4651-bc7c-cdc4f2010343'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='588.863281' xlink:href='#f7a36802-bdfb-46f0-8c5c-feb1bcd4cc13'/>\n<use y='340.382812' x='592.756348' xlink:href='#dcad652e-1793-4162-a8f7-cc7a4e8bab1a'/>\n</g>\n<g style='fill:rgb(32.156863%,32.156863%,32.156863%);fill-opacity:1;'>\n<use y='349.703125' x='312.023438' xlink:href='#cf02e886-39c1-499e-a942-733b8f141a75'/>\n<use y='349.703125' x='316.472656' xlink:href='#697b2392-7ba9-4bae-b0d6-558241e34ac0'/>\n<use y='349.703125' x='320.472656' xlink:href='#cf02e886-39c1-499e-a942-733b8f141a75'/>\n<use y='349.703125' x='324.921875' xlink:href='#9ab7684f-97e9-41d3-8629-23259e00ef9c'/>\n<use y='349.703125' x='326.699219' xlink:href='#d431d41e-7dbd-41b6-9281-c859236d2e74'/>\n<use y='349.703125' x='328.476562' xlink:href='#cf02e886-39c1-499e-a942-733b8f141a75'/>\n<use y='349.703125' x='332.925781' xlink:href='#1c1596df-497c-4b93-bc97-54230f9b0ab0'/>\n<use y='349.703125' x='337.375' xlink:href='#d431d41e-7dbd-41b6-9281-c859236d2e74'/>\n<use y='349.703125' x='339.152344' xlink:href='#737da28f-6997-4e0b-b914-592bf03a4bdf'/>\n<use y='349.703125' x='343.601562' xlink:href='#13624127-21b3-4c96-a874-86c356b316d6'/>\n<use y='349.703125' x='348.9375' xlink:href='#9ab7684f-97e9-41d3-8629-23259e00ef9c'/>\n<use y='349.703125' x='350.714844' xlink:href='#689c0526-fc39-4e82-a3fc-0242eb89922a'/>\n<use y='349.703125' x='354.714844' xlink:href='#737da28f-6997-4e0b-b914-592bf03a4bdf'/>\n<use y='349.703125' x='359.164062' xlink:href='#247980f7-11e7-4929-ac75-228d3e79ddd0'/>\n</g>\n</g>\n</svg>\n","value":"#huri.plot.GGView{:plot-command [[:<- :g [:data.frame {:availableBikes [:c 8 7 10 11 8 0 5 8 10 10 6 5 6 4 5 13 8 9 9 5 5 13 7 6 11 2 13 5 7 10 5 6 10 3 3 3 3 12 4 15 18 6 11 10 3 9 9 8 3 17 6 4 5 8 7 4 4 18 16 9 9 14 19 3 8 9 3 4 6 5]}]] [[:library :ggplot2] [:library :scales] [:library :grid] [:library :RColorBrewer] [:library :ggrepel] [:<- :palette [:brewer.pal \"Greys\" {:n 9}]] {:color.background :palette[2]} {:color.grid.major :palette[3]} {:color.axis.text :palette[6]} {:color.axis.title :palette[7]} {:color.title :palette[9]}] [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:ggplot :g [:aes :availableBikes {}]] [:geom_histogram {:binwidth 19/10, :alpha 0.75, :fill \"#c0392b\"}]] [:geom_vline [:aes {:xintercept [:mean :availableBikes]}] {:linetype \"dashed\", :color \"#c0392b\", :size 0.5}]] [:geom_hline {:yintercept 0, :size 0.4, :colour \"black\"}]] [:scale_x_continuous {:labels :comma}]] [:scale_y_continuous {:labels :comma}]] [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:theme_bw {:base_size 9}] [:theme {:panel.background [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:plot.background [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:panel.border [:element_rect {:color :color.background}]}]] [:theme {:panel.grid.major [:element_line {:color :color.grid.major, :size 0.25}]}]] [:theme {:panel.grid.minor [:element_blank]}]] [:theme {:axis.ticks [:element_blank]}]] [:theme {:legend.background [:element_rect {:fill :color.background}]}]] [:theme {:legend.key [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:legend.text [:element_text {:size 7, :color :color.axis.title}]}]] [:theme {:legend.title [:element_blank]}]] [:theme {:plot.title [:element_text {:size 10, :color :color.title, :vjust 1.25}]}]] [:theme {:axis.text.x [:element_text {:size 7, :color :color.axis.text}]}]] [:theme {:axis.text.y [:element_text {:size 7, :color :color.axis.text}]}]] [:theme {:axis.title.x [:element_text {:size 8, :color :color.axis.title, :vjust 0}]}]] [:theme {:axis.title.y [:element_text {:size 8, :color :color.axis.title, :vjust 1.25}]}]] [:theme {:plot.margin [:unit [:c 0.35 0.2 0.3 0.35] \"cm\"]}]]] [:theme {:legend.position \"none\"}]] [:labs {:y \"\", :title \"\", :x \"availableBikes\"}]]], :options {:width 9, :height 5}}"}
;; <=

;; @@
(->> df
     (where {:city "San Francisco"})
     (bar-chart :stationName [:availableDocks :availableBikes] {:flip? true :height 12}))
;; @@
;; =>
;;; {"type":"html","content":"<?xml version='1.0' encoding='UTF-8'?>\n<svg version='1.1' viewBox='0 0 648 864' height='864pt' width='648pt' xmlns:xlink='http://www.w3.org/1999/xlink' xmlns='http://www.w3.org/2000/svg'>\n<defs>\n<g>\n<symbol id='10ada928-62a3-4347-8204-d0ef630b0d3d' overflow='visible'>\n<path d='M 0.21875 0 L 0.21875 -5.015625 L 4.21875 -5.015625 L 4.21875 0 Z M 3.578125 -0.625 L 3.578125 -4.390625 L 0.859375 -4.390625 L 0.859375 -0.625 Z M 3.578125 -0.625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='0373f860-79fb-49c3-aba0-0e9aac3a40fa' overflow='visible'>\n<path d='M 2.46875 -0.578125 C 2.695312 -0.578125 2.882812 -0.601562 3.03125 -0.65625 C 3.289062 -0.738281 3.503906 -0.910156 3.671875 -1.171875 C 3.804688 -1.367188 3.90625 -1.625 3.96875 -1.9375 C 4 -2.132812 4.015625 -2.3125 4.015625 -2.46875 C 4.015625 -3.09375 3.890625 -3.578125 3.640625 -3.921875 C 3.398438 -4.265625 3.003906 -4.4375 2.453125 -4.4375 L 1.25 -4.4375 L 1.25 -0.578125 Z M 0.5625 -5.015625 L 2.59375 -5.015625 C 3.289062 -5.015625 3.828125 -4.769531 4.203125 -4.28125 C 4.546875 -3.84375 4.71875 -3.28125 4.71875 -2.59375 C 4.71875 -2.050781 4.613281 -1.566406 4.40625 -1.140625 C 4.0625 -0.378906 3.457031 0 2.59375 0 L 0.5625 0 Z M 0.5625 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='bf66ab75-0709-4ec2-abbe-4a01f33a231b' overflow='visible'>\n<path d='M 0.921875 -0.96875 C 0.921875 -0.789062 0.984375 -0.648438 1.109375 -0.546875 C 1.242188 -0.453125 1.398438 -0.40625 1.578125 -0.40625 C 1.796875 -0.40625 2.003906 -0.453125 2.203125 -0.546875 C 2.546875 -0.710938 2.71875 -0.988281 2.71875 -1.375 L 2.71875 -1.859375 C 2.644531 -1.816406 2.546875 -1.78125 2.421875 -1.75 C 2.304688 -1.71875 2.191406 -1.691406 2.078125 -1.671875 L 1.703125 -1.625 C 1.484375 -1.59375 1.316406 -1.546875 1.203125 -1.484375 C 1.015625 -1.378906 0.921875 -1.207031 0.921875 -0.96875 Z M 2.40625 -2.21875 C 2.550781 -2.238281 2.648438 -2.296875 2.703125 -2.390625 C 2.722656 -2.453125 2.734375 -2.53125 2.734375 -2.625 C 2.734375 -2.84375 2.660156 -3 2.515625 -3.09375 C 2.367188 -3.1875 2.15625 -3.234375 1.875 -3.234375 C 1.550781 -3.234375 1.320312 -3.144531 1.1875 -2.96875 C 1.113281 -2.875 1.066406 -2.734375 1.046875 -2.546875 L 0.46875 -2.546875 C 0.476562 -2.992188 0.625 -3.304688 0.90625 -3.484375 C 1.1875 -3.660156 1.515625 -3.75 1.890625 -3.75 C 2.316406 -3.75 2.664062 -3.664062 2.9375 -3.5 C 3.207031 -3.332031 3.34375 -3.078125 3.34375 -2.734375 L 3.34375 -0.625 C 3.34375 -0.5625 3.351562 -0.507812 3.375 -0.46875 C 3.40625 -0.4375 3.460938 -0.421875 3.546875 -0.421875 C 3.566406 -0.421875 3.59375 -0.421875 3.625 -0.421875 C 3.664062 -0.421875 3.707031 -0.425781 3.75 -0.4375 L 3.75 0.015625 C 3.644531 0.046875 3.566406 0.0625 3.515625 0.0625 C 3.472656 0.0703125 3.40625 0.078125 3.3125 0.078125 C 3.101562 0.078125 2.953125 0.00390625 2.859375 -0.140625 C 2.804688 -0.222656 2.769531 -0.335938 2.75 -0.484375 C 2.625 -0.316406 2.441406 -0.171875 2.203125 -0.046875 C 1.972656 0.0664062 1.71875 0.125 1.4375 0.125 C 1.09375 0.125 0.8125 0.0195312 0.59375 -0.1875 C 0.382812 -0.394531 0.28125 -0.648438 0.28125 -0.953125 C 0.28125 -1.296875 0.382812 -1.5625 0.59375 -1.75 C 0.8125 -1.9375 1.09375 -2.050781 1.4375 -2.09375 Z M 1.90625 -3.75 Z M 1.90625 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='cab96616-0e1a-49dc-99de-457a3e10394b' overflow='visible'>\n<path d='M 0.75 -3.65625 L 1.734375 -0.6875 L 2.75 -3.65625 L 3.421875 -3.65625 L 2.046875 0 L 1.390625 0 L 0.03125 -3.65625 Z M 0.75 -3.65625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='02f0d219-92cd-46e0-960d-e11470244484' overflow='visible'>\n<path d='M 0.453125 -3.640625 L 1.078125 -3.640625 L 1.078125 0 L 0.453125 0 Z M 0.453125 -5.015625 L 1.078125 -5.015625 L 1.078125 -4.328125 L 0.453125 -4.328125 Z M 0.453125 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='0fbdc780-6d22-480c-b581-916d1f4f6afb' overflow='visible'>\n<path d='M 0.8125 -1.15625 C 0.832031 -0.945312 0.882812 -0.785156 0.96875 -0.671875 C 1.125 -0.472656 1.394531 -0.375 1.78125 -0.375 C 2.007812 -0.375 2.207031 -0.421875 2.375 -0.515625 C 2.550781 -0.617188 2.640625 -0.773438 2.640625 -0.984375 C 2.640625 -1.140625 2.570312 -1.257812 2.4375 -1.34375 C 2.34375 -1.394531 2.164062 -1.453125 1.90625 -1.515625 L 1.421875 -1.640625 C 1.109375 -1.722656 0.875 -1.8125 0.71875 -1.90625 C 0.457031 -2.070312 0.328125 -2.300781 0.328125 -2.59375 C 0.328125 -2.9375 0.453125 -3.210938 0.703125 -3.421875 C 0.953125 -3.640625 1.285156 -3.75 1.703125 -3.75 C 2.253906 -3.75 2.65625 -3.585938 2.90625 -3.265625 C 3.050781 -3.066406 3.125 -2.847656 3.125 -2.609375 L 2.546875 -2.609375 C 2.535156 -2.742188 2.484375 -2.867188 2.390625 -2.984375 C 2.253906 -3.148438 2.007812 -3.234375 1.65625 -3.234375 C 1.425781 -3.234375 1.25 -3.1875 1.125 -3.09375 C 1.007812 -3.007812 0.953125 -2.894531 0.953125 -2.75 C 0.953125 -2.582031 1.03125 -2.453125 1.1875 -2.359375 C 1.28125 -2.304688 1.414062 -2.257812 1.59375 -2.21875 L 2 -2.109375 C 2.4375 -2.003906 2.734375 -1.898438 2.890625 -1.796875 C 3.128906 -1.640625 3.25 -1.394531 3.25 -1.0625 C 3.25 -0.738281 3.125 -0.457031 2.875 -0.21875 C 2.625 0.0195312 2.25 0.140625 1.75 0.140625 C 1.207031 0.140625 0.820312 0.015625 0.59375 -0.234375 C 0.363281 -0.484375 0.238281 -0.789062 0.21875 -1.15625 Z M 1.71875 -3.75 Z M 1.71875 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='e452c12d-9540-4335-96a8-acd1ffe1ffd3' overflow='visible'>\n<path d='' style='stroke:none;'/>\n</symbol>\n<symbol id='23581864-4fe4-4ac1-9c9f-a4fd125eecc9' overflow='visible'>\n<path d='M 0.578125 -4.6875 L 1.203125 -4.6875 L 1.203125 -3.65625 L 1.78125 -3.65625 L 1.78125 -3.15625 L 1.203125 -3.15625 L 1.203125 -0.765625 C 1.203125 -0.640625 1.242188 -0.554688 1.328125 -0.515625 C 1.367188 -0.484375 1.445312 -0.46875 1.5625 -0.46875 C 1.59375 -0.46875 1.625 -0.46875 1.65625 -0.46875 C 1.695312 -0.476562 1.738281 -0.484375 1.78125 -0.484375 L 1.78125 0 C 1.707031 0.0195312 1.632812 0.0351562 1.5625 0.046875 C 1.488281 0.0546875 1.40625 0.0625 1.3125 0.0625 C 1.019531 0.0625 0.820312 -0.0078125 0.71875 -0.15625 C 0.625 -0.3125 0.578125 -0.503906 0.578125 -0.734375 L 0.578125 -3.15625 L 0.078125 -3.15625 L 0.078125 -3.65625 L 0.578125 -3.65625 Z M 0.578125 -4.6875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='d4e43642-a0de-4bdc-af18-ff27a755bb94' overflow='visible'>\n<path d='M 3 -1.5 C 3 -1.070312 2.9375 -0.742188 2.8125 -0.515625 C 2.582031 -0.078125 2.140625 0.140625 1.484375 0.140625 C 1.109375 0.140625 0.785156 0.0351562 0.515625 -0.171875 C 0.253906 -0.378906 0.125 -0.742188 0.125 -1.265625 L 0.125 -1.625 L 0.765625 -1.625 L 0.765625 -1.265625 C 0.765625 -0.984375 0.820312 -0.773438 0.9375 -0.640625 C 1.0625 -0.503906 1.253906 -0.4375 1.515625 -0.4375 C 1.867188 -0.4375 2.101562 -0.5625 2.21875 -0.8125 C 2.289062 -0.96875 2.328125 -1.253906 2.328125 -1.671875 L 2.328125 -5.015625 L 3 -5.015625 Z M 3 -1.5 ' style='stroke:none;'/>\n</symbol>\n<symbol id='bdd46fbd-d377-4bba-91c8-3814d2922bd5' overflow='visible'>\n<path d='M 1.859375 -3.765625 C 2.273438 -3.765625 2.609375 -3.664062 2.859375 -3.46875 C 3.117188 -3.269531 3.28125 -2.925781 3.34375 -2.4375 L 2.734375 -2.4375 C 2.703125 -2.65625 2.617188 -2.835938 2.484375 -2.984375 C 2.359375 -3.140625 2.148438 -3.21875 1.859375 -3.21875 C 1.472656 -3.21875 1.191406 -3.023438 1.015625 -2.640625 C 0.910156 -2.390625 0.859375 -2.082031 0.859375 -1.71875 C 0.859375 -1.351562 0.9375 -1.046875 1.09375 -0.796875 C 1.25 -0.546875 1.492188 -0.421875 1.828125 -0.421875 C 2.078125 -0.421875 2.273438 -0.5 2.421875 -0.65625 C 2.578125 -0.8125 2.679688 -1.023438 2.734375 -1.296875 L 3.34375 -1.296875 C 3.269531 -0.804688 3.09375 -0.445312 2.8125 -0.21875 C 2.539062 0 2.195312 0.109375 1.78125 0.109375 C 1.300781 0.109375 0.914062 -0.0625 0.625 -0.40625 C 0.34375 -0.757812 0.203125 -1.203125 0.203125 -1.734375 C 0.203125 -2.367188 0.359375 -2.863281 0.671875 -3.21875 C 0.984375 -3.582031 1.378906 -3.765625 1.859375 -3.765625 Z M 1.765625 -3.75 Z M 1.765625 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='091b1d69-7c75-46b6-825f-da4b0b842949' overflow='visible'>\n<path d='M 0.4375 -5.015625 L 1.03125 -5.015625 L 1.03125 -2.109375 L 2.609375 -3.65625 L 3.390625 -3.65625 L 2 -2.296875 L 3.46875 0 L 2.6875 0 L 1.546875 -1.84375 L 1.03125 -1.375 L 1.03125 0 L 0.4375 0 Z M 0.4375 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='4809c55e-e7db-4fc1-aecb-b5fec5f904b7' overflow='visible'>\n<path d='M 1.90625 -0.390625 C 2.3125 -0.390625 2.585938 -0.546875 2.734375 -0.859375 C 2.890625 -1.171875 2.96875 -1.515625 2.96875 -1.890625 C 2.96875 -2.234375 2.914062 -2.507812 2.8125 -2.71875 C 2.632812 -3.0625 2.332031 -3.234375 1.90625 -3.234375 C 1.53125 -3.234375 1.253906 -3.085938 1.078125 -2.796875 C 0.910156 -2.515625 0.828125 -2.171875 0.828125 -1.765625 C 0.828125 -1.367188 0.910156 -1.039062 1.078125 -0.78125 C 1.253906 -0.519531 1.53125 -0.390625 1.90625 -0.390625 Z M 1.921875 -3.765625 C 2.398438 -3.765625 2.800781 -3.609375 3.125 -3.296875 C 3.445312 -2.984375 3.609375 -2.519531 3.609375 -1.90625 C 3.609375 -1.3125 3.460938 -0.820312 3.171875 -0.4375 C 2.890625 -0.0507812 2.445312 0.140625 1.84375 0.140625 C 1.34375 0.140625 0.941406 -0.03125 0.640625 -0.375 C 0.347656 -0.71875 0.203125 -1.175781 0.203125 -1.75 C 0.203125 -2.363281 0.359375 -2.851562 0.671875 -3.21875 C 0.984375 -3.582031 1.398438 -3.765625 1.921875 -3.765625 Z M 1.90625 -3.75 Z M 1.90625 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='6ac45e00-029c-4471-9991-7ff5ffbeaceb' overflow='visible'>\n<path d='M 0.453125 -3.65625 L 1.03125 -3.65625 L 1.03125 -3.140625 C 1.207031 -3.359375 1.390625 -3.515625 1.578125 -3.609375 C 1.773438 -3.703125 1.992188 -3.75 2.234375 -3.75 C 2.753906 -3.75 3.101562 -3.566406 3.28125 -3.203125 C 3.382812 -3.003906 3.4375 -2.71875 3.4375 -2.34375 L 3.4375 0 L 2.8125 0 L 2.8125 -2.3125 C 2.8125 -2.53125 2.773438 -2.707031 2.703125 -2.84375 C 2.597656 -3.070312 2.398438 -3.1875 2.109375 -3.1875 C 1.960938 -3.1875 1.84375 -3.171875 1.75 -3.140625 C 1.582031 -3.085938 1.4375 -2.988281 1.3125 -2.84375 C 1.207031 -2.71875 1.140625 -2.59375 1.109375 -2.46875 C 1.078125 -2.34375 1.0625 -2.160156 1.0625 -1.921875 L 1.0625 0 L 0.453125 0 Z M 1.890625 -3.75 Z M 1.890625 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='6a3a0a32-3bcc-44d9-8288-0e6d0c801f40' overflow='visible'>\n<path d='M 0.859375 -5.015625 L 1.8125 -0.9375 L 2.953125 -5.015625 L 3.6875 -5.015625 L 4.8125 -0.9375 L 5.765625 -5.015625 L 6.515625 -5.015625 L 5.1875 0 L 4.484375 0 L 3.3125 -4.15625 L 2.15625 0 L 1.4375 0 L 0.125 -5.015625 Z M 0.859375 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='1da6ad6a-cbc9-46ad-bddc-b5660b76e4c7' overflow='visible'>\n<path d='M 0.453125 -5.03125 L 1.0625 -5.03125 L 1.0625 -3.171875 C 1.207031 -3.347656 1.335938 -3.476562 1.453125 -3.5625 C 1.648438 -3.6875 1.898438 -3.75 2.203125 -3.75 C 2.734375 -3.75 3.09375 -3.5625 3.28125 -3.1875 C 3.382812 -2.988281 3.4375 -2.707031 3.4375 -2.34375 L 3.4375 0 L 2.796875 0 L 2.796875 -2.3125 C 2.796875 -2.570312 2.765625 -2.769531 2.703125 -2.90625 C 2.585938 -3.101562 2.378906 -3.203125 2.078125 -3.203125 C 1.816406 -3.203125 1.582031 -3.113281 1.375 -2.9375 C 1.164062 -2.757812 1.0625 -2.425781 1.0625 -1.9375 L 1.0625 0 L 0.453125 0 Z M 0.453125 -5.03125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='3093d11f-7144-4487-8aed-c2dd55c92efa' overflow='visible'>\n<path d='M 1.75 -3.71875 C 2.03125 -3.71875 2.28125 -3.648438 2.5 -3.515625 C 2.613281 -3.429688 2.726562 -3.316406 2.84375 -3.171875 L 2.84375 -3.625 L 3.421875 -3.625 L 3.421875 -0.296875 C 3.421875 0.171875 3.351562 0.535156 3.21875 0.796875 C 2.957031 1.296875 2.472656 1.546875 1.765625 1.546875 C 1.367188 1.546875 1.035156 1.457031 0.765625 1.28125 C 0.503906 1.101562 0.359375 0.828125 0.328125 0.453125 L 0.953125 0.453125 C 0.972656 0.617188 1.03125 0.742188 1.125 0.828125 C 1.257812 0.972656 1.476562 1.046875 1.78125 1.046875 C 2.25 1.046875 2.554688 0.878906 2.703125 0.546875 C 2.796875 0.347656 2.835938 -0.00390625 2.828125 -0.515625 C 2.703125 -0.328125 2.550781 -0.1875 2.375 -0.09375 C 2.207031 0 1.984375 0.046875 1.703125 0.046875 C 1.296875 0.046875 0.941406 -0.09375 0.640625 -0.375 C 0.347656 -0.65625 0.203125 -1.125 0.203125 -1.78125 C 0.203125 -2.382812 0.351562 -2.859375 0.65625 -3.203125 C 0.957031 -3.546875 1.320312 -3.71875 1.75 -3.71875 Z M 2.84375 -1.84375 C 2.84375 -2.300781 2.75 -2.640625 2.5625 -2.859375 C 2.382812 -3.078125 2.148438 -3.1875 1.859375 -3.1875 C 1.421875 -3.1875 1.125 -2.984375 0.96875 -2.578125 C 0.882812 -2.359375 0.84375 -2.070312 0.84375 -1.71875 C 0.84375 -1.3125 0.925781 -1 1.09375 -0.78125 C 1.257812 -0.5625 1.484375 -0.453125 1.765625 -0.453125 C 2.210938 -0.453125 2.523438 -0.65625 2.703125 -1.0625 C 2.796875 -1.28125 2.84375 -1.539062 2.84375 -1.84375 Z M 1.8125 -3.75 Z M 1.8125 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='49d7c5ef-7a75-4d2b-8182-60e011b9efda' overflow='visible'>\n<path d='M 0.53125 -5.015625 L 1.203125 -5.015625 L 1.203125 -2.578125 L 3.640625 -5.015625 L 4.578125 -5.015625 L 2.5 -3 L 4.640625 0 L 3.75 0 L 2 -2.515625 L 1.203125 -1.75 L 1.203125 0 L 0.53125 0 Z M 0.53125 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='ad69ce23-6ab7-4c95-a6a3-a211d82c100e' overflow='visible'>\n<path d='M 1.96875 -3.75 C 2.226562 -3.75 2.476562 -3.6875 2.71875 -3.5625 C 2.96875 -3.4375 3.15625 -3.28125 3.28125 -3.09375 C 3.40625 -2.90625 3.488281 -2.6875 3.53125 -2.4375 C 3.570312 -2.269531 3.59375 -2.003906 3.59375 -1.640625 L 0.90625 -1.640625 C 0.914062 -1.265625 1.003906 -0.960938 1.171875 -0.734375 C 1.335938 -0.515625 1.59375 -0.40625 1.9375 -0.40625 C 2.25 -0.40625 2.503906 -0.507812 2.703125 -0.71875 C 2.804688 -0.84375 2.882812 -0.988281 2.9375 -1.15625 L 3.53125 -1.15625 C 3.519531 -1.019531 3.46875 -0.867188 3.375 -0.703125 C 3.289062 -0.535156 3.191406 -0.398438 3.078125 -0.296875 C 2.890625 -0.109375 2.65625 0.015625 2.375 0.078125 C 2.226562 0.109375 2.0625 0.125 1.875 0.125 C 1.425781 0.125 1.039062 -0.0351562 0.71875 -0.359375 C 0.40625 -0.691406 0.25 -1.160156 0.25 -1.765625 C 0.25 -2.347656 0.40625 -2.820312 0.71875 -3.1875 C 1.039062 -3.5625 1.457031 -3.75 1.96875 -3.75 Z M 2.953125 -2.125 C 2.929688 -2.382812 2.875 -2.597656 2.78125 -2.765625 C 2.613281 -3.054688 2.332031 -3.203125 1.9375 -3.203125 C 1.65625 -3.203125 1.414062 -3.097656 1.21875 -2.890625 C 1.03125 -2.691406 0.929688 -2.4375 0.921875 -2.125 Z M 1.921875 -3.75 Z M 1.921875 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='f5d3bfca-3a9d-488c-8604-94d3f39d80e9' overflow='visible'>\n<path d='M 0.46875 -3.65625 L 1.046875 -3.65625 L 1.046875 -3.03125 C 1.097656 -3.15625 1.21875 -3.304688 1.40625 -3.484375 C 1.59375 -3.660156 1.804688 -3.75 2.046875 -3.75 C 2.054688 -3.75 2.078125 -3.742188 2.109375 -3.734375 C 2.140625 -3.734375 2.1875 -3.726562 2.25 -3.71875 L 2.25 -3.078125 C 2.207031 -3.085938 2.171875 -3.09375 2.140625 -3.09375 C 2.117188 -3.09375 2.085938 -3.09375 2.046875 -3.09375 C 1.734375 -3.09375 1.492188 -2.992188 1.328125 -2.796875 C 1.160156 -2.597656 1.078125 -2.367188 1.078125 -2.109375 L 1.078125 0 L 0.46875 0 Z M 0.46875 -3.65625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='77d9585b-31e4-4e0c-8c9c-a8b207187d3a' overflow='visible'>\n<path d='M 2.734375 -3.65625 L 3.421875 -3.65625 C 3.328125 -3.425781 3.132812 -2.890625 2.84375 -2.046875 C 2.613281 -1.421875 2.425781 -0.910156 2.28125 -0.515625 C 1.925781 0.421875 1.675781 0.988281 1.53125 1.1875 C 1.382812 1.394531 1.132812 1.5 0.78125 1.5 C 0.6875 1.5 0.613281 1.492188 0.5625 1.484375 C 0.519531 1.484375 0.46875 1.472656 0.40625 1.453125 L 0.40625 0.890625 C 0.507812 0.921875 0.582031 0.9375 0.625 0.9375 C 0.675781 0.945312 0.722656 0.953125 0.765625 0.953125 C 0.878906 0.953125 0.960938 0.929688 1.015625 0.890625 C 1.066406 0.859375 1.113281 0.816406 1.15625 0.765625 C 1.164062 0.742188 1.203125 0.648438 1.265625 0.484375 C 1.335938 0.316406 1.390625 0.191406 1.421875 0.109375 L 0.078125 -3.65625 L 0.765625 -3.65625 L 1.75 -0.6875 Z M 1.75 -3.75 Z M 1.75 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='f5647a9d-2871-49b0-9631-fc19a67066b3' overflow='visible'>\n<path d='M 0.21875 0 C 0.238281 -0.414062 0.320312 -0.78125 0.46875 -1.09375 C 0.625 -1.40625 0.925781 -1.691406 1.375 -1.953125 L 2.03125 -2.328125 C 2.320312 -2.503906 2.523438 -2.648438 2.640625 -2.765625 C 2.828125 -2.953125 2.921875 -3.171875 2.921875 -3.421875 C 2.921875 -3.703125 2.832031 -3.925781 2.65625 -4.09375 C 2.488281 -4.257812 2.265625 -4.34375 1.984375 -4.34375 C 1.554688 -4.34375 1.265625 -4.1875 1.109375 -3.875 C 1.023438 -3.695312 0.976562 -3.457031 0.96875 -3.15625 L 0.34375 -3.15625 C 0.34375 -3.582031 0.421875 -3.929688 0.578125 -4.203125 C 0.835938 -4.671875 1.304688 -4.90625 1.984375 -4.90625 C 2.546875 -4.90625 2.957031 -4.753906 3.21875 -4.453125 C 3.476562 -4.148438 3.609375 -3.816406 3.609375 -3.453125 C 3.609375 -3.054688 3.46875 -2.71875 3.1875 -2.4375 C 3.03125 -2.28125 2.742188 -2.085938 2.328125 -1.859375 L 1.859375 -1.59375 C 1.640625 -1.46875 1.46875 -1.347656 1.34375 -1.234375 C 1.113281 -1.035156 0.96875 -0.816406 0.90625 -0.578125 L 3.578125 -0.578125 L 3.578125 0 Z M 0.21875 0 ' style='stroke:none;'/>\n</symbol>\n<symbol id='92bb5c1b-8d4d-40b4-b1ad-d33df39f3934' overflow='visible'>\n<path d='M 0.84375 -1.78125 C 0.84375 -1.394531 0.925781 -1.066406 1.09375 -0.796875 C 1.257812 -0.535156 1.523438 -0.40625 1.890625 -0.40625 C 2.171875 -0.40625 2.40625 -0.523438 2.59375 -0.765625 C 2.78125 -1.015625 2.875 -1.367188 2.875 -1.828125 C 2.875 -2.285156 2.773438 -2.625 2.578125 -2.84375 C 2.390625 -3.070312 2.160156 -3.1875 1.890625 -3.1875 C 1.578125 -3.1875 1.320312 -3.066406 1.125 -2.828125 C 0.9375 -2.597656 0.84375 -2.25 0.84375 -1.78125 Z M 1.765625 -3.71875 C 2.046875 -3.71875 2.28125 -3.660156 2.46875 -3.546875 C 2.582031 -3.484375 2.707031 -3.363281 2.84375 -3.1875 L 2.84375 -5.03125 L 3.4375 -5.03125 L 3.4375 0 L 2.875 0 L 2.875 -0.515625 C 2.738281 -0.285156 2.570312 -0.117188 2.375 -0.015625 C 2.175781 0.078125 1.953125 0.125 1.703125 0.125 C 1.296875 0.125 0.941406 -0.046875 0.640625 -0.390625 C 0.335938 -0.734375 0.1875 -1.1875 0.1875 -1.75 C 0.1875 -2.28125 0.320312 -2.738281 0.59375 -3.125 C 0.875 -3.519531 1.265625 -3.71875 1.765625 -3.71875 Z M 1.765625 -3.71875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='1fdff9cd-6269-45e2-8bed-b34b2792b662' overflow='visible'>\n<path d='M 0.984375 -1.625 C 0.992188 -1.332031 1.054688 -1.097656 1.171875 -0.921875 C 1.398438 -0.585938 1.800781 -0.421875 2.375 -0.421875 C 2.625 -0.421875 2.851562 -0.457031 3.0625 -0.53125 C 3.46875 -0.675781 3.671875 -0.929688 3.671875 -1.296875 C 3.671875 -1.566406 3.585938 -1.757812 3.421875 -1.875 C 3.242188 -1.988281 2.972656 -2.085938 2.609375 -2.171875 L 1.921875 -2.328125 C 1.484375 -2.429688 1.171875 -2.539062 0.984375 -2.65625 C 0.660156 -2.875 0.5 -3.191406 0.5 -3.609375 C 0.5 -4.054688 0.65625 -4.421875 0.96875 -4.703125 C 1.28125 -4.992188 1.722656 -5.140625 2.296875 -5.140625 C 2.828125 -5.140625 3.273438 -5.015625 3.640625 -4.765625 C 4.003906 -4.515625 4.1875 -4.109375 4.1875 -3.546875 L 3.5625 -3.546875 C 3.519531 -3.816406 3.445312 -4.023438 3.34375 -4.171875 C 3.132812 -4.429688 2.78125 -4.5625 2.28125 -4.5625 C 1.875 -4.5625 1.582031 -4.472656 1.40625 -4.296875 C 1.238281 -4.128906 1.15625 -3.9375 1.15625 -3.71875 C 1.15625 -3.46875 1.253906 -3.285156 1.453125 -3.171875 C 1.585938 -3.097656 1.894531 -3.007812 2.375 -2.90625 L 3.078125 -2.75 C 3.410156 -2.664062 3.671875 -2.554688 3.859375 -2.421875 C 4.179688 -2.191406 4.34375 -1.851562 4.34375 -1.40625 C 4.34375 -0.851562 4.140625 -0.457031 3.734375 -0.21875 C 3.328125 0.0195312 2.859375 0.140625 2.328125 0.140625 C 1.703125 0.140625 1.210938 -0.015625 0.859375 -0.328125 C 0.503906 -0.648438 0.332031 -1.082031 0.34375 -1.625 Z M 2.359375 -5.15625 Z M 2.359375 -5.15625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='2fa6a5ab-c667-4a45-88df-f391ba868cec' overflow='visible'>\n<path d='M 1.0625 -3.65625 L 1.0625 -1.234375 C 1.0625 -1.046875 1.09375 -0.890625 1.15625 -0.765625 C 1.257812 -0.546875 1.460938 -0.4375 1.765625 -0.4375 C 2.191406 -0.4375 2.484375 -0.628906 2.640625 -1.015625 C 2.722656 -1.222656 2.765625 -1.503906 2.765625 -1.859375 L 2.765625 -3.65625 L 3.390625 -3.65625 L 3.390625 0 L 2.796875 0 L 2.8125 -0.546875 C 2.726562 -0.398438 2.628906 -0.28125 2.515625 -0.1875 C 2.273438 0 1.988281 0.09375 1.65625 0.09375 C 1.132812 0.09375 0.78125 -0.078125 0.59375 -0.421875 C 0.488281 -0.609375 0.4375 -0.859375 0.4375 -1.171875 L 0.4375 -3.65625 Z M 1.90625 -3.75 Z M 1.90625 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='c918bbc5-1fab-432a-8fb7-2bea63973fb2' overflow='visible'>\n<path d='M 0.59375 -5.015625 L 2.859375 -5.015625 C 3.304688 -5.015625 3.664062 -4.890625 3.9375 -4.640625 C 4.207031 -4.390625 4.34375 -4.035156 4.34375 -3.578125 C 4.34375 -3.191406 4.21875 -2.851562 3.96875 -2.5625 C 3.726562 -2.269531 3.359375 -2.125 2.859375 -2.125 L 1.28125 -2.125 L 1.28125 0 L 0.59375 0 Z M 3.65625 -3.578125 C 3.65625 -3.941406 3.519531 -4.191406 3.25 -4.328125 C 3.101562 -4.398438 2.894531 -4.4375 2.625 -4.4375 L 1.28125 -4.4375 L 1.28125 -2.703125 L 2.625 -2.703125 C 2.9375 -2.703125 3.1875 -2.765625 3.375 -2.890625 C 3.5625 -3.023438 3.65625 -3.253906 3.65625 -3.578125 Z M 3.65625 -3.578125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='24baa935-3afb-42ca-b4ef-1ffc8c841b87' overflow='visible'>\n<path d='M 0.859375 -1.25 C 0.898438 -0.894531 1.066406 -0.648438 1.359375 -0.515625 C 1.503906 -0.453125 1.671875 -0.421875 1.859375 -0.421875 C 2.222656 -0.421875 2.492188 -0.535156 2.671875 -0.765625 C 2.847656 -0.992188 2.9375 -1.25 2.9375 -1.53125 C 2.9375 -1.875 2.832031 -2.140625 2.625 -2.328125 C 2.414062 -2.515625 2.164062 -2.609375 1.875 -2.609375 C 1.65625 -2.609375 1.46875 -2.566406 1.3125 -2.484375 C 1.164062 -2.398438 1.039062 -2.285156 0.9375 -2.140625 L 0.40625 -2.171875 L 0.78125 -4.8125 L 3.3125 -4.8125 L 3.3125 -4.21875 L 1.234375 -4.21875 L 1.03125 -2.859375 C 1.144531 -2.941406 1.253906 -3.003906 1.359375 -3.046875 C 1.535156 -3.128906 1.742188 -3.171875 1.984375 -3.171875 C 2.429688 -3.171875 2.8125 -3.023438 3.125 -2.734375 C 3.4375 -2.441406 3.59375 -2.070312 3.59375 -1.625 C 3.59375 -1.164062 3.445312 -0.757812 3.15625 -0.40625 C 2.875 -0.0507812 2.421875 0.125 1.796875 0.125 C 1.398438 0.125 1.046875 0.015625 0.734375 -0.203125 C 0.429688 -0.429688 0.257812 -0.78125 0.21875 -1.25 Z M 0.859375 -1.25 ' style='stroke:none;'/>\n</symbol>\n<symbol id='07ea776c-aee7-462b-8d45-7502d730dcff' overflow='visible'>\n<path d='M 0.546875 -5.015625 L 1.234375 -5.015625 L 1.234375 -2.953125 L 3.84375 -2.953125 L 3.84375 -5.015625 L 4.53125 -5.015625 L 4.53125 0 L 3.84375 0 L 3.84375 -2.34375 L 1.234375 -2.34375 L 1.234375 0 L 0.546875 0 Z M 0.546875 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='16022362-e0a3-4b9e-a0c6-392d55078d8b' overflow='visible'>\n<path d='M 0.734375 -3.65625 L 1.4375 -0.78125 L 2.15625 -3.65625 L 2.84375 -3.65625 L 3.5625 -0.796875 L 4.3125 -3.65625 L 4.921875 -3.65625 L 3.859375 0 L 3.21875 0 L 2.484375 -2.828125 L 1.75 0 L 1.125 0 L 0.0625 -3.65625 Z M 0.734375 -3.65625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='9932acbc-8b99-4f31-be5f-e87ed7486a2e' overflow='visible'>\n<path d='M 2.421875 -2.90625 C 2.703125 -2.90625 2.925781 -2.941406 3.09375 -3.015625 C 3.34375 -3.140625 3.46875 -3.363281 3.46875 -3.6875 C 3.46875 -4.019531 3.332031 -4.242188 3.0625 -4.359375 C 2.914062 -4.421875 2.691406 -4.453125 2.390625 -4.453125 L 1.1875 -4.453125 L 1.1875 -2.90625 Z M 2.65625 -0.578125 C 3.070312 -0.578125 3.367188 -0.695312 3.546875 -0.9375 C 3.648438 -1.09375 3.703125 -1.28125 3.703125 -1.5 C 3.703125 -1.863281 3.539062 -2.113281 3.21875 -2.25 C 3.039062 -2.3125 2.8125 -2.34375 2.53125 -2.34375 L 1.1875 -2.34375 L 1.1875 -0.578125 Z M 0.515625 -5.015625 L 2.671875 -5.015625 C 3.265625 -5.015625 3.679688 -4.84375 3.921875 -4.5 C 4.066406 -4.289062 4.140625 -4.050781 4.140625 -3.78125 C 4.140625 -3.457031 4.050781 -3.195312 3.875 -3 C 3.78125 -2.894531 3.644531 -2.796875 3.46875 -2.703125 C 3.726562 -2.609375 3.921875 -2.5 4.046875 -2.375 C 4.273438 -2.15625 4.390625 -1.851562 4.390625 -1.46875 C 4.390625 -1.144531 4.285156 -0.851562 4.078125 -0.59375 C 3.773438 -0.195312 3.296875 0 2.640625 0 L 0.515625 0 Z M 0.515625 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='d5cf4064-f02e-44b4-b47e-bb8c182a8372' overflow='visible'>\n<path d='M 2.65625 -5.15625 C 3.289062 -5.15625 3.78125 -4.988281 4.125 -4.65625 C 4.476562 -4.320312 4.675781 -3.941406 4.71875 -3.515625 L 4.046875 -3.515625 C 3.972656 -3.835938 3.820312 -4.09375 3.59375 -4.28125 C 3.375 -4.46875 3.0625 -4.5625 2.65625 -4.5625 C 2.164062 -4.5625 1.765625 -4.390625 1.453125 -4.046875 C 1.148438 -3.703125 1 -3.171875 1 -2.453125 C 1 -1.859375 1.132812 -1.378906 1.40625 -1.015625 C 1.6875 -0.648438 2.097656 -0.46875 2.640625 -0.46875 C 3.140625 -0.46875 3.519531 -0.660156 3.78125 -1.046875 C 3.925781 -1.242188 4.03125 -1.507812 4.09375 -1.84375 L 4.765625 -1.84375 C 4.703125 -1.3125 4.503906 -0.867188 4.171875 -0.515625 C 3.773438 -0.0859375 3.238281 0.125 2.5625 0.125 C 1.976562 0.125 1.488281 -0.0507812 1.09375 -0.40625 C 0.570312 -0.863281 0.3125 -1.582031 0.3125 -2.5625 C 0.3125 -3.300781 0.503906 -3.910156 0.890625 -4.390625 C 1.316406 -4.898438 1.90625 -5.15625 2.65625 -5.15625 Z M 2.515625 -5.15625 Z M 2.515625 -5.15625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='0fb24c34-4cf3-4174-93f7-26ffb2ed5101' overflow='visible'>\n<path d='M 0.46875 -5.015625 L 1.078125 -5.015625 L 1.078125 0 L 0.46875 0 Z M 0.46875 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='0dc9cbb1-8da9-4eff-b854-b2296c9c9035' overflow='visible'>\n<path d='M 0.453125 -3.65625 L 1.0625 -3.65625 L 1.0625 -3.140625 C 1.207031 -3.316406 1.335938 -3.445312 1.453125 -3.53125 C 1.660156 -3.675781 1.890625 -3.75 2.140625 -3.75 C 2.429688 -3.75 2.664062 -3.675781 2.84375 -3.53125 C 2.945312 -3.445312 3.039062 -3.328125 3.125 -3.171875 C 3.257812 -3.359375 3.421875 -3.5 3.609375 -3.59375 C 3.796875 -3.695312 4.003906 -3.75 4.234375 -3.75 C 4.722656 -3.75 5.054688 -3.566406 5.234375 -3.203125 C 5.328125 -3.015625 5.375 -2.757812 5.375 -2.4375 L 5.375 0 L 4.734375 0 L 4.734375 -2.546875 C 4.734375 -2.785156 4.671875 -2.953125 4.546875 -3.046875 C 4.429688 -3.140625 4.285156 -3.1875 4.109375 -3.1875 C 3.867188 -3.1875 3.660156 -3.101562 3.484375 -2.9375 C 3.304688 -2.78125 3.21875 -2.507812 3.21875 -2.125 L 3.21875 0 L 2.59375 0 L 2.59375 -2.390625 C 2.59375 -2.640625 2.566406 -2.820312 2.515625 -2.9375 C 2.421875 -3.101562 2.242188 -3.1875 1.984375 -3.1875 C 1.753906 -3.1875 1.539062 -3.097656 1.34375 -2.921875 C 1.15625 -2.742188 1.0625 -2.414062 1.0625 -1.9375 L 1.0625 0 L 0.453125 0 Z M 0.453125 -3.65625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='7610a7f7-8be4-4668-99a4-528f7d39ba3f' overflow='visible'>\n<path d='M 0.515625 -5.015625 L 1.484375 -5.015625 L 2.9375 -0.78125 L 4.359375 -5.015625 L 5.328125 -5.015625 L 5.328125 0 L 4.6875 0 L 4.6875 -2.96875 C 4.6875 -3.070312 4.6875 -3.238281 4.6875 -3.46875 C 4.695312 -3.707031 4.703125 -3.960938 4.703125 -4.234375 L 3.265625 0 L 2.59375 0 L 1.15625 -4.234375 L 1.15625 -4.078125 C 1.15625 -3.953125 1.15625 -3.765625 1.15625 -3.515625 C 1.164062 -3.265625 1.171875 -3.082031 1.171875 -2.96875 L 1.171875 0 L 0.515625 0 Z M 0.515625 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='84ec0c58-94e7-44be-bbb5-f16a483ad50a' overflow='visible'>\n<path d='M 0.59375 -5.015625 L 4.265625 -5.015625 L 4.265625 -4.40625 L 1.265625 -4.40625 L 1.265625 -2.875 L 4.03125 -2.875 L 4.03125 -2.296875 L 1.265625 -2.296875 L 1.265625 -0.59375 L 4.3125 -0.59375 L 4.3125 0 L 0.59375 0 Z M 2.453125 -5.015625 Z M 2.453125 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='c00721a7-52a3-4489-a759-7de6cac3a30d' overflow='visible'>\n<path d='M 0.40625 -5.03125 L 1 -5.03125 L 1 -3.21875 C 1.132812 -3.394531 1.296875 -3.523438 1.484375 -3.609375 C 1.671875 -3.703125 1.875 -3.75 2.09375 -3.75 C 2.550781 -3.75 2.921875 -3.59375 3.203125 -3.28125 C 3.484375 -2.96875 3.625 -2.507812 3.625 -1.90625 C 3.625 -1.320312 3.484375 -0.835938 3.203125 -0.453125 C 2.921875 -0.078125 2.535156 0.109375 2.046875 0.109375 C 1.765625 0.109375 1.53125 0.046875 1.34375 -0.078125 C 1.226562 -0.160156 1.101562 -0.289062 0.96875 -0.46875 L 0.96875 0 L 0.40625 0 Z M 2 -0.421875 C 2.332031 -0.421875 2.582031 -0.550781 2.75 -0.8125 C 2.914062 -1.082031 3 -1.4375 3 -1.875 C 3 -2.257812 2.914062 -2.578125 2.75 -2.828125 C 2.582031 -3.078125 2.335938 -3.203125 2.015625 -3.203125 C 1.734375 -3.203125 1.488281 -3.097656 1.28125 -2.890625 C 1.070312 -2.679688 0.96875 -2.34375 0.96875 -1.875 C 0.96875 -1.53125 1.007812 -1.25 1.09375 -1.03125 C 1.257812 -0.625 1.5625 -0.421875 2 -0.421875 Z M 2 -0.421875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='61e31a73-cdee-4d48-be2c-f33e2453bdf4' overflow='visible'>\n<path d='M 0.9375 -5.015625 L 2.375 -0.75 L 3.796875 -5.015625 L 4.5625 -5.015625 L 2.734375 0 L 2 0 L 0.171875 -5.015625 Z M 0.9375 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='415a53e9-54f6-46d5-9b86-9f2d483a23ee' overflow='visible'>\n<path d='M 1.0625 -4.3125 L 0.453125 -4.3125 L 0.453125 -5.015625 L 1.0625 -5.015625 Z M -0.125 0.921875 C 0.144531 0.910156 0.304688 0.882812 0.359375 0.84375 C 0.421875 0.8125 0.453125 0.691406 0.453125 0.484375 L 0.453125 -3.640625 L 1.0625 -3.640625 L 1.0625 0.546875 C 1.0625 0.804688 1.019531 1.003906 0.9375 1.140625 C 0.789062 1.367188 0.519531 1.484375 0.125 1.484375 C 0.09375 1.484375 0.0546875 1.476562 0.015625 1.46875 C -0.015625 1.46875 -0.0625 1.460938 -0.125 1.453125 Z M -0.125 0.921875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='5062135a-1b6b-4c59-ae8d-4c5af49dcc8d' overflow='visible'>\n<path d='M 4.1875 -5.015625 L 4.1875 -4.421875 L 2.5 -4.421875 L 2.5 0 L 1.8125 0 L 1.8125 -4.421875 L 0.109375 -4.421875 L 0.109375 -5.015625 Z M 4.1875 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='20f10433-62cf-4427-9d4a-66c5c001be2b' overflow='visible'>\n<path d='M 3.65625 -4.8125 L 3.65625 -4.28125 C 3.5 -4.125 3.289062 -3.851562 3.03125 -3.46875 C 2.769531 -3.09375 2.539062 -2.6875 2.34375 -2.25 C 2.144531 -1.820312 1.992188 -1.4375 1.890625 -1.09375 C 1.828125 -0.863281 1.742188 -0.5 1.640625 0 L 0.953125 0 C 1.109375 -0.9375 1.453125 -1.867188 1.984375 -2.796875 C 2.296875 -3.335938 2.625 -3.800781 2.96875 -4.1875 L 0.25 -4.1875 L 0.25 -4.8125 Z M 3.65625 -4.8125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='faf9c806-cffc-48dc-8fa4-0e89266f3dca' overflow='visible'>\n<path d='M 3.109375 -2.0625 L 2.34375 -4.28125 L 1.53125 -2.0625 Z M 2 -5.015625 L 2.765625 -5.015625 L 4.578125 0 L 3.84375 0 L 3.328125 -1.5 L 1.34375 -1.5 L 0.796875 0 L 0.109375 0 Z M 2.34375 -5.015625 Z M 2.34375 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='7ed5e8ce-843d-470d-a211-5a4659c90d39' overflow='visible'>\n<path d='M 2.875 -2.71875 C 3.1875 -2.71875 3.4375 -2.78125 3.625 -2.90625 C 3.8125 -3.039062 3.90625 -3.273438 3.90625 -3.609375 C 3.90625 -3.960938 3.773438 -4.203125 3.515625 -4.328125 C 3.378906 -4.398438 3.191406 -4.4375 2.953125 -4.4375 L 1.296875 -4.4375 L 1.296875 -2.71875 Z M 0.609375 -5.015625 L 2.9375 -5.015625 C 3.320312 -5.015625 3.640625 -4.960938 3.890625 -4.859375 C 4.359375 -4.640625 4.59375 -4.242188 4.59375 -3.671875 C 4.59375 -3.367188 4.53125 -3.117188 4.40625 -2.921875 C 4.28125 -2.734375 4.109375 -2.582031 3.890625 -2.46875 C 4.085938 -2.382812 4.234375 -2.273438 4.328125 -2.140625 C 4.429688 -2.015625 4.488281 -1.804688 4.5 -1.515625 L 4.515625 -0.84375 C 4.523438 -0.65625 4.539062 -0.515625 4.5625 -0.421875 C 4.601562 -0.265625 4.675781 -0.160156 4.78125 -0.109375 L 4.78125 0 L 3.953125 0 C 3.929688 -0.0390625 3.910156 -0.09375 3.890625 -0.15625 C 3.878906 -0.226562 3.867188 -0.363281 3.859375 -0.5625 L 3.828125 -1.40625 C 3.804688 -1.726562 3.679688 -1.945312 3.453125 -2.0625 C 3.328125 -2.125 3.125 -2.15625 2.84375 -2.15625 L 1.296875 -2.15625 L 1.296875 0 L 0.609375 0 Z M 0.609375 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='15f4a5d1-e31f-45ed-8218-3f8fc09d4b20' overflow='visible'>\n<path d='M 0.59375 -5.015625 L 4.078125 -5.015625 L 4.078125 -4.40625 L 1.28125 -4.40625 L 1.28125 -2.875 L 3.75 -2.875 L 3.75 -2.28125 L 1.28125 -2.28125 L 1.28125 0 L 0.59375 0 Z M 0.59375 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='a8f15154-4753-4c2e-889a-bf0f8eab37f5' overflow='visible'>\n<path d='M 2.703125 -5.15625 C 3.171875 -5.15625 3.578125 -5.0625 3.921875 -4.875 C 4.421875 -4.613281 4.726562 -4.15625 4.84375 -3.5 L 4.171875 -3.5 C 4.085938 -3.863281 3.914062 -4.128906 3.65625 -4.296875 C 3.40625 -4.460938 3.082031 -4.546875 2.6875 -4.546875 C 2.21875 -4.546875 1.820312 -4.367188 1.5 -4.015625 C 1.1875 -3.671875 1.03125 -3.15625 1.03125 -2.46875 C 1.03125 -1.875 1.160156 -1.390625 1.421875 -1.015625 C 1.691406 -0.640625 2.117188 -0.453125 2.703125 -0.453125 C 3.160156 -0.453125 3.535156 -0.582031 3.828125 -0.84375 C 4.128906 -1.101562 4.285156 -1.53125 4.296875 -2.125 L 2.71875 -2.125 L 2.71875 -2.6875 L 4.921875 -2.6875 L 4.921875 0 L 4.484375 0 L 4.328125 -0.640625 C 4.097656 -0.390625 3.894531 -0.21875 3.71875 -0.125 C 3.414062 0.0507812 3.035156 0.140625 2.578125 0.140625 C 1.984375 0.140625 1.472656 -0.0507812 1.046875 -0.4375 C 0.578125 -0.925781 0.34375 -1.59375 0.34375 -2.4375 C 0.34375 -3.269531 0.566406 -3.9375 1.015625 -4.4375 C 1.453125 -4.914062 2.015625 -5.15625 2.703125 -5.15625 Z M 2.59375 -5.15625 Z M 2.59375 -5.15625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='498ee0e9-30bd-433e-a432-1fddd37c32a1' overflow='visible'>\n<path d='M 2.3125 -1.734375 L 2.3125 -3.953125 L 0.75 -1.734375 Z M 2.328125 0 L 2.328125 -1.203125 L 0.171875 -1.203125 L 0.171875 -1.796875 L 2.421875 -4.90625 L 2.9375 -4.90625 L 2.9375 -1.734375 L 3.65625 -1.734375 L 3.65625 -1.203125 L 2.9375 -1.203125 L 2.9375 0 Z M 2.328125 0 ' style='stroke:none;'/>\n</symbol>\n<symbol id='1862368e-e783-4c5e-9649-e2093e4fd672' overflow='visible'>\n<path d='M 0.171875 -0.484375 L 2.34375 -3.109375 L 0.34375 -3.109375 L 0.34375 -3.65625 L 3.171875 -3.65625 L 3.171875 -3.15625 L 1.015625 -0.546875 L 3.234375 -0.546875 L 3.234375 0 L 0.171875 0 Z M 1.765625 -3.75 Z M 1.765625 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='9826757d-31e9-4d33-9633-ddb1b135594a' overflow='visible'>\n<path d='M 2.078125 -5.109375 C 1.710938 -4.410156 1.476562 -3.894531 1.375 -3.5625 C 1.207031 -3.0625 1.125 -2.488281 1.125 -1.84375 C 1.125 -1.175781 1.21875 -0.570312 1.40625 -0.03125 C 1.519531 0.3125 1.742188 0.796875 2.078125 1.421875 L 1.671875 1.421875 C 1.335938 0.910156 1.128906 0.582031 1.046875 0.4375 C 0.972656 0.289062 0.890625 0.09375 0.796875 -0.15625 C 0.660156 -0.488281 0.566406 -0.847656 0.515625 -1.234375 C 0.492188 -1.429688 0.484375 -1.617188 0.484375 -1.796875 C 0.484375 -2.472656 0.585938 -3.070312 0.796875 -3.59375 C 0.929688 -3.9375 1.210938 -4.441406 1.640625 -5.109375 Z M 2.078125 -5.109375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='5e521fb0-e350-4c29-a6ab-cff23c4eec3a' overflow='visible'>\n<path d='M 0.25 1.421875 C 0.601562 0.722656 0.832031 0.210938 0.9375 -0.109375 C 1.101562 -0.597656 1.1875 -1.175781 1.1875 -1.84375 C 1.1875 -2.5 1.09375 -3.097656 0.90625 -3.640625 C 0.789062 -3.984375 0.566406 -4.472656 0.234375 -5.109375 L 0.640625 -5.109375 C 0.992188 -4.546875 1.207031 -4.195312 1.28125 -4.0625 C 1.351562 -3.9375 1.4375 -3.757812 1.53125 -3.53125 C 1.632812 -3.238281 1.707031 -2.953125 1.75 -2.671875 C 1.800781 -2.398438 1.828125 -2.132812 1.828125 -1.875 C 1.828125 -1.195312 1.71875 -0.597656 1.5 -0.078125 C 1.375 0.265625 1.097656 0.765625 0.671875 1.421875 Z M 0.25 1.421875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='734cb640-de7a-40b0-b39b-e8216d0e6a8e' overflow='visible'>\n<path d='M 1.265625 -5.015625 L 1.265625 -1.921875 C 1.265625 -1.554688 1.335938 -1.253906 1.484375 -1.015625 C 1.679688 -0.648438 2.019531 -0.46875 2.5 -0.46875 C 3.070312 -0.46875 3.460938 -0.660156 3.671875 -1.046875 C 3.785156 -1.265625 3.84375 -1.554688 3.84375 -1.921875 L 3.84375 -5.015625 L 4.53125 -5.015625 L 4.53125 -2.203125 C 4.53125 -1.585938 4.445312 -1.113281 4.28125 -0.78125 C 3.976562 -0.164062 3.40625 0.140625 2.5625 0.140625 C 1.707031 0.140625 1.128906 -0.164062 0.828125 -0.78125 C 0.660156 -1.113281 0.578125 -1.585938 0.578125 -2.203125 L 0.578125 -5.015625 Z M 2.5625 -5.015625 Z M 2.5625 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='5d4eee01-dc5a-410c-8a4f-b300c300a866' overflow='visible'>\n<path d='M 0.84375 -1.78125 C 0.84375 -1.46875 0.890625 -1.203125 0.984375 -0.984375 C 1.140625 -0.597656 1.421875 -0.40625 1.828125 -0.40625 C 2.253906 -0.40625 2.546875 -0.609375 2.703125 -1.015625 C 2.796875 -1.234375 2.84375 -1.515625 2.84375 -1.859375 C 2.84375 -2.179688 2.796875 -2.445312 2.703125 -2.65625 C 2.535156 -3.019531 2.238281 -3.203125 1.8125 -3.203125 C 1.550781 -3.203125 1.320312 -3.082031 1.125 -2.84375 C 0.9375 -2.613281 0.84375 -2.257812 0.84375 -1.78125 Z M 1.75 -3.75 C 2.0625 -3.75 2.316406 -3.671875 2.515625 -3.515625 C 2.628906 -3.429688 2.738281 -3.3125 2.84375 -3.15625 L 2.84375 -3.65625 L 3.4375 -3.65625 L 3.4375 1.453125 L 2.8125 1.453125 L 2.8125 -0.421875 C 2.707031 -0.253906 2.566406 -0.125 2.390625 -0.03125 C 2.210938 0.0625 1.988281 0.109375 1.71875 0.109375 C 1.320312 0.109375 0.96875 -0.0390625 0.65625 -0.34375 C 0.351562 -0.644531 0.203125 -1.109375 0.203125 -1.734375 C 0.203125 -2.316406 0.34375 -2.796875 0.625 -3.171875 C 0.914062 -3.554688 1.289062 -3.75 1.75 -3.75 Z M 1.75 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='b69883fe-a677-48f2-9397-180599c78d22' overflow='visible'>\n<path d='M 0.53125 -5.015625 L 1.34375 -5.015625 L 3.875 -0.953125 L 3.875 -5.015625 L 4.515625 -5.015625 L 4.515625 0 L 3.75 0 L 1.1875 -4.0625 L 1.1875 0 L 0.53125 0 Z M 2.484375 -5.015625 Z M 2.484375 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='5766d641-c9b6-4bbb-afd0-e7cf221f48b7' overflow='visible'>\n<path d='M 2 -0.40625 C 2.28125 -0.40625 2.515625 -0.523438 2.703125 -0.765625 C 2.898438 -1.015625 3 -1.378906 3 -1.859375 C 3 -2.148438 2.957031 -2.398438 2.875 -2.609375 C 2.707031 -3.003906 2.414062 -3.203125 2 -3.203125 C 1.570312 -3.203125 1.28125 -2.992188 1.125 -2.578125 C 1.039062 -2.347656 1 -2.054688 1 -1.703125 C 1 -1.421875 1.039062 -1.179688 1.125 -0.984375 C 1.28125 -0.597656 1.570312 -0.40625 2 -0.40625 Z M 0.40625 -3.640625 L 1 -3.640625 L 1 -3.15625 C 1.125 -3.320312 1.257812 -3.453125 1.40625 -3.546875 C 1.613281 -3.679688 1.859375 -3.75 2.140625 -3.75 C 2.554688 -3.75 2.90625 -3.585938 3.1875 -3.265625 C 3.476562 -2.953125 3.625 -2.5 3.625 -1.90625 C 3.625 -1.113281 3.414062 -0.546875 3 -0.203125 C 2.738281 0.015625 2.429688 0.125 2.078125 0.125 C 1.804688 0.125 1.578125 0.0625 1.390625 -0.0625 C 1.273438 -0.125 1.148438 -0.238281 1.015625 -0.40625 L 1.015625 1.453125 L 0.40625 1.453125 Z M 0.40625 -3.640625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='20af18b1-c0b4-4b01-987c-e266a13cf66e' overflow='visible'>\n<path d='M 0.140625 -5.015625 L 0.9375 -5.015625 L 2.375 -2.609375 L 3.828125 -5.015625 L 4.625 -5.015625 L 2.71875 -2.03125 L 2.71875 0 L 2.046875 0 L 2.046875 -2.03125 Z M 2.390625 -5.015625 Z M 2.390625 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='97113d21-a497-49f4-9dd1-da562099ee05' overflow='visible'>\n<path d='M 0.609375 -4.21875 C 0.617188 -4.46875 0.660156 -4.65625 0.734375 -4.78125 C 0.878906 -4.988281 1.15625 -5.09375 1.5625 -5.09375 C 1.601562 -5.09375 1.644531 -5.09375 1.6875 -5.09375 C 1.726562 -5.09375 1.773438 -5.085938 1.828125 -5.078125 L 1.828125 -4.515625 C 1.765625 -4.523438 1.71875 -4.53125 1.6875 -4.53125 C 1.65625 -4.53125 1.628906 -4.53125 1.609375 -4.53125 C 1.421875 -4.53125 1.304688 -4.476562 1.265625 -4.375 C 1.234375 -4.28125 1.21875 -4.035156 1.21875 -3.640625 L 1.828125 -3.640625 L 1.828125 -3.15625 L 1.21875 -3.15625 L 1.21875 0 L 0.609375 0 L 0.609375 -3.15625 L 0.09375 -3.15625 L 0.09375 -3.640625 L 0.609375 -3.640625 Z M 0.609375 -4.21875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='43a18a7e-1975-42b3-84e0-9606dd71d1a9' overflow='visible'>\n<path d='M 1.8125 0.140625 C 1.238281 0.140625 0.820312 -0.0195312 0.5625 -0.34375 C 0.300781 -0.664062 0.171875 -1.050781 0.171875 -1.5 L 0.8125 -1.5 C 0.832031 -1.1875 0.890625 -0.957031 0.984375 -0.8125 C 1.148438 -0.550781 1.441406 -0.421875 1.859375 -0.421875 C 2.179688 -0.421875 2.4375 -0.507812 2.625 -0.6875 C 2.820312 -0.863281 2.921875 -1.085938 2.921875 -1.359375 C 2.921875 -1.691406 2.816406 -1.925781 2.609375 -2.0625 C 2.410156 -2.195312 2.125 -2.265625 1.75 -2.265625 C 1.71875 -2.265625 1.675781 -2.265625 1.625 -2.265625 C 1.582031 -2.265625 1.539062 -2.265625 1.5 -2.265625 L 1.5 -2.796875 C 1.5625 -2.796875 1.613281 -2.789062 1.65625 -2.78125 C 1.707031 -2.78125 1.757812 -2.78125 1.8125 -2.78125 C 2.039062 -2.78125 2.226562 -2.816406 2.375 -2.890625 C 2.632812 -3.023438 2.765625 -3.253906 2.765625 -3.578125 C 2.765625 -3.816406 2.675781 -4.003906 2.5 -4.140625 C 2.332031 -4.273438 2.128906 -4.34375 1.890625 -4.34375 C 1.484375 -4.34375 1.203125 -4.207031 1.046875 -3.9375 C 0.960938 -3.78125 0.914062 -3.5625 0.90625 -3.28125 L 0.296875 -3.28125 C 0.296875 -3.644531 0.367188 -3.957031 0.515625 -4.21875 C 0.765625 -4.675781 1.203125 -4.90625 1.828125 -4.90625 C 2.328125 -4.90625 2.710938 -4.789062 2.984375 -4.5625 C 3.265625 -4.34375 3.40625 -4.023438 3.40625 -3.609375 C 3.40625 -3.304688 3.320312 -3.0625 3.15625 -2.875 C 3.0625 -2.757812 2.929688 -2.664062 2.765625 -2.59375 C 3.023438 -2.519531 3.226562 -2.382812 3.375 -2.1875 C 3.519531 -1.988281 3.59375 -1.742188 3.59375 -1.453125 C 3.59375 -0.984375 3.4375 -0.597656 3.125 -0.296875 C 2.820312 -0.00390625 2.382812 0.140625 1.8125 0.140625 Z M 1.8125 0.140625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='bbcfccc7-5384-4cb4-bf6e-2e0c71777eab' overflow='visible'>\n<path d='M 4.265625 -3.046875 C 4.265625 -3.191406 4.21875 -3.320312 4.125 -3.4375 C 4.03125 -3.550781 3.898438 -3.609375 3.734375 -3.609375 C 3.472656 -3.609375 3.25 -3.453125 3.0625 -3.140625 C 2.875 -2.835938 2.78125 -2.535156 2.78125 -2.234375 C 2.78125 -2.003906 2.828125 -1.832031 2.921875 -1.71875 C 3.015625 -1.601562 3.140625 -1.546875 3.296875 -1.546875 C 3.597656 -1.546875 3.832031 -1.722656 4 -2.078125 C 4.175781 -2.441406 4.265625 -2.765625 4.265625 -3.046875 Z M 3.59375 0.140625 C 2.632812 0.140625 1.898438 -0.15625 1.390625 -0.75 C 0.972656 -1.21875 0.765625 -1.765625 0.765625 -2.390625 C 0.765625 -3.140625 1.015625 -3.773438 1.515625 -4.296875 C 2.078125 -4.867188 2.828125 -5.15625 3.765625 -5.15625 C 4.578125 -5.15625 5.210938 -4.953125 5.671875 -4.546875 C 6.109375 -4.160156 6.328125 -3.65625 6.328125 -3.03125 C 6.328125 -2.507812 6.171875 -2.066406 5.859375 -1.703125 C 5.546875 -1.335938 5.195312 -1.15625 4.8125 -1.15625 C 4.59375 -1.15625 4.425781 -1.207031 4.3125 -1.3125 C 4.207031 -1.425781 4.15625 -1.554688 4.15625 -1.703125 C 4.15625 -1.734375 4.15625 -1.757812 4.15625 -1.78125 C 4.164062 -1.8125 4.175781 -1.84375 4.1875 -1.875 C 4.0625 -1.613281 3.90625 -1.425781 3.71875 -1.3125 C 3.539062 -1.207031 3.351562 -1.15625 3.15625 -1.15625 C 2.875 -1.15625 2.644531 -1.253906 2.46875 -1.453125 C 2.289062 -1.660156 2.203125 -1.921875 2.203125 -2.234375 C 2.203125 -2.691406 2.347656 -3.09375 2.640625 -3.4375 C 2.941406 -3.78125 3.265625 -3.953125 3.609375 -3.953125 C 3.847656 -3.953125 4.039062 -3.882812 4.1875 -3.75 C 4.34375 -3.625 4.425781 -3.460938 4.4375 -3.265625 L 4.59375 -3.84375 L 5.109375 -3.84375 L 4.65625 -2.34375 C 4.613281 -2.226562 4.582031 -2.132812 4.5625 -2.0625 C 4.550781 -2 4.546875 -1.9375 4.546875 -1.875 C 4.546875 -1.769531 4.578125 -1.695312 4.640625 -1.65625 C 4.703125 -1.613281 4.773438 -1.59375 4.859375 -1.59375 C 5.117188 -1.59375 5.347656 -1.726562 5.546875 -2 C 5.753906 -2.269531 5.859375 -2.625 5.859375 -3.0625 C 5.859375 -3.582031 5.675781 -3.992188 5.3125 -4.296875 C 4.945312 -4.597656 4.453125 -4.75 3.828125 -4.75 C 2.953125 -4.75 2.285156 -4.5 1.828125 -4 C 1.441406 -3.5625 1.25 -3.019531 1.25 -2.375 C 1.25 -1.800781 1.4375 -1.320312 1.8125 -0.9375 C 2.25 -0.476562 2.867188 -0.25 3.671875 -0.25 C 4.054688 -0.25 4.410156 -0.304688 4.734375 -0.421875 C 5.054688 -0.546875 5.351562 -0.707031 5.625 -0.90625 L 5.859375 -0.609375 C 5.648438 -0.421875 5.34375 -0.25 4.9375 -0.09375 C 4.53125 0.0625 4.082031 0.140625 3.59375 0.140625 Z M 3.59375 0.140625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='351f40d4-2fc8-4a8b-8fbf-02a4e78e4533' overflow='visible'>\n<path d='M 1.890625 -4.890625 C 2.523438 -4.890625 2.984375 -4.628906 3.265625 -4.109375 C 3.484375 -3.703125 3.59375 -3.148438 3.59375 -2.453125 C 3.59375 -1.785156 3.492188 -1.234375 3.296875 -0.796875 C 3.015625 -0.171875 2.546875 0.140625 1.890625 0.140625 C 1.304688 0.140625 0.867188 -0.113281 0.578125 -0.625 C 0.335938 -1.050781 0.21875 -1.625 0.21875 -2.34375 C 0.21875 -2.90625 0.289062 -3.382812 0.4375 -3.78125 C 0.707031 -4.519531 1.191406 -4.890625 1.890625 -4.890625 Z M 1.890625 -0.421875 C 2.203125 -0.421875 2.453125 -0.5625 2.640625 -0.84375 C 2.835938 -1.132812 2.9375 -1.664062 2.9375 -2.4375 C 2.9375 -2.988281 2.863281 -3.441406 2.71875 -3.796875 C 2.582031 -4.160156 2.320312 -4.34375 1.9375 -4.34375 C 1.570312 -4.34375 1.304688 -4.171875 1.140625 -3.828125 C 0.972656 -3.484375 0.890625 -2.984375 0.890625 -2.328125 C 0.890625 -1.828125 0.941406 -1.425781 1.046875 -1.125 C 1.210938 -0.65625 1.492188 -0.421875 1.890625 -0.421875 Z M 1.890625 -0.421875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='3a3a64a1-d6d0-4f66-a33e-d8444bd84bc5' overflow='visible'>\n<path d='M 0.671875 -3.46875 L 0.671875 -3.9375 C 1.117188 -3.976562 1.425781 -4.050781 1.59375 -4.15625 C 1.769531 -4.257812 1.90625 -4.5 2 -4.875 L 2.484375 -4.875 L 2.484375 0 L 1.828125 0 L 1.828125 -3.46875 Z M 0.671875 -3.46875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='b9582e8f-07fc-4016-b5ef-907d7a9dea00' overflow='visible'>\n<path d='M 0 -0.265625 L -5.734375 -0.265625 L -5.734375 -4.8125 L 0 -4.8125 Z M -0.71875 -4.09375 L -5.015625 -4.09375 L -5.015625 -0.984375 L -0.71875 -0.984375 Z M -0.71875 -4.09375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='b9fbad53-4522-4c6c-8250-7b116e75e87a' overflow='visible'>\n<path d='M -1.3125 -0.9375 C -1.082031 -0.957031 -0.90625 -1.015625 -0.78125 -1.109375 C -0.550781 -1.285156 -0.4375 -1.59375 -0.4375 -2.03125 C -0.4375 -2.289062 -0.492188 -2.519531 -0.609375 -2.71875 C -0.722656 -2.914062 -0.894531 -3.015625 -1.125 -3.015625 C -1.3125 -3.015625 -1.453125 -2.9375 -1.546875 -2.78125 C -1.597656 -2.675781 -1.660156 -2.472656 -1.734375 -2.171875 L -1.875 -1.625 C -1.96875 -1.257812 -2.066406 -0.992188 -2.171875 -0.828125 C -2.367188 -0.523438 -2.632812 -0.375 -2.96875 -0.375 C -3.363281 -0.375 -3.679688 -0.515625 -3.921875 -0.796875 C -4.171875 -1.085938 -4.296875 -1.472656 -4.296875 -1.953125 C -4.296875 -2.585938 -4.109375 -3.039062 -3.734375 -3.3125 C -3.503906 -3.488281 -3.253906 -3.578125 -2.984375 -3.578125 L -2.984375 -2.90625 C -3.140625 -2.894531 -3.285156 -2.835938 -3.421875 -2.734375 C -3.597656 -2.578125 -3.6875 -2.300781 -3.6875 -1.90625 C -3.6875 -1.632812 -3.632812 -1.429688 -3.53125 -1.296875 C -3.4375 -1.160156 -3.304688 -1.09375 -3.140625 -1.09375 C -2.953125 -1.09375 -2.804688 -1.179688 -2.703125 -1.359375 C -2.640625 -1.460938 -2.582031 -1.617188 -2.53125 -1.828125 L -2.421875 -2.28125 C -2.296875 -2.789062 -2.175781 -3.128906 -2.0625 -3.296875 C -1.882812 -3.578125 -1.601562 -3.71875 -1.21875 -3.71875 C -0.84375 -3.71875 -0.519531 -3.570312 -0.25 -3.28125 C 0.0195312 -3 0.15625 -2.570312 0.15625 -2 C 0.15625 -1.375 0.015625 -0.929688 -0.265625 -0.671875 C -0.546875 -0.410156 -0.894531 -0.273438 -1.3125 -0.265625 Z M -4.28125 -1.96875 Z M -4.28125 -1.96875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='4d9c9b9b-614a-4330-9440-bf809579b53c' overflow='visible'>\n<path d='M -5.359375 -0.65625 L -5.359375 -1.375 L -4.1875 -1.375 L -4.1875 -2.03125 L -3.609375 -2.03125 L -3.609375 -1.375 L -0.875 -1.375 C -0.726562 -1.375 -0.632812 -1.421875 -0.59375 -1.515625 C -0.5625 -1.566406 -0.546875 -1.660156 -0.546875 -1.796875 C -0.546875 -1.828125 -0.546875 -1.859375 -0.546875 -1.890625 C -0.546875 -1.929688 -0.550781 -1.976562 -0.5625 -2.03125 L 0 -2.03125 C 0.0195312 -1.957031 0.0351562 -1.875 0.046875 -1.78125 C 0.0546875 -1.695312 0.0625 -1.601562 0.0625 -1.5 C 0.0625 -1.175781 -0.0195312 -0.953125 -0.1875 -0.828125 C -0.351562 -0.710938 -0.570312 -0.65625 -0.84375 -0.65625 L -3.609375 -0.65625 L -3.609375 -0.09375 L -4.1875 -0.09375 L -4.1875 -0.65625 Z M -5.359375 -0.65625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='4100fe2b-0aa9-4ff3-aa7d-ed4a3c2cf642' overflow='visible'>\n<path d='M -1.109375 -1.0625 C -0.910156 -1.0625 -0.75 -1.132812 -0.625 -1.28125 C -0.507812 -1.425781 -0.453125 -1.601562 -0.453125 -1.8125 C -0.453125 -2.050781 -0.507812 -2.289062 -0.625 -2.53125 C -0.820312 -2.914062 -1.132812 -3.109375 -1.5625 -3.109375 L -2.125 -3.109375 C -2.070312 -3.023438 -2.023438 -2.914062 -1.984375 -2.78125 C -1.953125 -2.644531 -1.929688 -2.507812 -1.921875 -2.375 L -1.859375 -1.953125 C -1.828125 -1.703125 -1.773438 -1.507812 -1.703125 -1.375 C -1.578125 -1.164062 -1.378906 -1.0625 -1.109375 -1.0625 Z M -2.53125 -2.765625 C -2.550781 -2.921875 -2.617188 -3.023438 -2.734375 -3.078125 C -2.796875 -3.109375 -2.890625 -3.125 -3.015625 -3.125 C -3.253906 -3.125 -3.425781 -3.039062 -3.53125 -2.875 C -3.632812 -2.707031 -3.6875 -2.460938 -3.6875 -2.140625 C -3.6875 -1.773438 -3.585938 -1.515625 -3.390625 -1.359375 C -3.285156 -1.273438 -3.125 -1.21875 -2.90625 -1.1875 L -2.90625 -0.53125 C -3.425781 -0.550781 -3.785156 -0.71875 -3.984375 -1.03125 C -4.179688 -1.351562 -4.28125 -1.726562 -4.28125 -2.15625 C -4.28125 -2.644531 -4.1875 -3.046875 -4 -3.359375 C -3.8125 -3.660156 -3.519531 -3.8125 -3.125 -3.8125 L -0.71875 -3.8125 C -0.644531 -3.8125 -0.585938 -3.828125 -0.546875 -3.859375 C -0.503906 -3.890625 -0.484375 -3.953125 -0.484375 -4.046875 C -0.484375 -4.078125 -0.484375 -4.113281 -0.484375 -4.15625 C -0.484375 -4.195312 -0.488281 -4.238281 -0.5 -4.28125 L 0.015625 -4.28125 C 0.046875 -4.164062 0.0664062 -4.078125 0.078125 -4.015625 C 0.0859375 -3.960938 0.09375 -3.890625 0.09375 -3.796875 C 0.09375 -3.554688 0.00390625 -3.378906 -0.171875 -3.265625 C -0.265625 -3.210938 -0.394531 -3.171875 -0.5625 -3.140625 C -0.375 -3.003906 -0.207031 -2.800781 -0.0625 -2.53125 C 0.0703125 -2.257812 0.140625 -1.960938 0.140625 -1.640625 C 0.140625 -1.253906 0.0234375 -0.9375 -0.203125 -0.6875 C -0.441406 -0.445312 -0.738281 -0.328125 -1.09375 -0.328125 C -1.488281 -0.328125 -1.789062 -0.445312 -2 -0.6875 C -2.21875 -0.925781 -2.347656 -1.242188 -2.390625 -1.640625 Z M -4.28125 -2.171875 Z M -4.28125 -2.171875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='943094b7-c061-430d-bd6d-8bfc1fe5ee25' overflow='visible'>\n<path d='M -4.171875 -0.515625 L -4.171875 -1.234375 L 0 -1.234375 L 0 -0.515625 Z M -5.734375 -0.515625 L -5.734375 -1.234375 L -4.9375 -1.234375 L -4.9375 -0.515625 Z M -5.734375 -0.515625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='2924fa4a-4cf0-40b8-93dc-3dec1b54e117' overflow='visible'>\n<path d='M -0.453125 -2.171875 C -0.453125 -2.640625 -0.628906 -2.957031 -0.984375 -3.125 C -1.335938 -3.300781 -1.726562 -3.390625 -2.15625 -3.390625 C -2.550781 -3.390625 -2.867188 -3.328125 -3.109375 -3.203125 C -3.492188 -3.003906 -3.6875 -2.664062 -3.6875 -2.1875 C -3.6875 -1.757812 -3.519531 -1.445312 -3.1875 -1.25 C -2.863281 -1.050781 -2.472656 -0.953125 -2.015625 -0.953125 C -1.566406 -0.953125 -1.191406 -1.050781 -0.890625 -1.25 C -0.597656 -1.445312 -0.453125 -1.753906 -0.453125 -2.171875 Z M -4.3125 -2.203125 C -4.3125 -2.742188 -4.128906 -3.195312 -3.765625 -3.5625 C -3.410156 -3.9375 -2.882812 -4.125 -2.1875 -4.125 C -1.507812 -4.125 -0.945312 -3.957031 -0.5 -3.625 C -0.0625 -3.300781 0.15625 -2.796875 0.15625 -2.109375 C 0.15625 -1.535156 -0.0351562 -1.078125 -0.421875 -0.734375 C -0.816406 -0.398438 -1.34375 -0.234375 -2 -0.234375 C -2.707031 -0.234375 -3.269531 -0.410156 -3.6875 -0.765625 C -4.101562 -1.117188 -4.3125 -1.597656 -4.3125 -2.203125 Z M -4.28125 -2.1875 Z M -4.28125 -2.1875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='703657dc-fa2c-4449-8af9-fb0f288f2801' overflow='visible'>\n<path d='M -4.1875 -0.515625 L -4.1875 -1.1875 L -3.59375 -1.1875 C -3.832031 -1.382812 -4.003906 -1.59375 -4.109375 -1.8125 C -4.222656 -2.03125 -4.28125 -2.273438 -4.28125 -2.546875 C -4.28125 -3.140625 -4.070312 -3.539062 -3.65625 -3.75 C -3.425781 -3.863281 -3.101562 -3.921875 -2.6875 -3.921875 L 0 -3.921875 L 0 -3.21875 L -2.640625 -3.21875 C -2.890625 -3.21875 -3.09375 -3.175781 -3.25 -3.09375 C -3.507812 -2.96875 -3.640625 -2.742188 -3.640625 -2.421875 C -3.640625 -2.253906 -3.625 -2.117188 -3.59375 -2.015625 C -3.539062 -1.816406 -3.425781 -1.644531 -3.25 -1.5 C -3.113281 -1.382812 -2.972656 -1.304688 -2.828125 -1.265625 C -2.679688 -1.234375 -2.46875 -1.21875 -2.1875 -1.21875 L 0 -1.21875 L 0 -0.515625 Z M -4.28125 -2.171875 Z M -4.28125 -2.171875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='a2e32e90-f1de-4a37-bebf-4170f2f3c768' overflow='visible'>\n<path d='M -5.734375 -0.609375 L -5.734375 -1.53125 L -1.09375 -4.421875 L -5.734375 -4.421875 L -5.734375 -5.171875 L 0 -5.171875 L 0 -4.296875 L -4.640625 -1.359375 L 0 -1.359375 L 0 -0.609375 Z M -5.734375 -2.84375 Z M -5.734375 -2.84375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='d788cce7-7fbd-4064-a1b3-6dd04b74886c' overflow='visible'>\n<path d='M -4.1875 -0.515625 L -4.1875 -1.21875 L -3.59375 -1.21875 C -3.800781 -1.382812 -3.953125 -1.535156 -4.046875 -1.671875 C -4.203125 -1.898438 -4.28125 -2.160156 -4.28125 -2.453125 C -4.28125 -2.785156 -4.195312 -3.054688 -4.03125 -3.265625 C -3.9375 -3.378906 -3.800781 -3.484375 -3.625 -3.578125 C -3.84375 -3.734375 -4.003906 -3.914062 -4.109375 -4.125 C -4.222656 -4.332031 -4.28125 -4.566406 -4.28125 -4.828125 C -4.28125 -5.390625 -4.078125 -5.773438 -3.671875 -5.984375 C -3.453125 -6.085938 -3.15625 -6.140625 -2.78125 -6.140625 L 0 -6.140625 L 0 -5.421875 L -2.90625 -5.421875 C -3.1875 -5.421875 -3.378906 -5.347656 -3.484375 -5.203125 C -3.585938 -5.066406 -3.640625 -4.898438 -3.640625 -4.703125 C -3.640625 -4.421875 -3.546875 -4.179688 -3.359375 -3.984375 C -3.171875 -3.785156 -2.863281 -3.6875 -2.4375 -3.6875 L 0 -3.6875 L 0 -2.96875 L -2.734375 -2.96875 C -3.015625 -2.96875 -3.222656 -2.9375 -3.359375 -2.875 C -3.546875 -2.757812 -3.640625 -2.554688 -3.640625 -2.265625 C -3.640625 -2.003906 -3.535156 -1.765625 -3.328125 -1.546875 C -3.128906 -1.328125 -2.757812 -1.21875 -2.21875 -1.21875 L 0 -1.21875 L 0 -0.515625 Z M -4.1875 -0.515625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='964e6bdd-5edd-4610-885d-171b6cc0bcd0' overflow='visible'>\n<path d='M -4.28125 -2.265625 C -4.28125 -2.554688 -4.207031 -2.84375 -4.0625 -3.125 C -3.925781 -3.40625 -3.75 -3.617188 -3.53125 -3.765625 C -3.320312 -3.898438 -3.078125 -3.992188 -2.796875 -4.046875 C -2.597656 -4.085938 -2.289062 -4.109375 -1.875 -4.109375 L -1.875 -1.03125 C -1.445312 -1.050781 -1.101562 -1.15625 -0.84375 -1.34375 C -0.59375 -1.53125 -0.46875 -1.816406 -0.46875 -2.203125 C -0.46875 -2.578125 -0.585938 -2.875 -0.828125 -3.09375 C -0.960938 -3.21875 -1.125 -3.304688 -1.3125 -3.359375 L -1.3125 -4.046875 C -1.15625 -4.023438 -0.984375 -3.960938 -0.796875 -3.859375 C -0.609375 -3.753906 -0.457031 -3.640625 -0.34375 -3.515625 C -0.132812 -3.304688 0.0078125 -3.039062 0.09375 -2.71875 C 0.132812 -2.550781 0.15625 -2.363281 0.15625 -2.15625 C 0.15625 -1.632812 -0.03125 -1.191406 -0.40625 -0.828125 C -0.789062 -0.460938 -1.328125 -0.28125 -2.015625 -0.28125 C -2.679688 -0.28125 -3.222656 -0.460938 -3.640625 -0.828125 C -4.066406 -1.191406 -4.28125 -1.671875 -4.28125 -2.265625 Z M -2.421875 -3.375 C -2.734375 -3.351562 -2.976562 -3.289062 -3.15625 -3.1875 C -3.5 -2.988281 -3.671875 -2.664062 -3.671875 -2.21875 C -3.671875 -1.894531 -3.550781 -1.625 -3.3125 -1.40625 C -3.082031 -1.1875 -2.785156 -1.070312 -2.421875 -1.0625 Z M -4.28125 -2.1875 Z M -4.28125 -2.1875 ' style='stroke:none;'/>\n</symbol>\n</g>\n<clipPath id='8688ced5-4ca3-4d26-ba79-2be96d5c06b1'>\n<path d='M 177.566406 22.472656 L 551 22.472656 L 551 836 L 177.566406 836 Z M 177.566406 22.472656 '/>\n</clipPath>\n<clipPath id='ea181e57-1309-4391-b91f-7cd154578091'>\n<path d='M 177.566406 821 L 551 821 L 551 822 L 177.566406 822 Z M 177.566406 821 '/>\n</clipPath>\n<clipPath id='21bdfb35-a5ee-4be6-bfe1-3b389c8f27e1'>\n<path d='M 177.566406 798 L 551 798 L 551 799 L 177.566406 799 Z M 177.566406 798 '/>\n</clipPath>\n<clipPath id='d4a307f6-8c0b-4c5d-997f-b45f85cf858a'>\n<path d='M 177.566406 775 L 551 775 L 551 776 L 177.566406 776 Z M 177.566406 775 '/>\n</clipPath>\n<clipPath id='db97a5d4-f75c-42da-be0b-5a2431c7eeea'>\n<path d='M 177.566406 751 L 551 751 L 551 753 L 177.566406 753 Z M 177.566406 751 '/>\n</clipPath>\n<clipPath id='3c54a704-b477-4ae9-8694-e66fe0f53f0f'>\n<path d='M 177.566406 728 L 551 728 L 551 730 L 177.566406 730 Z M 177.566406 728 '/>\n</clipPath>\n<clipPath id='f033cb94-7529-49c1-971e-6617f162420a'>\n<path d='M 177.566406 705 L 551 705 L 551 707 L 177.566406 707 Z M 177.566406 705 '/>\n</clipPath>\n<clipPath id='5d214a60-59bd-4330-a50c-d547c968a2b4'>\n<path d='M 177.566406 682 L 551 682 L 551 684 L 177.566406 684 Z M 177.566406 682 '/>\n</clipPath>\n<clipPath id='92e0c7dd-6898-4053-9674-ee4b2c447052'>\n<path d='M 177.566406 659 L 551 659 L 551 661 L 177.566406 661 Z M 177.566406 659 '/>\n</clipPath>\n<clipPath id='f4b27b81-b278-46a1-8e0a-964edfbde525'>\n<path d='M 177.566406 636 L 551 636 L 551 637 L 177.566406 637 Z M 177.566406 636 '/>\n</clipPath>\n<clipPath id='28c99b31-4f19-4e9f-95a4-44e1782cf14b'>\n<path d='M 177.566406 613 L 551 613 L 551 614 L 177.566406 614 Z M 177.566406 613 '/>\n</clipPath>\n<clipPath id='e65b7eda-dc5e-43a4-977d-8f86a167b81a'>\n<path d='M 177.566406 590 L 551 590 L 551 591 L 177.566406 591 Z M 177.566406 590 '/>\n</clipPath>\n<clipPath id='8800702d-cbb8-48cc-ac9e-311740ae0b12'>\n<path d='M 177.566406 567 L 551 567 L 551 568 L 177.566406 568 Z M 177.566406 567 '/>\n</clipPath>\n<clipPath id='c54e1c28-06db-43a0-85f0-acaac82007e9'>\n<path d='M 177.566406 544 L 551 544 L 551 545 L 177.566406 545 Z M 177.566406 544 '/>\n</clipPath>\n<clipPath id='f88deb29-67f7-47f8-b492-3adf13113143'>\n<path d='M 177.566406 521 L 551 521 L 551 522 L 177.566406 522 Z M 177.566406 521 '/>\n</clipPath>\n<clipPath id='0e5666c0-980e-4d26-807b-078abe4280a0'>\n<path d='M 177.566406 497 L 551 497 L 551 499 L 177.566406 499 Z M 177.566406 497 '/>\n</clipPath>\n<clipPath id='b35af624-a324-4047-9322-315d95b0dd90'>\n<path d='M 177.566406 474 L 551 474 L 551 476 L 177.566406 476 Z M 177.566406 474 '/>\n</clipPath>\n<clipPath id='1ca23490-7063-485d-9faf-ec987c843042'>\n<path d='M 177.566406 451 L 551 451 L 551 453 L 177.566406 453 Z M 177.566406 451 '/>\n</clipPath>\n<clipPath id='e45e0bc7-33af-49c4-bfa2-b8664585bccb'>\n<path d='M 177.566406 428 L 551 428 L 551 430 L 177.566406 430 Z M 177.566406 428 '/>\n</clipPath>\n<clipPath id='49a803bc-1a7e-426f-9e14-cb3de3e38f60'>\n<path d='M 177.566406 405 L 551 405 L 551 407 L 177.566406 407 Z M 177.566406 405 '/>\n</clipPath>\n<clipPath id='fd9e9157-a5ea-47b3-8597-453b11cb4632'>\n<path d='M 177.566406 382 L 551 382 L 551 383 L 177.566406 383 Z M 177.566406 382 '/>\n</clipPath>\n<clipPath id='50cfc14a-1ca5-4ca6-b756-7d1d38fd6042'>\n<path d='M 177.566406 359 L 551 359 L 551 360 L 177.566406 360 Z M 177.566406 359 '/>\n</clipPath>\n<clipPath id='c6acc125-f608-4d64-b8ca-3bfb6ca12d53'>\n<path d='M 177.566406 336 L 551 336 L 551 337 L 177.566406 337 Z M 177.566406 336 '/>\n</clipPath>\n<clipPath id='7834c12f-8749-4083-badc-dc4a301aef4d'>\n<path d='M 177.566406 313 L 551 313 L 551 314 L 177.566406 314 Z M 177.566406 313 '/>\n</clipPath>\n<clipPath id='4d742c82-1d32-424c-8285-fa35b20aa39e'>\n<path d='M 177.566406 290 L 551 290 L 551 291 L 177.566406 291 Z M 177.566406 290 '/>\n</clipPath>\n<clipPath id='16fa5a7e-f758-4183-b780-ba029a920392'>\n<path d='M 177.566406 266 L 551 266 L 551 268 L 177.566406 268 Z M 177.566406 266 '/>\n</clipPath>\n<clipPath id='5a173510-141e-4c44-85d4-65022dcd4fc8'>\n<path d='M 177.566406 243 L 551 243 L 551 245 L 177.566406 245 Z M 177.566406 243 '/>\n</clipPath>\n<clipPath id='562cfd81-7aca-4d43-83fb-a32a38047c3d'>\n<path d='M 177.566406 220 L 551 220 L 551 222 L 177.566406 222 Z M 177.566406 220 '/>\n</clipPath>\n<clipPath id='4139c31d-e90a-4708-a3ce-1c1598f8d766'>\n<path d='M 177.566406 197 L 551 197 L 551 199 L 177.566406 199 Z M 177.566406 197 '/>\n</clipPath>\n<clipPath id='d736ee5c-0330-4a35-bb82-072fc9b245ff'>\n<path d='M 177.566406 174 L 551 174 L 551 176 L 177.566406 176 Z M 177.566406 174 '/>\n</clipPath>\n<clipPath id='7f8ffcca-e57a-47b1-9ab8-0905dab22ffd'>\n<path d='M 177.566406 151 L 551 151 L 551 153 L 177.566406 153 Z M 177.566406 151 '/>\n</clipPath>\n<clipPath id='0626dd37-d763-49c4-a06e-e6740edaa232'>\n<path d='M 177.566406 128 L 551 128 L 551 129 L 177.566406 129 Z M 177.566406 128 '/>\n</clipPath>\n<clipPath id='0432eb9c-5042-4bda-a3af-849cae2a962a'>\n<path d='M 177.566406 105 L 551 105 L 551 106 L 177.566406 106 Z M 177.566406 105 '/>\n</clipPath>\n<clipPath id='5219ba6a-59b8-441b-8479-5cb6b59e2758'>\n<path d='M 177.566406 82 L 551 82 L 551 83 L 177.566406 83 Z M 177.566406 82 '/>\n</clipPath>\n<clipPath id='aa1e372c-5cf7-43ee-8eb2-b3cc9901abfb'>\n<path d='M 177.566406 59 L 551 59 L 551 60 L 177.566406 60 Z M 177.566406 59 '/>\n</clipPath>\n<clipPath id='8aa50603-24a3-45bf-90b0-7088b1f926ee'>\n<path d='M 177.566406 36 L 551 36 L 551 37 L 177.566406 37 Z M 177.566406 36 '/>\n</clipPath>\n<clipPath id='3cd02c80-092c-453e-aa1a-33cd1b7a0e60'>\n<path d='M 194 22.472656 L 195 22.472656 L 195 836 L 194 836 Z M 194 22.472656 '/>\n</clipPath>\n<clipPath id='2b4b4fea-13ec-437c-a4bd-53be8514524e'>\n<path d='M 267 22.472656 L 269 22.472656 L 269 836 L 267 836 Z M 267 22.472656 '/>\n</clipPath>\n<clipPath id='829f0e55-5554-4b14-9892-75ff9e8e54a4'>\n<path d='M 341 22.472656 L 343 22.472656 L 343 836 L 341 836 Z M 341 22.472656 '/>\n</clipPath>\n<clipPath id='ceb8aa23-9d7c-4106-add7-2b4ea5cd8538'>\n<path d='M 415 22.472656 L 416 22.472656 L 416 836 L 415 836 Z M 415 22.472656 '/>\n</clipPath>\n<clipPath id='9a0685f6-c931-4639-be94-734c35726b25'>\n<path d='M 488 22.472656 L 490 22.472656 L 490 836 L 488 836 Z M 488 22.472656 '/>\n</clipPath>\n<clipPath id='21d0e783-7024-4e95-846e-231138c6e9eb'>\n<path d='M 177.566406 22.472656 L 551 22.472656 L 551 836 L 177.566406 836 Z M 177.566406 22.472656 '/>\n</clipPath>\n</defs>\n<g id='902a6a6a-82f4-407e-a292-cd04d60fe9df'>\n<rect style='fill:rgb(100%,100%,100%);fill-opacity:1;stroke:none;' height='864' width='648' y='0' x='0'/>\n<rect style='fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke:none;' height='864' width='648' y='0' x='0'/>\n<path d='M 0 864 L 648 864 L 648 0 L 0 0 Z M 0 864 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g clip-rule='nonzero' clip-path='url(#8688ced5-4ca3-4d26-ba79-2be96d5c06b1)'>\n<path d='M 177.566406 835.332031 L 550.300781 835.332031 L 550.300781 22.472656 L 177.566406 22.472656 Z M 177.566406 835.332031 ' style='fill-rule:nonzero;fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#ea181e57-1309-4391-b91f-7cd154578091)'>\n<path d='M 177.566406 821.476562 L 550.300781 821.476562 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#21bdfb35-a5ee-4be6-bfe1-3b389c8f27e1)'>\n<path d='M 177.566406 798.382812 L 550.300781 798.382812 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#d4a307f6-8c0b-4c5d-997f-b45f85cf858a)'>\n<path d='M 177.566406 775.289062 L 550.300781 775.289062 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#db97a5d4-f75c-42da-be0b-5a2431c7eeea)'>\n<path d='M 177.566406 752.199219 L 550.300781 752.199219 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#3c54a704-b477-4ae9-8694-e66fe0f53f0f)'>\n<path d='M 177.566406 729.105469 L 550.300781 729.105469 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#f033cb94-7529-49c1-971e-6617f162420a)'>\n<path d='M 177.566406 706.011719 L 550.300781 706.011719 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#5d214a60-59bd-4330-a50c-d547c968a2b4)'>\n<path d='M 177.566406 682.921875 L 550.300781 682.921875 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#92e0c7dd-6898-4053-9674-ee4b2c447052)'>\n<path d='M 177.566406 659.828125 L 550.300781 659.828125 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#f4b27b81-b278-46a1-8e0a-964edfbde525)'>\n<path d='M 177.566406 636.734375 L 550.300781 636.734375 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#28c99b31-4f19-4e9f-95a4-44e1782cf14b)'>\n<path d='M 177.566406 613.644531 L 550.300781 613.644531 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#e65b7eda-dc5e-43a4-977d-8f86a167b81a)'>\n<path d='M 177.566406 590.550781 L 550.300781 590.550781 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#8800702d-cbb8-48cc-ac9e-311740ae0b12)'>\n<path d='M 177.566406 567.457031 L 550.300781 567.457031 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#c54e1c28-06db-43a0-85f0-acaac82007e9)'>\n<path d='M 177.566406 544.363281 L 550.300781 544.363281 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#f88deb29-67f7-47f8-b492-3adf13113143)'>\n<path d='M 177.566406 521.273438 L 550.300781 521.273438 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#0e5666c0-980e-4d26-807b-078abe4280a0)'>\n<path d='M 177.566406 498.179688 L 550.300781 498.179688 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#b35af624-a324-4047-9322-315d95b0dd90)'>\n<path d='M 177.566406 475.085938 L 550.300781 475.085938 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#1ca23490-7063-485d-9faf-ec987c843042)'>\n<path d='M 177.566406 451.996094 L 550.300781 451.996094 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#e45e0bc7-33af-49c4-bfa2-b8664585bccb)'>\n<path d='M 177.566406 428.902344 L 550.300781 428.902344 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#49a803bc-1a7e-426f-9e14-cb3de3e38f60)'>\n<path d='M 177.566406 405.808594 L 550.300781 405.808594 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#fd9e9157-a5ea-47b3-8597-453b11cb4632)'>\n<path d='M 177.566406 382.71875 L 550.300781 382.71875 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#50cfc14a-1ca5-4ca6-b756-7d1d38fd6042)'>\n<path d='M 177.566406 359.625 L 550.300781 359.625 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#c6acc125-f608-4d64-b8ca-3bfb6ca12d53)'>\n<path d='M 177.566406 336.53125 L 550.300781 336.53125 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#7834c12f-8749-4083-badc-dc4a301aef4d)'>\n<path d='M 177.566406 313.4375 L 550.300781 313.4375 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#4d742c82-1d32-424c-8285-fa35b20aa39e)'>\n<path d='M 177.566406 290.347656 L 550.300781 290.347656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#16fa5a7e-f758-4183-b780-ba029a920392)'>\n<path d='M 177.566406 267.253906 L 550.300781 267.253906 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#5a173510-141e-4c44-85d4-65022dcd4fc8)'>\n<path d='M 177.566406 244.160156 L 550.300781 244.160156 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#562cfd81-7aca-4d43-83fb-a32a38047c3d)'>\n<path d='M 177.566406 221.070312 L 550.300781 221.070312 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#4139c31d-e90a-4708-a3ce-1c1598f8d766)'>\n<path d='M 177.566406 197.976562 L 550.300781 197.976562 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#d736ee5c-0330-4a35-bb82-072fc9b245ff)'>\n<path d='M 177.566406 174.882812 L 550.300781 174.882812 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#7f8ffcca-e57a-47b1-9ab8-0905dab22ffd)'>\n<path d='M 177.566406 151.792969 L 550.300781 151.792969 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#0626dd37-d763-49c4-a06e-e6740edaa232)'>\n<path d='M 177.566406 128.699219 L 550.300781 128.699219 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#0432eb9c-5042-4bda-a3af-849cae2a962a)'>\n<path d='M 177.566406 105.605469 L 550.300781 105.605469 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#5219ba6a-59b8-441b-8479-5cb6b59e2758)'>\n<path d='M 177.566406 82.511719 L 550.300781 82.511719 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#aa1e372c-5cf7-43ee-8eb2-b3cc9901abfb)'>\n<path d='M 177.566406 59.421875 L 550.300781 59.421875 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#8aa50603-24a3-45bf-90b0-7088b1f926ee)'>\n<path d='M 177.566406 36.328125 L 550.300781 36.328125 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#3cd02c80-092c-453e-aa1a-33cd1b7a0e60)'>\n<path d='M 194.507812 835.332031 L 194.507812 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#2b4b4fea-13ec-437c-a4bd-53be8514524e)'>\n<path d='M 268.171875 835.332031 L 268.171875 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#829f0e55-5554-4b14-9892-75ff9e8e54a4)'>\n<path d='M 341.835938 835.332031 L 341.835938 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#ceb8aa23-9d7c-4106-add7-2b4ea5cd8538)'>\n<path d='M 415.496094 835.332031 L 415.496094 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#9a0685f6-c931-4639-be94-734c35726b25)'>\n<path d='M 489.160156 835.332031 L 489.160156 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<path d='M 194.507812 821.476562 L 356.566406 821.476562 L 356.566406 811.085938 L 194.507812 811.085938 Z M 194.507812 821.476562 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 831.867188 L 238.707031 831.867188 L 238.707031 821.476562 L 194.507812 821.476562 Z M 194.507812 831.867188 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 798.382812 L 356.566406 798.382812 L 356.566406 787.992188 L 194.507812 787.992188 Z M 194.507812 798.382812 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 808.773438 L 238.707031 808.773438 L 238.707031 798.382812 L 194.507812 798.382812 Z M 194.507812 808.773438 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 775.289062 L 341.832031 775.289062 L 341.832031 764.898438 L 194.507812 764.898438 Z M 194.507812 775.289062 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 785.683594 L 268.171875 785.683594 L 268.171875 775.292969 L 194.507812 775.292969 Z M 194.507812 785.683594 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 752.199219 L 282.902344 752.199219 L 282.902344 741.808594 L 194.507812 741.808594 Z M 194.507812 752.199219 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 762.589844 L 327.101562 762.589844 L 327.101562 752.199219 L 194.507812 752.199219 Z M 194.507812 762.589844 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 729.105469 L 356.566406 729.105469 L 356.566406 718.714844 L 194.507812 718.714844 Z M 194.507812 729.105469 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 739.496094 L 253.4375 739.496094 L 253.4375 729.105469 L 194.507812 729.105469 Z M 194.507812 739.496094 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 706.011719 L 371.296875 706.011719 L 371.296875 695.621094 L 194.507812 695.621094 Z M 194.507812 706.011719 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 716.40625 L 238.707031 716.40625 L 238.707031 706.015625 L 194.507812 706.015625 Z M 194.507812 716.40625 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 682.921875 L 371.296875 682.921875 L 371.296875 672.53125 L 194.507812 672.53125 Z M 194.507812 682.921875 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 693.3125 L 238.707031 693.3125 L 238.707031 682.921875 L 194.507812 682.921875 Z M 194.507812 693.3125 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 659.828125 L 253.4375 659.828125 L 253.4375 649.4375 L 194.507812 649.4375 Z M 194.507812 659.828125 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 670.21875 L 356.566406 670.21875 L 356.566406 659.828125 L 194.507812 659.828125 Z M 194.507812 670.21875 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 636.734375 L 371.296875 636.734375 L 371.296875 626.34375 L 194.507812 626.34375 Z M 194.507812 636.734375 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 647.125 L 238.707031 647.125 L 238.707031 636.734375 L 194.507812 636.734375 Z M 194.507812 647.125 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 613.644531 L 356.566406 613.644531 L 356.566406 603.253906 L 194.507812 603.253906 Z M 194.507812 613.644531 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 624.035156 L 253.4375 624.035156 L 253.4375 613.644531 L 194.507812 613.644531 Z M 194.507812 624.035156 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 590.550781 L 297.636719 590.550781 L 297.636719 580.160156 L 194.507812 580.160156 Z M 194.507812 590.550781 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 600.941406 L 312.367188 600.941406 L 312.367188 590.550781 L 194.507812 590.550781 Z M 194.507812 600.941406 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 567.457031 L 312.367188 567.457031 L 312.367188 557.066406 L 194.507812 557.066406 Z M 194.507812 567.457031 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 577.847656 L 341.832031 577.847656 L 341.832031 567.457031 L 194.507812 567.457031 Z M 194.507812 577.847656 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 544.363281 L 386.03125 544.363281 L 386.03125 533.972656 L 194.507812 533.972656 Z M 194.507812 544.363281 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 554.757812 L 282.902344 554.757812 L 282.902344 544.367188 L 194.507812 544.367188 Z M 194.507812 554.757812 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 521.273438 L 430.230469 521.273438 L 430.230469 510.882812 L 194.507812 510.882812 Z M 194.507812 521.273438 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 531.664062 L 238.707031 531.664062 L 238.707031 521.273438 L 194.507812 521.273438 Z M 194.507812 531.664062 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 498.179688 L 386.03125 498.179688 L 386.03125 487.789062 L 194.507812 487.789062 Z M 194.507812 498.179688 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 508.570312 L 282.902344 508.570312 L 282.902344 498.179688 L 194.507812 498.179688 Z M 194.507812 508.570312 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 475.085938 L 268.171875 475.085938 L 268.171875 464.695312 L 194.507812 464.695312 Z M 194.507812 475.085938 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 485.480469 L 400.761719 485.480469 L 400.761719 475.089844 L 194.507812 475.089844 Z M 194.507812 485.480469 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 451.996094 L 415.496094 451.996094 L 415.496094 441.605469 L 194.507812 441.605469 Z M 194.507812 451.996094 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 462.386719 L 253.4375 462.386719 L 253.4375 451.996094 L 194.507812 451.996094 Z M 194.507812 462.386719 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 428.902344 L 356.566406 428.902344 L 356.566406 418.511719 L 194.507812 418.511719 Z M 194.507812 428.902344 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 439.292969 L 312.367188 439.292969 L 312.367188 428.902344 L 194.507812 428.902344 Z M 194.507812 439.292969 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 405.808594 L 430.230469 405.808594 L 430.230469 395.417969 L 194.507812 395.417969 Z M 194.507812 405.808594 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 416.203125 L 238.707031 416.203125 L 238.707031 405.8125 L 194.507812 405.8125 Z M 194.507812 416.203125 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 382.71875 L 297.636719 382.71875 L 297.636719 372.328125 L 194.507812 372.328125 Z M 194.507812 382.71875 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 393.109375 L 371.296875 393.109375 L 371.296875 382.71875 L 194.507812 382.71875 Z M 194.507812 393.109375 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 359.625 L 341.832031 359.625 L 341.832031 349.234375 L 194.507812 349.234375 Z M 194.507812 359.625 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 370.015625 L 327.101562 370.015625 L 327.101562 359.625 L 194.507812 359.625 Z M 194.507812 370.015625 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 336.53125 L 238.707031 336.53125 L 238.707031 326.140625 L 194.507812 326.140625 Z M 194.507812 336.53125 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 346.921875 L 430.230469 346.921875 L 430.230469 336.53125 L 194.507812 336.53125 Z M 194.507812 346.921875 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 313.4375 L 341.832031 313.4375 L 341.832031 303.046875 L 194.507812 303.046875 Z M 194.507812 313.4375 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 323.832031 L 327.101562 323.832031 L 327.101562 313.441406 L 194.507812 313.441406 Z M 194.507812 323.832031 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 290.347656 L 371.296875 290.347656 L 371.296875 279.957031 L 194.507812 279.957031 Z M 194.507812 290.347656 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 300.738281 L 297.636719 300.738281 L 297.636719 290.347656 L 194.507812 290.347656 Z M 194.507812 300.738281 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 267.253906 L 253.4375 267.253906 L 253.4375 256.863281 L 194.507812 256.863281 Z M 194.507812 267.253906 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 277.644531 L 415.496094 277.644531 L 415.496094 267.253906 L 194.507812 267.253906 Z M 194.507812 277.644531 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 244.160156 L 415.496094 244.160156 L 415.496094 233.769531 L 194.507812 233.769531 Z M 194.507812 244.160156 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 254.554688 L 253.4375 254.554688 L 253.4375 244.164062 L 194.507812 244.164062 Z M 194.507812 254.554688 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 221.070312 L 400.761719 221.070312 L 400.761719 210.679688 L 194.507812 210.679688 Z M 194.507812 221.070312 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 231.460938 L 327.101562 231.460938 L 327.101562 221.070312 L 194.507812 221.070312 Z M 194.507812 231.460938 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 197.976562 L 415.496094 197.976562 L 415.496094 187.585938 L 194.507812 187.585938 Z M 194.507812 197.976562 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 208.367188 L 312.367188 208.367188 L 312.367188 197.976562 L 194.507812 197.976562 Z M 194.507812 208.367188 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 174.882812 L 268.171875 174.882812 L 268.171875 164.492188 L 194.507812 164.492188 Z M 194.507812 174.882812 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 185.277344 L 459.695312 185.277344 L 459.695312 174.886719 L 194.507812 174.886719 Z M 194.507812 185.277344 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 151.792969 L 268.171875 151.792969 L 268.171875 141.402344 L 194.507812 141.402344 Z M 194.507812 151.792969 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 162.183594 L 459.695312 162.183594 L 459.695312 151.792969 L 194.507812 151.792969 Z M 194.507812 162.183594 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 128.699219 L 253.4375 128.699219 L 253.4375 118.308594 L 194.507812 118.308594 Z M 194.507812 128.699219 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 139.089844 L 474.425781 139.089844 L 474.425781 128.699219 L 194.507812 128.699219 Z M 194.507812 139.089844 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 105.605469 L 386.03125 105.605469 L 386.03125 95.214844 L 194.507812 95.214844 Z M 194.507812 105.605469 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 115.996094 L 341.832031 115.996094 L 341.832031 105.605469 L 194.507812 105.605469 Z M 194.507812 115.996094 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 82.511719 L 341.832031 82.511719 L 341.832031 72.121094 L 194.507812 72.121094 Z M 194.507812 82.511719 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 92.90625 L 444.960938 92.90625 L 444.960938 82.515625 L 194.507812 82.515625 Z M 194.507812 92.90625 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 59.421875 L 533.355469 59.421875 L 533.355469 49.03125 L 194.507812 49.03125 Z M 194.507812 59.421875 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 69.8125 L 253.4375 69.8125 L 253.4375 59.421875 L 194.507812 59.421875 Z M 194.507812 69.8125 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 194.507812 36.328125 L 459.695312 36.328125 L 459.695312 25.9375 L 194.507812 25.9375 Z M 194.507812 36.328125 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<path d='M 194.507812 46.71875 L 327.101562 46.71875 L 327.101562 36.328125 L 194.507812 36.328125 Z M 194.507812 46.71875 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<g clip-rule='nonzero' clip-path='url(#21d0e783-7024-4e95-846e-231138c6e9eb)'>\n<path d='M 177.566406 835.332031 L 550.300781 835.332031 L 550.300781 22.472656 L 177.566406 22.472656 Z M 177.566406 835.332031 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='823.984375' x='120.621094' xlink:href='#0373f860-79fb-49c3-aba0-0e9aac3a40fa'/>\n<use y='823.984375' x='125.67627' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='823.984375' x='129.569336' xlink:href='#cab96616-0e1a-49dc-99de-457a3e10394b'/>\n<use y='823.984375' x='133.069336' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='823.984375' x='134.624512' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='823.984375' x='138.124512' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='823.984375' x='140.069336' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='823.984375' x='143.962402' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='823.984375' x='145.907227' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='823.984375' x='147.852051' xlink:href='#d4e43642-a0de-4bdc-af18-ff27a755bb94'/>\n<use y='823.984375' x='151.352051' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='823.984375' x='155.245117' xlink:href='#bdd46fbd-d377-4bba-91c8-3814d2922bd5'/>\n<use y='823.984375' x='158.745117' xlink:href='#091b1d69-7c75-46b6-825f-da4b0b842949'/>\n<use y='823.984375' x='162.245117' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='823.984375' x='165.745117' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='823.984375' x='169.638184' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='800.890625' x='104.660156' xlink:href='#6a3a0a32-3bcc-44d9-8288-0e6d0c801f40'/>\n<use y='800.890625' x='111.26709' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='800.890625' x='115.160156' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='800.890625' x='118.660156' xlink:href='#1da6ad6a-cbc9-46ad-bddc-b5660b76e4c7'/>\n<use y='800.890625' x='122.553223' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='800.890625' x='124.108398' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='800.890625' x='128.001465' xlink:href='#3093d11f-7144-4487-8aed-c2dd55c92efa'/>\n<use y='800.890625' x='131.894531' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='800.890625' x='133.839355' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='800.890625' x='137.732422' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='800.890625' x='141.625488' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='800.890625' x='143.570312' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='800.890625' x='147.463379' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='800.890625' x='149.408203' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='800.890625' x='151.353027' xlink:href='#49d7c5ef-7a75-4d2b-8182-60e011b9efda'/>\n<use y='800.890625' x='156.021973' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='800.890625' x='159.915039' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='800.890625' x='163.808105' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='800.890625' x='166.13916' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='800.890625' x='170.032227' xlink:href='#77d9585b-31e4-4e0c-8c9c-a8b207187d3a'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='777.796875' x='117.492188' xlink:href='#f5647a9d-2871-49b0-9631-fc19a67066b3'/>\n<use y='777.796875' x='121.385254' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='777.796875' x='125.27832' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='777.796875' x='129.171387' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='777.796875' x='131.116211' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='777.796875' x='135.009277' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='777.796875' x='136.954102' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='777.796875' x='138.898926' xlink:href='#1fdff9cd-6269-45e2-8bed-b34b2792b662'/>\n<use y='777.796875' x='143.567871' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='777.796875' x='147.460938' xlink:href='#2fa6a5ab-c667-4a45-88df-f391ba868cec'/>\n<use y='777.796875' x='151.354004' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='777.796875' x='153.298828' xlink:href='#1da6ad6a-cbc9-46ad-bddc-b5660b76e4c7'/>\n<use y='777.796875' x='157.191895' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='777.796875' x='159.136719' xlink:href='#c918bbc5-1fab-432a-8fb7-2bea63973fb2'/>\n<use y='777.796875' x='163.805664' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='777.796875' x='167.69873' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='777.796875' x='170.029785' xlink:href='#091b1d69-7c75-46b6-825f-da4b0b842949'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='754.707031' x='129.953125' xlink:href='#24baa935-3afb-42ca-b4ef-1ffc8c841b87'/>\n<use y='754.707031' x='133.846191' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='754.707031' x='135.791016' xlink:href='#1da6ad6a-cbc9-46ad-bddc-b5660b76e4c7'/>\n<use y='754.707031' x='139.684082' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='754.707031' x='141.628906' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='754.707031' x='145.521973' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='754.707031' x='147.466797' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='754.707031' x='149.411621' xlink:href='#07ea776c-aee7-462b-8d45-7502d730dcff'/>\n<use y='754.707031' x='154.466797' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='754.707031' x='158.359863' xlink:href='#16022362-e0a3-4b9e-a0c6-392d55078d8b'/>\n<use y='754.707031' x='163.415039' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='754.707031' x='167.308105' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='754.707031' x='169.63916' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='731.613281' x='93.382812' xlink:href='#9932acbc-8b99-4f31-be5f-e87ed7486a2e'/>\n<use y='731.613281' x='98.051758' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='731.613281' x='100.382812' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='731.613281' x='104.275879' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='731.613281' x='108.168945' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='731.613281' x='112.062012' xlink:href='#16022362-e0a3-4b9e-a0c6-392d55078d8b'/>\n<use y='731.613281' x='117.117188' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='731.613281' x='121.010254' xlink:href='#77d9585b-31e4-4e0c-8c9c-a8b207187d3a'/>\n<use y='731.613281' x='124.510254' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='731.613281' x='126.455078' xlink:href='#1fdff9cd-6269-45e2-8bed-b34b2792b662'/>\n<use y='731.613281' x='131.124023' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='731.613281' x='133.068848' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='731.613281' x='135.013672' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='731.613281' x='138.906738' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='731.613281' x='140.851562' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='731.613281' x='142.796387' xlink:href='#9932acbc-8b99-4f31-be5f-e87ed7486a2e'/>\n<use y='731.613281' x='147.465332' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='731.613281' x='151.358398' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='731.613281' x='153.303223' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='731.613281' x='155.248047' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='731.613281' x='159.141113' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='731.613281' x='161.472168' xlink:href='#77d9585b-31e4-4e0c-8c9c-a8b207187d3a'/>\n<use y='731.613281' x='164.972168' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='731.613281' x='166.916992' xlink:href='#1fdff9cd-6269-45e2-8bed-b34b2792b662'/>\n<use y='731.613281' x='171.585938' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='708.519531' x='127.625' xlink:href='#d5cf4064-f02e-44b4-b47e-bb8c182a8372'/>\n<use y='708.519531' x='132.680176' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='708.519531' x='134.235352' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='708.519531' x='138.128418' xlink:href='#77d9585b-31e4-4e0c-8c9c-a8b207187d3a'/>\n<use y='708.519531' x='141.628418' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='708.519531' x='143.573242' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='708.519531' x='147.466309' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='708.519531' x='149.411133' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='708.519531' x='151.355957' xlink:href='#9932acbc-8b99-4f31-be5f-e87ed7486a2e'/>\n<use y='708.519531' x='156.024902' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='708.519531' x='159.917969' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='708.519531' x='161.862793' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='708.519531' x='163.807617' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='708.519531' x='167.700684' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='708.519531' x='170.031738' xlink:href='#77d9585b-31e4-4e0c-8c9c-a8b207187d3a'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='685.429688' x='87.5625' xlink:href='#d5cf4064-f02e-44b4-b47e-bb8c182a8372'/>\n<use y='685.429688' x='92.617676' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='685.429688' x='96.510742' xlink:href='#0dc9cbb1-8da9-4eff-b854-b2296c9c9035'/>\n<use y='685.429688' x='102.341797' xlink:href='#0dc9cbb1-8da9-4eff-b854-b2296c9c9035'/>\n<use y='685.429688' x='108.172852' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='685.429688' x='112.065918' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='685.429688' x='114.396973' xlink:href='#bdd46fbd-d377-4bba-91c8-3814d2922bd5'/>\n<use y='685.429688' x='117.896973' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='685.429688' x='119.452148' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='685.429688' x='123.345215' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='685.429688' x='124.900391' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='685.429688' x='126.845215' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='685.429688' x='130.738281' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='685.429688' x='132.683105' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='685.429688' x='134.62793' xlink:href='#7610a7f7-8be4-4668-99a4-528f7d39ba3f'/>\n<use y='685.429688' x='140.458984' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='685.429688' x='144.352051' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='685.429688' x='148.245117' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='685.429688' x='150.189941' xlink:href='#3093d11f-7144-4487-8aed-c2dd55c92efa'/>\n<use y='685.429688' x='154.083008' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='685.429688' x='157.976074' xlink:href='#0dc9cbb1-8da9-4eff-b854-b2296c9c9035'/>\n<use y='685.429688' x='163.807129' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='685.429688' x='167.700195' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='685.429688' x='170.03125' xlink:href='#77d9585b-31e4-4e0c-8c9c-a8b207187d3a'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='662.335938' x='101.550781' xlink:href='#84ec0c58-94e7-44be-bbb5-f16a483ad50a'/>\n<use y='662.335938' x='106.219727' xlink:href='#0dc9cbb1-8da9-4eff-b854-b2296c9c9035'/>\n<use y='662.335938' x='112.050781' xlink:href='#c00721a7-52a3-4489-a759-7de6cac3a30d'/>\n<use y='662.335938' x='115.943848' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='662.335938' x='119.836914' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='662.335938' x='122.167969' xlink:href='#bdd46fbd-d377-4bba-91c8-3814d2922bd5'/>\n<use y='662.335938' x='125.667969' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='662.335938' x='129.561035' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='662.335938' x='133.454102' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='662.335938' x='137.347168' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='662.335938' x='139.678223' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='662.335938' x='143.571289' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='662.335938' x='145.516113' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='662.335938' x='149.40918' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='662.335938' x='151.354004' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='662.335938' x='153.298828' xlink:href='#9932acbc-8b99-4f31-be5f-e87ed7486a2e'/>\n<use y='662.335938' x='157.967773' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='662.335938' x='160.298828' xlink:href='#77d9585b-31e4-4e0c-8c9c-a8b207187d3a'/>\n<use y='662.335938' x='163.798828' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='662.335938' x='167.691895' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='662.335938' x='171.584961' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='639.242188' x='92.210938' xlink:href='#84ec0c58-94e7-44be-bbb5-f16a483ad50a'/>\n<use y='639.242188' x='96.879883' xlink:href='#0dc9cbb1-8da9-4eff-b854-b2296c9c9035'/>\n<use y='639.242188' x='102.710938' xlink:href='#c00721a7-52a3-4489-a759-7de6cac3a30d'/>\n<use y='639.242188' x='106.604004' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='639.242188' x='110.49707' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='639.242188' x='112.828125' xlink:href='#bdd46fbd-d377-4bba-91c8-3814d2922bd5'/>\n<use y='639.242188' x='116.328125' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='639.242188' x='120.221191' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='639.242188' x='124.114258' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='639.242188' x='128.007324' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='639.242188' x='130.338379' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='639.242188' x='134.231445' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='639.242188' x='136.17627' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='639.242188' x='140.069336' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='639.242188' x='142.01416' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='639.242188' x='143.958984' xlink:href='#1fdff9cd-6269-45e2-8bed-b34b2792b662'/>\n<use y='639.242188' x='148.62793' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='639.242188' x='152.520996' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='639.242188' x='156.414062' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='639.242188' x='159.914062' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='639.242188' x='163.807129' xlink:href='#0dc9cbb1-8da9-4eff-b854-b2296c9c9035'/>\n<use y='639.242188' x='169.638184' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='616.152344' x='100.769531' xlink:href='#84ec0c58-94e7-44be-bbb5-f16a483ad50a'/>\n<use y='616.152344' x='105.438477' xlink:href='#0dc9cbb1-8da9-4eff-b854-b2296c9c9035'/>\n<use y='616.152344' x='111.269531' xlink:href='#c00721a7-52a3-4489-a759-7de6cac3a30d'/>\n<use y='616.152344' x='115.162598' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='616.152344' x='119.055664' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='616.152344' x='121.386719' xlink:href='#bdd46fbd-d377-4bba-91c8-3814d2922bd5'/>\n<use y='616.152344' x='124.886719' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='616.152344' x='128.779785' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='616.152344' x='132.672852' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='616.152344' x='136.565918' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='616.152344' x='138.896973' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='616.152344' x='142.790039' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='616.152344' x='144.734863' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='616.152344' x='148.62793' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='616.152344' x='150.572754' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='616.152344' x='152.517578' xlink:href='#61e31a73-cdee-4d48-be2c-f33e2453bdf4'/>\n<use y='616.152344' x='157.186523' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='616.152344' x='161.07959' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='616.152344' x='162.634766' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='616.152344' x='164.189941' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='616.152344' x='168.083008' xlink:href='#415a53e9-54f6-46d5-9b86-9f2d483a23ee'/>\n<use y='616.152344' x='169.638184' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='593.058594' x='121.777344' xlink:href='#5062135a-1b6b-4c59-ae8d-4c5af49dcc8d'/>\n<use y='593.058594' x='126.053223' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='593.058594' x='129.946289' xlink:href='#16022362-e0a3-4b9e-a0c6-392d55078d8b'/>\n<use y='593.058594' x='135.001465' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='593.058594' x='138.894531' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='593.058594' x='142.394531' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='593.058594' x='146.287598' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='593.058594' x='150.180664' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='593.058594' x='154.07373' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='593.058594' x='156.018555' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='593.058594' x='159.911621' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='593.058594' x='161.856445' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='593.058594' x='163.80127' xlink:href='#20f10433-62cf-4427-9d4a-66c5c001be2b'/>\n<use y='593.058594' x='167.694336' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='593.058594' x='169.63916' xlink:href='#1da6ad6a-cbc9-46ad-bddc-b5660b76e4c7'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='569.964844' x='111.675781' xlink:href='#c918bbc5-1fab-432a-8fb7-2bea63973fb2'/>\n<use y='569.964844' x='116.344727' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='569.964844' x='120.237793' xlink:href='#16022362-e0a3-4b9e-a0c6-392d55078d8b'/>\n<use y='569.964844' x='125.292969' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='569.964844' x='129.186035' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='569.964844' x='130.741211' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='569.964844' x='132.296387' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='569.964844' x='134.241211' xlink:href='#1fdff9cd-6269-45e2-8bed-b34b2792b662'/>\n<use y='569.964844' x='138.910156' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='569.964844' x='140.85498' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='569.964844' x='143.186035' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='569.964844' x='147.079102' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='569.964844' x='150.972168' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='569.964844' x='152.916992' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='569.964844' x='154.861816' xlink:href='#9932acbc-8b99-4f31-be5f-e87ed7486a2e'/>\n<use y='569.964844' x='159.530762' xlink:href='#faf9c806-cffc-48dc-8fa4-0e89266f3dca'/>\n<use y='569.964844' x='164.199707' xlink:href='#7ed5e8ce-843d-470d-a211-5a4659c90d39'/>\n<use y='569.964844' x='169.254883' xlink:href='#5062135a-1b6b-4c59-ae8d-4c5af49dcc8d'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='546.871094' x='129.175781' xlink:href='#f5647a9d-2871-49b0-9631-fc19a67066b3'/>\n<use y='546.871094' x='133.068848' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='546.871094' x='136.961914' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='546.871094' x='140.85498' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='546.871094' x='142.799805' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='546.871094' x='146.692871' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='546.871094' x='148.637695' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='546.871094' x='150.58252' xlink:href='#15f4a5d1-e31f-45ed-8218-3f8fc09d4b20'/>\n<use y='546.871094' x='154.858398' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='546.871094' x='158.751465' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='546.871094' x='160.306641' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='546.871094' x='163.806641' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='546.871094' x='167.699707' xlink:href='#0dc9cbb1-8da9-4eff-b854-b2296c9c9035'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='523.78125' x='124.507812' xlink:href='#9932acbc-8b99-4f31-be5f-e87ed7486a2e'/>\n<use y='523.78125' x='129.176758' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='523.78125' x='133.069824' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='523.78125' x='136.962891' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='523.78125' x='138.518066' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='523.78125' x='142.411133' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='523.78125' x='144.355957' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='523.78125' x='148.249023' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='523.78125' x='150.193848' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='523.78125' x='152.138672' xlink:href='#7610a7f7-8be4-4668-99a4-528f7d39ba3f'/>\n<use y='523.78125' x='157.969727' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='523.78125' x='161.862793' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='523.78125' x='164.193848' xlink:href='#091b1d69-7c75-46b6-825f-da4b0b842949'/>\n<use y='523.78125' x='167.693848' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='523.78125' x='171.586914' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='500.6875' x='98.835938' xlink:href='#84ec0c58-94e7-44be-bbb5-f16a483ad50a'/>\n<use y='500.6875' x='103.504883' xlink:href='#0dc9cbb1-8da9-4eff-b854-b2296c9c9035'/>\n<use y='500.6875' x='109.335938' xlink:href='#c00721a7-52a3-4489-a759-7de6cac3a30d'/>\n<use y='500.6875' x='113.229004' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='500.6875' x='117.12207' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='500.6875' x='119.453125' xlink:href='#bdd46fbd-d377-4bba-91c8-3814d2922bd5'/>\n<use y='500.6875' x='122.953125' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='500.6875' x='126.846191' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='500.6875' x='130.739258' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='500.6875' x='134.632324' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='500.6875' x='136.963379' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='500.6875' x='140.856445' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='500.6875' x='142.80127' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='500.6875' x='146.694336' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='500.6875' x='148.63916' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='500.6875' x='150.583984' xlink:href='#15f4a5d1-e31f-45ed-8218-3f8fc09d4b20'/>\n<use y='500.6875' x='154.859863' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='500.6875' x='158.75293' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='500.6875' x='160.308105' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='500.6875' x='163.808105' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='500.6875' x='167.701172' xlink:href='#0dc9cbb1-8da9-4eff-b854-b2296c9c9035'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='477.59375' x='63.410156' xlink:href='#a8f15154-4753-4c2e-889a-bf0f8eab37f5'/>\n<use y='477.59375' x='68.85498' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='477.59375' x='71.186035' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='477.59375' x='75.079102' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='477.59375' x='78.972168' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='477.59375' x='80.916992' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='477.59375' x='82.861816' xlink:href='#faf9c806-cffc-48dc-8fa4-0e89266f3dca'/>\n<use y='477.59375' x='87.530762' xlink:href='#cab96616-0e1a-49dc-99de-457a3e10394b'/>\n<use y='477.59375' x='91.030762' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='477.59375' x='94.923828' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='477.59375' x='98.816895' xlink:href='#2fa6a5ab-c667-4a45-88df-f391ba868cec'/>\n<use y='477.59375' x='102.709961' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='477.59375' x='106.603027' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='477.59375' x='108.547852' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='477.59375' x='112.440918' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='477.59375' x='114.385742' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='477.59375' x='116.330566' xlink:href='#d5cf4064-f02e-44b4-b47e-bb8c182a8372'/>\n<use y='477.59375' x='121.385742' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='477.59375' x='125.278809' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='477.59375' x='126.833984' xlink:href='#2fa6a5ab-c667-4a45-88df-f391ba868cec'/>\n<use y='477.59375' x='130.727051' xlink:href='#0dc9cbb1-8da9-4eff-b854-b2296c9c9035'/>\n<use y='477.59375' x='136.558105' xlink:href='#c00721a7-52a3-4489-a759-7de6cac3a30d'/>\n<use y='477.59375' x='140.451172' xlink:href='#2fa6a5ab-c667-4a45-88df-f391ba868cec'/>\n<use y='477.59375' x='144.344238' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='477.59375' x='147.844238' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='477.59375' x='149.789062' xlink:href='#faf9c806-cffc-48dc-8fa4-0e89266f3dca'/>\n<use y='477.59375' x='154.458008' xlink:href='#cab96616-0e1a-49dc-99de-457a3e10394b'/>\n<use y='477.59375' x='157.958008' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='477.59375' x='161.851074' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='477.59375' x='165.744141' xlink:href='#2fa6a5ab-c667-4a45-88df-f391ba868cec'/>\n<use y='477.59375' x='169.637207' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='454.503906' x='128.003906' xlink:href='#07ea776c-aee7-462b-8d45-7502d730dcff'/>\n<use y='454.503906' x='133.059082' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='454.503906' x='136.952148' xlink:href='#16022362-e0a3-4b9e-a0c6-392d55078d8b'/>\n<use y='454.503906' x='142.007324' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='454.503906' x='145.900391' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='454.503906' x='148.231445' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='454.503906' x='152.124512' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='454.503906' x='154.069336' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='454.503906' x='157.962402' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='454.503906' x='159.907227' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='454.503906' x='161.852051' xlink:href='#f5647a9d-2871-49b0-9631-fc19a67066b3'/>\n<use y='454.503906' x='165.745117' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='454.503906' x='169.638184' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='431.410156' x='132.679688' xlink:href='#7610a7f7-8be4-4668-99a4-528f7d39ba3f'/>\n<use y='431.410156' x='138.510742' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='431.410156' x='142.403809' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='431.410156' x='144.734863' xlink:href='#091b1d69-7c75-46b6-825f-da4b0b842949'/>\n<use y='431.410156' x='148.234863' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='431.410156' x='152.12793' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='431.410156' x='154.072754' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='431.410156' x='156.017578' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='431.410156' x='159.910645' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='431.410156' x='161.855469' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='431.410156' x='163.800293' xlink:href='#498ee0e9-30bd-433e-a432-1fddd37c32a1'/>\n<use y='431.410156' x='167.693359' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='431.410156' x='169.638184' xlink:href='#1da6ad6a-cbc9-46ad-bddc-b5660b76e4c7'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='408.316406' x='60.714844' xlink:href='#7610a7f7-8be4-4668-99a4-528f7d39ba3f'/>\n<use y='408.316406' x='66.545898' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='408.316406' x='70.438965' xlink:href='#bdd46fbd-d377-4bba-91c8-3814d2922bd5'/>\n<use y='408.316406' x='73.938965' xlink:href='#1da6ad6a-cbc9-46ad-bddc-b5660b76e4c7'/>\n<use y='408.316406' x='77.832031' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='408.316406' x='81.725098' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='408.316406' x='85.618164' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='408.316406' x='87.17334' xlink:href='#bdd46fbd-d377-4bba-91c8-3814d2922bd5'/>\n<use y='408.316406' x='90.67334' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='408.316406' x='94.17334' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='408.316406' x='96.118164' xlink:href='#c918bbc5-1fab-432a-8fb7-2bea63973fb2'/>\n<use y='408.316406' x='100.787109' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='408.316406' x='102.342285' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='408.316406' x='106.235352' xlink:href='#1862368e-e783-4c5e-9649-e2093e4fd672'/>\n<use y='408.316406' x='109.735352' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='408.316406' x='113.628418' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='408.316406' x='115.573242' xlink:href='#9826757d-31e9-4d33-9633-ddb1b135594a'/>\n<use y='408.316406' x='117.904297' xlink:href='#7610a7f7-8be4-4668-99a4-528f7d39ba3f'/>\n<use y='408.316406' x='123.735352' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='408.316406' x='127.628418' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='408.316406' x='129.959473' xlink:href='#091b1d69-7c75-46b6-825f-da4b0b842949'/>\n<use y='408.316406' x='133.459473' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='408.316406' x='137.352539' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='408.316406' x='139.297363' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='408.316406' x='141.242188' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='408.316406' x='145.135254' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='408.316406' x='147.080078' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='408.316406' x='149.024902' xlink:href='#9932acbc-8b99-4f31-be5f-e87ed7486a2e'/>\n<use y='408.316406' x='153.693848' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='408.316406' x='157.586914' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='408.316406' x='159.531738' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='408.316406' x='161.476562' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='408.316406' x='165.369629' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='408.316406' x='167.700684' xlink:href='#77d9585b-31e4-4e0c-8c9c-a8b207187d3a'/>\n<use y='408.316406' x='171.200684' xlink:href='#5e521fb0-e350-4c29-a6ab-cff23c4eec3a'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='385.226562' x='127.617188' xlink:href='#c918bbc5-1fab-432a-8fb7-2bea63973fb2'/>\n<use y='385.226562' x='132.286133' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='385.226562' x='136.179199' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='385.226562' x='139.679199' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='385.226562' x='141.624023' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='385.226562' x='143.568848' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='385.226562' x='147.461914' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='385.226562' x='149.406738' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='385.226562' x='151.351562' xlink:href='#49d7c5ef-7a75-4d2b-8182-60e011b9efda'/>\n<use y='385.226562' x='156.020508' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='385.226562' x='159.913574' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='385.226562' x='163.806641' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='385.226562' x='166.137695' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='385.226562' x='170.030762' xlink:href='#77d9585b-31e4-4e0c-8c9c-a8b207187d3a'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='362.132812' x='79.761719' xlink:href='#c918bbc5-1fab-432a-8fb7-2bea63973fb2'/>\n<use y='362.132812' x='84.430664' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='362.132812' x='88.32373' xlink:href='#16022362-e0a3-4b9e-a0c6-392d55078d8b'/>\n<use y='362.132812' x='93.378906' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='362.132812' x='97.271973' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='362.132812' x='98.827148' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='362.132812' x='100.382324' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='362.132812' x='102.327148' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='362.132812' x='106.220215' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='362.132812' x='108.165039' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='362.132812' x='110.109863' xlink:href='#c918bbc5-1fab-432a-8fb7-2bea63973fb2'/>\n<use y='362.132812' x='114.778809' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='362.132812' x='118.671875' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='362.132812' x='122.171875' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='362.132812' x='124.116699' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='362.132812' x='126.061523' xlink:href='#9826757d-31e9-4d33-9633-ddb1b135594a'/>\n<use y='362.132812' x='128.392578' xlink:href='#734cb640-de7a-40b0-b39b-e8216d0e6a8e'/>\n<use y='362.132812' x='133.447754' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='362.132812' x='137.34082' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='362.132812' x='138.895996' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='362.132812' x='142.789062' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='362.132812' x='146.682129' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='362.132812' x='148.626953' xlink:href='#1fdff9cd-6269-45e2-8bed-b34b2792b662'/>\n<use y='362.132812' x='153.295898' xlink:href='#5d4eee01-dc5a-410c-8a4f-b300c300a866'/>\n<use y='362.132812' x='157.188965' xlink:href='#2fa6a5ab-c667-4a45-88df-f391ba868cec'/>\n<use y='362.132812' x='161.082031' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='362.132812' x='164.975098' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='362.132812' x='167.306152' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='362.132812' x='171.199219' xlink:href='#5e521fb0-e350-4c29-a6ab-cff23c4eec3a'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='339.039062' x='44.363281' xlink:href='#1fdff9cd-6269-45e2-8bed-b34b2792b662'/>\n<use y='339.039062' x='49.032227' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='339.039062' x='52.925293' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='339.039062' x='56.818359' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='339.039062' x='58.763184' xlink:href='#15f4a5d1-e31f-45ed-8218-3f8fc09d4b20'/>\n<use y='339.039062' x='63.039062' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='339.039062' x='65.370117' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='339.039062' x='69.263184' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='339.039062' x='73.15625' xlink:href='#bdd46fbd-d377-4bba-91c8-3814d2922bd5'/>\n<use y='339.039062' x='76.65625' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='339.039062' x='78.211426' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='339.039062' x='81.711426' xlink:href='#bdd46fbd-d377-4bba-91c8-3814d2922bd5'/>\n<use y='339.039062' x='85.211426' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='339.039062' x='89.104492' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='339.039062' x='91.049316' xlink:href='#d5cf4064-f02e-44b4-b47e-bb8c182a8372'/>\n<use y='339.039062' x='96.104492' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='339.039062' x='99.997559' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='339.039062' x='101.552734' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='339.039062' x='103.497559' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='339.039062' x='105.828613' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='339.039062' x='109.72168' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='339.039062' x='111.276855' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='339.039062' x='115.169922' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='339.039062' x='117.114746' xlink:href='#9826757d-31e9-4d33-9633-ddb1b135594a'/>\n<use y='339.039062' x='119.445801' xlink:href='#5062135a-1b6b-4c59-ae8d-4c5af49dcc8d'/>\n<use y='339.039062' x='123.72168' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='339.039062' x='127.614746' xlink:href='#16022362-e0a3-4b9e-a0c6-392d55078d8b'/>\n<use y='339.039062' x='132.669922' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='339.039062' x='136.562988' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='339.039062' x='140.062988' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='339.039062' x='143.956055' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='339.039062' x='147.849121' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='339.039062' x='151.742188' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='339.039062' x='153.687012' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='339.039062' x='157.580078' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='339.039062' x='159.524902' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='339.039062' x='161.469727' xlink:href='#498ee0e9-30bd-433e-a432-1fddd37c32a1'/>\n<use y='339.039062' x='165.362793' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='339.039062' x='167.307617' xlink:href='#1da6ad6a-cbc9-46ad-bddc-b5660b76e4c7'/>\n<use y='339.039062' x='171.200684' xlink:href='#5e521fb0-e350-4c29-a6ab-cff23c4eec3a'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='315.945312' x='100.785156' xlink:href='#1fdff9cd-6269-45e2-8bed-b34b2792b662'/>\n<use y='315.945312' x='105.454102' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='315.945312' x='109.347168' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='315.945312' x='113.240234' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='315.945312' x='115.185059' xlink:href='#15f4a5d1-e31f-45ed-8218-3f8fc09d4b20'/>\n<use y='315.945312' x='119.460938' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='315.945312' x='121.791992' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='315.945312' x='125.685059' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='315.945312' x='129.578125' xlink:href='#bdd46fbd-d377-4bba-91c8-3814d2922bd5'/>\n<use y='315.945312' x='133.078125' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='315.945312' x='134.633301' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='315.945312' x='138.133301' xlink:href='#bdd46fbd-d377-4bba-91c8-3814d2922bd5'/>\n<use y='315.945312' x='141.633301' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='315.945312' x='145.526367' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='315.945312' x='147.471191' xlink:href='#d5cf4064-f02e-44b4-b47e-bb8c182a8372'/>\n<use y='315.945312' x='152.526367' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='315.945312' x='154.081543' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='315.945312' x='156.026367' xlink:href='#77d9585b-31e4-4e0c-8c9c-a8b207187d3a'/>\n<use y='315.945312' x='159.526367' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='315.945312' x='161.471191' xlink:href='#07ea776c-aee7-462b-8d45-7502d730dcff'/>\n<use y='315.945312' x='166.526367' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='315.945312' x='170.419434' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='315.945312' x='171.974609' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='292.855469' x='91.824219' xlink:href='#1fdff9cd-6269-45e2-8bed-b34b2792b662'/>\n<use y='292.855469' x='96.493164' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='292.855469' x='100.38623' xlink:href='#2fa6a5ab-c667-4a45-88df-f391ba868cec'/>\n<use y='292.855469' x='104.279297' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='292.855469' x='106.224121' xlink:href='#1da6ad6a-cbc9-46ad-bddc-b5660b76e4c7'/>\n<use y='292.855469' x='110.117188' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='292.855469' x='112.062012' xlink:href='#61e31a73-cdee-4d48-be2c-f33e2453bdf4'/>\n<use y='292.855469' x='116.730957' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='292.855469' x='120.624023' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='292.855469' x='124.51709' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='292.855469' x='126.461914' xlink:href='#b69883fe-a677-48f2-9397-180599c78d22'/>\n<use y='292.855469' x='131.51709' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='292.855469' x='135.410156' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='292.855469' x='138.910156' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='292.855469' x='142.410156' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='292.855469' x='144.35498' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='292.855469' x='148.248047' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='292.855469' x='150.192871' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='292.855469' x='152.137695' xlink:href='#7610a7f7-8be4-4668-99a4-528f7d39ba3f'/>\n<use y='292.855469' x='157.96875' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='292.855469' x='161.861816' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='292.855469' x='164.192871' xlink:href='#091b1d69-7c75-46b6-825f-da4b0b842949'/>\n<use y='292.855469' x='167.692871' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='292.855469' x='171.585938' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='269.761719' x='122.175781' xlink:href='#1fdff9cd-6269-45e2-8bed-b34b2792b662'/>\n<use y='269.761719' x='126.844727' xlink:href='#5766d641-c9b6-4bbb-afd0-e7cf221f48b7'/>\n<use y='269.761719' x='130.737793' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='269.761719' x='134.630859' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='269.761719' x='138.523926' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='269.761719' x='140.85498' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='269.761719' x='142.799805' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='269.761719' x='146.692871' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='269.761719' x='148.637695' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='269.761719' x='150.58252' xlink:href='#15f4a5d1-e31f-45ed-8218-3f8fc09d4b20'/>\n<use y='269.761719' x='154.858398' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='269.761719' x='158.751465' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='269.761719' x='160.306641' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='269.761719' x='163.806641' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='269.761719' x='167.699707' xlink:href='#0dc9cbb1-8da9-4eff-b854-b2296c9c9035'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='246.667969' x='24.023438' xlink:href='#20af18b1-c0b4-4b01-987c-e266a13cf66e'/>\n<use y='246.667969' x='28.692383' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='246.667969' x='32.585449' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='246.667969' x='34.916504' xlink:href='#c00721a7-52a3-4489-a759-7de6cac3a30d'/>\n<use y='246.667969' x='38.80957' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='246.667969' x='42.702637' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='246.667969' x='44.647461' xlink:href='#9932acbc-8b99-4f31-be5f-e87ed7486a2e'/>\n<use y='246.667969' x='49.316406' xlink:href='#2fa6a5ab-c667-4a45-88df-f391ba868cec'/>\n<use y='246.667969' x='53.209473' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='246.667969' x='57.102539' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='246.667969' x='60.995605' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='246.667969' x='64.888672' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='246.667969' x='66.833496' xlink:href='#d5cf4064-f02e-44b4-b47e-bb8c182a8372'/>\n<use y='246.667969' x='71.888672' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='246.667969' x='75.781738' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='246.667969' x='79.674805' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='246.667969' x='81.619629' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='246.667969' x='85.512695' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='246.667969' x='87.84375' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='246.667969' x='89.788574' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='246.667969' x='93.681641' xlink:href='#97113d21-a497-49f4-9dd1-da562099ee05'/>\n<use y='246.667969' x='95.626465' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='246.667969' x='97.571289' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='246.667969' x='99.516113' xlink:href='#1da6ad6a-cbc9-46ad-bddc-b5660b76e4c7'/>\n<use y='246.667969' x='103.40918' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='246.667969' x='107.302246' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='246.667969' x='109.24707' xlink:href='#faf9c806-cffc-48dc-8fa4-0e89266f3dca'/>\n<use y='246.667969' x='113.916016' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='246.667969' x='116.24707' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='246.667969' x='118.191895' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='246.667969' x='121.691895' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='246.667969' x='123.636719' xlink:href='#9826757d-31e9-4d33-9633-ddb1b135594a'/>\n<use y='246.667969' x='125.967773' xlink:href='#43a18a7e-1975-42b3-84e0-9606dd71d1a9'/>\n<use y='246.667969' x='129.86084' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='246.667969' x='132.191895' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='246.667969' x='136.084961' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='246.667969' x='138.029785' xlink:href='#bbcfccc7-5384-4cb4-bf6e-2e0c71777eab'/>\n<use y='246.667969' x='145.135742' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='246.667969' x='147.080566' xlink:href='#07ea776c-aee7-462b-8d45-7502d730dcff'/>\n<use y='246.667969' x='152.135742' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='246.667969' x='156.028809' xlink:href='#16022362-e0a3-4b9e-a0c6-392d55078d8b'/>\n<use y='246.667969' x='161.083984' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='246.667969' x='164.977051' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='246.667969' x='167.308105' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='246.667969' x='171.201172' xlink:href='#5e521fb0-e350-4c29-a6ab-cff23c4eec3a'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='223.578125' x='67.339844' xlink:href='#d5cf4064-f02e-44b4-b47e-bb8c182a8372'/>\n<use y='223.578125' x='72.39502' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='223.578125' x='73.950195' xlink:href='#cab96616-0e1a-49dc-99de-457a3e10394b'/>\n<use y='223.578125' x='77.450195' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='223.578125' x='79.005371' xlink:href='#bdd46fbd-d377-4bba-91c8-3814d2922bd5'/>\n<use y='223.578125' x='82.505371' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='223.578125' x='84.450195' xlink:href='#d5cf4064-f02e-44b4-b47e-bb8c182a8372'/>\n<use y='223.578125' x='89.505371' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='223.578125' x='93.398438' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='223.578125' x='97.291504' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='223.578125' x='99.236328' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='223.578125' x='103.129395' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='223.578125' x='105.460449' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='223.578125' x='107.405273' xlink:href='#9932acbc-8b99-4f31-be5f-e87ed7486a2e'/>\n<use y='223.578125' x='112.074219' xlink:href='#faf9c806-cffc-48dc-8fa4-0e89266f3dca'/>\n<use y='223.578125' x='116.743164' xlink:href='#7ed5e8ce-843d-470d-a211-5a4659c90d39'/>\n<use y='223.578125' x='121.79834' xlink:href='#5062135a-1b6b-4c59-ae8d-4c5af49dcc8d'/>\n<use y='223.578125' x='126.074219' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='223.578125' x='128.019043' xlink:href='#9826757d-31e9-4d33-9633-ddb1b135594a'/>\n<use y='223.578125' x='130.350098' xlink:href='#20f10433-62cf-4427-9d4a-66c5c001be2b'/>\n<use y='223.578125' x='134.243164' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='223.578125' x='136.187988' xlink:href='#1da6ad6a-cbc9-46ad-bddc-b5660b76e4c7'/>\n<use y='223.578125' x='140.081055' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='223.578125' x='142.025879' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='223.578125' x='145.918945' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='223.578125' x='147.86377' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='223.578125' x='149.808594' xlink:href='#7610a7f7-8be4-4668-99a4-528f7d39ba3f'/>\n<use y='223.578125' x='155.639648' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='223.578125' x='159.532715' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='223.578125' x='161.86377' xlink:href='#091b1d69-7c75-46b6-825f-da4b0b842949'/>\n<use y='223.578125' x='165.36377' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='223.578125' x='169.256836' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='223.578125' x='171.20166' xlink:href='#5e521fb0-e350-4c29-a6ab-cff23c4eec3a'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='200.484375' x='110.492188' xlink:href='#a8f15154-4753-4c2e-889a-bf0f8eab37f5'/>\n<use y='200.484375' x='115.937012' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='200.484375' x='119.830078' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='200.484375' x='121.385254' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='200.484375' x='125.27832' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='200.484375' x='129.171387' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='200.484375' x='133.064453' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='200.484375' x='135.009277' xlink:href='#a8f15154-4753-4c2e-889a-bf0f8eab37f5'/>\n<use y='200.484375' x='140.454102' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='200.484375' x='144.347168' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='200.484375' x='146.291992' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='200.484375' x='150.185059' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='200.484375' x='152.129883' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='200.484375' x='156.022949' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='200.484375' x='157.967773' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='200.484375' x='159.912598' xlink:href='#c918bbc5-1fab-432a-8fb7-2bea63973fb2'/>\n<use y='200.484375' x='164.581543' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='200.484375' x='168.474609' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='200.484375' x='170.029785' xlink:href='#091b1d69-7c75-46b6-825f-da4b0b842949'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='177.390625' x='61.496094' xlink:href='#07ea776c-aee7-462b-8d45-7502d730dcff'/>\n<use y='177.390625' x='66.55127' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='177.390625' x='70.444336' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='177.390625' x='72.775391' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='177.390625' x='75.106445' xlink:href='#77d9585b-31e4-4e0c-8c9c-a8b207187d3a'/>\n<use y='177.390625' x='78.606445' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='177.390625' x='80.55127' xlink:href='#9932acbc-8b99-4f31-be5f-e87ed7486a2e'/>\n<use y='177.390625' x='85.220215' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='177.390625' x='87.55127' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='177.390625' x='89.106445' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='177.390625' x='92.999512' xlink:href='#3093d11f-7144-4487-8aed-c2dd55c92efa'/>\n<use y='177.390625' x='96.892578' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='177.390625' x='100.785645' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='177.390625' x='104.285645' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='177.390625' x='106.230469' xlink:href='#c918bbc5-1fab-432a-8fb7-2bea63973fb2'/>\n<use y='177.390625' x='110.899414' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='177.390625' x='112.45459' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='177.390625' x='116.347656' xlink:href='#1862368e-e783-4c5e-9649-e2093e4fd672'/>\n<use y='177.390625' x='119.847656' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='177.390625' x='123.740723' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='177.390625' x='125.685547' xlink:href='#9826757d-31e9-4d33-9633-ddb1b135594a'/>\n<use y='177.390625' x='128.016602' xlink:href='#15f4a5d1-e31f-45ed-8218-3f8fc09d4b20'/>\n<use y='177.390625' x='132.29248' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='177.390625' x='136.185547' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='177.390625' x='138.516602' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='177.390625' x='140.847656' xlink:href='#77d9585b-31e4-4e0c-8c9c-a8b207187d3a'/>\n<use y='177.390625' x='144.347656' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='177.390625' x='146.29248' xlink:href='#9932acbc-8b99-4f31-be5f-e87ed7486a2e'/>\n<use y='177.390625' x='150.961426' xlink:href='#2fa6a5ab-c667-4a45-88df-f391ba868cec'/>\n<use y='177.390625' x='154.854492' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='177.390625' x='156.409668' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='177.390625' x='157.964844' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='177.390625' x='161.85791' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='177.390625' x='163.413086' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='177.390625' x='167.306152' xlink:href='#3093d11f-7144-4487-8aed-c2dd55c92efa'/>\n<use y='177.390625' x='171.199219' xlink:href='#5e521fb0-e350-4c29-a6ab-cff23c4eec3a'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='154.300781' x='44.359375' xlink:href='#1fdff9cd-6269-45e2-8bed-b34b2792b662'/>\n<use y='154.300781' x='49.02832' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='154.300781' x='52.921387' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='154.300781' x='56.814453' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='154.300781' x='58.759277' xlink:href='#15f4a5d1-e31f-45ed-8218-3f8fc09d4b20'/>\n<use y='154.300781' x='63.035156' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='154.300781' x='65.366211' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='154.300781' x='69.259277' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='154.300781' x='73.152344' xlink:href='#bdd46fbd-d377-4bba-91c8-3814d2922bd5'/>\n<use y='154.300781' x='76.652344' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='154.300781' x='78.20752' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='154.300781' x='81.70752' xlink:href='#bdd46fbd-d377-4bba-91c8-3814d2922bd5'/>\n<use y='154.300781' x='85.20752' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='154.300781' x='89.100586' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='154.300781' x='91.04541' xlink:href='#d5cf4064-f02e-44b4-b47e-bb8c182a8372'/>\n<use y='154.300781' x='96.100586' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='154.300781' x='99.993652' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='154.300781' x='101.548828' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='154.300781' x='103.493652' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='154.300781' x='105.824707' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='154.300781' x='109.717773' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='154.300781' x='111.272949' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='154.300781' x='115.166016' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='154.300781' x='117.11084' xlink:href='#f5647a9d-2871-49b0-9631-fc19a67066b3'/>\n<use y='154.300781' x='121.003906' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='154.300781' x='122.94873' xlink:href='#9826757d-31e9-4d33-9633-ddb1b135594a'/>\n<use y='154.300781' x='125.279785' xlink:href='#43a18a7e-1975-42b3-84e0-9606dd71d1a9'/>\n<use y='154.300781' x='129.172852' xlink:href='#43a18a7e-1975-42b3-84e0-9606dd71d1a9'/>\n<use y='154.300781' x='133.065918' xlink:href='#351f40d4-2fc8-4a8b-8fbf-02a4e78e4533'/>\n<use y='154.300781' x='136.958984' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='154.300781' x='138.903809' xlink:href='#5062135a-1b6b-4c59-ae8d-4c5af49dcc8d'/>\n<use y='154.300781' x='143.179688' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='154.300781' x='147.072754' xlink:href='#16022362-e0a3-4b9e-a0c6-392d55078d8b'/>\n<use y='154.300781' x='152.12793' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='154.300781' x='156.020996' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='154.300781' x='159.520996' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='154.300781' x='163.414062' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='154.300781' x='167.307129' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='154.300781' x='171.200195' xlink:href='#5e521fb0-e350-4c29-a6ab-cff23c4eec3a'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='131.207031' x='119.84375' xlink:href='#1fdff9cd-6269-45e2-8bed-b34b2792b662'/>\n<use y='131.207031' x='124.512695' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='131.207031' x='126.45752' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='131.207031' x='130.350586' xlink:href='#2fa6a5ab-c667-4a45-88df-f391ba868cec'/>\n<use y='131.207031' x='134.243652' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='131.207031' x='138.136719' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='131.207031' x='140.467773' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='131.207031' x='142.412598' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='131.207031' x='144.357422' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='131.207031' x='148.250488' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='131.207031' x='150.195312' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='131.207031' x='152.140137' xlink:href='#7610a7f7-8be4-4668-99a4-528f7d39ba3f'/>\n<use y='131.207031' x='157.971191' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='131.207031' x='161.864258' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='131.207031' x='164.195312' xlink:href='#091b1d69-7c75-46b6-825f-da4b0b842949'/>\n<use y='131.207031' x='167.695312' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='131.207031' x='171.588379' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='108.113281' x='21.035156' xlink:href='#5062135a-1b6b-4c59-ae8d-4c5af49dcc8d'/>\n<use y='108.113281' x='25.311035' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='108.113281' x='29.204102' xlink:href='#0dc9cbb1-8da9-4eff-b854-b2296c9c9035'/>\n<use y='108.113281' x='35.035156' xlink:href='#5766d641-c9b6-4bbb-afd0-e7cf221f48b7'/>\n<use y='108.113281' x='38.928223' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='108.113281' x='42.821289' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='108.113281' x='45.152344' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='108.113281' x='49.04541' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='108.113281' x='51.376465' xlink:href='#77d9585b-31e4-4e0c-8c9c-a8b207187d3a'/>\n<use y='108.113281' x='54.876465' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='108.113281' x='56.821289' xlink:href='#5062135a-1b6b-4c59-ae8d-4c5af49dcc8d'/>\n<use y='108.113281' x='61.097168' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='108.113281' x='63.428223' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='108.113281' x='67.321289' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='108.113281' x='71.214355' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='108.113281' x='74.714355' xlink:href='#c00721a7-52a3-4489-a759-7de6cac3a30d'/>\n<use y='108.113281' x='78.607422' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='108.113281' x='82.500488' xlink:href='#77d9585b-31e4-4e0c-8c9c-a8b207187d3a'/>\n<use y='108.113281' x='86.000488' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='108.113281' x='87.945312' xlink:href='#5062135a-1b6b-4c59-ae8d-4c5af49dcc8d'/>\n<use y='108.113281' x='92.221191' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='108.113281' x='96.114258' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='108.113281' x='98.445312' xlink:href='#0dc9cbb1-8da9-4eff-b854-b2296c9c9035'/>\n<use y='108.113281' x='104.276367' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='108.113281' x='105.831543' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='108.113281' x='109.724609' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='108.113281' x='113.617676' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='108.113281' x='115.172852' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='108.113281' x='117.117676' xlink:href='#9826757d-31e9-4d33-9633-ddb1b135594a'/>\n<use y='108.113281' x='119.44873' xlink:href='#07ea776c-aee7-462b-8d45-7502d730dcff'/>\n<use y='108.113281' x='124.503906' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='108.113281' x='128.396973' xlink:href='#16022362-e0a3-4b9e-a0c6-392d55078d8b'/>\n<use y='108.113281' x='133.452148' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='108.113281' x='137.345215' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='108.113281' x='139.67627' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='108.113281' x='143.569336' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='108.113281' x='145.51416' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='108.113281' x='149.407227' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='108.113281' x='151.352051' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='108.113281' x='153.296875' xlink:href='#9932acbc-8b99-4f31-be5f-e87ed7486a2e'/>\n<use y='108.113281' x='157.96582' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='108.113281' x='161.858887' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='108.113281' x='165.751953' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='108.113281' x='167.307129' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='108.113281' x='171.200195' xlink:href='#5e521fb0-e350-4c29-a6ab-cff23c4eec3a'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='85.019531' x='119.828125' xlink:href='#f5647a9d-2871-49b0-9631-fc19a67066b3'/>\n<use y='85.019531' x='123.721191' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='85.019531' x='127.614258' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n<use y='85.019531' x='131.507324' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='85.019531' x='133.452148' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='85.019531' x='137.345215' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='85.019531' x='139.290039' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='85.019531' x='141.234863' xlink:href='#5062135a-1b6b-4c59-ae8d-4c5af49dcc8d'/>\n<use y='85.019531' x='145.510742' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='85.019531' x='149.403809' xlink:href='#16022362-e0a3-4b9e-a0c6-392d55078d8b'/>\n<use y='85.019531' x='154.458984' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='85.019531' x='158.352051' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='85.019531' x='161.852051' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='85.019531' x='165.745117' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='85.019531' x='169.638184' xlink:href='#92bb5c1b-8d4d-40b4-b1ad-d33df39f3934'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='61.929688' x='128.785156' xlink:href='#7610a7f7-8be4-4668-99a4-528f7d39ba3f'/>\n<use y='61.929688' x='134.616211' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='61.929688' x='138.509277' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='61.929688' x='140.840332' xlink:href='#091b1d69-7c75-46b6-825f-da4b0b842949'/>\n<use y='61.929688' x='144.340332' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='61.929688' x='148.233398' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='61.929688' x='150.178223' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='61.929688' x='152.123047' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='61.929688' x='156.016113' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='61.929688' x='157.960938' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='61.929688' x='159.905762' xlink:href='#3a3a64a1-d6d0-4f66-a33e-d8444bd84bc5'/>\n<use y='61.929688' x='163.798828' xlink:href='#351f40d4-2fc8-4a8b-8fbf-02a4e78e4533'/>\n<use y='61.929688' x='167.691895' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='61.929688' x='169.636719' xlink:href='#1da6ad6a-cbc9-46ad-bddc-b5660b76e4c7'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='38.835938' x='112.839844' xlink:href='#7610a7f7-8be4-4668-99a4-528f7d39ba3f'/>\n<use y='38.835938' x='118.670898' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='38.835938' x='122.563965' xlink:href='#f5d3bfca-3a9d-488c-8604-94d3f39d80e9'/>\n<use y='38.835938' x='124.89502' xlink:href='#091b1d69-7c75-46b6-825f-da4b0b842949'/>\n<use y='38.835938' x='128.39502' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='38.835938' x='132.288086' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='38.835938' x='134.23291' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='38.835938' x='136.177734' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='38.835938' x='140.070801' xlink:href='#23581864-4fe4-4ac1-9c9f-a4fd125eecc9'/>\n<use y='38.835938' x='142.015625' xlink:href='#e452c12d-9540-4335-96a8-acd1ffe1ffd3'/>\n<use y='38.835938' x='143.960449' xlink:href='#1fdff9cd-6269-45e2-8bed-b34b2792b662'/>\n<use y='38.835938' x='148.629395' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='38.835938' x='152.522461' xlink:href='#6ac45e00-029c-4471-9991-7ff5ffbeaceb'/>\n<use y='38.835938' x='156.415527' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n<use y='38.835938' x='159.915527' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='38.835938' x='163.808594' xlink:href='#0dc9cbb1-8da9-4eff-b854-b2296c9c9035'/>\n<use y='38.835938' x='169.639648' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='844.382812' x='192.5625' xlink:href='#351f40d4-2fc8-4a8b-8fbf-02a4e78e4533'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='844.382812' x='266.226562' xlink:href='#24baa935-3afb-42ca-b4ef-1ffc8c841b87'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='844.382812' x='337.941406' xlink:href='#3a3a64a1-d6d0-4f66-a33e-d8444bd84bc5'/>\n<use y='844.382812' x='341.834473' xlink:href='#351f40d4-2fc8-4a8b-8fbf-02a4e78e4533'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='844.382812' x='411.601562' xlink:href='#3a3a64a1-d6d0-4f66-a33e-d8444bd84bc5'/>\n<use y='844.382812' x='415.494629' xlink:href='#24baa935-3afb-42ca-b4ef-1ffc8c841b87'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='844.382812' x='485.265625' xlink:href='#f5647a9d-2871-49b0-9631-fc19a67066b3'/>\n<use y='844.382812' x='489.158691' xlink:href='#351f40d4-2fc8-4a8b-8fbf-02a4e78e4533'/>\n</g>\n<g style='fill:rgb(32.156863%,32.156863%,32.156863%);fill-opacity:1;'>\n<use y='451.355469' x='17.449219' xlink:href='#b9fbad53-4522-4c6c-8250-7b116e75e87a'/>\n<use y='447.355469' x='17.449219' xlink:href='#4d9c9b9b-614a-4330-9440-bf809579b53c'/>\n<use y='445.132812' x='17.449219' xlink:href='#4100fe2b-0aa9-4ff3-aa7d-ed4a3c2cf642'/>\n<use y='440.683594' x='17.449219' xlink:href='#4d9c9b9b-614a-4330-9440-bf809579b53c'/>\n<use y='438.460938' x='17.449219' xlink:href='#943094b7-c061-430d-bd6d-8bfc1fe5ee25'/>\n<use y='436.683594' x='17.449219' xlink:href='#2924fa4a-4cf0-40b8-93dc-3dec1b54e117'/>\n<use y='432.234375' x='17.449219' xlink:href='#703657dc-fa2c-4449-8af9-fb0f288f2801'/>\n<use y='427.785156' x='17.449219' xlink:href='#a2e32e90-f1de-4a37-bebf-4170f2f3c768'/>\n<use y='422.007812' x='17.449219' xlink:href='#4100fe2b-0aa9-4ff3-aa7d-ed4a3c2cf642'/>\n<use y='417.558594' x='17.449219' xlink:href='#d788cce7-7fbd-4064-a1b3-6dd04b74886c'/>\n<use y='410.894531' x='17.449219' xlink:href='#964e6bdd-5edd-4610-885d-171b6cc0bcd0'/>\n</g>\n<path d='M 558.804688 452.59375 L 633.828125 452.59375 L 633.828125 405.210938 L 558.804688 405.210938 Z M 558.804688 452.59375 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;'/>\n<path d='M 563.054688 431.0625 L 580.335938 431.0625 L 580.335938 413.78125 L 563.054688 413.78125 Z M 563.054688 431.0625 ' style='fill-rule:nonzero;fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 563.765625 430.351562 L 579.628906 430.351562 L 579.628906 414.488281 L 563.765625 414.488281 Z M 563.765625 430.351562 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:1;'/>\n<path d='M 563.054688 448.34375 L 580.335938 448.34375 L 580.335938 431.0625 L 563.054688 431.0625 Z M 563.054688 448.34375 ' style='fill-rule:nonzero;fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 563.765625 447.632812 L 579.628906 447.632812 L 579.628906 431.769531 L 563.765625 431.769531 Z M 563.765625 447.632812 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,74.901961%,76.862745%);fill-opacity:1;'/>\n<g style='fill:rgb(32.156863%,32.156863%,32.156863%);fill-opacity:1;'>\n<use y='424.929688' x='582.496094' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='424.929688' x='586.38916' xlink:href='#cab96616-0e1a-49dc-99de-457a3e10394b'/>\n<use y='424.929688' x='589.88916' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='424.929688' x='593.782227' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='424.929688' x='595.337402' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='424.929688' x='596.892578' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='424.929688' x='600.785645' xlink:href='#c00721a7-52a3-4489-a759-7de6cac3a30d'/>\n<use y='424.929688' x='604.678711' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='424.929688' x='606.233887' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='424.929688' x='610.126953' xlink:href='#9932acbc-8b99-4f31-be5f-e87ed7486a2e'/>\n<use y='424.929688' x='614.795898' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='424.929688' x='616.351074' xlink:href='#091b1d69-7c75-46b6-825f-da4b0b842949'/>\n<use y='424.929688' x='619.851074' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='424.929688' x='623.744141' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n</g>\n<g style='fill:rgb(32.156863%,32.156863%,32.156863%);fill-opacity:1;'>\n<use y='442.210938' x='582.496094' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='442.210938' x='586.38916' xlink:href='#cab96616-0e1a-49dc-99de-457a3e10394b'/>\n<use y='442.210938' x='589.88916' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='442.210938' x='593.782227' xlink:href='#02f0d219-92cd-46e0-960d-e11470244484'/>\n<use y='442.210938' x='595.337402' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='442.210938' x='596.892578' xlink:href='#bf66ab75-0709-4ec2-abbe-4a01f33a231b'/>\n<use y='442.210938' x='600.785645' xlink:href='#c00721a7-52a3-4489-a759-7de6cac3a30d'/>\n<use y='442.210938' x='604.678711' xlink:href='#0fb24c34-4cf3-4174-93f7-26ffb2ed5101'/>\n<use y='442.210938' x='606.233887' xlink:href='#ad69ce23-6ab7-4c95-a6a3-a211d82c100e'/>\n<use y='442.210938' x='610.126953' xlink:href='#0373f860-79fb-49c3-aba0-0e9aac3a40fa'/>\n<use y='442.210938' x='615.182129' xlink:href='#4809c55e-e7db-4fc1-aecb-b5fec5f904b7'/>\n<use y='442.210938' x='619.075195' xlink:href='#bdd46fbd-d377-4bba-91c8-3814d2922bd5'/>\n<use y='442.210938' x='622.575195' xlink:href='#091b1d69-7c75-46b6-825f-da4b0b842949'/>\n<use y='442.210938' x='626.075195' xlink:href='#0fbdc780-6d22-480c-b581-916d1f4f6afb'/>\n</g>\n</g>\n</svg>\n","value":"#huri.plot.GGView{:plot-command [[:<- :g [:data.frame {:stationName [:c \"Powell Street BART\" \"Clay at Battery\" \"Davis at Jackson\" \"Commercial at Montgomery\" \"Washington at Kearny\" \"Post at Kearny\" \"Embarcadero at Vallejo\" \"Spear at Folsom\" \"Harry Bridges Plaza (Ferry Building)\" \"Embarcadero at Folsom\" \"Embarcadero at Bryant\" \"Temporary Transbay Terminal (Howard at Beale)\" \"Beale at Market\" \"5th at Howard\" \"San Francisco City Hall\" \"Golden Gate at Polk\" \"Embarcadero at Sansome\" \"2nd at Townsend\" \"2nd at Folsom\" \"Howard at 2nd\" \"2nd at South Park\" \"Townsend at 7th\" \"South Van Ness at Market\" \"Market at 10th\" \"Yerba Buena Center of the Arts (3rd @ Howard)\" \"San Francisco Caltrain 2 (330 Townsend)\" \"San Francisco Caltrain (Townsend at 4th)\" \"Powell at Post (Union Square)\" \"Civic Center BART (7th at Market)\" \"Grant Avenue at Columbus Avenue\" \"Steuart at Market\" \"Mechanics Plaza (Market at Battery)\" \"Market at 4th\" \"Market at Sansome\" \"Broadway St at Battery St\" \"Powell Street BART\" \"Clay at Battery\" \"Davis at Jackson\" \"Commercial at Montgomery\" \"Washington at Kearny\" \"Post at Kearny\" \"Embarcadero at Vallejo\" \"Spear at Folsom\" \"Harry Bridges Plaza (Ferry Building)\" \"Embarcadero at Folsom\" \"Embarcadero at Bryant\" \"Temporary Transbay Terminal (Howard at Beale)\" \"Beale at Market\" \"5th at Howard\" \"San Francisco City Hall\" \"Golden Gate at Polk\" \"Embarcadero at Sansome\" \"2nd at Townsend\" \"2nd at Folsom\" \"Howard at 2nd\" \"2nd at South Park\" \"Townsend at 7th\" \"South Van Ness at Market\" \"Market at 10th\" \"Yerba Buena Center of the Arts (3rd @ Howard)\" \"San Francisco Caltrain 2 (330 Townsend)\" \"San Francisco Caltrain (Townsend at 4th)\" \"Powell at Post (Union Square)\" \"Civic Center BART (7th at Market)\" \"Grant Avenue at Columbus Avenue\" \"Steuart at Market\" \"Mechanics Plaza (Market at Battery)\" \"Market at 4th\" \"Market at Sansome\" \"Broadway St at Battery St\"], :y__auto [:c 8 12 11 12 11 7 11 4 5 13 4 13 16 6 10 15 12 10 13 15 10 7 12 23 15 5 3 10 14 5 4 16 11 18 11 10 3 3 3 3 12 4 15 18 6 11 10 3 9 9 8 3 17 6 4 5 8 7 4 4 18 16 9 9 14 19 3 8 9 4], :series__auto [:c \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableDocks\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\" \"availableBikes\"]}]] [[:library :ggplot2] [:library :scales] [:library :grid] [:library :RColorBrewer] [:library :ggrepel] [:<- :palette [:brewer.pal \"Greys\" {:n 9}]] {:color.background :palette[2]} {:color.grid.major :palette[3]} {:color.axis.text :palette[6]} {:color.axis.title :palette[7]} {:color.title :palette[9]}] [:+ [:+ [:+ [:+ [:+ [:ggplot :g [:aes {:x [:reorder :stationName :y__auto], :y :y__auto, :fill :series__auto}]] [:geom_bar {:stat \"identity\", :position \"dodge\"}]] [:coord_flip]] [:scale_y_continuous {:labels :comma}]] [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:theme_bw {:base_size 9}] [:theme {:panel.background [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:plot.background [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:panel.border [:element_rect {:color :color.background}]}]] [:theme {:panel.grid.major [:element_line {:color :color.grid.major, :size 0.25}]}]] [:theme {:panel.grid.minor [:element_blank]}]] [:theme {:axis.ticks [:element_blank]}]] [:theme {:legend.background [:element_rect {:fill :color.background}]}]] [:theme {:legend.key [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:legend.text [:element_text {:size 7, :color :color.axis.title}]}]] [:theme {:legend.title [:element_blank]}]] [:theme {:plot.title [:element_text {:size 10, :color :color.title, :vjust 1.25}]}]] [:theme {:axis.text.x [:element_text {:size 7, :color :color.axis.text}]}]] [:theme {:axis.text.y [:element_text {:size 7, :color :color.axis.text}]}]] [:theme {:axis.title.x [:element_text {:size 8, :color :color.axis.title, :vjust 0}]}]] [:theme {:axis.title.y [:element_text {:size 8, :color :color.axis.title, :vjust 1.25}]}]] [:theme {:plot.margin [:unit [:c 0.35 0.2 0.3 0.35] \"cm\"]}]]] [:labs {:y \"\", :title \"\", :x \"stationName\"}]]], :options {:width 9, :height 12}}"}
;; <=

;; @@
(->> df
     (where {:city "San Jose"})
     (scatter-plot :longitude :latitude {:label :stationName}))
;; @@
;; =>
;;; {"type":"html","content":"<?xml version='1.0' encoding='UTF-8'?>\n<svg version='1.1' viewBox='0 0 648 360' height='360pt' width='648pt' xmlns:xlink='http://www.w3.org/1999/xlink' xmlns='http://www.w3.org/2000/svg'>\n<defs>\n<g>\n<symbol id='70877cea-4f4b-4892-8e60-5add7a9dcf92' overflow='visible'>\n<path d='M 0.328125 0 L 0.328125 -7.140625 L 5.984375 -7.140625 L 5.984375 0 Z M 5.09375 -0.890625 L 5.09375 -6.25 L 1.21875 -6.25 L 1.21875 -0.890625 Z M 5.09375 -0.890625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='bce029c5-6a04-4949-a7f2-9ceac4d60995' overflow='visible'>\n<path d='M 1.390625 -2.296875 C 1.410156 -1.898438 1.503906 -1.570312 1.671875 -1.3125 C 1.992188 -0.84375 2.5625 -0.609375 3.375 -0.609375 C 3.738281 -0.609375 4.066406 -0.660156 4.359375 -0.765625 C 4.941406 -0.960938 5.234375 -1.320312 5.234375 -1.84375 C 5.234375 -2.226562 5.109375 -2.503906 4.859375 -2.671875 C 4.617188 -2.835938 4.234375 -2.976562 3.703125 -3.09375 L 2.734375 -3.3125 C 2.109375 -3.457031 1.664062 -3.613281 1.40625 -3.78125 C 0.945312 -4.082031 0.71875 -4.53125 0.71875 -5.125 C 0.71875 -5.757812 0.9375 -6.28125 1.375 -6.6875 C 1.820312 -7.101562 2.453125 -7.3125 3.265625 -7.3125 C 4.015625 -7.3125 4.648438 -7.128906 5.171875 -6.765625 C 5.703125 -6.410156 5.96875 -5.835938 5.96875 -5.046875 L 5.046875 -5.046875 C 5.003906 -5.429688 4.90625 -5.722656 4.75 -5.921875 C 4.445312 -6.296875 3.941406 -6.484375 3.234375 -6.484375 C 2.671875 -6.484375 2.265625 -6.363281 2.015625 -6.125 C 1.765625 -5.882812 1.640625 -5.601562 1.640625 -5.28125 C 1.640625 -4.9375 1.785156 -4.679688 2.078125 -4.515625 C 2.265625 -4.410156 2.695312 -4.28125 3.375 -4.125 L 4.375 -3.90625 C 4.851562 -3.789062 5.222656 -3.640625 5.484375 -3.453125 C 5.941406 -3.117188 6.171875 -2.632812 6.171875 -2 C 6.171875 -1.207031 5.882812 -0.640625 5.3125 -0.296875 C 4.738281 0.0351562 4.070312 0.203125 3.3125 0.203125 C 2.414062 0.203125 1.71875 -0.0195312 1.21875 -0.46875 C 0.71875 -0.914062 0.472656 -1.523438 0.484375 -2.296875 Z M 3.34375 -7.328125 Z M 3.34375 -7.328125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='32da805b-7aeb-40df-b595-c31cd6eac50b' overflow='visible'>\n<path d='M 1.3125 -1.390625 C 1.3125 -1.128906 1.398438 -0.925781 1.578125 -0.78125 C 1.765625 -0.632812 1.988281 -0.5625 2.25 -0.5625 C 2.550781 -0.5625 2.847656 -0.632812 3.140625 -0.78125 C 3.628906 -1.019531 3.875 -1.40625 3.875 -1.9375 L 3.875 -2.65625 C 3.757812 -2.582031 3.617188 -2.519531 3.453125 -2.46875 C 3.285156 -2.425781 3.117188 -2.394531 2.953125 -2.375 L 2.4375 -2.3125 C 2.113281 -2.269531 1.875 -2.203125 1.71875 -2.109375 C 1.445312 -1.960938 1.3125 -1.722656 1.3125 -1.390625 Z M 3.4375 -3.15625 C 3.632812 -3.175781 3.765625 -3.257812 3.828125 -3.40625 C 3.867188 -3.488281 3.890625 -3.597656 3.890625 -3.734375 C 3.890625 -4.035156 3.78125 -4.253906 3.5625 -4.390625 C 3.351562 -4.523438 3.050781 -4.59375 2.65625 -4.59375 C 2.207031 -4.59375 1.882812 -4.46875 1.6875 -4.21875 C 1.582031 -4.082031 1.515625 -3.878906 1.484375 -3.609375 L 0.671875 -3.609375 C 0.679688 -4.253906 0.882812 -4.703125 1.28125 -4.953125 C 1.6875 -5.203125 2.15625 -5.328125 2.6875 -5.328125 C 3.289062 -5.328125 3.785156 -5.210938 4.171875 -4.984375 C 4.554688 -4.742188 4.75 -4.378906 4.75 -3.890625 L 4.75 -0.890625 C 4.75 -0.804688 4.765625 -0.734375 4.796875 -0.671875 C 4.835938 -0.617188 4.914062 -0.59375 5.03125 -0.59375 C 5.070312 -0.59375 5.113281 -0.59375 5.15625 -0.59375 C 5.207031 -0.601562 5.265625 -0.613281 5.328125 -0.625 L 5.328125 0.03125 C 5.191406 0.0625 5.085938 0.0820312 5.015625 0.09375 C 4.941406 0.101562 4.84375 0.109375 4.71875 0.109375 C 4.414062 0.109375 4.195312 0.00390625 4.0625 -0.203125 C 3.988281 -0.316406 3.9375 -0.476562 3.90625 -0.6875 C 3.726562 -0.457031 3.472656 -0.253906 3.140625 -0.078125 C 2.804688 0.0859375 2.441406 0.171875 2.046875 0.171875 C 1.554688 0.171875 1.160156 0.0234375 0.859375 -0.265625 C 0.554688 -0.554688 0.40625 -0.921875 0.40625 -1.359375 C 0.40625 -1.847656 0.550781 -2.222656 0.84375 -2.484375 C 1.144531 -2.753906 1.539062 -2.921875 2.03125 -2.984375 Z M 2.703125 -5.328125 Z M 2.703125 -5.328125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='b5ffbc07-c018-4471-8d17-28477c1e76f5' overflow='visible'>\n<path d='M 0.640625 -5.203125 L 1.46875 -5.203125 L 1.46875 -4.46875 C 1.71875 -4.769531 1.976562 -4.988281 2.25 -5.125 C 2.53125 -5.257812 2.835938 -5.328125 3.171875 -5.328125 C 3.910156 -5.328125 4.410156 -5.066406 4.671875 -4.546875 C 4.816406 -4.265625 4.890625 -3.863281 4.890625 -3.34375 L 4.890625 0 L 4 0 L 4 -3.28125 C 4 -3.59375 3.953125 -3.847656 3.859375 -4.046875 C 3.703125 -4.367188 3.421875 -4.53125 3.015625 -4.53125 C 2.804688 -4.53125 2.632812 -4.507812 2.5 -4.46875 C 2.257812 -4.394531 2.046875 -4.253906 1.859375 -4.046875 C 1.710938 -3.867188 1.617188 -3.6875 1.578125 -3.5 C 1.535156 -3.320312 1.515625 -3.0625 1.515625 -2.71875 L 1.515625 0 L 0.640625 0 Z M 2.703125 -5.328125 Z M 2.703125 -5.328125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='ea17262e-0acb-4c28-9950-d8bcf6c27f90' overflow='visible'>\n<path d='' style='stroke:none;'/>\n</symbol>\n<symbol id='fefb0bdb-46bf-4fe8-ad02-31d0d222e074' overflow='visible'>\n<path d='M 4.265625 -2.125 C 4.265625 -1.519531 4.175781 -1.050781 4 -0.71875 C 3.664062 -0.113281 3.035156 0.1875 2.109375 0.1875 C 1.578125 0.1875 1.117188 0.0390625 0.734375 -0.25 C 0.359375 -0.539062 0.171875 -1.054688 0.171875 -1.796875 L 0.171875 -2.296875 L 1.078125 -2.296875 L 1.078125 -1.796875 C 1.078125 -1.398438 1.160156 -1.101562 1.328125 -0.90625 C 1.503906 -0.71875 1.773438 -0.625 2.140625 -0.625 C 2.660156 -0.625 3 -0.800781 3.15625 -1.15625 C 3.257812 -1.375 3.3125 -1.78125 3.3125 -2.375 L 3.3125 -7.140625 L 4.265625 -7.140625 Z M 4.265625 -2.125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='dbf65fae-ae06-4862-99fa-000972f5777e' overflow='visible'>\n<path d='M 2.703125 -0.5625 C 3.285156 -0.5625 3.679688 -0.78125 3.890625 -1.21875 C 4.109375 -1.65625 4.21875 -2.144531 4.21875 -2.6875 C 4.21875 -3.175781 4.140625 -3.570312 3.984375 -3.875 C 3.742188 -4.351562 3.320312 -4.59375 2.71875 -4.59375 C 2.1875 -4.59375 1.796875 -4.390625 1.546875 -3.984375 C 1.304688 -3.578125 1.1875 -3.082031 1.1875 -2.5 C 1.1875 -1.945312 1.304688 -1.484375 1.546875 -1.109375 C 1.796875 -0.742188 2.179688 -0.5625 2.703125 -0.5625 Z M 2.734375 -5.359375 C 3.410156 -5.359375 3.976562 -5.132812 4.4375 -4.6875 C 4.90625 -4.238281 5.140625 -3.582031 5.140625 -2.71875 C 5.140625 -1.875 4.929688 -1.175781 4.515625 -0.625 C 4.109375 -0.0820312 3.476562 0.1875 2.625 0.1875 C 1.90625 0.1875 1.332031 -0.0507812 0.90625 -0.53125 C 0.488281 -1.019531 0.28125 -1.671875 0.28125 -2.484375 C 0.28125 -3.359375 0.503906 -4.054688 0.953125 -4.578125 C 1.398438 -5.097656 1.992188 -5.359375 2.734375 -5.359375 Z M 2.71875 -5.328125 Z M 2.71875 -5.328125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='f52b2695-c4db-41f6-88d2-f3aa491e2fcd' overflow='visible'>\n<path d='M 1.15625 -1.640625 C 1.1875 -1.347656 1.257812 -1.125 1.375 -0.96875 C 1.59375 -0.6875 1.976562 -0.546875 2.53125 -0.546875 C 2.851562 -0.546875 3.132812 -0.613281 3.375 -0.75 C 3.625 -0.894531 3.75 -1.113281 3.75 -1.40625 C 3.75 -1.625 3.648438 -1.796875 3.453125 -1.921875 C 3.328125 -1.984375 3.078125 -2.0625 2.703125 -2.15625 L 2.015625 -2.34375 C 1.566406 -2.445312 1.238281 -2.566406 1.03125 -2.703125 C 0.65625 -2.941406 0.46875 -3.269531 0.46875 -3.6875 C 0.46875 -4.175781 0.644531 -4.570312 1 -4.875 C 1.351562 -5.1875 1.832031 -5.34375 2.4375 -5.34375 C 3.21875 -5.34375 3.78125 -5.113281 4.125 -4.65625 C 4.34375 -4.363281 4.445312 -4.046875 4.4375 -3.703125 L 3.609375 -3.703125 C 3.597656 -3.898438 3.53125 -4.082031 3.40625 -4.25 C 3.207031 -4.476562 2.859375 -4.59375 2.359375 -4.59375 C 2.023438 -4.59375 1.773438 -4.53125 1.609375 -4.40625 C 1.441406 -4.28125 1.359375 -4.113281 1.359375 -3.90625 C 1.359375 -3.675781 1.46875 -3.492188 1.6875 -3.359375 C 1.820312 -3.273438 2.015625 -3.203125 2.265625 -3.140625 L 2.84375 -3 C 3.46875 -2.851562 3.890625 -2.707031 4.109375 -2.5625 C 4.441406 -2.332031 4.609375 -1.976562 4.609375 -1.5 C 4.609375 -1.039062 4.429688 -0.640625 4.078125 -0.296875 C 3.734375 0.0351562 3.203125 0.203125 2.484375 0.203125 C 1.710938 0.203125 1.164062 0.0234375 0.84375 -0.328125 C 0.519531 -0.679688 0.347656 -1.117188 0.328125 -1.640625 Z M 2.453125 -5.328125 Z M 2.453125 -5.328125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='1ca36580-885e-47e5-b102-89cd46a0408c' overflow='visible'>\n<path d='M 2.8125 -5.328125 C 3.175781 -5.328125 3.53125 -5.238281 3.875 -5.0625 C 4.226562 -4.882812 4.492188 -4.660156 4.671875 -4.390625 C 4.847656 -4.128906 4.96875 -3.820312 5.03125 -3.46875 C 5.082031 -3.226562 5.109375 -2.847656 5.109375 -2.328125 L 1.28125 -2.328125 C 1.300781 -1.796875 1.425781 -1.367188 1.65625 -1.046875 C 1.894531 -0.734375 2.257812 -0.578125 2.75 -0.578125 C 3.207031 -0.578125 3.570312 -0.726562 3.84375 -1.03125 C 4 -1.195312 4.109375 -1.398438 4.171875 -1.640625 L 5.03125 -1.640625 C 5.007812 -1.441406 4.929688 -1.222656 4.796875 -0.984375 C 4.671875 -0.753906 4.53125 -0.566406 4.375 -0.421875 C 4.101562 -0.160156 3.773438 0.015625 3.390625 0.109375 C 3.179688 0.160156 2.941406 0.1875 2.671875 0.1875 C 2.023438 0.1875 1.472656 -0.046875 1.015625 -0.515625 C 0.566406 -0.992188 0.34375 -1.65625 0.34375 -2.5 C 0.34375 -3.332031 0.566406 -4.007812 1.015625 -4.53125 C 1.472656 -5.0625 2.070312 -5.328125 2.8125 -5.328125 Z M 4.203125 -3.015625 C 4.171875 -3.398438 4.085938 -3.703125 3.953125 -3.921875 C 3.710938 -4.347656 3.3125 -4.5625 2.75 -4.5625 C 2.351562 -4.5625 2.019531 -4.414062 1.75 -4.125 C 1.476562 -3.832031 1.332031 -3.460938 1.3125 -3.015625 Z M 2.71875 -5.328125 Z M 2.71875 -5.328125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='14eb422a-af12-430d-b989-46e4e0205c3c' overflow='visible'>\n<path d='M 3.5 -0.828125 C 3.820312 -0.828125 4.09375 -0.859375 4.3125 -0.921875 C 4.6875 -1.054688 4.992188 -1.300781 5.234375 -1.65625 C 5.421875 -1.945312 5.554688 -2.316406 5.640625 -2.765625 C 5.691406 -3.035156 5.71875 -3.285156 5.71875 -3.515625 C 5.71875 -4.398438 5.539062 -5.085938 5.1875 -5.578125 C 4.832031 -6.066406 4.265625 -6.3125 3.484375 -6.3125 L 1.78125 -6.3125 L 1.78125 -0.828125 Z M 0.796875 -7.140625 L 3.6875 -7.140625 C 4.675781 -7.140625 5.441406 -6.789062 5.984375 -6.09375 C 6.460938 -5.46875 6.703125 -4.660156 6.703125 -3.671875 C 6.703125 -2.921875 6.5625 -2.238281 6.28125 -1.625 C 5.769531 -0.539062 4.90625 0 3.6875 0 L 0.796875 0 Z M 0.796875 -7.140625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='1dfb9be4-eb19-440b-95a7-a669197ba06e' overflow='visible'>\n<path d='M 0.640625 -5.1875 L 1.53125 -5.1875 L 1.53125 0 L 0.640625 0 Z M 0.640625 -7.140625 L 1.53125 -7.140625 L 1.53125 -6.140625 L 0.640625 -6.140625 Z M 0.640625 -7.140625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='9df5463d-624a-43ad-b012-5e8f19bd92d7' overflow='visible'>\n<path d='M 0.671875 -5.203125 L 1.5 -5.203125 L 1.5 -4.3125 C 1.5625 -4.476562 1.722656 -4.6875 1.984375 -4.9375 C 2.253906 -5.195312 2.566406 -5.328125 2.921875 -5.328125 C 2.929688 -5.328125 2.957031 -5.320312 3 -5.3125 C 3.039062 -5.3125 3.109375 -5.304688 3.203125 -5.296875 L 3.203125 -4.375 C 3.148438 -4.382812 3.097656 -4.390625 3.046875 -4.390625 C 3.003906 -4.390625 2.957031 -4.390625 2.90625 -4.390625 C 2.46875 -4.390625 2.128906 -4.25 1.890625 -3.96875 C 1.660156 -3.6875 1.546875 -3.363281 1.546875 -3 L 1.546875 0 L 0.671875 0 Z M 0.671875 -5.203125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='94ca4bf9-8a65-48f8-8fad-c8bf73c19929' overflow='visible'>\n<path d='M 1.203125 -2.546875 C 1.203125 -1.984375 1.316406 -1.515625 1.546875 -1.140625 C 1.785156 -0.765625 2.164062 -0.578125 2.6875 -0.578125 C 3.09375 -0.578125 3.425781 -0.75 3.6875 -1.09375 C 3.945312 -1.445312 4.078125 -1.945312 4.078125 -2.59375 C 4.078125 -3.257812 3.941406 -3.75 3.671875 -4.0625 C 3.398438 -4.375 3.066406 -4.53125 2.671875 -4.53125 C 2.234375 -4.53125 1.878906 -4.363281 1.609375 -4.03125 C 1.335938 -3.695312 1.203125 -3.203125 1.203125 -2.546875 Z M 2.515625 -5.296875 C 2.910156 -5.296875 3.242188 -5.210938 3.515625 -5.046875 C 3.671875 -4.953125 3.847656 -4.78125 4.046875 -4.53125 L 4.046875 -7.15625 L 4.890625 -7.15625 L 4.890625 0 L 4.09375 0 L 4.09375 -0.71875 C 3.894531 -0.40625 3.65625 -0.175781 3.375 -0.03125 C 3.09375 0.113281 2.773438 0.1875 2.421875 0.1875 C 1.835938 0.1875 1.332031 -0.0546875 0.90625 -0.546875 C 0.476562 -1.035156 0.265625 -1.6875 0.265625 -2.5 C 0.265625 -3.25 0.457031 -3.898438 0.84375 -4.453125 C 1.238281 -5.015625 1.796875 -5.296875 2.515625 -5.296875 Z M 2.515625 -5.296875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='8e5eef14-8132-481d-b70b-2824ac7ffc6c' overflow='visible'>\n<path d='M 3.765625 -7.328125 C 4.671875 -7.328125 5.375 -7.085938 5.875 -6.609375 C 6.375 -6.140625 6.648438 -5.601562 6.703125 -5 L 5.765625 -5 C 5.648438 -5.457031 5.429688 -5.820312 5.109375 -6.09375 C 4.796875 -6.363281 4.351562 -6.5 3.78125 -6.5 C 3.070312 -6.5 2.5 -6.25 2.0625 -5.75 C 1.632812 -5.257812 1.421875 -4.503906 1.421875 -3.484375 C 1.421875 -2.640625 1.613281 -1.957031 2 -1.4375 C 2.394531 -0.925781 2.976562 -0.671875 3.75 -0.671875 C 4.46875 -0.671875 5.015625 -0.941406 5.390625 -1.484375 C 5.585938 -1.773438 5.734375 -2.15625 5.828125 -2.625 L 6.765625 -2.625 C 6.679688 -1.875 6.40625 -1.242188 5.9375 -0.734375 C 5.363281 -0.117188 4.597656 0.1875 3.640625 0.1875 C 2.804688 0.1875 2.109375 -0.0625 1.546875 -0.5625 C 0.804688 -1.226562 0.4375 -2.253906 0.4375 -3.640625 C 0.4375 -4.691406 0.710938 -5.554688 1.265625 -6.234375 C 1.867188 -6.960938 2.703125 -7.328125 3.765625 -7.328125 Z M 3.578125 -7.328125 Z M 3.578125 -7.328125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='abb23039-e577-4ed0-beb1-d21bc1020176' overflow='visible'>\n<path d='M 0.671875 -7.140625 L 1.546875 -7.140625 L 1.546875 0 L 0.671875 0 Z M 0.671875 -7.140625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='1cb78ce3-365a-41f3-832a-54634e13752c' overflow='visible'>\n<path d='M 0.8125 -6.65625 L 1.703125 -6.65625 L 1.703125 -5.203125 L 2.53125 -5.203125 L 2.53125 -4.484375 L 1.703125 -4.484375 L 1.703125 -1.09375 C 1.703125 -0.914062 1.765625 -0.796875 1.890625 -0.734375 C 1.953125 -0.691406 2.0625 -0.671875 2.21875 -0.671875 C 2.257812 -0.671875 2.304688 -0.671875 2.359375 -0.671875 C 2.410156 -0.679688 2.46875 -0.6875 2.53125 -0.6875 L 2.53125 0 C 2.425781 0.03125 2.320312 0.0507812 2.21875 0.0625 C 2.113281 0.0703125 1.992188 0.078125 1.859375 0.078125 C 1.453125 0.078125 1.175781 -0.0234375 1.03125 -0.234375 C 0.882812 -0.441406 0.8125 -0.710938 0.8125 -1.046875 L 0.8125 -4.484375 L 0.109375 -4.484375 L 0.109375 -5.203125 L 0.8125 -5.203125 Z M 0.8125 -6.65625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='7d6ec591-ef49-4be4-b5f2-cb97084cdbb5' overflow='visible'>\n<path d='M 1.0625 -5.203125 L 2.453125 -0.96875 L 3.90625 -5.203125 L 4.875 -5.203125 L 2.90625 0 L 1.96875 0 L 0.046875 -5.203125 Z M 1.0625 -5.203125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='07d2081b-8e74-404f-b60a-bda15f060dbc' overflow='visible'>\n<path d='M 2.65625 -5.359375 C 3.238281 -5.359375 3.710938 -5.210938 4.078125 -4.921875 C 4.441406 -4.640625 4.664062 -4.148438 4.75 -3.453125 L 3.890625 -3.453125 C 3.835938 -3.773438 3.71875 -4.039062 3.53125 -4.25 C 3.351562 -4.46875 3.0625 -4.578125 2.65625 -4.578125 C 2.09375 -4.578125 1.691406 -4.300781 1.453125 -3.75 C 1.296875 -3.40625 1.21875 -2.972656 1.21875 -2.453125 C 1.21875 -1.929688 1.328125 -1.488281 1.546875 -1.125 C 1.765625 -0.769531 2.113281 -0.59375 2.59375 -0.59375 C 2.957031 -0.59375 3.242188 -0.703125 3.453125 -0.921875 C 3.660156 -1.148438 3.804688 -1.457031 3.890625 -1.84375 L 4.75 -1.84375 C 4.644531 -1.15625 4.398438 -0.648438 4.015625 -0.328125 C 3.628906 -0.00390625 3.128906 0.15625 2.515625 0.15625 C 1.835938 0.15625 1.296875 -0.09375 0.890625 -0.59375 C 0.484375 -1.09375 0.28125 -1.710938 0.28125 -2.453125 C 0.28125 -3.367188 0.503906 -4.082031 0.953125 -4.59375 C 1.398438 -5.101562 1.96875 -5.359375 2.65625 -5.359375 Z M 2.515625 -5.328125 Z M 2.515625 -5.328125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='44e7ce1f-601c-4c57-9df2-a5fdc2850847' overflow='visible'>\n<path d='M 4.421875 -2.921875 L 3.34375 -6.078125 L 2.1875 -2.921875 Z M 2.828125 -7.140625 L 3.921875 -7.140625 L 6.515625 0 L 5.453125 0 L 4.734375 -2.140625 L 1.90625 -2.140625 L 1.140625 0 L 0.140625 0 Z M 3.328125 -7.140625 Z M 3.328125 -7.140625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='9e535106-c458-47bf-a194-050224a1abb3' overflow='visible'>\n<path d='M 0.640625 -5.203125 L 1.5 -5.203125 L 1.5 -4.46875 C 1.707031 -4.71875 1.898438 -4.90625 2.078125 -5.03125 C 2.359375 -5.226562 2.679688 -5.328125 3.046875 -5.328125 C 3.460938 -5.328125 3.796875 -5.222656 4.046875 -5.015625 C 4.191406 -4.898438 4.320312 -4.726562 4.4375 -4.5 C 4.632812 -4.78125 4.863281 -4.988281 5.125 -5.125 C 5.382812 -5.257812 5.679688 -5.328125 6.015625 -5.328125 C 6.710938 -5.328125 7.1875 -5.070312 7.4375 -4.5625 C 7.570312 -4.289062 7.640625 -3.925781 7.640625 -3.46875 L 7.640625 0 L 6.734375 0 L 6.734375 -3.609375 C 6.734375 -3.960938 6.644531 -4.203125 6.46875 -4.328125 C 6.300781 -4.460938 6.09375 -4.53125 5.84375 -4.53125 C 5.5 -4.53125 5.203125 -4.410156 4.953125 -4.171875 C 4.703125 -3.941406 4.578125 -3.5625 4.578125 -3.03125 L 4.578125 0 L 3.6875 0 L 3.6875 -3.390625 C 3.6875 -3.742188 3.644531 -4.003906 3.5625 -4.171875 C 3.4375 -4.410156 3.191406 -4.53125 2.828125 -4.53125 C 2.492188 -4.53125 2.191406 -4.398438 1.921875 -4.140625 C 1.648438 -3.890625 1.515625 -3.429688 1.515625 -2.765625 L 1.515625 0 L 0.640625 0 Z M 0.640625 -5.203125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='08a486c0-2d6a-4d56-9616-54db256de353' overflow='visible'>\n<path d='M 0.578125 -7.15625 L 1.421875 -7.15625 L 1.421875 -4.578125 C 1.609375 -4.828125 1.832031 -5.015625 2.09375 -5.140625 C 2.363281 -5.273438 2.65625 -5.34375 2.96875 -5.34375 C 3.625 -5.34375 4.148438 -5.117188 4.546875 -4.671875 C 4.953125 -4.222656 5.15625 -3.566406 5.15625 -2.703125 C 5.15625 -1.878906 4.953125 -1.191406 4.546875 -0.640625 C 4.148438 -0.0976562 3.601562 0.171875 2.90625 0.171875 C 2.507812 0.171875 2.175781 0.0703125 1.90625 -0.125 C 1.738281 -0.238281 1.5625 -0.421875 1.375 -0.671875 L 1.375 0 L 0.578125 0 Z M 2.84375 -0.609375 C 3.320312 -0.609375 3.675781 -0.796875 3.90625 -1.171875 C 4.144531 -1.546875 4.265625 -2.039062 4.265625 -2.65625 C 4.265625 -3.207031 4.144531 -3.660156 3.90625 -4.015625 C 3.675781 -4.378906 3.332031 -4.5625 2.875 -4.5625 C 2.476562 -4.5625 2.128906 -4.410156 1.828125 -4.109375 C 1.523438 -3.816406 1.375 -3.332031 1.375 -2.65625 C 1.375 -2.164062 1.4375 -1.769531 1.5625 -1.46875 C 1.789062 -0.894531 2.21875 -0.609375 2.84375 -0.609375 Z M 2.84375 -0.609375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='553fc5e8-e5a3-452c-8e81-1d777c4746a7' overflow='visible'>\n<path d='M 0.84375 -7.140625 L 4.0625 -7.140625 C 4.695312 -7.140625 5.207031 -6.957031 5.59375 -6.59375 C 5.988281 -6.238281 6.1875 -5.738281 6.1875 -5.09375 C 6.1875 -4.53125 6.007812 -4.039062 5.65625 -3.625 C 5.3125 -3.21875 4.78125 -3.015625 4.0625 -3.015625 L 1.8125 -3.015625 L 1.8125 0 L 0.84375 0 Z M 5.203125 -5.09375 C 5.203125 -5.613281 5.007812 -5.96875 4.625 -6.15625 C 4.40625 -6.257812 4.109375 -6.3125 3.734375 -6.3125 L 1.8125 -6.3125 L 1.8125 -3.828125 L 3.734375 -3.828125 C 4.171875 -3.828125 4.523438 -3.921875 4.796875 -4.109375 C 5.066406 -4.296875 5.203125 -4.625 5.203125 -5.09375 Z M 5.203125 -5.09375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='4d9ce2bd-e255-4439-9642-bb53937b4394' overflow='visible'>\n<path d='M 1.203125 -2.53125 C 1.203125 -2.082031 1.265625 -1.707031 1.390625 -1.40625 C 1.617188 -0.863281 2.019531 -0.59375 2.59375 -0.59375 C 3.207031 -0.59375 3.628906 -0.875 3.859375 -1.4375 C 3.984375 -1.757812 4.046875 -2.164062 4.046875 -2.65625 C 4.046875 -3.101562 3.972656 -3.476562 3.828125 -3.78125 C 3.597656 -4.289062 3.179688 -4.546875 2.578125 -4.546875 C 2.203125 -4.546875 1.878906 -4.378906 1.609375 -4.046875 C 1.335938 -3.722656 1.203125 -3.21875 1.203125 -2.53125 Z M 2.5 -5.328125 C 2.925781 -5.328125 3.289062 -5.21875 3.59375 -5 C 3.75 -4.882812 3.898438 -4.710938 4.046875 -4.484375 L 4.046875 -5.203125 L 4.890625 -5.203125 L 4.890625 2.078125 L 4 2.078125 L 4 -0.59375 C 3.851562 -0.363281 3.648438 -0.175781 3.390625 -0.03125 C 3.140625 0.101562 2.820312 0.171875 2.4375 0.171875 C 1.882812 0.171875 1.390625 -0.046875 0.953125 -0.484375 C 0.515625 -0.921875 0.296875 -1.582031 0.296875 -2.46875 C 0.296875 -3.300781 0.5 -3.984375 0.90625 -4.515625 C 1.3125 -5.054688 1.84375 -5.328125 2.5 -5.328125 Z M 2.5 -5.328125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='8bad88ae-7a7a-47aa-bdfe-88ec054c732b' overflow='visible'>\n<path d='M 1.515625 -5.203125 L 1.515625 -1.75 C 1.515625 -1.476562 1.554688 -1.257812 1.640625 -1.09375 C 1.796875 -0.78125 2.085938 -0.625 2.515625 -0.625 C 3.117188 -0.625 3.53125 -0.898438 3.75 -1.453125 C 3.875 -1.742188 3.9375 -2.144531 3.9375 -2.65625 L 3.9375 -5.203125 L 4.8125 -5.203125 L 4.8125 0 L 3.984375 0 L 4 -0.765625 C 3.882812 -0.566406 3.742188 -0.398438 3.578125 -0.265625 C 3.242188 0.00390625 2.835938 0.140625 2.359375 0.140625 C 1.617188 0.140625 1.113281 -0.109375 0.84375 -0.609375 C 0.695312 -0.867188 0.625 -1.222656 0.625 -1.671875 L 0.625 -5.203125 Z M 2.71875 -5.328125 Z M 2.71875 -5.328125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='8144c006-357a-4df4-8253-57c8a075ec89' overflow='visible'>\n<path d='M 0.953125 -4.921875 L 0.953125 -5.59375 C 1.585938 -5.65625 2.03125 -5.757812 2.28125 -5.90625 C 2.53125 -6.050781 2.710938 -6.394531 2.828125 -6.9375 L 3.53125 -6.9375 L 3.53125 0 L 2.59375 0 L 2.59375 -4.921875 Z M 0.953125 -4.921875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='2c470afe-b012-4401-a6ab-6d28e3e61cdc' overflow='visible'>\n<path d='M 2.84375 -0.59375 C 3.25 -0.59375 3.585938 -0.757812 3.859375 -1.09375 C 4.128906 -1.4375 4.265625 -1.953125 4.265625 -2.640625 C 4.265625 -3.046875 4.203125 -3.398438 4.078125 -3.703125 C 3.847656 -4.273438 3.4375 -4.5625 2.84375 -4.5625 C 2.226562 -4.5625 1.8125 -4.257812 1.59375 -3.65625 C 1.476562 -3.332031 1.421875 -2.921875 1.421875 -2.421875 C 1.421875 -2.015625 1.476562 -1.671875 1.59375 -1.390625 C 1.820312 -0.859375 2.238281 -0.59375 2.84375 -0.59375 Z M 0.578125 -5.1875 L 1.421875 -5.1875 L 1.421875 -4.484375 C 1.597656 -4.722656 1.789062 -4.910156 2 -5.046875 C 2.289062 -5.234375 2.632812 -5.328125 3.03125 -5.328125 C 3.625 -5.328125 4.125 -5.097656 4.53125 -4.640625 C 4.945312 -4.191406 5.15625 -3.550781 5.15625 -2.71875 C 5.15625 -1.582031 4.859375 -0.769531 4.265625 -0.28125 C 3.890625 0.0195312 3.453125 0.171875 2.953125 0.171875 C 2.554688 0.171875 2.226562 0.0859375 1.96875 -0.078125 C 1.8125 -0.179688 1.640625 -0.347656 1.453125 -0.578125 L 1.453125 2.078125 L 0.578125 2.078125 Z M 0.578125 -5.1875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='848a7f06-2ec9-4d02-87ce-3a59ec7f6c9f' overflow='visible'>\n<path d='M 1.046875 -5.203125 L 2.046875 -1.109375 L 3.0625 -5.203125 L 4.046875 -5.203125 L 5.0625 -1.125 L 6.125 -5.203125 L 7 -5.203125 L 5.484375 0 L 4.578125 0 L 3.53125 -4.03125 L 2.5 0 L 1.59375 0 L 0.09375 -5.203125 Z M 1.046875 -5.203125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='2bb104cd-2d57-457c-b53e-938fc695bd72' overflow='visible'>\n<path d='M 3.890625 -5.203125 L 4.859375 -5.203125 C 4.734375 -4.867188 4.457031 -4.109375 4.03125 -2.921875 C 3.71875 -2.023438 3.453125 -1.296875 3.234375 -0.734375 C 2.734375 0.597656 2.378906 1.410156 2.171875 1.703125 C 1.960938 1.992188 1.609375 2.140625 1.109375 2.140625 C 0.984375 2.140625 0.882812 2.132812 0.8125 2.125 C 0.75 2.113281 0.671875 2.09375 0.578125 2.0625 L 0.578125 1.265625 C 0.722656 1.304688 0.828125 1.332031 0.890625 1.34375 C 0.960938 1.351562 1.023438 1.359375 1.078125 1.359375 C 1.242188 1.359375 1.363281 1.332031 1.4375 1.28125 C 1.519531 1.226562 1.585938 1.160156 1.640625 1.078125 C 1.648438 1.054688 1.703125 0.925781 1.796875 0.6875 C 1.898438 0.445312 1.976562 0.269531 2.03125 0.15625 L 0.109375 -5.203125 L 1.09375 -5.203125 L 2.484375 -0.96875 Z M 2.484375 -5.328125 Z M 2.484375 -5.328125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='7d61d690-8adf-4ed0-94d5-2a74856d0189' overflow='visible'>\n<path d='M 0.78125 -7.140625 L 1.765625 -7.140625 L 1.765625 -4.1875 L 5.46875 -4.1875 L 5.46875 -7.140625 L 6.453125 -7.140625 L 6.453125 0 L 5.46875 0 L 5.46875 -3.34375 L 1.765625 -3.34375 L 1.765625 0 L 0.78125 0 Z M 0.78125 -7.140625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='41cdac45-8b65-4365-af0a-db5a1b22687b' overflow='visible'>\n<path d='M 0.734375 -7.140625 L 2.125 -7.140625 L 4.171875 -1.109375 L 6.203125 -7.140625 L 7.578125 -7.140625 L 7.578125 0 L 6.65625 0 L 6.65625 -4.21875 C 6.65625 -4.363281 6.65625 -4.601562 6.65625 -4.9375 C 6.664062 -5.269531 6.671875 -5.628906 6.671875 -6.015625 L 4.640625 0 L 3.6875 0 L 1.640625 -6.015625 L 1.640625 -5.796875 C 1.640625 -5.628906 1.640625 -5.363281 1.640625 -5 C 1.648438 -4.644531 1.65625 -4.382812 1.65625 -4.21875 L 1.65625 0 L 0.734375 0 Z M 0.734375 -7.140625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='e0a92b58-cd5b-47e2-b121-ebaa6da8e043' overflow='visible'>\n<path d='M 0.765625 -7.140625 L 1.71875 -7.140625 L 1.71875 -0.84375 L 5.34375 -0.84375 L 5.34375 0 L 0.765625 0 Z M 0.765625 -7.140625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='33ba5ef5-6dd1-4a6d-9573-82b5f55d1905' overflow='visible'>\n<path d='M 0.765625 -7.140625 L 1.703125 -7.140625 L 1.703125 -3.65625 L 5.1875 -7.140625 L 6.515625 -7.140625 L 3.546875 -4.265625 L 6.59375 0 L 5.34375 0 L 2.84375 -3.578125 L 1.703125 -2.484375 L 1.703125 0 L 0.765625 0 Z M 0.765625 -7.140625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='95ba0e07-9c88-4540-b5de-6e3d71a035f5' overflow='visible'>\n<path d='M 1.8125 -7.140625 L 1.8125 -2.71875 C 1.8125 -2.207031 1.90625 -1.78125 2.09375 -1.4375 C 2.382812 -0.914062 2.875 -0.65625 3.5625 -0.65625 C 4.375 -0.65625 4.929688 -0.9375 5.234375 -1.5 C 5.390625 -1.800781 5.46875 -2.207031 5.46875 -2.71875 L 5.46875 -7.140625 L 6.453125 -7.140625 L 6.453125 -3.125 C 6.453125 -2.25 6.332031 -1.578125 6.09375 -1.109375 C 5.65625 -0.242188 4.835938 0.1875 3.640625 0.1875 C 2.429688 0.1875 1.613281 -0.242188 1.1875 -1.109375 C 0.945312 -1.578125 0.828125 -2.25 0.828125 -3.125 L 0.828125 -7.140625 Z M 3.640625 -7.140625 Z M 3.640625 -7.140625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='8ac2822a-ad76-460c-a782-6cf3fcc0f821' overflow='visible'>\n<path d='M 3.296875 -2.46875 L 3.296875 -5.625 L 1.0625 -2.46875 Z M 3.3125 0 L 3.3125 -1.703125 L 0.25 -1.703125 L 0.25 -2.5625 L 3.4375 -6.984375 L 4.171875 -6.984375 L 4.171875 -2.46875 L 5.203125 -2.46875 L 5.203125 -1.703125 L 4.171875 -1.703125 L 4.171875 0 Z M 3.3125 0 ' style='stroke:none;'/>\n</symbol>\n<symbol id='fea978f8-7687-4c73-94fe-43f4f0bc3a1e' overflow='visible'>\n<path d='M 0.640625 -7.15625 L 1.515625 -7.15625 L 1.515625 -4.5 C 1.722656 -4.757812 1.910156 -4.941406 2.078125 -5.046875 C 2.359375 -5.234375 2.707031 -5.328125 3.125 -5.328125 C 3.882812 -5.328125 4.398438 -5.066406 4.671875 -4.546875 C 4.816406 -4.253906 4.890625 -3.851562 4.890625 -3.34375 L 4.890625 0 L 3.984375 0 L 3.984375 -3.28125 C 3.984375 -3.664062 3.9375 -3.945312 3.84375 -4.125 C 3.675781 -4.40625 3.375 -4.546875 2.9375 -4.546875 C 2.582031 -4.546875 2.253906 -4.421875 1.953125 -4.171875 C 1.660156 -3.921875 1.515625 -3.453125 1.515625 -2.765625 L 1.515625 0 L 0.640625 0 Z M 0.640625 -7.15625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='70b7e0c8-9a8c-45c5-972f-8a072643f9a7' overflow='visible'>\n<path d='M 0.625 -7.140625 L 1.46875 -7.140625 L 1.46875 -3 L 3.703125 -5.203125 L 4.828125 -5.203125 L 2.828125 -3.25 L 4.9375 0 L 3.8125 0 L 2.203125 -2.625 L 1.46875 -1.953125 L 1.46875 0 L 0.625 0 Z M 0.625 -7.140625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='2a022af8-3a0b-46ac-afc9-a6c6e4fc0ea8' overflow='visible'>\n<path d='M 3.84375 -7.328125 C 4.507812 -7.328125 5.085938 -7.195312 5.578125 -6.9375 C 6.285156 -6.5625 6.722656 -5.910156 6.890625 -4.984375 L 5.921875 -4.984375 C 5.804688 -5.503906 5.566406 -5.878906 5.203125 -6.109375 C 4.835938 -6.347656 4.375 -6.46875 3.8125 -6.46875 C 3.15625 -6.46875 2.597656 -6.21875 2.140625 -5.71875 C 1.691406 -5.226562 1.46875 -4.492188 1.46875 -3.515625 C 1.46875 -2.660156 1.65625 -1.96875 2.03125 -1.4375 C 2.40625 -0.90625 3.015625 -0.640625 3.859375 -0.640625 C 4.503906 -0.640625 5.035156 -0.828125 5.453125 -1.203125 C 5.878906 -1.578125 6.097656 -2.179688 6.109375 -3.015625 L 3.875 -3.015625 L 3.875 -3.8125 L 7 -3.8125 L 7 0 L 6.375 0 L 6.140625 -0.921875 C 5.816406 -0.554688 5.53125 -0.304688 5.28125 -0.171875 C 4.851562 0.0664062 4.3125 0.1875 3.65625 0.1875 C 2.8125 0.1875 2.085938 -0.0820312 1.484375 -0.625 C 0.816406 -1.3125 0.484375 -2.253906 0.484375 -3.453125 C 0.484375 -4.648438 0.804688 -5.601562 1.453125 -6.3125 C 2.066406 -6.988281 2.863281 -7.328125 3.84375 -7.328125 Z M 3.6875 -7.328125 Z M 3.6875 -7.328125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='998d2628-c24e-486f-9464-6673a1f907c5' overflow='visible'>\n<path d='M 2.265625 -7.140625 L 3 -7.140625 L 0.734375 0 L 0 0 Z M 2.265625 -7.140625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='0aec2617-bc8d-41ac-b492-83548ada7e54' overflow='visible'>\n<path d='M 0.40625 -3.21875 L 2.859375 -3.21875 L 2.859375 -2.328125 L 0.40625 -2.328125 Z M 0.40625 -3.21875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='60e4ab49-c824-444f-89d7-e92f0b7622cc' overflow='visible'>\n<path d='M 1.328125 -1.6875 C 1.347656 -1.195312 1.53125 -0.863281 1.875 -0.6875 C 2.050781 -0.582031 2.253906 -0.53125 2.484375 -0.53125 C 2.898438 -0.53125 3.253906 -0.703125 3.546875 -1.046875 C 3.847656 -1.398438 4.054688 -2.109375 4.171875 -3.171875 C 3.984375 -2.867188 3.742188 -2.65625 3.453125 -2.53125 C 3.171875 -2.40625 2.863281 -2.34375 2.53125 -2.34375 C 1.863281 -2.34375 1.332031 -2.550781 0.9375 -2.96875 C 0.550781 -3.382812 0.359375 -3.921875 0.359375 -4.578125 C 0.359375 -5.210938 0.550781 -5.769531 0.9375 -6.25 C 1.320312 -6.726562 1.890625 -6.96875 2.640625 -6.96875 C 3.648438 -6.96875 4.347656 -6.507812 4.734375 -5.59375 C 4.953125 -5.09375 5.0625 -4.46875 5.0625 -3.71875 C 5.0625 -2.863281 4.929688 -2.109375 4.671875 -1.453125 C 4.242188 -0.359375 3.523438 0.1875 2.515625 0.1875 C 1.835938 0.1875 1.320312 0.0078125 0.96875 -0.34375 C 0.613281 -0.695312 0.4375 -1.144531 0.4375 -1.6875 Z M 2.65625 -3.109375 C 3 -3.109375 3.3125 -3.222656 3.59375 -3.453125 C 3.882812 -3.679688 4.03125 -4.082031 4.03125 -4.65625 C 4.03125 -5.164062 3.898438 -5.546875 3.640625 -5.796875 C 3.378906 -6.046875 3.050781 -6.171875 2.65625 -6.171875 C 2.226562 -6.171875 1.890625 -6.023438 1.640625 -5.734375 C 1.390625 -5.453125 1.265625 -5.070312 1.265625 -4.59375 C 1.265625 -4.132812 1.375 -3.769531 1.59375 -3.5 C 1.8125 -3.238281 2.164062 -3.109375 2.65625 -3.109375 Z M 2.65625 -3.109375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='d62847a4-b70e-498f-b650-0fa63f23d7cf' overflow='visible'>\n<path d='M 4.078125 -3.875 C 4.535156 -3.875 4.894531 -3.960938 5.15625 -4.140625 C 5.414062 -4.316406 5.546875 -4.644531 5.546875 -5.125 C 5.546875 -5.632812 5.363281 -5.976562 5 -6.15625 C 4.800781 -6.257812 4.535156 -6.3125 4.203125 -6.3125 L 1.84375 -6.3125 L 1.84375 -3.875 Z M 0.875 -7.140625 L 4.171875 -7.140625 C 4.722656 -7.140625 5.175781 -7.0625 5.53125 -6.90625 C 6.195312 -6.59375 6.53125 -6.03125 6.53125 -5.21875 C 6.53125 -4.789062 6.441406 -4.4375 6.265625 -4.15625 C 6.085938 -3.882812 5.84375 -3.671875 5.53125 -3.515625 C 5.800781 -3.398438 6.003906 -3.25 6.140625 -3.0625 C 6.285156 -2.875 6.367188 -2.570312 6.390625 -2.15625 L 6.421875 -1.203125 C 6.429688 -0.929688 6.457031 -0.726562 6.5 -0.59375 C 6.550781 -0.363281 6.648438 -0.21875 6.796875 -0.15625 L 6.796875 0 L 5.625 0 C 5.582031 -0.0625 5.550781 -0.140625 5.53125 -0.234375 C 5.519531 -0.335938 5.503906 -0.523438 5.484375 -0.796875 L 5.4375 -2 C 5.414062 -2.457031 5.238281 -2.769531 4.90625 -2.9375 C 4.726562 -3.019531 4.441406 -3.0625 4.046875 -3.0625 L 1.84375 -3.0625 L 1.84375 0 L 0.875 0 Z M 0.875 -7.140625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='7f521d5f-886f-451e-abfa-034e01bcff17' overflow='visible'>\n<path d='M 0.21875 0 L 0.21875 -5.015625 L 4.21875 -5.015625 L 4.21875 0 Z M 3.578125 -0.625 L 3.578125 -4.390625 L 0.859375 -4.390625 L 0.859375 -0.625 Z M 3.578125 -0.625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='17812813-511b-484d-b2cd-c0e84459a10c' overflow='visible'>\n<path d='M 1.8125 0.140625 C 1.238281 0.140625 0.820312 -0.0195312 0.5625 -0.34375 C 0.300781 -0.664062 0.171875 -1.050781 0.171875 -1.5 L 0.8125 -1.5 C 0.832031 -1.1875 0.890625 -0.957031 0.984375 -0.8125 C 1.148438 -0.550781 1.441406 -0.421875 1.859375 -0.421875 C 2.179688 -0.421875 2.4375 -0.507812 2.625 -0.6875 C 2.820312 -0.863281 2.921875 -1.085938 2.921875 -1.359375 C 2.921875 -1.691406 2.816406 -1.925781 2.609375 -2.0625 C 2.410156 -2.195312 2.125 -2.265625 1.75 -2.265625 C 1.71875 -2.265625 1.675781 -2.265625 1.625 -2.265625 C 1.582031 -2.265625 1.539062 -2.265625 1.5 -2.265625 L 1.5 -2.796875 C 1.5625 -2.796875 1.613281 -2.789062 1.65625 -2.78125 C 1.707031 -2.78125 1.757812 -2.78125 1.8125 -2.78125 C 2.039062 -2.78125 2.226562 -2.816406 2.375 -2.890625 C 2.632812 -3.023438 2.765625 -3.253906 2.765625 -3.578125 C 2.765625 -3.816406 2.675781 -4.003906 2.5 -4.140625 C 2.332031 -4.273438 2.128906 -4.34375 1.890625 -4.34375 C 1.484375 -4.34375 1.203125 -4.207031 1.046875 -3.9375 C 0.960938 -3.78125 0.914062 -3.5625 0.90625 -3.28125 L 0.296875 -3.28125 C 0.296875 -3.644531 0.367188 -3.957031 0.515625 -4.21875 C 0.765625 -4.675781 1.203125 -4.90625 1.828125 -4.90625 C 2.328125 -4.90625 2.710938 -4.789062 2.984375 -4.5625 C 3.265625 -4.34375 3.40625 -4.023438 3.40625 -3.609375 C 3.40625 -3.304688 3.320312 -3.0625 3.15625 -2.875 C 3.0625 -2.757812 2.929688 -2.664062 2.765625 -2.59375 C 3.023438 -2.519531 3.226562 -2.382812 3.375 -2.1875 C 3.519531 -1.988281 3.59375 -1.742188 3.59375 -1.453125 C 3.59375 -0.984375 3.4375 -0.597656 3.125 -0.296875 C 2.820312 -0.00390625 2.382812 0.140625 1.8125 0.140625 Z M 1.8125 0.140625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='1e281205-001a-46d9-bec2-f59d8e6ffad0' overflow='visible'>\n<path d='M 3.65625 -4.8125 L 3.65625 -4.28125 C 3.5 -4.125 3.289062 -3.851562 3.03125 -3.46875 C 2.769531 -3.09375 2.539062 -2.6875 2.34375 -2.25 C 2.144531 -1.820312 1.992188 -1.4375 1.890625 -1.09375 C 1.828125 -0.863281 1.742188 -0.5 1.640625 0 L 0.953125 0 C 1.109375 -0.9375 1.453125 -1.867188 1.984375 -2.796875 C 2.296875 -3.335938 2.625 -3.800781 2.96875 -4.1875 L 0.25 -4.1875 L 0.25 -4.8125 Z M 3.65625 -4.8125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='22c87d1e-6a10-484e-8a95-086a601e3558' overflow='visible'>\n<path d='M 0.59375 -0.75 L 1.3125 -0.75 L 1.3125 0 L 0.59375 0 Z M 0.59375 -0.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='ea7daa72-002e-4fd2-86cc-efe8d5035022' overflow='visible'>\n<path d='M 1.890625 -4.890625 C 2.523438 -4.890625 2.984375 -4.628906 3.265625 -4.109375 C 3.484375 -3.703125 3.59375 -3.148438 3.59375 -2.453125 C 3.59375 -1.785156 3.492188 -1.234375 3.296875 -0.796875 C 3.015625 -0.171875 2.546875 0.140625 1.890625 0.140625 C 1.304688 0.140625 0.867188 -0.113281 0.578125 -0.625 C 0.335938 -1.050781 0.21875 -1.625 0.21875 -2.34375 C 0.21875 -2.90625 0.289062 -3.382812 0.4375 -3.78125 C 0.707031 -4.519531 1.191406 -4.890625 1.890625 -4.890625 Z M 1.890625 -0.421875 C 2.203125 -0.421875 2.453125 -0.5625 2.640625 -0.84375 C 2.835938 -1.132812 2.9375 -1.664062 2.9375 -2.4375 C 2.9375 -2.988281 2.863281 -3.441406 2.71875 -3.796875 C 2.582031 -4.160156 2.320312 -4.34375 1.9375 -4.34375 C 1.570312 -4.34375 1.304688 -4.171875 1.140625 -3.828125 C 0.972656 -3.484375 0.890625 -2.984375 0.890625 -2.328125 C 0.890625 -1.828125 0.941406 -1.425781 1.046875 -1.125 C 1.210938 -0.65625 1.492188 -0.421875 1.890625 -0.421875 Z M 1.890625 -0.421875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='bf8171fa-331e-4c5e-ab9f-95e9f07300d7' overflow='visible'>\n<path d='M 0.859375 -1.25 C 0.898438 -0.894531 1.066406 -0.648438 1.359375 -0.515625 C 1.503906 -0.453125 1.671875 -0.421875 1.859375 -0.421875 C 2.222656 -0.421875 2.492188 -0.535156 2.671875 -0.765625 C 2.847656 -0.992188 2.9375 -1.25 2.9375 -1.53125 C 2.9375 -1.875 2.832031 -2.140625 2.625 -2.328125 C 2.414062 -2.515625 2.164062 -2.609375 1.875 -2.609375 C 1.65625 -2.609375 1.46875 -2.566406 1.3125 -2.484375 C 1.164062 -2.398438 1.039062 -2.285156 0.9375 -2.140625 L 0.40625 -2.171875 L 0.78125 -4.8125 L 3.3125 -4.8125 L 3.3125 -4.21875 L 1.234375 -4.21875 L 1.03125 -2.859375 C 1.144531 -2.941406 1.253906 -3.003906 1.359375 -3.046875 C 1.535156 -3.128906 1.742188 -3.171875 1.984375 -3.171875 C 2.429688 -3.171875 2.8125 -3.023438 3.125 -2.734375 C 3.4375 -2.441406 3.59375 -2.070312 3.59375 -1.625 C 3.59375 -1.164062 3.445312 -0.757812 3.15625 -0.40625 C 2.875 -0.0507812 2.421875 0.125 1.796875 0.125 C 1.398438 0.125 1.046875 0.015625 0.734375 -0.203125 C 0.429688 -0.429688 0.257812 -0.78125 0.21875 -1.25 Z M 0.859375 -1.25 ' style='stroke:none;'/>\n</symbol>\n<symbol id='a9ce67cc-8d4e-4732-a9bb-4b4bdd3e3785' overflow='visible'>\n<path d='M 2.3125 -1.734375 L 2.3125 -3.953125 L 0.75 -1.734375 Z M 2.328125 0 L 2.328125 -1.203125 L 0.171875 -1.203125 L 0.171875 -1.796875 L 2.421875 -4.90625 L 2.9375 -4.90625 L 2.9375 -1.734375 L 3.65625 -1.734375 L 3.65625 -1.203125 L 2.9375 -1.203125 L 2.9375 0 Z M 2.328125 0 ' style='stroke:none;'/>\n</symbol>\n<symbol id='2338419e-ace9-4091-ba20-2e98a11c37c9' overflow='visible'>\n<path d='M 0.296875 -2.265625 L 2 -2.265625 L 2 -1.640625 L 0.296875 -1.640625 Z M 0.296875 -2.265625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='286ae716-d55e-4ebd-8921-8b7558ff08f9' overflow='visible'>\n<path d='M 0.671875 -3.46875 L 0.671875 -3.9375 C 1.117188 -3.976562 1.425781 -4.050781 1.59375 -4.15625 C 1.769531 -4.257812 1.90625 -4.5 2 -4.875 L 2.484375 -4.875 L 2.484375 0 L 1.828125 0 L 1.828125 -3.46875 Z M 0.671875 -3.46875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='c55069bc-2767-481a-ab7a-58c251e200a8' overflow='visible'>\n<path d='M 0.21875 0 C 0.238281 -0.414062 0.320312 -0.78125 0.46875 -1.09375 C 0.625 -1.40625 0.925781 -1.691406 1.375 -1.953125 L 2.03125 -2.328125 C 2.320312 -2.503906 2.523438 -2.648438 2.640625 -2.765625 C 2.828125 -2.953125 2.921875 -3.171875 2.921875 -3.421875 C 2.921875 -3.703125 2.832031 -3.925781 2.65625 -4.09375 C 2.488281 -4.257812 2.265625 -4.34375 1.984375 -4.34375 C 1.554688 -4.34375 1.265625 -4.1875 1.109375 -3.875 C 1.023438 -3.695312 0.976562 -3.457031 0.96875 -3.15625 L 0.34375 -3.15625 C 0.34375 -3.582031 0.421875 -3.929688 0.578125 -4.203125 C 0.835938 -4.671875 1.304688 -4.90625 1.984375 -4.90625 C 2.546875 -4.90625 2.957031 -4.753906 3.21875 -4.453125 C 3.476562 -4.148438 3.609375 -3.816406 3.609375 -3.453125 C 3.609375 -3.054688 3.46875 -2.71875 3.1875 -2.4375 C 3.03125 -2.28125 2.742188 -2.085938 2.328125 -1.859375 L 1.859375 -1.59375 C 1.640625 -1.46875 1.46875 -1.347656 1.34375 -1.234375 C 1.113281 -1.035156 0.96875 -0.816406 0.90625 -0.578125 L 3.578125 -0.578125 L 3.578125 0 Z M 0.21875 0 ' style='stroke:none;'/>\n</symbol>\n<symbol id='b3ec2faa-e4a2-41cc-a4d2-078152e5e884' overflow='visible'>\n<path d='M 0.9375 -1.1875 C 0.945312 -0.84375 1.078125 -0.609375 1.328125 -0.484375 C 1.453125 -0.410156 1.59375 -0.375 1.75 -0.375 C 2.039062 -0.375 2.289062 -0.492188 2.5 -0.734375 C 2.707031 -0.984375 2.851562 -1.484375 2.9375 -2.234375 C 2.800781 -2.015625 2.628906 -1.859375 2.421875 -1.765625 C 2.222656 -1.679688 2.007812 -1.640625 1.78125 -1.640625 C 1.3125 -1.640625 0.9375 -1.785156 0.65625 -2.078125 C 0.382812 -2.378906 0.25 -2.757812 0.25 -3.21875 C 0.25 -3.664062 0.382812 -4.054688 0.65625 -4.390625 C 0.925781 -4.734375 1.328125 -4.90625 1.859375 -4.90625 C 2.566406 -4.90625 3.054688 -4.582031 3.328125 -3.9375 C 3.484375 -3.582031 3.5625 -3.140625 3.5625 -2.609375 C 3.5625 -2.015625 3.46875 -1.484375 3.28125 -1.015625 C 2.988281 -0.242188 2.484375 0.140625 1.765625 0.140625 C 1.285156 0.140625 0.921875 0.015625 0.671875 -0.234375 C 0.429688 -0.484375 0.3125 -0.800781 0.3125 -1.1875 Z M 1.859375 -2.1875 C 2.109375 -2.1875 2.332031 -2.265625 2.53125 -2.421875 C 2.726562 -2.585938 2.828125 -2.867188 2.828125 -3.265625 C 2.828125 -3.628906 2.738281 -3.898438 2.5625 -4.078125 C 2.382812 -4.253906 2.15625 -4.34375 1.875 -4.34375 C 1.570312 -4.34375 1.332031 -4.238281 1.15625 -4.03125 C 0.976562 -3.832031 0.890625 -3.566406 0.890625 -3.234375 C 0.890625 -2.910156 0.96875 -2.65625 1.125 -2.46875 C 1.28125 -2.28125 1.523438 -2.1875 1.859375 -2.1875 Z M 1.859375 -2.1875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='cb0041e0-c7c9-4e38-8e74-0f0d21366b3f' overflow='visible'>\n<path d='M 1.90625 -2.84375 C 2.175781 -2.84375 2.382812 -2.914062 2.53125 -3.0625 C 2.6875 -3.21875 2.765625 -3.398438 2.765625 -3.609375 C 2.765625 -3.796875 2.691406 -3.960938 2.546875 -4.109375 C 2.398438 -4.265625 2.179688 -4.34375 1.890625 -4.34375 C 1.585938 -4.34375 1.367188 -4.265625 1.234375 -4.109375 C 1.109375 -3.960938 1.046875 -3.785156 1.046875 -3.578125 C 1.046875 -3.347656 1.128906 -3.164062 1.296875 -3.03125 C 1.472656 -2.90625 1.675781 -2.84375 1.90625 -2.84375 Z M 1.9375 -0.421875 C 2.226562 -0.421875 2.46875 -0.492188 2.65625 -0.640625 C 2.84375 -0.796875 2.9375 -1.03125 2.9375 -1.34375 C 2.9375 -1.65625 2.835938 -1.894531 2.640625 -2.0625 C 2.453125 -2.226562 2.203125 -2.3125 1.890625 -2.3125 C 1.597656 -2.3125 1.359375 -2.226562 1.171875 -2.0625 C 0.992188 -1.894531 0.90625 -1.660156 0.90625 -1.359375 C 0.90625 -1.109375 0.988281 -0.890625 1.15625 -0.703125 C 1.320312 -0.515625 1.582031 -0.421875 1.9375 -0.421875 Z M 1.0625 -2.609375 C 0.894531 -2.679688 0.765625 -2.765625 0.671875 -2.859375 C 0.484375 -3.046875 0.390625 -3.285156 0.390625 -3.578125 C 0.390625 -3.941406 0.519531 -4.253906 0.78125 -4.515625 C 1.050781 -4.773438 1.429688 -4.90625 1.921875 -4.90625 C 2.390625 -4.90625 2.753906 -4.78125 3.015625 -4.53125 C 3.285156 -4.289062 3.421875 -4.003906 3.421875 -3.671875 C 3.421875 -3.367188 3.34375 -3.117188 3.1875 -2.921875 C 3.101562 -2.816406 2.96875 -2.710938 2.78125 -2.609375 C 2.988281 -2.515625 3.148438 -2.410156 3.265625 -2.296875 C 3.484375 -2.066406 3.59375 -1.769531 3.59375 -1.40625 C 3.59375 -0.976562 3.445312 -0.613281 3.15625 -0.3125 C 2.863281 -0.0078125 2.453125 0.140625 1.921875 0.140625 C 1.453125 0.140625 1.050781 0.0078125 0.71875 -0.25 C 0.382812 -0.507812 0.21875 -0.882812 0.21875 -1.375 C 0.21875 -1.664062 0.289062 -1.914062 0.4375 -2.125 C 0.582031 -2.332031 0.789062 -2.492188 1.0625 -2.609375 Z M 1.0625 -2.609375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='b1b6faec-1bc4-439d-9399-b4e8b951d09d' overflow='visible'>\n<path d='M 0.265625 0 L 0.265625 -5.734375 L 4.8125 -5.734375 L 4.8125 0 Z M 4.09375 -0.71875 L 4.09375 -5.015625 L 0.984375 -5.015625 L 0.984375 -0.71875 Z M 4.09375 -0.71875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='32f5d44c-7199-47ed-a0d2-b25b9ae392b0' overflow='visible'>\n<path d='M 0.53125 -5.734375 L 1.234375 -5.734375 L 1.234375 0 L 0.53125 0 Z M 0.53125 -5.734375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='d9f662c6-1ff3-470c-8801-3db4416f7959' overflow='visible'>\n<path d='M 2.171875 -0.453125 C 2.640625 -0.453125 2.957031 -0.628906 3.125 -0.984375 C 3.300781 -1.335938 3.390625 -1.726562 3.390625 -2.15625 C 3.390625 -2.550781 3.328125 -2.867188 3.203125 -3.109375 C 3.003906 -3.492188 2.664062 -3.6875 2.1875 -3.6875 C 1.757812 -3.6875 1.445312 -3.519531 1.25 -3.1875 C 1.050781 -2.863281 0.953125 -2.472656 0.953125 -2.015625 C 0.953125 -1.566406 1.050781 -1.191406 1.25 -0.890625 C 1.445312 -0.597656 1.753906 -0.453125 2.171875 -0.453125 Z M 2.203125 -4.3125 C 2.742188 -4.3125 3.195312 -4.128906 3.5625 -3.765625 C 3.9375 -3.410156 4.125 -2.882812 4.125 -2.1875 C 4.125 -1.507812 3.957031 -0.945312 3.625 -0.5 C 3.300781 -0.0625 2.796875 0.15625 2.109375 0.15625 C 1.535156 0.15625 1.078125 -0.0351562 0.734375 -0.421875 C 0.398438 -0.816406 0.234375 -1.34375 0.234375 -2 C 0.234375 -2.707031 0.410156 -3.269531 0.765625 -3.6875 C 1.117188 -4.101562 1.597656 -4.3125 2.203125 -4.3125 Z M 2.1875 -4.28125 Z M 2.1875 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='e88f8e61-4684-4b7c-a537-ac4239130370' overflow='visible'>\n<path d='M 0.515625 -4.1875 L 1.1875 -4.1875 L 1.1875 -3.59375 C 1.382812 -3.832031 1.59375 -4.003906 1.8125 -4.109375 C 2.03125 -4.222656 2.273438 -4.28125 2.546875 -4.28125 C 3.140625 -4.28125 3.539062 -4.070312 3.75 -3.65625 C 3.863281 -3.425781 3.921875 -3.101562 3.921875 -2.6875 L 3.921875 0 L 3.21875 0 L 3.21875 -2.640625 C 3.21875 -2.890625 3.175781 -3.09375 3.09375 -3.25 C 2.96875 -3.507812 2.742188 -3.640625 2.421875 -3.640625 C 2.253906 -3.640625 2.117188 -3.625 2.015625 -3.59375 C 1.816406 -3.539062 1.644531 -3.425781 1.5 -3.25 C 1.382812 -3.113281 1.304688 -2.972656 1.265625 -2.828125 C 1.234375 -2.679688 1.21875 -2.46875 1.21875 -2.1875 L 1.21875 0 L 0.515625 0 Z M 2.171875 -4.28125 Z M 2.171875 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='b1172a5b-18ca-456c-b911-aaa414368bde' overflow='visible'>\n<path d='M 2 -4.265625 C 2.320312 -4.265625 2.609375 -4.179688 2.859375 -4.015625 C 2.984375 -3.921875 3.117188 -3.789062 3.265625 -3.625 L 3.265625 -4.140625 L 3.90625 -4.140625 L 3.90625 -0.34375 C 3.90625 0.1875 3.828125 0.609375 3.671875 0.921875 C 3.378906 1.484375 2.828125 1.765625 2.015625 1.765625 C 1.566406 1.765625 1.1875 1.664062 0.875 1.46875 C 0.570312 1.269531 0.40625 0.953125 0.375 0.515625 L 1.078125 0.515625 C 1.117188 0.703125 1.1875 0.847656 1.28125 0.953125 C 1.445312 1.109375 1.695312 1.1875 2.03125 1.1875 C 2.570312 1.1875 2.925781 1 3.09375 0.625 C 3.195312 0.394531 3.242188 -0.00390625 3.234375 -0.578125 C 3.085938 -0.367188 2.914062 -0.210938 2.71875 -0.109375 C 2.519531 -0.00390625 2.257812 0.046875 1.9375 0.046875 C 1.488281 0.046875 1.09375 -0.113281 0.75 -0.4375 C 0.40625 -0.757812 0.234375 -1.289062 0.234375 -2.03125 C 0.234375 -2.726562 0.40625 -3.273438 0.75 -3.671875 C 1.09375 -4.066406 1.507812 -4.265625 2 -4.265625 Z M 3.265625 -2.109375 C 3.265625 -2.628906 3.15625 -3.015625 2.9375 -3.265625 C 2.726562 -3.515625 2.457031 -3.640625 2.125 -3.640625 C 1.625 -3.640625 1.285156 -3.40625 1.109375 -2.9375 C 1.015625 -2.695312 0.96875 -2.375 0.96875 -1.96875 C 0.96875 -1.5 1.0625 -1.140625 1.25 -0.890625 C 1.4375 -0.648438 1.691406 -0.53125 2.015625 -0.53125 C 2.523438 -0.53125 2.882812 -0.753906 3.09375 -1.203125 C 3.207031 -1.460938 3.265625 -1.765625 3.265625 -2.109375 Z M 2.078125 -4.28125 Z M 2.078125 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='f67809ad-9474-48c6-9510-b883f0824063' overflow='visible'>\n<path d='M 0.515625 -4.171875 L 1.234375 -4.171875 L 1.234375 0 L 0.515625 0 Z M 0.515625 -5.734375 L 1.234375 -5.734375 L 1.234375 -4.9375 L 0.515625 -4.9375 Z M 0.515625 -5.734375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='6e5b542a-aeb6-467f-9944-bf290d889f91' overflow='visible'>\n<path d='M 0.65625 -5.359375 L 1.375 -5.359375 L 1.375 -4.1875 L 2.03125 -4.1875 L 2.03125 -3.609375 L 1.375 -3.609375 L 1.375 -0.875 C 1.375 -0.726562 1.421875 -0.632812 1.515625 -0.59375 C 1.566406 -0.5625 1.660156 -0.546875 1.796875 -0.546875 C 1.828125 -0.546875 1.859375 -0.546875 1.890625 -0.546875 C 1.929688 -0.546875 1.976562 -0.550781 2.03125 -0.5625 L 2.03125 0 C 1.957031 0.0195312 1.875 0.0351562 1.78125 0.046875 C 1.695312 0.0546875 1.601562 0.0625 1.5 0.0625 C 1.175781 0.0625 0.953125 -0.0195312 0.828125 -0.1875 C 0.710938 -0.351562 0.65625 -0.570312 0.65625 -0.84375 L 0.65625 -3.609375 L 0.09375 -3.609375 L 0.09375 -4.1875 L 0.65625 -4.1875 Z M 0.65625 -5.359375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='00ed052e-ee8f-40e7-a440-c34b980bff9e' overflow='visible'>\n<path d='M 1.21875 -4.1875 L 1.21875 -1.40625 C 1.21875 -1.195312 1.253906 -1.023438 1.328125 -0.890625 C 1.453125 -0.640625 1.679688 -0.515625 2.015625 -0.515625 C 2.503906 -0.515625 2.835938 -0.734375 3.015625 -1.171875 C 3.117188 -1.398438 3.171875 -1.71875 3.171875 -2.125 L 3.171875 -4.1875 L 3.875 -4.1875 L 3.875 0 L 3.203125 0 L 3.21875 -0.625 C 3.125 -0.457031 3.007812 -0.320312 2.875 -0.21875 C 2.601562 0 2.273438 0.109375 1.890625 0.109375 C 1.296875 0.109375 0.890625 -0.0859375 0.671875 -0.484375 C 0.554688 -0.703125 0.5 -0.988281 0.5 -1.34375 L 0.5 -4.1875 Z M 2.1875 -4.28125 Z M 2.1875 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='38b5cae8-fefa-45db-914c-9209977c81d6' overflow='visible'>\n<path d='M 0.96875 -2.046875 C 0.96875 -1.597656 1.0625 -1.222656 1.25 -0.921875 C 1.4375 -0.617188 1.738281 -0.46875 2.15625 -0.46875 C 2.488281 -0.46875 2.757812 -0.609375 2.96875 -0.890625 C 3.175781 -1.171875 3.28125 -1.570312 3.28125 -2.09375 C 3.28125 -2.625 3.171875 -3.015625 2.953125 -3.265625 C 2.734375 -3.515625 2.46875 -3.640625 2.15625 -3.640625 C 1.800781 -3.640625 1.515625 -3.503906 1.296875 -3.234375 C 1.078125 -2.960938 0.96875 -2.566406 0.96875 -2.046875 Z M 2.015625 -4.265625 C 2.335938 -4.265625 2.609375 -4.195312 2.828125 -4.0625 C 2.953125 -3.976562 3.09375 -3.835938 3.25 -3.640625 L 3.25 -5.765625 L 3.921875 -5.765625 L 3.921875 0 L 3.296875 0 L 3.296875 -0.578125 C 3.128906 -0.328125 2.929688 -0.140625 2.703125 -0.015625 C 2.484375 0.0976562 2.226562 0.15625 1.9375 0.15625 C 1.476562 0.15625 1.078125 -0.0390625 0.734375 -0.4375 C 0.390625 -0.832031 0.21875 -1.351562 0.21875 -2 C 0.21875 -2.613281 0.375 -3.144531 0.6875 -3.59375 C 1 -4.039062 1.441406 -4.265625 2.015625 -4.265625 Z M 2.015625 -4.265625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='594d09ad-0ef4-4938-8584-121081640ffc' overflow='visible'>\n<path d='M 2.265625 -4.28125 C 2.554688 -4.28125 2.84375 -4.207031 3.125 -4.0625 C 3.40625 -3.925781 3.617188 -3.75 3.765625 -3.53125 C 3.898438 -3.320312 3.992188 -3.078125 4.046875 -2.796875 C 4.085938 -2.597656 4.109375 -2.289062 4.109375 -1.875 L 1.03125 -1.875 C 1.050781 -1.445312 1.15625 -1.101562 1.34375 -0.84375 C 1.53125 -0.59375 1.816406 -0.46875 2.203125 -0.46875 C 2.578125 -0.46875 2.875 -0.585938 3.09375 -0.828125 C 3.21875 -0.960938 3.304688 -1.125 3.359375 -1.3125 L 4.046875 -1.3125 C 4.023438 -1.15625 3.960938 -0.984375 3.859375 -0.796875 C 3.753906 -0.609375 3.640625 -0.457031 3.515625 -0.34375 C 3.304688 -0.132812 3.039062 0.0078125 2.71875 0.09375 C 2.550781 0.132812 2.363281 0.15625 2.15625 0.15625 C 1.632812 0.15625 1.191406 -0.03125 0.828125 -0.40625 C 0.460938 -0.789062 0.28125 -1.328125 0.28125 -2.015625 C 0.28125 -2.679688 0.460938 -3.222656 0.828125 -3.640625 C 1.191406 -4.066406 1.671875 -4.28125 2.265625 -4.28125 Z M 3.375 -2.421875 C 3.351562 -2.734375 3.289062 -2.976562 3.1875 -3.15625 C 2.988281 -3.5 2.664062 -3.671875 2.21875 -3.671875 C 1.894531 -3.671875 1.625 -3.550781 1.40625 -3.3125 C 1.1875 -3.082031 1.070312 -2.785156 1.0625 -2.421875 Z M 2.1875 -4.28125 Z M 2.1875 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='83039200-d91c-4a93-bb1e-5864e7f7a0f4' overflow='visible'>\n<path d='M 0 -0.265625 L -5.734375 -0.265625 L -5.734375 -4.8125 L 0 -4.8125 Z M -0.71875 -4.09375 L -5.015625 -4.09375 L -5.015625 -0.984375 L -0.71875 -0.984375 Z M -0.71875 -4.09375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='9f87ea93-b03c-4322-8f8f-68a8dceaeff5' overflow='visible'>\n<path d='M -5.734375 -0.53125 L -5.734375 -1.234375 L 0 -1.234375 L 0 -0.53125 Z M -5.734375 -0.53125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='161fc93a-9677-4906-91e1-c383682f999b' overflow='visible'>\n<path d='M -1.109375 -1.0625 C -0.910156 -1.0625 -0.75 -1.132812 -0.625 -1.28125 C -0.507812 -1.425781 -0.453125 -1.601562 -0.453125 -1.8125 C -0.453125 -2.050781 -0.507812 -2.289062 -0.625 -2.53125 C -0.820312 -2.914062 -1.132812 -3.109375 -1.5625 -3.109375 L -2.125 -3.109375 C -2.070312 -3.023438 -2.023438 -2.914062 -1.984375 -2.78125 C -1.953125 -2.644531 -1.929688 -2.507812 -1.921875 -2.375 L -1.859375 -1.953125 C -1.828125 -1.703125 -1.773438 -1.507812 -1.703125 -1.375 C -1.578125 -1.164062 -1.378906 -1.0625 -1.109375 -1.0625 Z M -2.53125 -2.765625 C -2.550781 -2.921875 -2.617188 -3.023438 -2.734375 -3.078125 C -2.796875 -3.109375 -2.890625 -3.125 -3.015625 -3.125 C -3.253906 -3.125 -3.425781 -3.039062 -3.53125 -2.875 C -3.632812 -2.707031 -3.6875 -2.460938 -3.6875 -2.140625 C -3.6875 -1.773438 -3.585938 -1.515625 -3.390625 -1.359375 C -3.285156 -1.273438 -3.125 -1.21875 -2.90625 -1.1875 L -2.90625 -0.53125 C -3.425781 -0.550781 -3.785156 -0.71875 -3.984375 -1.03125 C -4.179688 -1.351562 -4.28125 -1.726562 -4.28125 -2.15625 C -4.28125 -2.644531 -4.1875 -3.046875 -4 -3.359375 C -3.8125 -3.660156 -3.519531 -3.8125 -3.125 -3.8125 L -0.71875 -3.8125 C -0.644531 -3.8125 -0.585938 -3.828125 -0.546875 -3.859375 C -0.503906 -3.890625 -0.484375 -3.953125 -0.484375 -4.046875 C -0.484375 -4.078125 -0.484375 -4.113281 -0.484375 -4.15625 C -0.484375 -4.195312 -0.488281 -4.238281 -0.5 -4.28125 L 0.015625 -4.28125 C 0.046875 -4.164062 0.0664062 -4.078125 0.078125 -4.015625 C 0.0859375 -3.960938 0.09375 -3.890625 0.09375 -3.796875 C 0.09375 -3.554688 0.00390625 -3.378906 -0.171875 -3.265625 C -0.265625 -3.210938 -0.394531 -3.171875 -0.5625 -3.140625 C -0.375 -3.003906 -0.207031 -2.800781 -0.0625 -2.53125 C 0.0703125 -2.257812 0.140625 -1.960938 0.140625 -1.640625 C 0.140625 -1.253906 0.0234375 -0.9375 -0.203125 -0.6875 C -0.441406 -0.445312 -0.738281 -0.328125 -1.09375 -0.328125 C -1.488281 -0.328125 -1.789062 -0.445312 -2 -0.6875 C -2.21875 -0.925781 -2.347656 -1.242188 -2.390625 -1.640625 Z M -4.28125 -2.171875 Z M -4.28125 -2.171875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='656118ca-ecde-4e78-b1a8-db1b921ce903' overflow='visible'>\n<path d='M -5.359375 -0.65625 L -5.359375 -1.375 L -4.1875 -1.375 L -4.1875 -2.03125 L -3.609375 -2.03125 L -3.609375 -1.375 L -0.875 -1.375 C -0.726562 -1.375 -0.632812 -1.421875 -0.59375 -1.515625 C -0.5625 -1.566406 -0.546875 -1.660156 -0.546875 -1.796875 C -0.546875 -1.828125 -0.546875 -1.859375 -0.546875 -1.890625 C -0.546875 -1.929688 -0.550781 -1.976562 -0.5625 -2.03125 L 0 -2.03125 C 0.0195312 -1.957031 0.0351562 -1.875 0.046875 -1.78125 C 0.0546875 -1.695312 0.0625 -1.601562 0.0625 -1.5 C 0.0625 -1.175781 -0.0195312 -0.953125 -0.1875 -0.828125 C -0.351562 -0.710938 -0.570312 -0.65625 -0.84375 -0.65625 L -3.609375 -0.65625 L -3.609375 -0.09375 L -4.1875 -0.09375 L -4.1875 -0.65625 Z M -5.359375 -0.65625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='8e8a5c3a-126f-4244-83d2-087d58413494' overflow='visible'>\n<path d='M -4.171875 -0.515625 L -4.171875 -1.234375 L 0 -1.234375 L 0 -0.515625 Z M -5.734375 -0.515625 L -5.734375 -1.234375 L -4.9375 -1.234375 L -4.9375 -0.515625 Z M -5.734375 -0.515625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='a4db1a0c-d25a-4720-ae54-ef4bdef43247' overflow='visible'>\n<path d='M -4.1875 -1.21875 L -1.40625 -1.21875 C -1.195312 -1.21875 -1.023438 -1.253906 -0.890625 -1.328125 C -0.640625 -1.453125 -0.515625 -1.679688 -0.515625 -2.015625 C -0.515625 -2.503906 -0.734375 -2.835938 -1.171875 -3.015625 C -1.398438 -3.117188 -1.71875 -3.171875 -2.125 -3.171875 L -4.1875 -3.171875 L -4.1875 -3.875 L 0 -3.875 L 0 -3.203125 L -0.625 -3.21875 C -0.457031 -3.125 -0.320312 -3.007812 -0.21875 -2.875 C 0 -2.601562 0.109375 -2.273438 0.109375 -1.890625 C 0.109375 -1.296875 -0.0859375 -0.890625 -0.484375 -0.671875 C -0.703125 -0.554688 -0.988281 -0.5 -1.34375 -0.5 L -4.1875 -0.5 Z M -4.28125 -2.1875 Z M -4.28125 -2.1875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='8465bf5c-70f2-476a-b848-55f9a6126c9f' overflow='visible'>\n<path d='M -2.046875 -0.96875 C -1.597656 -0.96875 -1.222656 -1.0625 -0.921875 -1.25 C -0.617188 -1.4375 -0.46875 -1.738281 -0.46875 -2.15625 C -0.46875 -2.488281 -0.609375 -2.757812 -0.890625 -2.96875 C -1.171875 -3.175781 -1.570312 -3.28125 -2.09375 -3.28125 C -2.625 -3.28125 -3.015625 -3.171875 -3.265625 -2.953125 C -3.515625 -2.734375 -3.640625 -2.46875 -3.640625 -2.15625 C -3.640625 -1.800781 -3.503906 -1.515625 -3.234375 -1.296875 C -2.960938 -1.078125 -2.566406 -0.96875 -2.046875 -0.96875 Z M -4.265625 -2.015625 C -4.265625 -2.335938 -4.195312 -2.609375 -4.0625 -2.828125 C -3.976562 -2.953125 -3.835938 -3.09375 -3.640625 -3.25 L -5.765625 -3.25 L -5.765625 -3.921875 L 0 -3.921875 L 0 -3.296875 L -0.578125 -3.296875 C -0.328125 -3.128906 -0.140625 -2.929688 -0.015625 -2.703125 C 0.0976562 -2.484375 0.15625 -2.226562 0.15625 -1.9375 C 0.15625 -1.476562 -0.0390625 -1.078125 -0.4375 -0.734375 C -0.832031 -0.390625 -1.351562 -0.21875 -2 -0.21875 C -2.613281 -0.21875 -3.144531 -0.375 -3.59375 -0.6875 C -4.039062 -1 -4.265625 -1.441406 -4.265625 -2.015625 Z M -4.265625 -2.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='f8915848-23fc-406d-9f37-20632f361258' overflow='visible'>\n<path d='M -4.28125 -2.265625 C -4.28125 -2.554688 -4.207031 -2.84375 -4.0625 -3.125 C -3.925781 -3.40625 -3.75 -3.617188 -3.53125 -3.765625 C -3.320312 -3.898438 -3.078125 -3.992188 -2.796875 -4.046875 C -2.597656 -4.085938 -2.289062 -4.109375 -1.875 -4.109375 L -1.875 -1.03125 C -1.445312 -1.050781 -1.101562 -1.15625 -0.84375 -1.34375 C -0.59375 -1.53125 -0.46875 -1.816406 -0.46875 -2.203125 C -0.46875 -2.578125 -0.585938 -2.875 -0.828125 -3.09375 C -0.960938 -3.21875 -1.125 -3.304688 -1.3125 -3.359375 L -1.3125 -4.046875 C -1.15625 -4.023438 -0.984375 -3.960938 -0.796875 -3.859375 C -0.609375 -3.753906 -0.457031 -3.640625 -0.34375 -3.515625 C -0.132812 -3.304688 0.0078125 -3.039062 0.09375 -2.71875 C 0.132812 -2.550781 0.15625 -2.363281 0.15625 -2.15625 C 0.15625 -1.632812 -0.03125 -1.191406 -0.40625 -0.828125 C -0.789062 -0.460938 -1.328125 -0.28125 -2.015625 -0.28125 C -2.679688 -0.28125 -3.222656 -0.460938 -3.640625 -0.828125 C -4.066406 -1.191406 -4.28125 -1.671875 -4.28125 -2.265625 Z M -2.421875 -3.375 C -2.734375 -3.351562 -2.976562 -3.289062 -3.15625 -3.1875 C -3.5 -2.988281 -3.671875 -2.664062 -3.671875 -2.21875 C -3.671875 -1.894531 -3.550781 -1.625 -3.3125 -1.40625 C -3.082031 -1.1875 -2.785156 -1.070312 -2.421875 -1.0625 Z M -4.28125 -2.1875 Z M -4.28125 -2.1875 ' style='stroke:none;'/>\n</symbol>\n</g>\n<clipPath id='ef8a7269-59c7-4c84-b66c-fca49ad15323'>\n<path d='M 46.480469 22.472656 L 643 22.472656 L 643 332 L 46.480469 332 Z M 46.480469 22.472656 '/>\n</clipPath>\n<clipPath id='8cc8e740-92a2-465f-99f0-3695a0614bdb'>\n<path d='M 46.480469 313 L 643 313 L 643 315 L 46.480469 315 Z M 46.480469 313 '/>\n</clipPath>\n<clipPath id='7b1ffb36-0ad8-4db8-8210-7b8513622233'>\n<path d='M 46.480469 252 L 643 252 L 643 253 L 46.480469 253 Z M 46.480469 252 '/>\n</clipPath>\n<clipPath id='57e0b41d-5afe-413f-ad97-6ba858b98c65'>\n<path d='M 46.480469 190 L 643 190 L 643 192 L 46.480469 192 Z M 46.480469 190 '/>\n</clipPath>\n<clipPath id='90d98df6-4c3b-4f09-9961-97e10d141554'>\n<path d='M 46.480469 129 L 643 129 L 643 131 L 46.480469 131 Z M 46.480469 129 '/>\n</clipPath>\n<clipPath id='22933304-d25a-4fe8-ba55-4cf12b0af7a6'>\n<path d='M 46.480469 68 L 643 68 L 643 69 L 46.480469 69 Z M 46.480469 68 '/>\n</clipPath>\n<clipPath id='685cec40-cf29-45df-81d1-94c8cf5cc6ba'>\n<path d='M 182 22.472656 L 184 22.472656 L 184 332 L 182 332 Z M 182 22.472656 '/>\n</clipPath>\n<clipPath id='ebb63304-142d-490e-9d9f-1f66f61fb9e3'>\n<path d='M 373 22.472656 L 375 22.472656 L 375 332 L 373 332 Z M 373 22.472656 '/>\n</clipPath>\n<clipPath id='0ecfa58e-b5c6-4dc3-a442-83183992611b'>\n<path d='M 564 22.472656 L 565 22.472656 L 565 332 L 564 332 Z M 564 22.472656 '/>\n</clipPath>\n<clipPath id='ac7fdd66-d93b-4cb4-9d8b-7f268e0de2df'>\n<path d='M 46.480469 22.472656 L 643 22.472656 L 643 332 L 46.480469 332 Z M 46.480469 22.472656 '/>\n</clipPath>\n</defs>\n<g id='d0a1a87f-4b4d-441c-a188-ff1065d502b2'>\n<rect style='fill:rgb(100%,100%,100%);fill-opacity:1;stroke:none;' height='360' width='648' y='0' x='0'/>\n<rect style='fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke:none;' height='360' width='648' y='0' x='0'/>\n<path d='M 0 360 L 648 360 L 648 0 L 0 0 Z M 0 360 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g clip-rule='nonzero' clip-path='url(#ef8a7269-59c7-4c84-b66c-fca49ad15323)'>\n<path d='M 46.480469 331.332031 L 642.332031 331.332031 L 642.332031 22.472656 L 46.480469 22.472656 Z M 46.480469 331.332031 ' style='fill-rule:nonzero;fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#8cc8e740-92a2-465f-99f0-3695a0614bdb)'>\n<path d='M 46.480469 314 L 642.332031 314 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#7b1ffb36-0ad8-4db8-8210-7b8513622233)'>\n<path d='M 46.480469 252.613281 L 642.332031 252.613281 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#57e0b41d-5afe-413f-ad97-6ba858b98c65)'>\n<path d='M 46.480469 191.222656 L 642.332031 191.222656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#90d98df6-4c3b-4f09-9961-97e10d141554)'>\n<path d='M 46.480469 129.835938 L 642.332031 129.835938 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#22933304-d25a-4fe8-ba55-4cf12b0af7a6)'>\n<path d='M 46.480469 68.445312 L 642.332031 68.445312 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#685cec40-cf29-45df-81d1-94c8cf5cc6ba)'>\n<path d='M 182.972656 331.332031 L 182.972656 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#ebb63304-142d-490e-9d9f-1f66f61fb9e3)'>\n<path d='M 373.8125 331.332031 L 373.8125 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#0ecfa58e-b5c6-4dc3-a442-83183992611b)'>\n<path d='M 564.65625 331.332031 L 564.65625 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<path d='M 150.921875 317.292969 C 150.921875 319.898438 147.011719 319.898438 147.011719 317.292969 C 147.011719 314.6875 150.921875 314.6875 150.921875 317.292969 ' style='fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 395.253906 305.433594 C 395.253906 308.039062 391.34375 308.039062 391.34375 305.433594 C 391.34375 302.824219 395.253906 302.824219 395.253906 305.433594 ' style='fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 282.21875 265.039062 C 282.21875 267.644531 278.308594 267.644531 278.308594 265.039062 C 278.308594 262.433594 282.21875 262.433594 282.21875 265.039062 ' style='fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 314.699219 296.628906 C 314.699219 299.234375 310.789062 299.234375 310.789062 296.628906 C 310.789062 294.023438 314.699219 294.023438 314.699219 296.628906 ' style='fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 298.019531 231.484375 C 298.019531 234.089844 294.109375 234.089844 294.109375 231.484375 C 294.109375 228.875 298.019531 228.875 298.019531 231.484375 ' style='fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 434.109375 267.371094 C 434.109375 269.976562 430.199219 269.976562 430.199219 267.371094 C 430.199219 264.765625 434.109375 264.765625 434.109375 267.371094 ' style='fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 455.332031 311.976562 C 455.332031 314.582031 451.421875 314.582031 451.421875 311.976562 C 451.421875 309.371094 455.332031 309.371094 455.332031 311.976562 ' style='fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 285.789062 83.890625 C 285.789062 86.5 281.878906 86.5 281.878906 83.890625 C 281.878906 81.285156 285.789062 81.285156 285.789062 83.890625 ' style='fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 433.117188 223.257812 C 433.117188 225.863281 429.207031 225.863281 429.207031 223.257812 C 429.207031 220.652344 433.117188 220.652344 433.117188 223.257812 ' style='fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 458.59375 241.746094 C 458.59375 244.355469 454.683594 244.355469 454.683594 241.746094 C 454.683594 239.140625 458.59375 239.140625 458.59375 241.746094 ' style='fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 492.351562 279.527344 C 492.351562 282.132812 488.445312 282.132812 488.445312 279.527344 C 488.445312 276.917969 492.351562 276.917969 492.351562 279.527344 ' style='fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 376.972656 199.804688 C 376.972656 202.414062 373.0625 202.414062 373.0625 199.804688 C 373.0625 197.199219 376.972656 197.199219 376.972656 199.804688 ' style='fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 183.324219 280.949219 C 183.324219 283.554688 179.414062 283.554688 179.414062 280.949219 C 179.414062 278.34375 183.324219 278.34375 183.324219 280.949219 ' style='fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 617.203125 265.441406 C 617.203125 268.050781 613.292969 268.050781 613.292969 265.441406 C 613.292969 262.835938 617.203125 262.835938 617.203125 265.441406 ' style='fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 75.519531 36.511719 C 75.519531 39.117188 71.609375 39.117188 71.609375 36.511719 C 71.609375 33.90625 75.519531 33.90625 75.519531 36.511719 ' style='fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 268.574219 157.765625 C 268.574219 160.375 264.664062 160.375 264.664062 157.765625 C 264.664062 155.160156 268.574219 155.160156 268.574219 157.765625 ' style='fill-rule:nonzero;fill:rgb(75.294118%,22.352941%,16.862745%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 143.015625 313.632812 L 148.964844 317.292969 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(40%,40%,40%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 56.605469 313.632812 L 206.09375 313.632812 L 206.007812 313.632812 L 206.355469 313.617188 L 206.695312 313.550781 L 207.019531 313.425781 L 207.320312 313.25 L 207.589844 313.03125 L 207.820312 312.773438 L 208.007812 312.476562 L 208.144531 312.160156 L 208.226562 311.820312 L 208.253906 311.472656 L 208.253906 301.453125 L 208.226562 301.105469 L 208.144531 300.769531 L 208.007812 300.449219 L 207.820312 300.15625 L 207.589844 299.894531 L 207.320312 299.675781 L 207.019531 299.503906 L 206.695312 299.378906 L 206.355469 299.308594 L 206.09375 299.292969 L 56.605469 299.292969 L 56.863281 299.308594 L 56.519531 299.296875 L 56.171875 299.335938 L 55.839844 299.433594 L 55.523438 299.582031 L 55.238281 299.78125 L 54.988281 300.019531 L 54.777344 300.300781 L 54.617188 300.605469 L 54.507812 300.9375 L 54.453125 301.28125 L 54.445312 301.453125 L 54.445312 311.472656 L 54.453125 311.300781 L 54.453125 311.648438 L 54.507812 311.992188 L 54.617188 312.320312 L 54.777344 312.628906 L 54.988281 312.90625 L 55.238281 313.148438 L 55.523438 313.34375 L 55.839844 313.492188 L 56.171875 313.589844 L 56.519531 313.632812 Z M 56.605469 313.632812 ' style='fill-rule:nonzero;fill:rgb(100%,100%,100%);fill-opacity:1;stroke-width:0.533489;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g style='fill:rgb(0%,0%,0%);fill-opacity:1;'>\n<use y='310.035156' x='58.042969' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='310.035156' x='64.681625' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='310.035156' x='70.217087' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='310.035156' x='75.752548' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='310.035156' x='78.517853' xlink:href='#fefb0bdb-46bf-4fe8-ad02-31d0d222e074'/>\n<use y='310.035156' x='83.494415' xlink:href='#dbf65fae-ae06-4862-99fa-000972f5777e'/>\n<use y='310.035156' x='89.029877' xlink:href='#f52b2695-c4db-41f6-88d2-f3aa491e2fcd'/>\n<use y='310.035156' x='94.006439' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='310.035156' x='99.541901' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='310.035156' x='102.307205' xlink:href='#14eb422a-af12-430d-b989-46e4e0205c3c'/>\n<use y='310.035156' x='109.495041' xlink:href='#1dfb9be4-eb19-440b-95a7-a669197ba06e'/>\n<use y='310.035156' x='111.706314' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n<use y='310.035156' x='115.020782' xlink:href='#1dfb9be4-eb19-440b-95a7-a669197ba06e'/>\n<use y='310.035156' x='117.232056' xlink:href='#94ca4bf9-8a65-48f8-8fad-c8bf73c19929'/>\n<use y='310.035156' x='122.767517' xlink:href='#dbf65fae-ae06-4862-99fa-000972f5777e'/>\n<use y='310.035156' x='128.302979' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='310.035156' x='133.83844' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='310.035156' x='136.603745' xlink:href='#8e5eef14-8132-481d-b70b-2824ac7ffc6c'/>\n<use y='310.035156' x='143.79158' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='310.035156' x='149.327042' xlink:href='#abb23039-e577-4ed0-beb1-d21bc1020176'/>\n<use y='310.035156' x='151.538315' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='310.035156' x='154.303619' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n<use y='310.035156' x='157.618088' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='310.035156' x='163.153549' xlink:href='#1dfb9be4-eb19-440b-95a7-a669197ba06e'/>\n<use y='310.035156' x='165.364822' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='310.035156' x='170.900284' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='310.035156' x='173.665588' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='310.035156' x='180.304245' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='310.035156' x='183.06955' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='310.035156' x='188.605011' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='310.035156' x='191.370316' xlink:href='#1dfb9be4-eb19-440b-95a7-a669197ba06e'/>\n<use y='310.035156' x='193.581589' xlink:href='#dbf65fae-ae06-4862-99fa-000972f5777e'/>\n<use y='310.035156' x='199.11705' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n</g>\n<path d='M 393.738281 301.796875 L 393.300781 305.433594 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(40%,40%,40%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 343.933594 301.796875 L 445.28125 301.796875 L 445.195312 301.796875 L 445.542969 301.78125 L 445.882812 301.710938 L 446.207031 301.589844 L 446.507812 301.414062 L 446.777344 301.195312 L 447.007812 300.933594 L 447.195312 300.640625 L 447.332031 300.320312 L 447.414062 299.984375 L 447.441406 299.636719 L 447.441406 289.617188 L 447.414062 289.269531 L 447.332031 288.933594 L 447.195312 288.613281 L 447.007812 288.320312 L 446.777344 288.058594 L 446.507812 287.839844 L 446.207031 287.664062 L 445.882812 287.542969 L 445.542969 287.472656 L 445.28125 287.457031 L 343.933594 287.457031 L 344.195312 287.472656 L 343.847656 287.457031 L 343.503906 287.5 L 343.167969 287.597656 L 342.855469 287.746094 L 342.570312 287.941406 L 342.316406 288.183594 L 342.109375 288.460938 L 341.949219 288.769531 L 341.835938 289.097656 L 341.78125 289.441406 L 341.773438 289.617188 L 341.773438 299.636719 L 341.78125 299.464844 L 341.78125 299.808594 L 341.835938 300.152344 L 341.949219 300.484375 L 342.109375 300.792969 L 342.316406 301.070312 L 342.570312 301.308594 L 342.855469 301.507812 L 343.167969 301.65625 L 343.503906 301.753906 L 343.847656 301.796875 Z M 343.933594 301.796875 ' style='fill-rule:nonzero;fill:rgb(100%,100%,100%);fill-opacity:1;stroke-width:0.533489;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g style='fill:rgb(0%,0%,0%);fill-opacity:1;'>\n<use y='298.195312' x='345.375' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='298.195312' x='352.013657' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='298.195312' x='357.549118' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='298.195312' x='363.084579' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='298.195312' x='365.849884' xlink:href='#fefb0bdb-46bf-4fe8-ad02-31d0d222e074'/>\n<use y='298.195312' x='370.826447' xlink:href='#dbf65fae-ae06-4862-99fa-000972f5777e'/>\n<use y='298.195312' x='376.361908' xlink:href='#f52b2695-c4db-41f6-88d2-f3aa491e2fcd'/>\n<use y='298.195312' x='381.33847' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='298.195312' x='386.873932' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='298.195312' x='389.639236' xlink:href='#8e5eef14-8132-481d-b70b-2824ac7ffc6c'/>\n<use y='298.195312' x='396.827072' xlink:href='#1dfb9be4-eb19-440b-95a7-a669197ba06e'/>\n<use y='298.195312' x='399.038345' xlink:href='#7d6ec591-ef49-4be4-b5f2-cb97084cdbb5'/>\n<use y='298.195312' x='404.014908' xlink:href='#1dfb9be4-eb19-440b-95a7-a669197ba06e'/>\n<use y='298.195312' x='406.226181' xlink:href='#07d2081b-8e74-404f-b60a-bda15f060dbc'/>\n<use y='298.195312' x='411.202744' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='298.195312' x='413.968048' xlink:href='#8e5eef14-8132-481d-b70b-2824ac7ffc6c'/>\n<use y='298.195312' x='421.155884' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='298.195312' x='426.691345' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='298.195312' x='432.226807' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='298.195312' x='434.992111' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='298.195312' x='440.527573' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n</g>\n<path d='M 265.390625 261.410156 L 280.265625 265.039062 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(40%,40%,40%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 181.714844 261.410156 L 290.269531 261.410156 L 290.183594 261.410156 L 290.53125 261.394531 L 290.871094 261.324219 L 291.195312 261.203125 L 291.496094 261.027344 L 291.765625 260.808594 L 291.996094 260.546875 L 292.183594 260.253906 L 292.320312 259.933594 L 292.402344 259.597656 L 292.429688 259.25 L 292.429688 249.230469 L 292.402344 248.882812 L 292.320312 248.546875 L 292.183594 248.226562 L 291.996094 247.933594 L 291.765625 247.671875 L 291.496094 247.453125 L 291.195312 247.277344 L 290.871094 247.15625 L 290.53125 247.085938 L 290.269531 247.070312 L 181.714844 247.070312 L 181.976562 247.085938 L 181.628906 247.070312 L 181.285156 247.113281 L 180.949219 247.210938 L 180.636719 247.359375 L 180.347656 247.558594 L 180.097656 247.796875 L 179.890625 248.074219 L 179.726562 248.382812 L 179.617188 248.714844 L 179.5625 249.054688 L 179.554688 249.230469 L 179.554688 259.25 L 179.5625 259.078125 L 179.5625 259.425781 L 179.617188 259.769531 L 179.726562 260.097656 L 179.890625 260.40625 L 180.097656 260.683594 L 180.347656 260.925781 L 180.636719 261.121094 L 180.949219 261.269531 L 181.285156 261.367188 L 181.628906 261.410156 Z M 181.714844 261.410156 ' style='fill-rule:nonzero;fill:rgb(100%,100%,100%);fill-opacity:1;stroke-width:0.533489;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g style='fill:rgb(0%,0%,0%);fill-opacity:1;'>\n<use y='257.8125' x='183.15625' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='257.8125' x='189.794907' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='257.8125' x='195.330368' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='257.8125' x='200.865829' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='257.8125' x='203.631134' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='257.8125' x='209.166595' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='257.8125' x='211.9319' xlink:href='#8e5eef14-8132-481d-b70b-2824ac7ffc6c'/>\n<use y='257.8125' x='219.119736' xlink:href='#abb23039-e577-4ed0-beb1-d21bc1020176'/>\n<use y='257.8125' x='221.331009' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='257.8125' x='226.86647' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n<use y='257.8125' x='230.180939' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='257.8125' x='235.7164' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='257.8125' x='238.481705' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='257.8125' x='244.017166' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='257.8125' x='246.782471' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='257.8125' x='249.547775' xlink:href='#44e7ce1f-601c-4c57-9df2-a5fdc2850847'/>\n<use y='257.8125' x='256.186432' xlink:href='#abb23039-e577-4ed0-beb1-d21bc1020176'/>\n<use y='257.8125' x='258.397705' xlink:href='#9e535106-c458-47bf-a194-050224a1abb3'/>\n<use y='257.8125' x='266.688736' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='257.8125' x='272.224197' xlink:href='#94ca4bf9-8a65-48f8-8fad-c8bf73c19929'/>\n<use y='257.8125' x='277.759659' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='257.8125' x='283.29512' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n</g>\n<path d='M 312.464844 282.992188 L 312.746094 296.628906 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(40%,40%,40%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 268.546875 282.992188 L 356.089844 282.992188 L 356.003906 282.992188 L 356.351562 282.976562 L 356.691406 282.90625 L 357.015625 282.785156 L 357.316406 282.609375 L 357.585938 282.390625 L 357.816406 282.128906 L 358.003906 281.835938 L 358.140625 281.515625 L 358.222656 281.179688 L 358.25 280.832031 L 358.25 270.8125 L 358.222656 270.464844 L 358.140625 270.128906 L 358.003906 269.808594 L 357.816406 269.515625 L 357.585938 269.253906 L 357.316406 269.035156 L 357.015625 268.859375 L 356.691406 268.738281 L 356.351562 268.667969 L 356.089844 268.652344 L 268.546875 268.652344 L 268.804688 268.667969 L 268.460938 268.652344 L 268.113281 268.695312 L 267.78125 268.792969 L 267.464844 268.941406 L 267.179688 269.136719 L 266.929688 269.378906 L 266.71875 269.65625 L 266.558594 269.964844 L 266.449219 270.292969 L 266.394531 270.636719 L 266.386719 270.8125 L 266.386719 280.832031 L 266.394531 280.660156 L 266.394531 281.007812 L 266.449219 281.347656 L 266.558594 281.679688 L 266.71875 281.988281 L 266.929688 282.265625 L 267.179688 282.503906 L 267.464844 282.703125 L 267.78125 282.851562 L 268.113281 282.949219 L 268.460938 282.992188 Z M 268.546875 282.992188 ' style='fill-rule:nonzero;fill:rgb(100%,100%,100%);fill-opacity:1;stroke-width:0.533489;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g style='fill:rgb(0%,0%,0%);fill-opacity:1;'>\n<use y='279.390625' x='269.988281' xlink:href='#44e7ce1f-601c-4c57-9df2-a5fdc2850847'/>\n<use y='279.390625' x='276.626938' xlink:href='#94ca4bf9-8a65-48f8-8fad-c8bf73c19929'/>\n<use y='279.390625' x='282.162399' xlink:href='#dbf65fae-ae06-4862-99fa-000972f5777e'/>\n<use y='279.390625' x='287.697861' xlink:href='#08a486c0-2d6a-4d56-9616-54db256de353'/>\n<use y='279.390625' x='293.233322' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='279.390625' x='298.768784' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='279.390625' x='301.534088' xlink:href='#dbf65fae-ae06-4862-99fa-000972f5777e'/>\n<use y='279.390625' x='307.06955' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='279.390625' x='312.605011' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='279.390625' x='315.370316' xlink:href='#44e7ce1f-601c-4c57-9df2-a5fdc2850847'/>\n<use y='279.390625' x='322.008972' xlink:href='#abb23039-e577-4ed0-beb1-d21bc1020176'/>\n<use y='279.390625' x='324.220245' xlink:href='#9e535106-c458-47bf-a194-050224a1abb3'/>\n<use y='279.390625' x='332.511276' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='279.390625' x='338.046738' xlink:href='#94ca4bf9-8a65-48f8-8fad-c8bf73c19929'/>\n<use y='279.390625' x='343.582199' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='279.390625' x='349.117661' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n</g>\n<path d='M 299.679688 231.6875 L 296.066406 231.484375 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(40%,40%,40%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 301.839844 241.367188 L 386.613281 241.367188 L 386.527344 241.367188 L 386.875 241.351562 L 387.214844 241.285156 L 387.539062 241.160156 L 387.839844 240.984375 L 388.109375 240.765625 L 388.339844 240.507812 L 388.527344 240.210938 L 388.664062 239.894531 L 388.746094 239.554688 L 388.773438 239.207031 L 388.773438 229.1875 L 388.746094 228.839844 L 388.664062 228.503906 L 388.527344 228.183594 L 388.339844 227.890625 L 388.109375 227.628906 L 387.839844 227.410156 L 387.539062 227.238281 L 387.214844 227.113281 L 386.875 227.042969 L 386.613281 227.027344 L 301.839844 227.027344 L 302.101562 227.042969 L 301.753906 227.03125 L 301.40625 227.070312 L 301.074219 227.167969 L 300.761719 227.316406 L 300.472656 227.515625 L 300.222656 227.753906 L 300.015625 228.035156 L 299.851562 228.339844 L 299.742188 228.671875 L 299.6875 229.015625 L 299.679688 229.1875 L 299.679688 239.207031 L 299.6875 239.035156 L 299.6875 239.382812 L 299.742188 239.726562 L 299.851562 240.054688 L 300.015625 240.363281 L 300.222656 240.640625 L 300.472656 240.882812 L 300.761719 241.078125 L 301.074219 241.226562 L 301.40625 241.324219 L 301.753906 241.367188 Z M 301.839844 241.367188 ' style='fill-rule:nonzero;fill:rgb(100%,100%,100%);fill-opacity:1;stroke-width:0.533489;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g style='fill:rgb(0%,0%,0%);fill-opacity:1;'>\n<use y='237.769531' x='303.277344' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='237.769531' x='309.916' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='237.769531' x='315.451462' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='237.769531' x='320.986923' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='237.769531' x='323.752228' xlink:href='#553fc5e8-e5a3-452c-8e81-1d777c4746a7'/>\n<use y='237.769531' x='330.390884' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='237.769531' x='335.926346' xlink:href='#94ca4bf9-8a65-48f8-8fad-c8bf73c19929'/>\n<use y='237.769531' x='341.461807' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n<use y='237.769531' x='344.776276' xlink:href='#dbf65fae-ae06-4862-99fa-000972f5777e'/>\n<use y='237.769531' x='350.311737' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='237.769531' x='353.077042' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='237.769531' x='359.715698' xlink:href='#4d9ce2bd-e255-4439-9642-bb53937b4394'/>\n<use y='237.769531' x='365.25116' xlink:href='#8bad88ae-7a7a-47aa-bdfe-88ec054c732b'/>\n<use y='237.769531' x='370.786621' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='237.769531' x='376.322083' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n<use y='237.769531' x='379.636551' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n</g>\n<path d='M 426.9375 262.910156 L 432.152344 267.371094 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(40%,40%,40%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 367.582031 262.910156 L 469.515625 262.910156 L 469.429688 262.910156 L 469.777344 262.894531 L 470.117188 262.828125 L 470.441406 262.703125 L 470.742188 262.527344 L 471.011719 262.308594 L 471.242188 262.050781 L 471.429688 261.753906 L 471.566406 261.433594 L 471.648438 261.097656 L 471.675781 260.75 L 471.675781 250.730469 L 471.648438 250.382812 L 471.566406 250.046875 L 471.429688 249.726562 L 471.242188 249.433594 L 471.011719 249.171875 L 470.742188 248.953125 L 470.441406 248.78125 L 470.117188 248.65625 L 469.777344 248.585938 L 469.515625 248.570312 L 367.582031 248.570312 L 367.84375 248.585938 L 367.496094 248.574219 L 367.148438 248.613281 L 366.816406 248.710938 L 366.5 248.859375 L 366.214844 249.058594 L 365.964844 249.296875 L 365.757812 249.578125 L 365.59375 249.882812 L 365.484375 250.214844 L 365.429688 250.558594 L 365.421875 250.730469 L 365.421875 260.75 L 365.429688 260.578125 L 365.429688 260.925781 L 365.484375 261.269531 L 365.59375 261.597656 L 365.757812 261.90625 L 365.964844 262.183594 L 366.214844 262.425781 L 366.5 262.621094 L 366.816406 262.769531 L 367.148438 262.867188 L 367.496094 262.910156 Z M 367.582031 262.910156 ' style='fill-rule:nonzero;fill:rgb(100%,100%,100%);fill-opacity:1;stroke-width:0.533489;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g style='fill:rgb(0%,0%,0%);fill-opacity:1;'>\n<use y='259.3125' x='369.023438' xlink:href='#553fc5e8-e5a3-452c-8e81-1d777c4746a7'/>\n<use y='259.3125' x='375.662094' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='259.3125' x='381.197556' xlink:href='#f52b2695-c4db-41f6-88d2-f3aa491e2fcd'/>\n<use y='259.3125' x='386.174118' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='259.3125' x='391.709579' xlink:href='#dbf65fae-ae06-4862-99fa-000972f5777e'/>\n<use y='259.3125' x='397.245041' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='259.3125' x='400.010345' xlink:href='#94ca4bf9-8a65-48f8-8fad-c8bf73c19929'/>\n<use y='259.3125' x='405.545807' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='259.3125' x='411.081268' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='259.3125' x='413.846573' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='259.3125' x='420.485229' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='259.3125' x='426.020691' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='259.3125' x='431.556152' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='259.3125' x='434.321457' xlink:href='#44e7ce1f-601c-4c57-9df2-a5fdc2850847'/>\n<use y='259.3125' x='440.960114' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='259.3125' x='446.495575' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='259.3125' x='449.26088' xlink:href='#dbf65fae-ae06-4862-99fa-000972f5777e'/>\n<use y='259.3125' x='454.796341' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='259.3125' x='460.331802' xlink:href='#1dfb9be4-eb19-440b-95a7-a669197ba06e'/>\n<use y='259.3125' x='462.543076' xlink:href='#dbf65fae-ae06-4862-99fa-000972f5777e'/>\n</g>\n<path d='M 461.132812 313.269531 L 453.375 311.976562 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(40%,40%,40%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 459.253906 327.609375 L 549 327.609375 L 548.914062 327.609375 L 549.257812 327.59375 L 549.601562 327.523438 L 549.925781 327.402344 L 550.226562 327.226562 L 550.496094 327.007812 L 550.726562 326.746094 L 550.910156 326.453125 L 551.046875 326.132812 L 551.132812 325.796875 L 551.160156 325.449219 L 551.160156 315.429688 L 551.132812 315.082031 L 551.046875 314.746094 L 550.910156 314.425781 L 550.726562 314.132812 L 550.496094 313.871094 L 550.226562 313.652344 L 549.925781 313.476562 L 549.601562 313.355469 L 549.257812 313.285156 L 549 313.269531 L 459.253906 313.269531 L 459.511719 313.285156 L 459.164062 313.269531 L 458.820312 313.3125 L 458.488281 313.410156 L 458.171875 313.558594 L 457.886719 313.757812 L 457.636719 313.996094 L 457.425781 314.273438 L 457.265625 314.582031 L 457.15625 314.914062 L 457.101562 315.253906 L 457.09375 315.429688 L 457.09375 325.449219 L 457.101562 325.277344 L 457.101562 325.625 L 457.15625 325.96875 L 457.265625 326.296875 L 457.425781 326.605469 L 457.636719 326.882812 L 457.886719 327.125 L 458.171875 327.320312 L 458.488281 327.46875 L 458.820312 327.566406 L 459.164062 327.609375 Z M 459.253906 327.609375 ' style='fill-rule:nonzero;fill:rgb(100%,100%,100%);fill-opacity:1;stroke-width:0.533489;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g style='fill:rgb(0%,0%,0%);fill-opacity:1;'>\n<use y='324.011719' x='460.691406' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='324.011719' x='467.330063' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='324.011719' x='472.865524' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='324.011719' x='478.400986' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='324.011719' x='481.16629' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='324.011719' x='487.804947' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='324.011719' x='493.340408' xlink:href='#abb23039-e577-4ed0-beb1-d21bc1020176'/>\n<use y='324.011719' x='495.551682' xlink:href='#7d6ec591-ef49-4be4-b5f2-cb97084cdbb5'/>\n<use y='324.011719' x='500.528244' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='324.011719' x='506.063705' xlink:href='#94ca4bf9-8a65-48f8-8fad-c8bf73c19929'/>\n<use y='324.011719' x='511.599167' xlink:href='#dbf65fae-ae06-4862-99fa-000972f5777e'/>\n<use y='324.011719' x='517.134628' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n<use y='324.011719' x='520.449097' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='324.011719' x='523.214401' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='324.011719' x='528.749863' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='324.011719' x='531.515167' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='324.011719' x='534.280472' xlink:href='#8144c006-357a-4df4-8253-57c8a075ec89'/>\n<use y='324.011719' x='539.815933' xlink:href='#f52b2695-c4db-41f6-88d2-f3aa491e2fcd'/>\n<use y='324.011719' x='544.792496' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n</g>\n<path d='M 283.539062 87.496094 L 283.832031 83.890625 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(40%,40%,40%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 257.441406 101.835938 L 308.464844 101.835938 L 308.378906 101.835938 L 308.726562 101.820312 L 309.066406 101.75 L 309.390625 101.628906 L 309.691406 101.453125 L 309.960938 101.234375 L 310.191406 100.972656 L 310.375 100.679688 L 310.511719 100.359375 L 310.597656 100.023438 L 310.625 99.675781 L 310.625 89.65625 L 310.597656 89.308594 L 310.511719 88.972656 L 310.375 88.652344 L 310.191406 88.359375 L 309.960938 88.097656 L 309.691406 87.878906 L 309.390625 87.703125 L 309.066406 87.582031 L 308.726562 87.511719 L 308.464844 87.496094 L 257.441406 87.496094 L 257.703125 87.511719 L 257.355469 87.496094 L 257.007812 87.539062 L 256.675781 87.636719 L 256.363281 87.785156 L 256.074219 87.984375 L 255.824219 88.222656 L 255.617188 88.5 L 255.453125 88.808594 L 255.34375 89.140625 L 255.289062 89.480469 L 255.28125 89.65625 L 255.28125 99.675781 L 255.289062 99.503906 L 255.289062 99.851562 L 255.34375 100.191406 L 255.453125 100.523438 L 255.617188 100.832031 L 255.824219 101.109375 L 256.074219 101.347656 L 256.363281 101.546875 L 256.675781 101.695312 L 257.007812 101.792969 L 257.355469 101.835938 Z M 257.441406 101.835938 ' style='fill-rule:nonzero;fill:rgb(100%,100%,100%);fill-opacity:1;stroke-width:0.533489;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g style='fill:rgb(0%,0%,0%);fill-opacity:1;'>\n<use y='98.238281' x='258.882812' xlink:href='#fefb0bdb-46bf-4fe8-ad02-31d0d222e074'/>\n<use y='98.238281' x='263.859375' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='98.238281' x='269.394836' xlink:href='#2c470afe-b012-4401-a6ab-6d28e3e61cdc'/>\n<use y='98.238281' x='274.930298' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='98.238281' x='280.465759' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='98.238281' x='286.001221' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='98.238281' x='288.766525' xlink:href='#dbf65fae-ae06-4862-99fa-000972f5777e'/>\n<use y='98.238281' x='294.301987' xlink:href='#848a7f06-2ec9-4d02-87ce-3a59ec7f6c9f'/>\n<use y='98.238281' x='301.489822' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n</g>\n<path d='M 431.105469 219.464844 L 431.160156 223.257812 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(40%,40%,40%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 388.902344 219.464844 L 473.097656 219.464844 L 473.011719 219.464844 L 473.359375 219.449219 L 473.699219 219.382812 L 474.023438 219.257812 L 474.324219 219.085938 L 474.59375 218.863281 L 474.824219 218.605469 L 475.007812 218.308594 L 475.144531 217.992188 L 475.230469 217.652344 L 475.257812 217.304688 L 475.257812 207.285156 L 475.230469 206.9375 L 475.144531 206.601562 L 475.007812 206.28125 L 474.824219 205.988281 L 474.59375 205.726562 L 474.324219 205.507812 L 474.023438 205.335938 L 473.699219 205.210938 L 473.359375 205.140625 L 473.097656 205.125 L 388.902344 205.125 L 389.160156 205.140625 L 388.8125 205.128906 L 388.46875 205.167969 L 388.136719 205.265625 L 387.820312 205.414062 L 387.535156 205.613281 L 387.285156 205.851562 L 387.074219 206.132812 L 386.914062 206.4375 L 386.804688 206.769531 L 386.746094 207.113281 L 386.742188 207.285156 L 386.742188 217.304688 L 386.746094 217.132812 L 386.746094 217.480469 L 386.804688 217.824219 L 386.914062 218.152344 L 387.074219 218.460938 L 387.285156 218.738281 L 387.535156 218.980469 L 387.820312 219.175781 L 388.136719 219.324219 L 388.46875 219.421875 L 388.8125 219.464844 Z M 388.902344 219.464844 ' style='fill-rule:nonzero;fill:rgb(100%,100%,100%);fill-opacity:1;stroke-width:0.533489;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g style='fill:rgb(0%,0%,0%);fill-opacity:1;'>\n<use y='215.867188' x='390.34375' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='215.867188' x='396.982407' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='215.867188' x='402.517868' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='215.867188' x='408.053329' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='215.867188' x='410.818634' xlink:href='#fefb0bdb-46bf-4fe8-ad02-31d0d222e074'/>\n<use y='215.867188' x='415.795197' xlink:href='#dbf65fae-ae06-4862-99fa-000972f5777e'/>\n<use y='215.867188' x='421.330658' xlink:href='#f52b2695-c4db-41f6-88d2-f3aa491e2fcd'/>\n<use y='215.867188' x='426.30722' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='215.867188' x='431.842682' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='215.867188' x='434.607986' xlink:href='#8e5eef14-8132-481d-b70b-2824ac7ffc6c'/>\n<use y='215.867188' x='441.795822' xlink:href='#1dfb9be4-eb19-440b-95a7-a669197ba06e'/>\n<use y='215.867188' x='444.007095' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='215.867188' x='446.7724' xlink:href='#2bb104cd-2d57-457c-b53e-938fc695bd72'/>\n<use y='215.867188' x='451.748962' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='215.867188' x='454.514267' xlink:href='#7d61d690-8adf-4ed0-94d5-2a74856d0189'/>\n<use y='215.867188' x='461.702103' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='215.867188' x='467.237564' xlink:href='#abb23039-e577-4ed0-beb1-d21bc1020176'/>\n<use y='215.867188' x='469.448837' xlink:href='#abb23039-e577-4ed0-beb1-d21bc1020176'/>\n</g>\n<path d='M 478.898438 244.589844 L 456.640625 241.746094 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(40%,40%,40%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 481.058594 255.640625 L 537.589844 255.640625 L 537.503906 255.640625 L 537.851562 255.625 L 538.191406 255.554688 L 538.515625 255.433594 L 538.816406 255.257812 L 539.085938 255.039062 L 539.316406 254.777344 L 539.503906 254.484375 L 539.640625 254.164062 L 539.722656 253.828125 L 539.75 253.480469 L 539.75 243.460938 L 539.722656 243.113281 L 539.640625 242.777344 L 539.503906 242.457031 L 539.316406 242.164062 L 539.085938 241.902344 L 538.816406 241.683594 L 538.515625 241.507812 L 538.191406 241.386719 L 537.851562 241.316406 L 537.589844 241.300781 L 481.058594 241.300781 L 481.316406 241.316406 L 480.96875 241.304688 L 480.625 241.34375 L 480.292969 241.441406 L 479.976562 241.589844 L 479.691406 241.789062 L 479.441406 242.027344 L 479.230469 242.304688 L 479.070312 242.613281 L 478.960938 242.945312 L 478.902344 243.289062 L 478.898438 243.460938 L 478.898438 253.480469 L 478.902344 253.308594 L 478.902344 253.65625 L 478.960938 254 L 479.070312 254.328125 L 479.230469 254.636719 L 479.441406 254.914062 L 479.691406 255.15625 L 479.976562 255.351562 L 480.292969 255.5 L 480.625 255.597656 L 480.96875 255.640625 Z M 481.058594 255.640625 ' style='fill-rule:nonzero;fill:rgb(100%,100%,100%);fill-opacity:1;stroke-width:0.533489;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g style='fill:rgb(0%,0%,0%);fill-opacity:1;'>\n<use y='252.042969' x='482.496094' xlink:href='#41cdac45-8b65-4365-af0a-db5a1b22687b'/>\n<use y='252.042969' x='490.787125' xlink:href='#e0a92b58-cd5b-47e2-b121-ebaa6da8e043'/>\n<use y='252.042969' x='496.322586' xlink:href='#33ba5ef5-6dd1-4a6d-9573-82b5f55d1905'/>\n<use y='252.042969' x='502.961243' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='252.042969' x='505.726547' xlink:href='#e0a92b58-cd5b-47e2-b121-ebaa6da8e043'/>\n<use y='252.042969' x='511.262009' xlink:href='#1dfb9be4-eb19-440b-95a7-a669197ba06e'/>\n<use y='252.042969' x='513.473282' xlink:href='#08a486c0-2d6a-4d56-9616-54db256de353'/>\n<use y='252.042969' x='519.008743' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n<use y='252.042969' x='522.323212' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='252.042969' x='527.858673' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n<use y='252.042969' x='531.173141' xlink:href='#2bb104cd-2d57-457c-b53e-938fc695bd72'/>\n</g>\n<path d='M 501.949219 291.613281 L 490.398438 279.527344 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(40%,40%,40%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 454.808594 305.957031 L 562.796875 305.957031 L 562.710938 305.953125 L 563.058594 305.941406 L 563.398438 305.871094 L 563.722656 305.746094 L 564.023438 305.574219 L 564.292969 305.351562 L 564.523438 305.09375 L 564.710938 304.800781 L 564.847656 304.480469 L 564.929688 304.140625 L 564.957031 303.796875 L 564.957031 293.773438 L 564.929688 293.429688 L 564.847656 293.089844 L 564.710938 292.769531 L 564.523438 292.476562 L 564.292969 292.21875 L 564.023438 291.996094 L 563.722656 291.824219 L 563.398438 291.699219 L 563.058594 291.628906 L 562.796875 291.613281 L 454.808594 291.613281 L 455.066406 291.628906 L 454.71875 291.617188 L 454.375 291.660156 L 454.042969 291.753906 L 453.726562 291.90625 L 453.441406 292.101562 L 453.191406 292.34375 L 452.980469 292.621094 L 452.820312 292.929688 L 452.710938 293.257812 L 452.652344 293.601562 L 452.648438 293.773438 L 452.648438 303.796875 L 452.652344 303.621094 L 452.652344 303.96875 L 452.710938 304.3125 L 452.820312 304.640625 L 452.980469 304.949219 L 453.191406 305.226562 L 453.441406 305.46875 L 453.726562 305.667969 L 454.042969 305.816406 L 454.375 305.910156 L 454.71875 305.953125 Z M 454.808594 305.957031 ' style='fill-rule:nonzero;fill:rgb(100%,100%,100%);fill-opacity:1;stroke-width:0.533489;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g style='fill:rgb(0%,0%,0%);fill-opacity:1;'>\n<use y='302.355469' x='456.246094' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='302.355469' x='462.88475' xlink:href='#fefb0bdb-46bf-4fe8-ad02-31d0d222e074'/>\n<use y='302.355469' x='467.861313' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='302.355469' x='474.499969' xlink:href='#95ba0e07-9c88-4540-b5de-6e3d71a035f5'/>\n<use y='302.355469' x='481.687805' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='302.355469' x='484.45311' xlink:href='#8ac2822a-ad76-460c-a782-6cf3fcc0f821'/>\n<use y='302.355469' x='489.988571' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='302.355469' x='492.753876' xlink:href='#fea978f8-7687-4c73-94fe-43f4f0bc3a1e'/>\n<use y='302.355469' x='498.289337' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='302.355469' x='501.054642' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='302.355469' x='506.590103' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='302.355469' x='509.355408' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='302.355469' x='512.120712' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='302.355469' x='518.759369' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='302.355469' x='524.29483' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='302.355469' x='529.830292' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='302.355469' x='532.595596' xlink:href='#8e5eef14-8132-481d-b70b-2824ac7ffc6c'/>\n<use y='302.355469' x='539.783432' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='302.355469' x='545.318893' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n<use y='302.355469' x='548.633362' xlink:href='#abb23039-e577-4ed0-beb1-d21bc1020176'/>\n<use y='302.355469' x='550.844635' xlink:href='#dbf65fae-ae06-4862-99fa-000972f5777e'/>\n<use y='302.355469' x='556.380096' xlink:href='#f52b2695-c4db-41f6-88d2-f3aa491e2fcd'/>\n</g>\n<path d='M 373.382812 196.101562 L 375.015625 199.804688 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(40%,40%,40%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 336.429688 196.101562 L 404.023438 196.101562 L 403.9375 196.101562 L 404.28125 196.085938 L 404.625 196.015625 L 404.949219 195.894531 L 405.25 195.71875 L 405.519531 195.5 L 405.75 195.238281 L 405.933594 194.945312 L 406.070312 194.625 L 406.15625 194.289062 L 406.183594 193.941406 L 406.183594 183.921875 L 406.15625 183.574219 L 406.070312 183.238281 L 405.933594 182.917969 L 405.75 182.625 L 405.519531 182.363281 L 405.25 182.144531 L 404.949219 181.96875 L 404.625 181.847656 L 404.28125 181.777344 L 404.023438 181.761719 L 336.429688 181.761719 L 336.6875 181.777344 L 336.339844 181.761719 L 335.996094 181.804688 L 335.660156 181.902344 L 335.347656 182.050781 L 335.0625 182.246094 L 334.8125 182.488281 L 334.601562 182.765625 L 334.441406 183.074219 L 334.332031 183.402344 L 334.273438 183.746094 L 334.269531 183.921875 L 334.269531 193.941406 L 334.273438 193.769531 L 334.273438 194.117188 L 334.332031 194.457031 L 334.441406 194.789062 L 334.601562 195.097656 L 334.8125 195.375 L 335.0625 195.613281 L 335.347656 195.8125 L 335.660156 195.960938 L 335.996094 196.058594 L 336.339844 196.101562 Z M 336.429688 196.101562 ' style='fill-rule:nonzero;fill:rgb(100%,100%,100%);fill-opacity:1;stroke-width:0.533489;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g style='fill:rgb(0%,0%,0%);fill-opacity:1;'>\n<use y='192.5' x='337.871094' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='192.5' x='344.50975' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='192.5' x='347.275055' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='192.5' x='350.040359' xlink:href='#fefb0bdb-46bf-4fe8-ad02-31d0d222e074'/>\n<use y='192.5' x='355.016922' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='192.5' x='360.552383' xlink:href='#9e535106-c458-47bf-a194-050224a1abb3'/>\n<use y='192.5' x='368.843414' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='192.5' x='374.378876' xlink:href='#f52b2695-c4db-41f6-88d2-f3aa491e2fcd'/>\n<use y='192.5' x='379.355438' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='192.5' x='382.120743' xlink:href='#553fc5e8-e5a3-452c-8e81-1d777c4746a7'/>\n<use y='192.5' x='388.759399' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='192.5' x='394.294861' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n<use y='192.5' x='397.609329' xlink:href='#70b7e0c8-9a8c-45c5-972f-8a072643f9a7'/>\n</g>\n<path d='M 192.78125 284.558594 L 181.371094 280.949219 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(40%,40%,40%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 155.097656 298.898438 L 275.816406 298.898438 L 275.730469 298.898438 L 276.074219 298.882812 L 276.417969 298.8125 L 276.742188 298.691406 L 277.042969 298.515625 L 277.3125 298.296875 L 277.542969 298.035156 L 277.726562 297.742188 L 277.863281 297.421875 L 277.949219 297.085938 L 277.976562 296.738281 L 277.976562 286.71875 L 277.949219 286.371094 L 277.863281 286.035156 L 277.726562 285.714844 L 277.542969 285.421875 L 277.3125 285.160156 L 277.042969 284.941406 L 276.742188 284.765625 L 276.417969 284.644531 L 276.074219 284.574219 L 275.816406 284.558594 L 155.097656 284.558594 L 155.355469 284.574219 L 155.011719 284.558594 L 154.664062 284.601562 L 154.332031 284.699219 L 154.015625 284.847656 L 153.730469 285.046875 L 153.480469 285.285156 L 153.269531 285.5625 L 153.109375 285.871094 L 153 286.203125 L 152.945312 286.542969 L 152.9375 286.71875 L 152.9375 296.738281 L 152.945312 296.566406 L 152.945312 296.914062 L 153 297.257812 L 153.109375 297.585938 L 153.269531 297.894531 L 153.480469 298.171875 L 153.730469 298.414062 L 154.015625 298.609375 L 154.332031 298.757812 L 154.664062 298.855469 L 155.011719 298.898438 Z M 155.097656 298.898438 ' style='fill-rule:nonzero;fill:rgb(100%,100%,100%);fill-opacity:1;stroke-width:0.533489;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g style='fill:rgb(0%,0%,0%);fill-opacity:1;'>\n<use y='295.300781' x='156.539062' xlink:href='#44e7ce1f-601c-4c57-9df2-a5fdc2850847'/>\n<use y='295.300781' x='163.177719' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n<use y='295.300781' x='166.492188' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='295.300781' x='172.027649' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='295.300781' x='177.56311' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='295.300781' x='183.098572' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='295.300781' x='185.863876' xlink:href='#2a022af8-3a0b-46ac-afc9-a6c6e4fc0ea8'/>\n<use y='295.300781' x='193.605743' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n<use y='295.300781' x='196.920212' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='295.300781' x='202.455673' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='295.300781' x='207.991135' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='295.300781' x='213.526596' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='295.300781' x='216.291901' xlink:href='#998d2628-c24e-486f-9464-6673a1f907c5'/>\n<use y='295.300781' x='219.057205' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='295.300781' x='221.82251' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='295.300781' x='228.461166' xlink:href='#44e7ce1f-601c-4c57-9df2-a5fdc2850847'/>\n<use y='295.300781' x='235.099823' xlink:href='#553fc5e8-e5a3-452c-8e81-1d777c4746a7'/>\n<use y='295.300781' x='241.73848' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='295.300781' x='244.503784' xlink:href='#8e5eef14-8132-481d-b70b-2824ac7ffc6c'/>\n<use y='295.300781' x='251.69162' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='295.300781' x='257.227081' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='295.300781' x='262.762543' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='295.300781' x='265.527847' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='295.300781' x='271.063309' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n</g>\n<path d='M 596.246094 269.046875 L 615.246094 265.441406 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(40%,40%,40%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 496.164062 283.390625 L 620.753906 283.390625 L 620.667969 283.386719 L 621.015625 283.375 L 621.355469 283.304688 L 621.679688 283.179688 L 621.980469 283.007812 L 622.25 282.785156 L 622.480469 282.527344 L 622.667969 282.234375 L 622.804688 281.914062 L 622.886719 281.574219 L 622.914062 281.230469 L 622.914062 271.207031 L 622.886719 270.863281 L 622.804688 270.523438 L 622.667969 270.203125 L 622.480469 269.910156 L 622.25 269.652344 L 621.980469 269.429688 L 621.679688 269.257812 L 621.355469 269.132812 L 621.015625 269.0625 L 620.753906 269.046875 L 496.164062 269.046875 L 496.421875 269.0625 L 496.074219 269.050781 L 495.730469 269.09375 L 495.394531 269.1875 L 495.082031 269.335938 L 494.796875 269.535156 L 494.546875 269.777344 L 494.335938 270.054688 L 494.175781 270.363281 L 494.066406 270.691406 L 494.007812 271.035156 L 494.003906 271.207031 L 494.003906 281.230469 L 494.007812 281.054688 L 494.007812 281.402344 L 494.066406 281.746094 L 494.175781 282.074219 L 494.335938 282.382812 L 494.546875 282.660156 L 494.796875 282.902344 L 495.082031 283.097656 L 495.394531 283.25 L 495.730469 283.34375 L 496.074219 283.386719 Z M 496.164062 283.390625 ' style='fill-rule:nonzero;fill:rgb(100%,100%,100%);fill-opacity:1;stroke-width:0.533489;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g style='fill:rgb(0%,0%,0%);fill-opacity:1;'>\n<use y='279.789062' x='497.601562' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='279.789062' x='504.240219' xlink:href='#fefb0bdb-46bf-4fe8-ad02-31d0d222e074'/>\n<use y='279.789062' x='509.216782' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='279.789062' x='515.855438' xlink:href='#95ba0e07-9c88-4540-b5de-6e3d71a035f5'/>\n<use y='279.789062' x='523.043274' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='279.789062' x='525.808578' xlink:href='#0aec2617-bc8d-41ac-b492-83548ada7e54'/>\n<use y='279.789062' x='529.123047' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='279.789062' x='531.888351' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='279.789062' x='538.527008' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='279.789062' x='544.062469' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='279.789062' x='549.597931' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='279.789062' x='552.363235' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='279.789062' x='559.001892' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='279.789062' x='564.537354' xlink:href='#abb23039-e577-4ed0-beb1-d21bc1020176'/>\n<use y='279.789062' x='566.748627' xlink:href='#7d6ec591-ef49-4be4-b5f2-cb97084cdbb5'/>\n<use y='279.789062' x='571.725189' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='279.789062' x='577.260651' xlink:href='#94ca4bf9-8a65-48f8-8fad-c8bf73c19929'/>\n<use y='279.789062' x='582.796112' xlink:href='#dbf65fae-ae06-4862-99fa-000972f5777e'/>\n<use y='279.789062' x='588.331573' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n<use y='279.789062' x='591.646042' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='279.789062' x='594.411346' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='279.789062' x='599.946808' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='279.789062' x='602.712112' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='279.789062' x='605.477417' xlink:href='#60e4ab49-c824-444f-89d7-e92f0b7622cc'/>\n<use y='279.789062' x='611.012878' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='279.789062' x='613.778183' xlink:href='#fea978f8-7687-4c73-94fe-43f4f0bc3a1e'/>\n</g>\n<path d='M 77.171875 36.511719 L 73.566406 36.511719 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(40%,40%,40%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 79.332031 43.683594 L 226.039062 43.683594 L 225.953125 43.679688 L 226.300781 43.667969 L 226.640625 43.597656 L 226.964844 43.472656 L 227.269531 43.300781 L 227.535156 43.078125 L 227.765625 42.820312 L 227.953125 42.527344 L 228.089844 42.207031 L 228.171875 41.867188 L 228.199219 41.523438 L 228.199219 31.5 L 228.171875 31.15625 L 228.089844 30.816406 L 227.953125 30.496094 L 227.765625 30.203125 L 227.535156 29.945312 L 227.269531 29.722656 L 226.964844 29.550781 L 226.640625 29.425781 L 226.300781 29.355469 L 226.039062 29.339844 L 79.332031 29.339844 L 79.589844 29.355469 L 79.242188 29.34375 L 78.898438 29.386719 L 78.566406 29.480469 L 78.25 29.632812 L 77.964844 29.828125 L 77.714844 30.070312 L 77.503906 30.347656 L 77.34375 30.65625 L 77.234375 30.984375 L 77.179688 31.328125 L 77.171875 31.5 L 77.171875 41.523438 L 77.179688 41.347656 L 77.179688 41.695312 L 77.234375 42.039062 L 77.34375 42.367188 L 77.503906 42.675781 L 77.714844 42.953125 L 77.964844 43.195312 L 78.25 43.394531 L 78.566406 43.542969 L 78.898438 43.636719 L 79.242188 43.679688 Z M 79.332031 43.683594 ' style='fill-rule:nonzero;fill:rgb(100%,100%,100%);fill-opacity:1;stroke-width:0.533489;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g style='fill:rgb(0%,0%,0%);fill-opacity:1;'>\n<use y='40.082031' x='80.773438' xlink:href='#bce029c5-6a04-4949-a7f2-9ceac4d60995'/>\n<use y='40.082031' x='87.412094' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='40.082031' x='92.947556' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='40.082031' x='98.483017' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='40.082031' x='101.248322' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='40.082031' x='106.783783' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='40.082031' x='109.549088' xlink:href='#8e5eef14-8132-481d-b70b-2824ac7ffc6c'/>\n<use y='40.082031' x='116.736923' xlink:href='#abb23039-e577-4ed0-beb1-d21bc1020176'/>\n<use y='40.082031' x='118.948196' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='40.082031' x='124.483658' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n<use y='40.082031' x='127.798126' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='40.082031' x='133.333588' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='40.082031' x='136.098892' xlink:href='#8e5eef14-8132-481d-b70b-2824ac7ffc6c'/>\n<use y='40.082031' x='143.286728' xlink:href='#dbf65fae-ae06-4862-99fa-000972f5777e'/>\n<use y='40.082031' x='148.822189' xlink:href='#8bad88ae-7a7a-47aa-bdfe-88ec054c732b'/>\n<use y='40.082031' x='154.357651' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='40.082031' x='159.893112' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='40.082031' x='162.658417' xlink:href='#2bb104cd-2d57-457c-b53e-938fc695bd72'/>\n<use y='40.082031' x='167.634979' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='40.082031' x='170.400284' xlink:href='#8e5eef14-8132-481d-b70b-2824ac7ffc6c'/>\n<use y='40.082031' x='177.58812' xlink:href='#1dfb9be4-eb19-440b-95a7-a669197ba06e'/>\n<use y='40.082031' x='179.799393' xlink:href='#7d6ec591-ef49-4be4-b5f2-cb97084cdbb5'/>\n<use y='40.082031' x='184.775955' xlink:href='#1dfb9be4-eb19-440b-95a7-a669197ba06e'/>\n<use y='40.082031' x='186.987228' xlink:href='#07d2081b-8e74-404f-b60a-bda15f060dbc'/>\n<use y='40.082031' x='191.963791' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='40.082031' x='194.729095' xlink:href='#8e5eef14-8132-481d-b70b-2824ac7ffc6c'/>\n<use y='40.082031' x='201.916931' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='40.082031' x='207.452393' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='40.082031' x='212.987854' xlink:href='#1cb78ce3-365a-41f3-832a-54634e13752c'/>\n<use y='40.082031' x='215.753159' xlink:href='#1ca36580-885e-47e5-b102-89cd46a0408c'/>\n<use y='40.082031' x='221.28862' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n</g>\n<path d='M 266.617188 154.007812 L 266.617188 157.765625 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(40%,40%,40%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 238.066406 154.007812 L 295.160156 154.007812 L 295.074219 154.007812 L 295.421875 153.992188 L 295.761719 153.921875 L 296.085938 153.800781 L 296.386719 153.625 L 296.65625 153.40625 L 296.886719 153.144531 L 297.074219 152.851562 L 297.207031 152.53125 L 297.292969 152.195312 L 297.320312 151.847656 L 297.320312 141.828125 L 297.292969 141.480469 L 297.207031 141.144531 L 297.074219 140.824219 L 296.886719 140.53125 L 296.65625 140.269531 L 296.386719 140.050781 L 296.085938 139.875 L 295.761719 139.753906 L 295.421875 139.683594 L 295.160156 139.667969 L 238.066406 139.667969 L 238.328125 139.683594 L 237.980469 139.667969 L 237.636719 139.710938 L 237.300781 139.808594 L 236.988281 139.957031 L 236.703125 140.152344 L 236.449219 140.394531 L 236.242188 140.671875 L 236.082031 140.980469 L 235.96875 141.308594 L 235.914062 141.652344 L 235.90625 141.828125 L 235.90625 151.847656 L 235.914062 151.675781 L 235.914062 152.023438 L 235.96875 152.363281 L 236.082031 152.695312 L 236.242188 153.003906 L 236.449219 153.28125 L 236.703125 153.519531 L 236.988281 153.71875 L 237.300781 153.867188 L 237.636719 153.964844 L 237.980469 154.007812 Z M 238.066406 154.007812 ' style='fill-rule:nonzero;fill:rgb(100%,100%,100%);fill-opacity:1;stroke-width:0.533489;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g style='fill:rgb(0%,0%,0%);fill-opacity:1;'>\n<use y='150.40625' x='239.507812' xlink:href='#d62847a4-b70e-498f-b650-0fa63f23d7cf'/>\n<use y='150.40625' x='246.695648' xlink:href='#2bb104cd-2d57-457c-b53e-938fc695bd72'/>\n<use y='150.40625' x='251.672211' xlink:href='#abb23039-e577-4ed0-beb1-d21bc1020176'/>\n<use y='150.40625' x='253.883484' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='150.40625' x='259.418945' xlink:href='#b5ffbc07-c018-4471-8d17-28477c1e76f5'/>\n<use y='150.40625' x='264.954407' xlink:href='#94ca4bf9-8a65-48f8-8fad-c8bf73c19929'/>\n<use y='150.40625' x='270.489868' xlink:href='#ea17262e-0acb-4c28-9950-d8bcf6c27f90'/>\n<use y='150.40625' x='273.255173' xlink:href='#553fc5e8-e5a3-452c-8e81-1d777c4746a7'/>\n<use y='150.40625' x='279.893829' xlink:href='#32da805b-7aeb-40df-b595-c31cd6eac50b'/>\n<use y='150.40625' x='285.429291' xlink:href='#9df5463d-624a-43ad-b012-5e8f19bd92d7'/>\n<use y='150.40625' x='288.743759' xlink:href='#70b7e0c8-9a8c-45c5-972f-8a072643f9a7'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#ac7fdd66-d93b-4cb4-9d8b-7f268e0de2df)'>\n<path d='M 46.480469 331.332031 L 642.332031 331.332031 L 642.332031 22.472656 L 46.480469 22.472656 Z M 46.480469 331.332031 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='316.507812' x='21.035156' xlink:href='#17812813-511b-484d-b2cd-c0e84459a10c'/>\n<use y='316.507812' x='24.928223' xlink:href='#1e281205-001a-46d9-bec2-f59d8e6ffad0'/>\n<use y='316.507812' x='28.821289' xlink:href='#22c87d1e-6a10-484e-8a95-086a601e3558'/>\n<use y='316.507812' x='30.766113' xlink:href='#17812813-511b-484d-b2cd-c0e84459a10c'/>\n<use y='316.507812' x='34.65918' xlink:href='#17812813-511b-484d-b2cd-c0e84459a10c'/>\n<use y='316.507812' x='38.552246' xlink:href='#ea7daa72-002e-4fd2-86cc-efe8d5035022'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='255.121094' x='21.035156' xlink:href='#17812813-511b-484d-b2cd-c0e84459a10c'/>\n<use y='255.121094' x='24.928223' xlink:href='#1e281205-001a-46d9-bec2-f59d8e6ffad0'/>\n<use y='255.121094' x='28.821289' xlink:href='#22c87d1e-6a10-484e-8a95-086a601e3558'/>\n<use y='255.121094' x='30.766113' xlink:href='#17812813-511b-484d-b2cd-c0e84459a10c'/>\n<use y='255.121094' x='34.65918' xlink:href='#17812813-511b-484d-b2cd-c0e84459a10c'/>\n<use y='255.121094' x='38.552246' xlink:href='#bf8171fa-331e-4c5e-ab9f-95e9f07300d7'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='193.730469' x='21.035156' xlink:href='#17812813-511b-484d-b2cd-c0e84459a10c'/>\n<use y='193.730469' x='24.928223' xlink:href='#1e281205-001a-46d9-bec2-f59d8e6ffad0'/>\n<use y='193.730469' x='28.821289' xlink:href='#22c87d1e-6a10-484e-8a95-086a601e3558'/>\n<use y='193.730469' x='30.766113' xlink:href='#17812813-511b-484d-b2cd-c0e84459a10c'/>\n<use y='193.730469' x='34.65918' xlink:href='#a9ce67cc-8d4e-4732-a9bb-4b4bdd3e3785'/>\n<use y='193.730469' x='38.552246' xlink:href='#ea7daa72-002e-4fd2-86cc-efe8d5035022'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='132.34375' x='21.035156' xlink:href='#17812813-511b-484d-b2cd-c0e84459a10c'/>\n<use y='132.34375' x='24.928223' xlink:href='#1e281205-001a-46d9-bec2-f59d8e6ffad0'/>\n<use y='132.34375' x='28.821289' xlink:href='#22c87d1e-6a10-484e-8a95-086a601e3558'/>\n<use y='132.34375' x='30.766113' xlink:href='#17812813-511b-484d-b2cd-c0e84459a10c'/>\n<use y='132.34375' x='34.65918' xlink:href='#a9ce67cc-8d4e-4732-a9bb-4b4bdd3e3785'/>\n<use y='132.34375' x='38.552246' xlink:href='#bf8171fa-331e-4c5e-ab9f-95e9f07300d7'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='70.953125' x='21.035156' xlink:href='#17812813-511b-484d-b2cd-c0e84459a10c'/>\n<use y='70.953125' x='24.928223' xlink:href='#1e281205-001a-46d9-bec2-f59d8e6ffad0'/>\n<use y='70.953125' x='28.821289' xlink:href='#22c87d1e-6a10-484e-8a95-086a601e3558'/>\n<use y='70.953125' x='30.766113' xlink:href='#17812813-511b-484d-b2cd-c0e84459a10c'/>\n<use y='70.953125' x='34.65918' xlink:href='#bf8171fa-331e-4c5e-ab9f-95e9f07300d7'/>\n<use y='70.953125' x='38.552246' xlink:href='#ea7daa72-002e-4fd2-86cc-efe8d5035022'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='171.101562' xlink:href='#2338419e-ace9-4091-ba20-2e98a11c37c9'/>\n<use y='340.382812' x='173.432617' xlink:href='#286ae716-d55e-4ebd-8921-8b7558ff08f9'/>\n<use y='340.382812' x='177.325684' xlink:href='#c55069bc-2767-481a-ab7a-58c251e200a8'/>\n<use y='340.382812' x='181.21875' xlink:href='#286ae716-d55e-4ebd-8921-8b7558ff08f9'/>\n<use y='340.382812' x='185.111816' xlink:href='#22c87d1e-6a10-484e-8a95-086a601e3558'/>\n<use y='340.382812' x='187.056641' xlink:href='#b3ec2faa-e4a2-41cc-a4d2-078152e5e884'/>\n<use y='340.382812' x='190.949707' xlink:href='#ea7daa72-002e-4fd2-86cc-efe8d5035022'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='361.941406' xlink:href='#2338419e-ace9-4091-ba20-2e98a11c37c9'/>\n<use y='340.382812' x='364.272461' xlink:href='#286ae716-d55e-4ebd-8921-8b7558ff08f9'/>\n<use y='340.382812' x='368.165527' xlink:href='#c55069bc-2767-481a-ab7a-58c251e200a8'/>\n<use y='340.382812' x='372.058594' xlink:href='#286ae716-d55e-4ebd-8921-8b7558ff08f9'/>\n<use y='340.382812' x='375.95166' xlink:href='#22c87d1e-6a10-484e-8a95-086a601e3558'/>\n<use y='340.382812' x='377.896484' xlink:href='#cb0041e0-c7c9-4e38-8e74-0f0d21366b3f'/>\n<use y='340.382812' x='381.789551' xlink:href='#b3ec2faa-e4a2-41cc-a4d2-078152e5e884'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='552.785156' xlink:href='#2338419e-ace9-4091-ba20-2e98a11c37c9'/>\n<use y='340.382812' x='555.116211' xlink:href='#286ae716-d55e-4ebd-8921-8b7558ff08f9'/>\n<use y='340.382812' x='559.009277' xlink:href='#c55069bc-2767-481a-ab7a-58c251e200a8'/>\n<use y='340.382812' x='562.902344' xlink:href='#286ae716-d55e-4ebd-8921-8b7558ff08f9'/>\n<use y='340.382812' x='566.79541' xlink:href='#22c87d1e-6a10-484e-8a95-086a601e3558'/>\n<use y='340.382812' x='568.740234' xlink:href='#cb0041e0-c7c9-4e38-8e74-0f0d21366b3f'/>\n<use y='340.382812' x='572.633301' xlink:href='#cb0041e0-c7c9-4e38-8e74-0f0d21366b3f'/>\n</g>\n<g style='fill:rgb(32.156863%,32.156863%,32.156863%);fill-opacity:1;'>\n<use y='349.703125' x='328.171875' xlink:href='#32f5d44c-7199-47ed-a0d2-b25b9ae392b0'/>\n<use y='349.703125' x='329.949219' xlink:href='#d9f662c6-1ff3-470c-8801-3db4416f7959'/>\n<use y='349.703125' x='334.398438' xlink:href='#e88f8e61-4684-4b7c-a537-ac4239130370'/>\n<use y='349.703125' x='338.847656' xlink:href='#b1172a5b-18ca-456c-b911-aaa414368bde'/>\n<use y='349.703125' x='343.296875' xlink:href='#f67809ad-9474-48c6-9510-b883f0824063'/>\n<use y='349.703125' x='345.074219' xlink:href='#6e5b542a-aeb6-467f-9944-bf290d889f91'/>\n<use y='349.703125' x='347.296875' xlink:href='#00ed052e-ee8f-40e7-a440-c34b980bff9e'/>\n<use y='349.703125' x='351.746094' xlink:href='#38b5cae8-fefa-45db-914c-9209977c81d6'/>\n<use y='349.703125' x='356.195312' xlink:href='#594d09ad-0ef4-4938-8584-121081640ffc'/>\n</g>\n<g style='fill:rgb(32.156863%,32.156863%,32.156863%);fill-opacity:1;'>\n<use y='189.800781' x='17.449219' xlink:href='#9f87ea93-b03c-4322-8f8f-68a8dceaeff5'/>\n<use y='188.023438' x='17.449219' xlink:href='#161fc93a-9677-4906-91e1-c383682f999b'/>\n<use y='183.574219' x='17.449219' xlink:href='#656118ca-ecde-4e78-b1a8-db1b921ce903'/>\n<use y='181.351562' x='17.449219' xlink:href='#8e8a5c3a-126f-4244-83d2-087d58413494'/>\n<use y='179.574219' x='17.449219' xlink:href='#656118ca-ecde-4e78-b1a8-db1b921ce903'/>\n<use y='177.351562' x='17.449219' xlink:href='#a4db1a0c-d25a-4720-ae54-ef4bdef43247'/>\n<use y='172.902344' x='17.449219' xlink:href='#8465bf5c-70f2-476a-b848-55f9a6126c9f'/>\n<use y='168.453125' x='17.449219' xlink:href='#f8915848-23fc-406d-9f37-20632f361258'/>\n</g>\n</g>\n</svg>\n","value":"#huri.plot.GGView{:plot-command [[:<- :g [:data.frame {:longitude [:c -121.901782 -121.888979 -121.894902 -121.8932 -121.894074 -121.886943 -121.885831 -121.894715 -121.886995 -121.88566 -121.883891 -121.889937 -121.900084 -121.877349 -121.905733 -121.895617], :latitude [:c 37.329732 37.330698 37.333988 37.331415 37.336721 37.333798 37.330165 37.348742 37.337391 37.335885 37.332808 37.339301 37.332692 37.333955 37.352601 37.342725], :stationName [:c \"San Jose Diridon Caltrain Station\" \"San Jose Civic Center\" \"Santa Clara at Almaden\" \"Adobe on Almaden\" \"San Pedro Square\" \"Paseo de San Antonio\" \"San Salvador at 1st\" \"Japantown\" \"San Jose City Hall\" \"MLK Library\" \"SJSU 4th at San Carlos\" \"St James Park\" \"Arena Green / SAP Center\" \"SJSU - San Salvador at 9th\" \"Santa Clara County Civic Center\" \"Ryland Park\"]}]] [[:library :ggplot2] [:library :scales] [:library :grid] [:library :RColorBrewer] [:library :ggrepel] [:<- :palette [:brewer.pal \"Greys\" {:n 9}]] {:color.background :palette[2]} {:color.grid.major :palette[3]} {:color.axis.text :palette[6]} {:color.axis.title :palette[7]} {:color.title :palette[9]}] [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:ggplot :g [:aes {:x :longitude, :y :latitude}]] [:geom_point [:aes {}] {:alpha 0.5, :colour \"#c0392b\"}]] [:geom_label_repel [:aes {:label :stationName}] {:size 3.5, :show.legend false}]] [:scale_x_continuous {:labels :comma}]] [:scale_y_continuous {:labels :comma}]] [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:theme_bw {:base_size 9}] [:theme {:panel.background [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:plot.background [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:panel.border [:element_rect {:color :color.background}]}]] [:theme {:panel.grid.major [:element_line {:color :color.grid.major, :size 0.25}]}]] [:theme {:panel.grid.minor [:element_blank]}]] [:theme {:axis.ticks [:element_blank]}]] [:theme {:legend.background [:element_rect {:fill :color.background}]}]] [:theme {:legend.key [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:legend.text [:element_text {:size 7, :color :color.axis.title}]}]] [:theme {:legend.title [:element_blank]}]] [:theme {:plot.title [:element_text {:size 10, :color :color.title, :vjust 1.25}]}]] [:theme {:axis.text.x [:element_text {:size 7, :color :color.axis.text}]}]] [:theme {:axis.text.y [:element_text {:size 7, :color :color.axis.text}]}]] [:theme {:axis.title.x [:element_text {:size 8, :color :color.axis.title, :vjust 0}]}]] [:theme {:axis.title.y [:element_text {:size 8, :color :color.axis.title, :vjust 1.25}]}]] [:theme {:plot.margin [:unit [:c 0.35 0.2 0.3 0.35] \"cm\"]}]]] [:theme {:legend.position \"none\"}]] [:labs {:y \"latitude\", :title \"\", :x \"longitude\"}]]], :options {:width 9, :height 5}}"}
;; <=

;; @@
(line-chart :longitude :totalDocks {:trendline? true} df)
;; @@
;; =>
;;; {"type":"html","content":"<?xml version='1.0' encoding='UTF-8'?>\n<svg version='1.1' viewBox='0 0 648 360' height='360pt' width='648pt' xmlns:xlink='http://www.w3.org/1999/xlink' xmlns='http://www.w3.org/2000/svg'>\n<defs>\n<g>\n<symbol id='157ee8f7-0187-476f-ba68-8e5212306b2e' overflow='visible'>\n<path d='M 0.21875 0 L 0.21875 -5.015625 L 4.21875 -5.015625 L 4.21875 0 Z M 3.578125 -0.625 L 3.578125 -4.390625 L 0.859375 -4.390625 L 0.859375 -0.625 Z M 3.578125 -0.625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='a2970065-03ef-4de8-a2c4-ec63c1322b5e' overflow='visible'>\n<path d='M 0.671875 -3.46875 L 0.671875 -3.9375 C 1.117188 -3.976562 1.425781 -4.050781 1.59375 -4.15625 C 1.769531 -4.257812 1.90625 -4.5 2 -4.875 L 2.484375 -4.875 L 2.484375 0 L 1.828125 0 L 1.828125 -3.46875 Z M 0.671875 -3.46875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='67392821-102e-4f96-b49a-0500f2efd492' overflow='visible'>\n<path d='M 0.859375 -1.25 C 0.898438 -0.894531 1.066406 -0.648438 1.359375 -0.515625 C 1.503906 -0.453125 1.671875 -0.421875 1.859375 -0.421875 C 2.222656 -0.421875 2.492188 -0.535156 2.671875 -0.765625 C 2.847656 -0.992188 2.9375 -1.25 2.9375 -1.53125 C 2.9375 -1.875 2.832031 -2.140625 2.625 -2.328125 C 2.414062 -2.515625 2.164062 -2.609375 1.875 -2.609375 C 1.65625 -2.609375 1.46875 -2.566406 1.3125 -2.484375 C 1.164062 -2.398438 1.039062 -2.285156 0.9375 -2.140625 L 0.40625 -2.171875 L 0.78125 -4.8125 L 3.3125 -4.8125 L 3.3125 -4.21875 L 1.234375 -4.21875 L 1.03125 -2.859375 C 1.144531 -2.941406 1.253906 -3.003906 1.359375 -3.046875 C 1.535156 -3.128906 1.742188 -3.171875 1.984375 -3.171875 C 2.429688 -3.171875 2.8125 -3.023438 3.125 -2.734375 C 3.4375 -2.441406 3.59375 -2.070312 3.59375 -1.625 C 3.59375 -1.164062 3.445312 -0.757812 3.15625 -0.40625 C 2.875 -0.0507812 2.421875 0.125 1.796875 0.125 C 1.398438 0.125 1.046875 0.015625 0.734375 -0.203125 C 0.429688 -0.429688 0.257812 -0.78125 0.21875 -1.25 Z M 0.859375 -1.25 ' style='stroke:none;'/>\n</symbol>\n<symbol id='f3e1d9e8-165f-48e2-8fd8-ca8420a4e951' overflow='visible'>\n<path d='M 0.21875 0 C 0.238281 -0.414062 0.320312 -0.78125 0.46875 -1.09375 C 0.625 -1.40625 0.925781 -1.691406 1.375 -1.953125 L 2.03125 -2.328125 C 2.320312 -2.503906 2.523438 -2.648438 2.640625 -2.765625 C 2.828125 -2.953125 2.921875 -3.171875 2.921875 -3.421875 C 2.921875 -3.703125 2.832031 -3.925781 2.65625 -4.09375 C 2.488281 -4.257812 2.265625 -4.34375 1.984375 -4.34375 C 1.554688 -4.34375 1.265625 -4.1875 1.109375 -3.875 C 1.023438 -3.695312 0.976562 -3.457031 0.96875 -3.15625 L 0.34375 -3.15625 C 0.34375 -3.582031 0.421875 -3.929688 0.578125 -4.203125 C 0.835938 -4.671875 1.304688 -4.90625 1.984375 -4.90625 C 2.546875 -4.90625 2.957031 -4.753906 3.21875 -4.453125 C 3.476562 -4.148438 3.609375 -3.816406 3.609375 -3.453125 C 3.609375 -3.054688 3.46875 -2.71875 3.1875 -2.4375 C 3.03125 -2.28125 2.742188 -2.085938 2.328125 -1.859375 L 1.859375 -1.59375 C 1.640625 -1.46875 1.46875 -1.347656 1.34375 -1.234375 C 1.113281 -1.035156 0.96875 -0.816406 0.90625 -0.578125 L 3.578125 -0.578125 L 3.578125 0 Z M 0.21875 0 ' style='stroke:none;'/>\n</symbol>\n<symbol id='448f268c-ecc2-4793-aef2-6323c2e5595a' overflow='visible'>\n<path d='M 1.890625 -4.890625 C 2.523438 -4.890625 2.984375 -4.628906 3.265625 -4.109375 C 3.484375 -3.703125 3.59375 -3.148438 3.59375 -2.453125 C 3.59375 -1.785156 3.492188 -1.234375 3.296875 -0.796875 C 3.015625 -0.171875 2.546875 0.140625 1.890625 0.140625 C 1.304688 0.140625 0.867188 -0.113281 0.578125 -0.625 C 0.335938 -1.050781 0.21875 -1.625 0.21875 -2.34375 C 0.21875 -2.90625 0.289062 -3.382812 0.4375 -3.78125 C 0.707031 -4.519531 1.191406 -4.890625 1.890625 -4.890625 Z M 1.890625 -0.421875 C 2.203125 -0.421875 2.453125 -0.5625 2.640625 -0.84375 C 2.835938 -1.132812 2.9375 -1.664062 2.9375 -2.4375 C 2.9375 -2.988281 2.863281 -3.441406 2.71875 -3.796875 C 2.582031 -4.160156 2.320312 -4.34375 1.9375 -4.34375 C 1.570312 -4.34375 1.304688 -4.171875 1.140625 -3.828125 C 0.972656 -3.484375 0.890625 -2.984375 0.890625 -2.328125 C 0.890625 -1.828125 0.941406 -1.425781 1.046875 -1.125 C 1.210938 -0.65625 1.492188 -0.421875 1.890625 -0.421875 Z M 1.890625 -0.421875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='e8df3713-bdef-463b-8dcd-6de449d4a167' overflow='visible'>\n<path d='M 0.296875 -2.265625 L 2 -2.265625 L 2 -1.640625 L 0.296875 -1.640625 Z M 0.296875 -2.265625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='b9a8af37-51f1-4d71-92df-aef032fc4090' overflow='visible'>\n<path d='M 0.59375 -0.75 L 1.3125 -0.75 L 1.3125 0 L 0.59375 0 Z M 0.59375 -0.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='4e579ef6-5473-438f-9054-c33cefa794d6' overflow='visible'>\n<path d='M 2.3125 -1.734375 L 2.3125 -3.953125 L 0.75 -1.734375 Z M 2.328125 0 L 2.328125 -1.203125 L 0.171875 -1.203125 L 0.171875 -1.796875 L 2.421875 -4.90625 L 2.9375 -4.90625 L 2.9375 -1.734375 L 3.65625 -1.734375 L 3.65625 -1.203125 L 2.9375 -1.203125 L 2.9375 0 Z M 2.328125 0 ' style='stroke:none;'/>\n</symbol>\n<symbol id='5c166b9e-0229-4e56-a6d5-e7f25362e794' overflow='visible'>\n<path d='M 0.265625 0 L 0.265625 -5.734375 L 4.8125 -5.734375 L 4.8125 0 Z M 4.09375 -0.71875 L 4.09375 -5.015625 L 0.984375 -5.015625 L 0.984375 -0.71875 Z M 4.09375 -0.71875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='2611e663-5369-4154-985b-7a846e03cf55' overflow='visible'>\n<path d='M 0.53125 -5.734375 L 1.234375 -5.734375 L 1.234375 0 L 0.53125 0 Z M 0.53125 -5.734375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='e170efde-09e4-45db-b85c-38c2836e7a05' overflow='visible'>\n<path d='M 2.171875 -0.453125 C 2.640625 -0.453125 2.957031 -0.628906 3.125 -0.984375 C 3.300781 -1.335938 3.390625 -1.726562 3.390625 -2.15625 C 3.390625 -2.550781 3.328125 -2.867188 3.203125 -3.109375 C 3.003906 -3.492188 2.664062 -3.6875 2.1875 -3.6875 C 1.757812 -3.6875 1.445312 -3.519531 1.25 -3.1875 C 1.050781 -2.863281 0.953125 -2.472656 0.953125 -2.015625 C 0.953125 -1.566406 1.050781 -1.191406 1.25 -0.890625 C 1.445312 -0.597656 1.753906 -0.453125 2.171875 -0.453125 Z M 2.203125 -4.3125 C 2.742188 -4.3125 3.195312 -4.128906 3.5625 -3.765625 C 3.9375 -3.410156 4.125 -2.882812 4.125 -2.1875 C 4.125 -1.507812 3.957031 -0.945312 3.625 -0.5 C 3.300781 -0.0625 2.796875 0.15625 2.109375 0.15625 C 1.535156 0.15625 1.078125 -0.0351562 0.734375 -0.421875 C 0.398438 -0.816406 0.234375 -1.34375 0.234375 -2 C 0.234375 -2.707031 0.410156 -3.269531 0.765625 -3.6875 C 1.117188 -4.101562 1.597656 -4.3125 2.203125 -4.3125 Z M 2.1875 -4.28125 Z M 2.1875 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='ea7cdaa6-3682-4634-8a2e-bd2854741301' overflow='visible'>\n<path d='M 0.515625 -4.1875 L 1.1875 -4.1875 L 1.1875 -3.59375 C 1.382812 -3.832031 1.59375 -4.003906 1.8125 -4.109375 C 2.03125 -4.222656 2.273438 -4.28125 2.546875 -4.28125 C 3.140625 -4.28125 3.539062 -4.070312 3.75 -3.65625 C 3.863281 -3.425781 3.921875 -3.101562 3.921875 -2.6875 L 3.921875 0 L 3.21875 0 L 3.21875 -2.640625 C 3.21875 -2.890625 3.175781 -3.09375 3.09375 -3.25 C 2.96875 -3.507812 2.742188 -3.640625 2.421875 -3.640625 C 2.253906 -3.640625 2.117188 -3.625 2.015625 -3.59375 C 1.816406 -3.539062 1.644531 -3.425781 1.5 -3.25 C 1.382812 -3.113281 1.304688 -2.972656 1.265625 -2.828125 C 1.234375 -2.679688 1.21875 -2.46875 1.21875 -2.1875 L 1.21875 0 L 0.515625 0 Z M 2.171875 -4.28125 Z M 2.171875 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='a890e55b-113a-44aa-990f-3b30c60d6c39' overflow='visible'>\n<path d='M 2 -4.265625 C 2.320312 -4.265625 2.609375 -4.179688 2.859375 -4.015625 C 2.984375 -3.921875 3.117188 -3.789062 3.265625 -3.625 L 3.265625 -4.140625 L 3.90625 -4.140625 L 3.90625 -0.34375 C 3.90625 0.1875 3.828125 0.609375 3.671875 0.921875 C 3.378906 1.484375 2.828125 1.765625 2.015625 1.765625 C 1.566406 1.765625 1.1875 1.664062 0.875 1.46875 C 0.570312 1.269531 0.40625 0.953125 0.375 0.515625 L 1.078125 0.515625 C 1.117188 0.703125 1.1875 0.847656 1.28125 0.953125 C 1.445312 1.109375 1.695312 1.1875 2.03125 1.1875 C 2.570312 1.1875 2.925781 1 3.09375 0.625 C 3.195312 0.394531 3.242188 -0.00390625 3.234375 -0.578125 C 3.085938 -0.367188 2.914062 -0.210938 2.71875 -0.109375 C 2.519531 -0.00390625 2.257812 0.046875 1.9375 0.046875 C 1.488281 0.046875 1.09375 -0.113281 0.75 -0.4375 C 0.40625 -0.757812 0.234375 -1.289062 0.234375 -2.03125 C 0.234375 -2.726562 0.40625 -3.273438 0.75 -3.671875 C 1.09375 -4.066406 1.507812 -4.265625 2 -4.265625 Z M 3.265625 -2.109375 C 3.265625 -2.628906 3.15625 -3.015625 2.9375 -3.265625 C 2.726562 -3.515625 2.457031 -3.640625 2.125 -3.640625 C 1.625 -3.640625 1.285156 -3.40625 1.109375 -2.9375 C 1.015625 -2.695312 0.96875 -2.375 0.96875 -1.96875 C 0.96875 -1.5 1.0625 -1.140625 1.25 -0.890625 C 1.4375 -0.648438 1.691406 -0.53125 2.015625 -0.53125 C 2.523438 -0.53125 2.882812 -0.753906 3.09375 -1.203125 C 3.207031 -1.460938 3.265625 -1.765625 3.265625 -2.109375 Z M 2.078125 -4.28125 Z M 2.078125 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='1230cee7-dcd8-47f9-85ae-2d911a94583f' overflow='visible'>\n<path d='M 0.515625 -4.171875 L 1.234375 -4.171875 L 1.234375 0 L 0.515625 0 Z M 0.515625 -5.734375 L 1.234375 -5.734375 L 1.234375 -4.9375 L 0.515625 -4.9375 Z M 0.515625 -5.734375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='80625e48-25bc-4973-bfff-06aec7de99c7' overflow='visible'>\n<path d='M 0.65625 -5.359375 L 1.375 -5.359375 L 1.375 -4.1875 L 2.03125 -4.1875 L 2.03125 -3.609375 L 1.375 -3.609375 L 1.375 -0.875 C 1.375 -0.726562 1.421875 -0.632812 1.515625 -0.59375 C 1.566406 -0.5625 1.660156 -0.546875 1.796875 -0.546875 C 1.828125 -0.546875 1.859375 -0.546875 1.890625 -0.546875 C 1.929688 -0.546875 1.976562 -0.550781 2.03125 -0.5625 L 2.03125 0 C 1.957031 0.0195312 1.875 0.0351562 1.78125 0.046875 C 1.695312 0.0546875 1.601562 0.0625 1.5 0.0625 C 1.175781 0.0625 0.953125 -0.0195312 0.828125 -0.1875 C 0.710938 -0.351562 0.65625 -0.570312 0.65625 -0.84375 L 0.65625 -3.609375 L 0.09375 -3.609375 L 0.09375 -4.1875 L 0.65625 -4.1875 Z M 0.65625 -5.359375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='b51f8545-db8e-44a8-b1d1-3a4a29220138' overflow='visible'>\n<path d='M 1.21875 -4.1875 L 1.21875 -1.40625 C 1.21875 -1.195312 1.253906 -1.023438 1.328125 -0.890625 C 1.453125 -0.640625 1.679688 -0.515625 2.015625 -0.515625 C 2.503906 -0.515625 2.835938 -0.734375 3.015625 -1.171875 C 3.117188 -1.398438 3.171875 -1.71875 3.171875 -2.125 L 3.171875 -4.1875 L 3.875 -4.1875 L 3.875 0 L 3.203125 0 L 3.21875 -0.625 C 3.125 -0.457031 3.007812 -0.320312 2.875 -0.21875 C 2.601562 0 2.273438 0.109375 1.890625 0.109375 C 1.296875 0.109375 0.890625 -0.0859375 0.671875 -0.484375 C 0.554688 -0.703125 0.5 -0.988281 0.5 -1.34375 L 0.5 -4.1875 Z M 2.1875 -4.28125 Z M 2.1875 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='b4f99aa9-e806-40ce-9f4c-5f3f7d452127' overflow='visible'>\n<path d='M 0.96875 -2.046875 C 0.96875 -1.597656 1.0625 -1.222656 1.25 -0.921875 C 1.4375 -0.617188 1.738281 -0.46875 2.15625 -0.46875 C 2.488281 -0.46875 2.757812 -0.609375 2.96875 -0.890625 C 3.175781 -1.171875 3.28125 -1.570312 3.28125 -2.09375 C 3.28125 -2.625 3.171875 -3.015625 2.953125 -3.265625 C 2.734375 -3.515625 2.46875 -3.640625 2.15625 -3.640625 C 1.800781 -3.640625 1.515625 -3.503906 1.296875 -3.234375 C 1.078125 -2.960938 0.96875 -2.566406 0.96875 -2.046875 Z M 2.015625 -4.265625 C 2.335938 -4.265625 2.609375 -4.195312 2.828125 -4.0625 C 2.953125 -3.976562 3.09375 -3.835938 3.25 -3.640625 L 3.25 -5.765625 L 3.921875 -5.765625 L 3.921875 0 L 3.296875 0 L 3.296875 -0.578125 C 3.128906 -0.328125 2.929688 -0.140625 2.703125 -0.015625 C 2.484375 0.0976562 2.226562 0.15625 1.9375 0.15625 C 1.476562 0.15625 1.078125 -0.0390625 0.734375 -0.4375 C 0.390625 -0.832031 0.21875 -1.351562 0.21875 -2 C 0.21875 -2.613281 0.375 -3.144531 0.6875 -3.59375 C 1 -4.039062 1.441406 -4.265625 2.015625 -4.265625 Z M 2.015625 -4.265625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='ecca8276-7c93-4f15-b798-55a92d0dafb7' overflow='visible'>\n<path d='M 2.265625 -4.28125 C 2.554688 -4.28125 2.84375 -4.207031 3.125 -4.0625 C 3.40625 -3.925781 3.617188 -3.75 3.765625 -3.53125 C 3.898438 -3.320312 3.992188 -3.078125 4.046875 -2.796875 C 4.085938 -2.597656 4.109375 -2.289062 4.109375 -1.875 L 1.03125 -1.875 C 1.050781 -1.445312 1.15625 -1.101562 1.34375 -0.84375 C 1.53125 -0.59375 1.816406 -0.46875 2.203125 -0.46875 C 2.578125 -0.46875 2.875 -0.585938 3.09375 -0.828125 C 3.21875 -0.960938 3.304688 -1.125 3.359375 -1.3125 L 4.046875 -1.3125 C 4.023438 -1.15625 3.960938 -0.984375 3.859375 -0.796875 C 3.753906 -0.609375 3.640625 -0.457031 3.515625 -0.34375 C 3.304688 -0.132812 3.039062 0.0078125 2.71875 0.09375 C 2.550781 0.132812 2.363281 0.15625 2.15625 0.15625 C 1.632812 0.15625 1.191406 -0.03125 0.828125 -0.40625 C 0.460938 -0.789062 0.28125 -1.328125 0.28125 -2.015625 C 0.28125 -2.679688 0.460938 -3.222656 0.828125 -3.640625 C 1.191406 -4.066406 1.671875 -4.28125 2.265625 -4.28125 Z M 3.375 -2.421875 C 3.351562 -2.734375 3.289062 -2.976562 3.1875 -3.15625 C 2.988281 -3.5 2.664062 -3.671875 2.21875 -3.671875 C 1.894531 -3.671875 1.625 -3.550781 1.40625 -3.3125 C 1.1875 -3.082031 1.070312 -2.785156 1.0625 -2.421875 Z M 2.1875 -4.28125 Z M 2.1875 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='81302d72-3a49-4695-9eeb-2ab69fd8e493' overflow='visible'>\n<path d='M 0 -0.265625 L -5.734375 -0.265625 L -5.734375 -4.8125 L 0 -4.8125 Z M -0.71875 -4.09375 L -5.015625 -4.09375 L -5.015625 -0.984375 L -0.71875 -0.984375 Z M -0.71875 -4.09375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='eec05952-c111-45bb-b949-243f180e0290' overflow='visible'>\n<path d='M -5.359375 -0.65625 L -5.359375 -1.375 L -4.1875 -1.375 L -4.1875 -2.03125 L -3.609375 -2.03125 L -3.609375 -1.375 L -0.875 -1.375 C -0.726562 -1.375 -0.632812 -1.421875 -0.59375 -1.515625 C -0.5625 -1.566406 -0.546875 -1.660156 -0.546875 -1.796875 C -0.546875 -1.828125 -0.546875 -1.859375 -0.546875 -1.890625 C -0.546875 -1.929688 -0.550781 -1.976562 -0.5625 -2.03125 L 0 -2.03125 C 0.0195312 -1.957031 0.0351562 -1.875 0.046875 -1.78125 C 0.0546875 -1.695312 0.0625 -1.601562 0.0625 -1.5 C 0.0625 -1.175781 -0.0195312 -0.953125 -0.1875 -0.828125 C -0.351562 -0.710938 -0.570312 -0.65625 -0.84375 -0.65625 L -3.609375 -0.65625 L -3.609375 -0.09375 L -4.1875 -0.09375 L -4.1875 -0.65625 Z M -5.359375 -0.65625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='d86e7e30-19c8-4988-954c-64173ace324a' overflow='visible'>\n<path d='M -0.453125 -2.171875 C -0.453125 -2.640625 -0.628906 -2.957031 -0.984375 -3.125 C -1.335938 -3.300781 -1.726562 -3.390625 -2.15625 -3.390625 C -2.550781 -3.390625 -2.867188 -3.328125 -3.109375 -3.203125 C -3.492188 -3.003906 -3.6875 -2.664062 -3.6875 -2.1875 C -3.6875 -1.757812 -3.519531 -1.445312 -3.1875 -1.25 C -2.863281 -1.050781 -2.472656 -0.953125 -2.015625 -0.953125 C -1.566406 -0.953125 -1.191406 -1.050781 -0.890625 -1.25 C -0.597656 -1.445312 -0.453125 -1.753906 -0.453125 -2.171875 Z M -4.3125 -2.203125 C -4.3125 -2.742188 -4.128906 -3.195312 -3.765625 -3.5625 C -3.410156 -3.9375 -2.882812 -4.125 -2.1875 -4.125 C -1.507812 -4.125 -0.945312 -3.957031 -0.5 -3.625 C -0.0625 -3.300781 0.15625 -2.796875 0.15625 -2.109375 C 0.15625 -1.535156 -0.0351562 -1.078125 -0.421875 -0.734375 C -0.816406 -0.398438 -1.34375 -0.234375 -2 -0.234375 C -2.707031 -0.234375 -3.269531 -0.410156 -3.6875 -0.765625 C -4.101562 -1.117188 -4.3125 -1.597656 -4.3125 -2.203125 Z M -4.28125 -2.1875 Z M -4.28125 -2.1875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='2e16d35b-9f55-4a98-9ba5-9b46928f5d80' overflow='visible'>\n<path d='M -1.109375 -1.0625 C -0.910156 -1.0625 -0.75 -1.132812 -0.625 -1.28125 C -0.507812 -1.425781 -0.453125 -1.601562 -0.453125 -1.8125 C -0.453125 -2.050781 -0.507812 -2.289062 -0.625 -2.53125 C -0.820312 -2.914062 -1.132812 -3.109375 -1.5625 -3.109375 L -2.125 -3.109375 C -2.070312 -3.023438 -2.023438 -2.914062 -1.984375 -2.78125 C -1.953125 -2.644531 -1.929688 -2.507812 -1.921875 -2.375 L -1.859375 -1.953125 C -1.828125 -1.703125 -1.773438 -1.507812 -1.703125 -1.375 C -1.578125 -1.164062 -1.378906 -1.0625 -1.109375 -1.0625 Z M -2.53125 -2.765625 C -2.550781 -2.921875 -2.617188 -3.023438 -2.734375 -3.078125 C -2.796875 -3.109375 -2.890625 -3.125 -3.015625 -3.125 C -3.253906 -3.125 -3.425781 -3.039062 -3.53125 -2.875 C -3.632812 -2.707031 -3.6875 -2.460938 -3.6875 -2.140625 C -3.6875 -1.773438 -3.585938 -1.515625 -3.390625 -1.359375 C -3.285156 -1.273438 -3.125 -1.21875 -2.90625 -1.1875 L -2.90625 -0.53125 C -3.425781 -0.550781 -3.785156 -0.71875 -3.984375 -1.03125 C -4.179688 -1.351562 -4.28125 -1.726562 -4.28125 -2.15625 C -4.28125 -2.644531 -4.1875 -3.046875 -4 -3.359375 C -3.8125 -3.660156 -3.519531 -3.8125 -3.125 -3.8125 L -0.71875 -3.8125 C -0.644531 -3.8125 -0.585938 -3.828125 -0.546875 -3.859375 C -0.503906 -3.890625 -0.484375 -3.953125 -0.484375 -4.046875 C -0.484375 -4.078125 -0.484375 -4.113281 -0.484375 -4.15625 C -0.484375 -4.195312 -0.488281 -4.238281 -0.5 -4.28125 L 0.015625 -4.28125 C 0.046875 -4.164062 0.0664062 -4.078125 0.078125 -4.015625 C 0.0859375 -3.960938 0.09375 -3.890625 0.09375 -3.796875 C 0.09375 -3.554688 0.00390625 -3.378906 -0.171875 -3.265625 C -0.265625 -3.210938 -0.394531 -3.171875 -0.5625 -3.140625 C -0.375 -3.003906 -0.207031 -2.800781 -0.0625 -2.53125 C 0.0703125 -2.257812 0.140625 -1.960938 0.140625 -1.640625 C 0.140625 -1.253906 0.0234375 -0.9375 -0.203125 -0.6875 C -0.441406 -0.445312 -0.738281 -0.328125 -1.09375 -0.328125 C -1.488281 -0.328125 -1.789062 -0.445312 -2 -0.6875 C -2.21875 -0.925781 -2.347656 -1.242188 -2.390625 -1.640625 Z M -4.28125 -2.171875 Z M -4.28125 -2.171875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='ebfc8873-0bdf-4eec-b31c-f827c657bd84' overflow='visible'>\n<path d='M -5.734375 -0.53125 L -5.734375 -1.234375 L 0 -1.234375 L 0 -0.53125 Z M -5.734375 -0.53125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='a53c2182-d636-40bb-a81f-7f465c765e8a' overflow='visible'>\n<path d='M -0.671875 -2.8125 C -0.671875 -3.070312 -0.695312 -3.289062 -0.75 -3.46875 C -0.851562 -3.769531 -1.046875 -4.015625 -1.328125 -4.203125 C -1.566406 -4.359375 -1.863281 -4.46875 -2.21875 -4.53125 C -2.4375 -4.570312 -2.640625 -4.59375 -2.828125 -4.59375 C -3.535156 -4.59375 -4.085938 -4.453125 -4.484375 -4.171875 C -4.878906 -3.890625 -5.078125 -3.4375 -5.078125 -2.8125 L -5.078125 -1.421875 L -0.671875 -1.421875 Z M -5.734375 -0.640625 L -5.734375 -2.96875 C -5.734375 -3.757812 -5.457031 -4.375 -4.90625 -4.8125 C -4.394531 -5.195312 -3.742188 -5.390625 -2.953125 -5.390625 C -2.347656 -5.390625 -1.800781 -5.273438 -1.3125 -5.046875 C -0.4375 -4.640625 0 -3.945312 0 -2.96875 L 0 -0.640625 Z M -5.734375 -0.640625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='227e7841-6a8d-4f9e-a328-84171d708352' overflow='visible'>\n<path d='M -4.3125 -2.125 C -4.3125 -2.601562 -4.195312 -2.988281 -3.96875 -3.28125 C -3.738281 -3.570312 -3.34375 -3.75 -2.78125 -3.8125 L -2.78125 -3.125 C -3.039062 -3.082031 -3.253906 -2.988281 -3.421875 -2.84375 C -3.585938 -2.695312 -3.671875 -2.457031 -3.671875 -2.125 C -3.671875 -1.675781 -3.453125 -1.359375 -3.015625 -1.171875 C -2.734375 -1.046875 -2.382812 -0.984375 -1.96875 -0.984375 C -1.550781 -0.984375 -1.195312 -1.070312 -0.90625 -1.25 C -0.625 -1.425781 -0.484375 -1.703125 -0.484375 -2.078125 C -0.484375 -2.367188 -0.570312 -2.597656 -0.75 -2.765625 C -0.925781 -2.941406 -1.171875 -3.0625 -1.484375 -3.125 L -1.484375 -3.8125 C -0.929688 -3.738281 -0.523438 -3.546875 -0.265625 -3.234375 C -0.00390625 -2.921875 0.125 -2.519531 0.125 -2.03125 C 0.125 -1.476562 -0.0703125 -1.039062 -0.46875 -0.71875 C -0.875 -0.394531 -1.378906 -0.234375 -1.984375 -0.234375 C -2.710938 -0.234375 -3.28125 -0.410156 -3.6875 -0.765625 C -4.101562 -1.117188 -4.3125 -1.570312 -4.3125 -2.125 Z M -4.28125 -2.015625 Z M -4.28125 -2.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='09147308-b746-42b4-984e-42459e277ea8' overflow='visible'>\n<path d='M -5.734375 -0.5 L -5.734375 -1.171875 L -2.40625 -1.171875 L -4.1875 -2.984375 L -4.1875 -3.875 L -2.625 -2.28125 L 0 -3.96875 L 0 -3.078125 L -2.109375 -1.765625 L -1.578125 -1.171875 L 0 -1.171875 L 0 -0.5 Z M -5.734375 -0.5 ' style='stroke:none;'/>\n</symbol>\n<symbol id='730d48d1-ccb7-4ae5-927f-afdc4205d583' overflow='visible'>\n<path d='M -1.3125 -0.9375 C -1.082031 -0.957031 -0.90625 -1.015625 -0.78125 -1.109375 C -0.550781 -1.285156 -0.4375 -1.59375 -0.4375 -2.03125 C -0.4375 -2.289062 -0.492188 -2.519531 -0.609375 -2.71875 C -0.722656 -2.914062 -0.894531 -3.015625 -1.125 -3.015625 C -1.3125 -3.015625 -1.453125 -2.9375 -1.546875 -2.78125 C -1.597656 -2.675781 -1.660156 -2.472656 -1.734375 -2.171875 L -1.875 -1.625 C -1.96875 -1.257812 -2.066406 -0.992188 -2.171875 -0.828125 C -2.367188 -0.523438 -2.632812 -0.375 -2.96875 -0.375 C -3.363281 -0.375 -3.679688 -0.515625 -3.921875 -0.796875 C -4.171875 -1.085938 -4.296875 -1.472656 -4.296875 -1.953125 C -4.296875 -2.585938 -4.109375 -3.039062 -3.734375 -3.3125 C -3.503906 -3.488281 -3.253906 -3.578125 -2.984375 -3.578125 L -2.984375 -2.90625 C -3.140625 -2.894531 -3.285156 -2.835938 -3.421875 -2.734375 C -3.597656 -2.578125 -3.6875 -2.300781 -3.6875 -1.90625 C -3.6875 -1.632812 -3.632812 -1.429688 -3.53125 -1.296875 C -3.4375 -1.160156 -3.304688 -1.09375 -3.140625 -1.09375 C -2.953125 -1.09375 -2.804688 -1.179688 -2.703125 -1.359375 C -2.640625 -1.460938 -2.582031 -1.617188 -2.53125 -1.828125 L -2.421875 -2.28125 C -2.296875 -2.789062 -2.175781 -3.128906 -2.0625 -3.296875 C -1.882812 -3.578125 -1.601562 -3.71875 -1.21875 -3.71875 C -0.84375 -3.71875 -0.519531 -3.570312 -0.25 -3.28125 C 0.0195312 -3 0.15625 -2.570312 0.15625 -2 C 0.15625 -1.375 0.015625 -0.929688 -0.265625 -0.671875 C -0.546875 -0.410156 -0.894531 -0.273438 -1.3125 -0.265625 Z M -4.28125 -1.96875 Z M -4.28125 -1.96875 ' style='stroke:none;'/>\n</symbol>\n</g>\n<clipPath id='8c16fec0-3c32-4e54-ac6d-6ce50f0072ea'>\n<path d='M 32.855469 22.472656 L 643 22.472656 L 643 332 L 32.855469 332 Z M 32.855469 22.472656 '/>\n</clipPath>\n<clipPath id='7275067a-df9a-4af5-9d53-1237d4418d7a'>\n<path d='M 32.855469 246 L 643 246 L 643 248 L 32.855469 248 Z M 32.855469 246 '/>\n</clipPath>\n<clipPath id='494cd928-db41-4145-9a9e-7f91952c5c7d'>\n<path d='M 32.855469 159 L 643 159 L 643 160 L 32.855469 160 Z M 32.855469 159 '/>\n</clipPath>\n<clipPath id='f133482b-c1d5-40b2-a8bf-18fd8bb9fd22'>\n<path d='M 32.855469 71 L 643 71 L 643 72 L 32.855469 72 Z M 32.855469 71 '/>\n</clipPath>\n<clipPath id='b87cc2ad-97f1-4f16-a26c-6b50da7c747d'>\n<path d='M 79 22.472656 L 81 22.472656 L 81 332 L 79 332 Z M 79 22.472656 '/>\n</clipPath>\n<clipPath id='25da0268-9892-4e4e-9949-87f54bb34f41'>\n<path d='M 284 22.472656 L 285 22.472656 L 285 332 L 284 332 Z M 284 22.472656 '/>\n</clipPath>\n<clipPath id='e1c3d546-97cc-429e-8b61-6cd0c3c2847c'>\n<path d='M 488 22.472656 L 490 22.472656 L 490 332 L 488 332 Z M 488 22.472656 '/>\n</clipPath>\n<clipPath id='26b47c95-6d10-4581-bab8-dee52a922910'>\n<path d='M 32.855469 22.472656 L 643 22.472656 L 643 332 L 32.855469 332 Z M 32.855469 22.472656 '/>\n</clipPath>\n</defs>\n<g id='78332058-2fda-4391-93c4-877c3d473843'>\n<rect style='fill:rgb(100%,100%,100%);fill-opacity:1;stroke:none;' height='360' width='648' y='0' x='0'/>\n<rect style='fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke:none;' height='360' width='648' y='0' x='0'/>\n<path d='M 0 360 L 648 360 L 648 0 L 0 0 Z M 0 360 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g clip-rule='nonzero' clip-path='url(#8c16fec0-3c32-4e54-ac6d-6ce50f0072ea)'>\n<path d='M 32.855469 331.332031 L 642.328125 331.332031 L 642.328125 22.472656 L 32.855469 22.472656 Z M 32.855469 331.332031 ' style='fill-rule:nonzero;fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#7275067a-df9a-4af5-9d53-1237d4418d7a)'>\n<path d='M 32.855469 247.097656 L 642.332031 247.097656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#494cd928-db41-4145-9a9e-7f91952c5c7d)'>\n<path d='M 32.855469 159.351562 L 642.332031 159.351562 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#f133482b-c1d5-40b2-a8bf-18fd8bb9fd22)'>\n<path d='M 32.855469 71.609375 L 642.332031 71.609375 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#b87cc2ad-97f1-4f16-a26c-6b50da7c747d)'>\n<path d='M 79.949219 331.332031 L 79.949219 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#25da0268-9892-4e4e-9949-87f54bb34f41)'>\n<path d='M 284.550781 331.332031 L 284.550781 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#e1c3d546-97cc-429e-8b61-6cd0c3c2847c)'>\n<path d='M 489.152344 331.332031 L 489.152344 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<path d='M 60.558594 176.902344 L 60.917969 106.707031 L 62.164062 36.511719 L 62.226562 176.902344 L 67.933594 106.707031 L 71.253906 176.902344 L 73.871094 176.902344 L 74.703125 247.097656 L 74.871094 176.902344 L 75.074219 247.097656 L 76.417969 176.902344 L 76.640625 247.097656 L 77.171875 247.097656 L 78.914062 176.902344 L 79.066406 247.097656 L 79.121094 36.511719 L 79.980469 247.097656 L 80.921875 176.902344 L 81.457031 247.097656 L 81.550781 247.097656 L 81.886719 176.902344 L 83.59375 176.902344 L 84.585938 106.707031 L 84.777344 176.902344 L 85.429688 106.707031 L 85.9375 106.707031 L 87.378906 247.097656 L 89.121094 176.902344 L 89.796875 176.902344 L 89.886719 36.511719 L 92.210938 247.097656 L 250.742188 247.097656 L 251.726562 36.511719 L 253.910156 247.097656 L 281.1875 247.097656 L 320.605469 106.707031 L 322.308594 317.292969 L 325.980469 317.292969 L 343.0625 247.097656 L 378.324219 247.097656 L 379.941406 106.707031 L 403.550781 317.292969 L 405.375 247.097656 L 406.675781 247.097656 L 410.675781 106.707031 L 435.46875 247.097656 L 585.589844 247.097656 L 589.632812 36.511719 L 591.371094 176.902344 L 595.9375 247.097656 L 596.671875 317.292969 L 596.863281 247.097656 L 597.515625 247.097656 L 598.410156 176.902344 L 601.75 247.097656 L 605.949219 247.097656 L 606.125 176.902344 L 607.933594 176.902344 L 614.628906 247.097656 ' style='fill:none;stroke-width:1.066978;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(75.294118%,22.352941%,16.862745%);stroke-opacity:1;stroke-miterlimit:1;'/>\n<path d='M 60.558594 132.886719 L 67.574219 142.121094 L 74.585938 148.992188 L 81.601562 153.050781 L 88.613281 154.5625 L 95.628906 154.324219 L 102.640625 153.304688 L 109.65625 152.109375 L 116.667969 151.046875 L 123.679688 150.265625 L 130.695312 149.824219 L 137.707031 149.753906 L 144.722656 150.042969 L 151.734375 150.675781 L 158.75 151.632812 L 165.761719 152.875 L 172.777344 154.371094 L 179.789062 156.082031 L 186.804688 157.976562 L 193.816406 160.007812 L 200.832031 162.144531 L 207.84375 164.347656 L 214.855469 166.582031 L 221.871094 168.816406 L 228.882812 171.019531 L 235.898438 173.167969 L 242.910156 175.242188 L 249.925781 177.230469 L 256.9375 179.117188 L 263.953125 180.90625 L 270.964844 182.59375 L 277.980469 184.195312 L 284.992188 185.71875 L 292.003906 187.175781 L 299.019531 188.578125 L 306.03125 189.9375 L 313.046875 191.25 L 320.058594 192.507812 L 327.074219 193.652344 L 334.085938 194.523438 L 341.101562 194.996094 L 348.113281 194.984375 L 355.128906 194.445312 L 362.140625 193.398438 L 369.15625 191.929688 L 376.167969 190.167969 L 383.179688 188.265625 L 390.195312 186.394531 L 397.207031 184.726562 L 404.222656 183.417969 L 411.234375 182.609375 L 418.25 182.101562 L 425.261719 181.675781 L 432.277344 181.347656 L 439.289062 181.117188 L 446.304688 180.992188 L 453.316406 180.96875 L 460.328125 181.054688 L 467.34375 181.25 L 474.355469 181.558594 L 481.371094 181.96875 L 488.382812 182.484375 L 495.398438 183.101562 L 502.410156 183.8125 L 509.425781 184.605469 L 516.4375 185.464844 L 523.453125 186.375 L 530.464844 187.3125 L 537.480469 188.242188 L 544.492188 189.132812 L 551.503906 189.925781 L 558.519531 190.5625 L 565.53125 190.976562 L 572.546875 191.09375 L 579.558594 190.828125 L 586.574219 190.121094 L 593.585938 188.914062 L 600.601562 187.1875 L 607.613281 184.933594 L 614.628906 182.152344 L 614.628906 260.265625 L 607.613281 257.703125 L 600.601562 255.6875 L 593.585938 254.253906 L 586.574219 253.402344 L 579.558594 253.101562 L 572.546875 253.292969 L 565.53125 253.898438 L 558.519531 254.84375 L 551.503906 256.042969 L 544.492188 257.425781 L 537.480469 258.917969 L 530.464844 260.472656 L 523.453125 262.039062 L 516.4375 263.582031 L 509.425781 265.074219 L 502.410156 266.488281 L 495.398438 267.808594 L 488.382812 269.019531 L 481.371094 270.105469 L 474.355469 271.058594 L 467.34375 271.871094 L 460.328125 272.53125 L 453.316406 273.039062 L 446.304688 273.390625 L 439.289062 273.578125 L 432.277344 273.601562 L 425.261719 273.460938 L 418.25 273.152344 L 411.234375 272.679688 L 404.222656 271.957031 L 397.207031 270.914062 L 390.195312 269.660156 L 383.179688 268.308594 L 376.167969 266.980469 L 369.15625 265.800781 L 362.140625 264.882812 L 355.128906 264.308594 L 348.113281 264.121094 L 341.101562 264.296875 L 334.085938 264.746094 L 327.074219 265.335938 L 320.058594 265.910156 L 313.046875 266.390625 L 306.03125 266.761719 L 299.019531 267.011719 L 292.003906 267.140625 L 284.992188 267.164062 L 277.980469 267.09375 L 270.964844 266.945312 L 263.953125 266.734375 L 256.9375 266.464844 L 249.925781 266.148438 L 242.910156 265.78125 L 235.898438 265.359375 L 228.882812 264.871094 L 221.871094 264.292969 L 214.855469 263.609375 L 207.84375 262.789062 L 200.832031 261.804688 L 193.816406 260.625 L 186.804688 259.210938 L 179.789062 257.527344 L 172.777344 255.542969 L 165.761719 253.21875 L 158.75 250.523438 L 151.734375 247.421875 L 144.722656 243.886719 L 137.707031 239.894531 L 130.695312 235.429688 L 123.679688 230.488281 L 116.667969 225.097656 L 109.65625 219.328125 L 102.640625 213.324219 L 95.628906 207.394531 L 88.613281 202.203125 L 81.601562 198.59375 L 74.585938 197.226562 L 67.574219 198.289062 L 60.558594 201.40625 Z M 60.558594 132.886719 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(0%,0%,0%);fill-opacity:0.25098;'/>\n<path d='M 60.558594 167.148438 L 67.574219 170.203125 L 74.585938 173.109375 L 81.601562 175.820312 L 88.613281 178.382812 L 95.628906 180.859375 L 102.640625 183.3125 L 109.65625 185.71875 L 116.667969 188.074219 L 123.679688 190.375 L 130.695312 192.628906 L 137.707031 194.824219 L 144.722656 196.964844 L 151.734375 199.050781 L 158.75 201.078125 L 165.761719 203.046875 L 172.777344 204.957031 L 179.789062 206.804688 L 186.804688 208.59375 L 193.816406 210.316406 L 200.832031 211.976562 L 207.84375 213.566406 L 214.855469 215.09375 L 221.871094 216.554688 L 228.882812 217.945312 L 235.898438 219.265625 L 242.910156 220.511719 L 249.925781 221.6875 L 256.9375 222.792969 L 263.953125 223.820312 L 270.964844 224.769531 L 277.980469 225.644531 L 284.992188 226.441406 L 292.003906 227.160156 L 299.019531 227.796875 L 306.03125 228.351562 L 313.046875 228.820312 L 320.058594 229.207031 L 327.074219 229.492188 L 334.085938 229.632812 L 341.101562 229.644531 L 348.113281 229.554688 L 355.128906 229.375 L 362.140625 229.140625 L 369.15625 228.863281 L 383.179688 228.285156 L 390.195312 228.027344 L 397.207031 227.820312 L 404.222656 227.6875 L 411.234375 227.644531 L 418.25 227.625 L 425.261719 227.570312 L 432.277344 227.476562 L 439.289062 227.347656 L 446.304688 227.191406 L 453.316406 227.003906 L 460.328125 226.792969 L 467.34375 226.5625 L 474.355469 226.308594 L 481.371094 226.035156 L 488.382812 225.753906 L 495.398438 225.457031 L 502.410156 225.148438 L 509.425781 224.839844 L 516.4375 224.523438 L 523.453125 224.207031 L 530.464844 223.890625 L 537.480469 223.582031 L 544.492188 223.277344 L 551.503906 222.984375 L 558.519531 222.703125 L 565.53125 222.4375 L 572.546875 222.191406 L 579.558594 221.964844 L 586.574219 221.761719 L 593.585938 221.585938 L 600.601562 221.4375 L 607.613281 221.320312 L 614.628906 221.210938 ' style='fill:none;stroke-width:2.133957;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:1;'/>\n<g clip-rule='nonzero' clip-path='url(#26b47c95-6d10-4581-bab8-dee52a922910)'>\n<path d='M 32.855469 331.332031 L 642.328125 331.332031 L 642.328125 22.472656 L 32.855469 22.472656 Z M 32.855469 331.332031 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='249.605469' x='21.035156' xlink:href='#a2970065-03ef-4de8-a2c4-ec63c1322b5e'/>\n<use y='249.605469' x='24.928223' xlink:href='#67392821-102e-4f96-b49a-0500f2efd492'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='161.859375' x='21.035156' xlink:href='#f3e1d9e8-165f-48e2-8fd8-ca8420a4e951'/>\n<use y='161.859375' x='24.928223' xlink:href='#448f268c-ecc2-4793-aef2-6323c2e5595a'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='74.117188' x='21.035156' xlink:href='#f3e1d9e8-165f-48e2-8fd8-ca8420a4e951'/>\n<use y='74.117188' x='24.928223' xlink:href='#67392821-102e-4f96-b49a-0500f2efd492'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='70.023438' xlink:href='#e8df3713-bdef-463b-8dcd-6de449d4a167'/>\n<use y='340.382812' x='72.354492' xlink:href='#a2970065-03ef-4de8-a2c4-ec63c1322b5e'/>\n<use y='340.382812' x='76.247559' xlink:href='#f3e1d9e8-165f-48e2-8fd8-ca8420a4e951'/>\n<use y='340.382812' x='80.140625' xlink:href='#f3e1d9e8-165f-48e2-8fd8-ca8420a4e951'/>\n<use y='340.382812' x='84.033691' xlink:href='#b9a8af37-51f1-4d71-92df-aef032fc4090'/>\n<use y='340.382812' x='85.978516' xlink:href='#4e579ef6-5473-438f-9054-c33cefa794d6'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='274.625' xlink:href='#e8df3713-bdef-463b-8dcd-6de449d4a167'/>\n<use y='340.382812' x='276.956055' xlink:href='#a2970065-03ef-4de8-a2c4-ec63c1322b5e'/>\n<use y='340.382812' x='280.849121' xlink:href='#f3e1d9e8-165f-48e2-8fd8-ca8420a4e951'/>\n<use y='340.382812' x='284.742188' xlink:href='#f3e1d9e8-165f-48e2-8fd8-ca8420a4e951'/>\n<use y='340.382812' x='288.635254' xlink:href='#b9a8af37-51f1-4d71-92df-aef032fc4090'/>\n<use y='340.382812' x='290.580078' xlink:href='#f3e1d9e8-165f-48e2-8fd8-ca8420a4e951'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='479.226562' xlink:href='#e8df3713-bdef-463b-8dcd-6de449d4a167'/>\n<use y='340.382812' x='481.557617' xlink:href='#a2970065-03ef-4de8-a2c4-ec63c1322b5e'/>\n<use y='340.382812' x='485.450684' xlink:href='#f3e1d9e8-165f-48e2-8fd8-ca8420a4e951'/>\n<use y='340.382812' x='489.34375' xlink:href='#f3e1d9e8-165f-48e2-8fd8-ca8420a4e951'/>\n<use y='340.382812' x='493.236816' xlink:href='#b9a8af37-51f1-4d71-92df-aef032fc4090'/>\n<use y='340.382812' x='495.181641' xlink:href='#448f268c-ecc2-4793-aef2-6323c2e5595a'/>\n</g>\n<g style='fill:rgb(32.156863%,32.156863%,32.156863%);fill-opacity:1;'>\n<use y='349.703125' x='321.359375' xlink:href='#2611e663-5369-4154-985b-7a846e03cf55'/>\n<use y='349.703125' x='323.136719' xlink:href='#e170efde-09e4-45db-b85c-38c2836e7a05'/>\n<use y='349.703125' x='327.585938' xlink:href='#ea7cdaa6-3682-4634-8a2e-bd2854741301'/>\n<use y='349.703125' x='332.035156' xlink:href='#a890e55b-113a-44aa-990f-3b30c60d6c39'/>\n<use y='349.703125' x='336.484375' xlink:href='#1230cee7-dcd8-47f9-85ae-2d911a94583f'/>\n<use y='349.703125' x='338.261719' xlink:href='#80625e48-25bc-4973-bfff-06aec7de99c7'/>\n<use y='349.703125' x='340.484375' xlink:href='#b51f8545-db8e-44a8-b1d1-3a4a29220138'/>\n<use y='349.703125' x='344.933594' xlink:href='#b4f99aa9-e806-40ce-9f4c-5f3f7d452127'/>\n<use y='349.703125' x='349.382812' xlink:href='#ecca8276-7c93-4f15-b798-55a92d0dafb7'/>\n</g>\n<g style='fill:rgb(32.156863%,32.156863%,32.156863%);fill-opacity:1;'>\n<use y='195.574219' x='17.449219' xlink:href='#eec05952-c111-45bb-b949-243f180e0290'/>\n<use y='193.351562' x='17.449219' xlink:href='#d86e7e30-19c8-4988-954c-64173ace324a'/>\n<use y='188.902344' x='17.449219' xlink:href='#eec05952-c111-45bb-b949-243f180e0290'/>\n<use y='186.679688' x='17.449219' xlink:href='#2e16d35b-9f55-4a98-9ba5-9b46928f5d80'/>\n<use y='182.230469' x='17.449219' xlink:href='#ebfc8873-0bdf-4eec-b31c-f827c657bd84'/>\n<use y='180.453125' x='17.449219' xlink:href='#a53c2182-d636-40bb-a81f-7f465c765e8a'/>\n<use y='174.675781' x='17.449219' xlink:href='#d86e7e30-19c8-4988-954c-64173ace324a'/>\n<use y='170.226562' x='17.449219' xlink:href='#227e7841-6a8d-4f9e-a328-84171d708352'/>\n<use y='166.226562' x='17.449219' xlink:href='#09147308-b746-42b4-984e-42459e277ea8'/>\n<use y='162.226562' x='17.449219' xlink:href='#730d48d1-ccb7-4ae5-927f-afdc4205d583'/>\n</g>\n</g>\n</svg>\n","value":"#huri.plot.GGView{:plot-command [[:<- :g [:data.frame {:longitude [:c -121.901782 -121.888979 -121.894902 -121.8932 -121.894074 -121.886943 -121.885831 -121.894715 -121.886995 -121.88566 -121.883891 -121.889937 -121.900084 -121.877349 -122.253755 -122.232089 -122.229951 -122.227424 -122.203288 -122.224728 -122.081896 -122.076713 -122.106758 -122.052476 -122.108338 -122.083678 -122.080623 -122.164759 -122.163093 -122.142805 -122.159504 -122.137619 -122.408433 -122.39997 -122.398436 -122.402923 -122.404767 -122.403452 -122.398525 -122.390373 -122.394148 -122.391034 -122.388013 -122.394643 -122.397086 -122.405127 -122.417324 -122.418603 -122.403234 -122.390288 -122.39644 -122.398108 -122.392738 -122.402717 -122.418954 -122.417385 -122.401014 -122.39547 -122.39528 -122.408499 -122.411748 -122.405942 -122.39461 -122.399051 -122.404966 -122.400811 -121.905733 -122.400862 -122.233051 -121.895617], :totalDocks [:c 27 15 11 19 15 15 15 15 15 19 19 15 19 15 15 27 15 15 15 15 15 23 23 15 15 11 15 23 11 15 11 15 19 15 15 15 15 19 15 19 23 19 15 23 19 15 19 23 15 27 19 19 15 15 19 27 19 23 19 19 23 19 23 19 19 27 15 15 15 15]}]] [[:library :ggplot2] [:library :scales] [:library :grid] [:library :RColorBrewer] [:library :ggrepel] [:<- :palette [:brewer.pal \"Greys\" {:n 9}]] {:color.background :palette[2]} {:color.grid.major :palette[3]} {:color.axis.text :palette[6]} {:color.axis.title :palette[7]} {:color.title :palette[9]}] [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:ggplot :g [:aes {:x :longitude, :y :totalDocks}]] [:geom_line {:colour \"#c0392b\"}]] [:geom_smooth [:aes {}] {:fill \"black\", :alpha 0.25, :colour \"black\"}]] [:scale_x_continuous {:labels :comma}]] [:scale_y_continuous {:labels :comma}]] [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:theme_bw {:base_size 9}] [:theme {:panel.background [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:plot.background [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:panel.border [:element_rect {:color :color.background}]}]] [:theme {:panel.grid.major [:element_line {:color :color.grid.major, :size 0.25}]}]] [:theme {:panel.grid.minor [:element_blank]}]] [:theme {:axis.ticks [:element_blank]}]] [:theme {:legend.background [:element_rect {:fill :color.background}]}]] [:theme {:legend.key [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:legend.text [:element_text {:size 7, :color :color.axis.title}]}]] [:theme {:legend.title [:element_blank]}]] [:theme {:plot.title [:element_text {:size 10, :color :color.title, :vjust 1.25}]}]] [:theme {:axis.text.x [:element_text {:size 7, :color :color.axis.text}]}]] [:theme {:axis.text.y [:element_text {:size 7, :color :color.axis.text}]}]] [:theme {:axis.title.x [:element_text {:size 8, :color :color.axis.title, :vjust 0}]}]] [:theme {:axis.title.y [:element_text {:size 8, :color :color.axis.title, :vjust 1.25}]}]] [:theme {:plot.margin [:unit [:c 0.35 0.2 0.3 0.35] \"cm\"]}]]] [:theme {:legend.position \"none\"}]] [:labs {:y \"totalDocks\", :title \"\", :x \"longitude\"}]]], :options {:width 9, :height 5}}"}
;; <=

;; @@
(scatter-plot :longitude :latitude {:group-by :city} df) 
;; @@
;; =>
;;; {"type":"html","content":"<?xml version='1.0' encoding='UTF-8'?>\n<svg version='1.1' viewBox='0 0 648 360' height='360pt' width='648pt' xmlns:xlink='http://www.w3.org/1999/xlink' xmlns='http://www.w3.org/2000/svg'>\n<defs>\n<g>\n<symbol id='e0306d94-74cf-46c6-bb0a-34f13ccbfc35' overflow='visible'>\n<path d='M 0.21875 0 L 0.21875 -5.015625 L 4.21875 -5.015625 L 4.21875 0 Z M 3.578125 -0.625 L 3.578125 -4.390625 L 0.859375 -4.390625 L 0.859375 -0.625 Z M 3.578125 -0.625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='76cb1015-cd3a-42f4-9d1e-4fbe417959cc' overflow='visible'>\n<path d='M 1.8125 0.140625 C 1.238281 0.140625 0.820312 -0.0195312 0.5625 -0.34375 C 0.300781 -0.664062 0.171875 -1.050781 0.171875 -1.5 L 0.8125 -1.5 C 0.832031 -1.1875 0.890625 -0.957031 0.984375 -0.8125 C 1.148438 -0.550781 1.441406 -0.421875 1.859375 -0.421875 C 2.179688 -0.421875 2.4375 -0.507812 2.625 -0.6875 C 2.820312 -0.863281 2.921875 -1.085938 2.921875 -1.359375 C 2.921875 -1.691406 2.816406 -1.925781 2.609375 -2.0625 C 2.410156 -2.195312 2.125 -2.265625 1.75 -2.265625 C 1.71875 -2.265625 1.675781 -2.265625 1.625 -2.265625 C 1.582031 -2.265625 1.539062 -2.265625 1.5 -2.265625 L 1.5 -2.796875 C 1.5625 -2.796875 1.613281 -2.789062 1.65625 -2.78125 C 1.707031 -2.78125 1.757812 -2.78125 1.8125 -2.78125 C 2.039062 -2.78125 2.226562 -2.816406 2.375 -2.890625 C 2.632812 -3.023438 2.765625 -3.253906 2.765625 -3.578125 C 2.765625 -3.816406 2.675781 -4.003906 2.5 -4.140625 C 2.332031 -4.273438 2.128906 -4.34375 1.890625 -4.34375 C 1.484375 -4.34375 1.203125 -4.207031 1.046875 -3.9375 C 0.960938 -3.78125 0.914062 -3.5625 0.90625 -3.28125 L 0.296875 -3.28125 C 0.296875 -3.644531 0.367188 -3.957031 0.515625 -4.21875 C 0.765625 -4.675781 1.203125 -4.90625 1.828125 -4.90625 C 2.328125 -4.90625 2.710938 -4.789062 2.984375 -4.5625 C 3.265625 -4.34375 3.40625 -4.023438 3.40625 -3.609375 C 3.40625 -3.304688 3.320312 -3.0625 3.15625 -2.875 C 3.0625 -2.757812 2.929688 -2.664062 2.765625 -2.59375 C 3.023438 -2.519531 3.226562 -2.382812 3.375 -2.1875 C 3.519531 -1.988281 3.59375 -1.742188 3.59375 -1.453125 C 3.59375 -0.984375 3.4375 -0.597656 3.125 -0.296875 C 2.820312 -0.00390625 2.382812 0.140625 1.8125 0.140625 Z M 1.8125 0.140625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='f8b137c6-b785-4887-819f-6509856f1052' overflow='visible'>\n<path d='M 3.65625 -4.8125 L 3.65625 -4.28125 C 3.5 -4.125 3.289062 -3.851562 3.03125 -3.46875 C 2.769531 -3.09375 2.539062 -2.6875 2.34375 -2.25 C 2.144531 -1.820312 1.992188 -1.4375 1.890625 -1.09375 C 1.828125 -0.863281 1.742188 -0.5 1.640625 0 L 0.953125 0 C 1.109375 -0.9375 1.453125 -1.867188 1.984375 -2.796875 C 2.296875 -3.335938 2.625 -3.800781 2.96875 -4.1875 L 0.25 -4.1875 L 0.25 -4.8125 Z M 3.65625 -4.8125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='f70967a2-45ae-42ca-9824-939a68c579b6' overflow='visible'>\n<path d='M 0.59375 -0.75 L 1.3125 -0.75 L 1.3125 0 L 0.59375 0 Z M 0.59375 -0.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='3fb7d8af-7976-4375-ab12-2f49a25b82a7' overflow='visible'>\n<path d='M 2.3125 -1.734375 L 2.3125 -3.953125 L 0.75 -1.734375 Z M 2.328125 0 L 2.328125 -1.203125 L 0.171875 -1.203125 L 0.171875 -1.796875 L 2.421875 -4.90625 L 2.9375 -4.90625 L 2.9375 -1.734375 L 3.65625 -1.734375 L 3.65625 -1.203125 L 2.9375 -1.203125 L 2.9375 0 Z M 2.328125 0 ' style='stroke:none;'/>\n</symbol>\n<symbol id='70c90257-a8ce-476e-8592-83a8154478a6' overflow='visible'>\n<path d='M 0.859375 -1.25 C 0.898438 -0.894531 1.066406 -0.648438 1.359375 -0.515625 C 1.503906 -0.453125 1.671875 -0.421875 1.859375 -0.421875 C 2.222656 -0.421875 2.492188 -0.535156 2.671875 -0.765625 C 2.847656 -0.992188 2.9375 -1.25 2.9375 -1.53125 C 2.9375 -1.875 2.832031 -2.140625 2.625 -2.328125 C 2.414062 -2.515625 2.164062 -2.609375 1.875 -2.609375 C 1.65625 -2.609375 1.46875 -2.566406 1.3125 -2.484375 C 1.164062 -2.398438 1.039062 -2.285156 0.9375 -2.140625 L 0.40625 -2.171875 L 0.78125 -4.8125 L 3.3125 -4.8125 L 3.3125 -4.21875 L 1.234375 -4.21875 L 1.03125 -2.859375 C 1.144531 -2.941406 1.253906 -3.003906 1.359375 -3.046875 C 1.535156 -3.128906 1.742188 -3.171875 1.984375 -3.171875 C 2.429688 -3.171875 2.8125 -3.023438 3.125 -2.734375 C 3.4375 -2.441406 3.59375 -2.070312 3.59375 -1.625 C 3.59375 -1.164062 3.445312 -0.757812 3.15625 -0.40625 C 2.875 -0.0507812 2.421875 0.125 1.796875 0.125 C 1.398438 0.125 1.046875 0.015625 0.734375 -0.203125 C 0.429688 -0.429688 0.257812 -0.78125 0.21875 -1.25 Z M 0.859375 -1.25 ' style='stroke:none;'/>\n</symbol>\n<symbol id='dffc1239-9538-4cf6-8a9d-20cfdf089427' overflow='visible'>\n<path d='M 2.046875 -4.921875 C 2.597656 -4.921875 2.976562 -4.773438 3.1875 -4.484375 C 3.40625 -4.203125 3.515625 -3.910156 3.515625 -3.609375 L 2.90625 -3.609375 C 2.863281 -3.804688 2.804688 -3.957031 2.734375 -4.0625 C 2.585938 -4.269531 2.363281 -4.375 2.0625 -4.375 C 1.71875 -4.375 1.441406 -4.210938 1.234375 -3.890625 C 1.035156 -3.578125 0.925781 -3.125 0.90625 -2.53125 C 1.039062 -2.738281 1.21875 -2.894531 1.4375 -3 C 1.625 -3.09375 1.84375 -3.140625 2.09375 -3.140625 C 2.5 -3.140625 2.851562 -3.003906 3.15625 -2.734375 C 3.46875 -2.472656 3.625 -2.082031 3.625 -1.5625 C 3.625 -1.125 3.476562 -0.734375 3.1875 -0.390625 C 2.894531 -0.046875 2.484375 0.125 1.953125 0.125 C 1.492188 0.125 1.097656 -0.046875 0.765625 -0.390625 C 0.429688 -0.742188 0.265625 -1.332031 0.265625 -2.15625 C 0.265625 -2.757812 0.335938 -3.273438 0.484375 -3.703125 C 0.765625 -4.515625 1.285156 -4.921875 2.046875 -4.921875 Z M 2 -0.421875 C 2.320312 -0.421875 2.5625 -0.53125 2.71875 -0.75 C 2.882812 -0.96875 2.96875 -1.222656 2.96875 -1.515625 C 2.96875 -1.765625 2.894531 -2.003906 2.75 -2.234375 C 2.613281 -2.460938 2.351562 -2.578125 1.96875 -2.578125 C 1.707031 -2.578125 1.476562 -2.488281 1.28125 -2.3125 C 1.082031 -2.132812 0.984375 -1.867188 0.984375 -1.515625 C 0.984375 -1.203125 1.070312 -0.941406 1.25 -0.734375 C 1.425781 -0.523438 1.675781 -0.421875 2 -0.421875 Z M 2 -0.421875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='b20091d9-4f1f-47e9-beda-eef096c91571' overflow='visible'>\n<path d='M 1.90625 -2.84375 C 2.175781 -2.84375 2.382812 -2.914062 2.53125 -3.0625 C 2.6875 -3.21875 2.765625 -3.398438 2.765625 -3.609375 C 2.765625 -3.796875 2.691406 -3.960938 2.546875 -4.109375 C 2.398438 -4.265625 2.179688 -4.34375 1.890625 -4.34375 C 1.585938 -4.34375 1.367188 -4.265625 1.234375 -4.109375 C 1.109375 -3.960938 1.046875 -3.785156 1.046875 -3.578125 C 1.046875 -3.347656 1.128906 -3.164062 1.296875 -3.03125 C 1.472656 -2.90625 1.675781 -2.84375 1.90625 -2.84375 Z M 1.9375 -0.421875 C 2.226562 -0.421875 2.46875 -0.492188 2.65625 -0.640625 C 2.84375 -0.796875 2.9375 -1.03125 2.9375 -1.34375 C 2.9375 -1.65625 2.835938 -1.894531 2.640625 -2.0625 C 2.453125 -2.226562 2.203125 -2.3125 1.890625 -2.3125 C 1.597656 -2.3125 1.359375 -2.226562 1.171875 -2.0625 C 0.992188 -1.894531 0.90625 -1.660156 0.90625 -1.359375 C 0.90625 -1.109375 0.988281 -0.890625 1.15625 -0.703125 C 1.320312 -0.515625 1.582031 -0.421875 1.9375 -0.421875 Z M 1.0625 -2.609375 C 0.894531 -2.679688 0.765625 -2.765625 0.671875 -2.859375 C 0.484375 -3.046875 0.390625 -3.285156 0.390625 -3.578125 C 0.390625 -3.941406 0.519531 -4.253906 0.78125 -4.515625 C 1.050781 -4.773438 1.429688 -4.90625 1.921875 -4.90625 C 2.390625 -4.90625 2.753906 -4.78125 3.015625 -4.53125 C 3.285156 -4.289062 3.421875 -4.003906 3.421875 -3.671875 C 3.421875 -3.367188 3.34375 -3.117188 3.1875 -2.921875 C 3.101562 -2.816406 2.96875 -2.710938 2.78125 -2.609375 C 2.988281 -2.515625 3.148438 -2.410156 3.265625 -2.296875 C 3.484375 -2.066406 3.59375 -1.769531 3.59375 -1.40625 C 3.59375 -0.976562 3.445312 -0.613281 3.15625 -0.3125 C 2.863281 -0.0078125 2.453125 0.140625 1.921875 0.140625 C 1.453125 0.140625 1.050781 0.0078125 0.71875 -0.25 C 0.382812 -0.507812 0.21875 -0.882812 0.21875 -1.375 C 0.21875 -1.664062 0.289062 -1.914062 0.4375 -2.125 C 0.582031 -2.332031 0.789062 -2.492188 1.0625 -2.609375 Z M 1.0625 -2.609375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='cf906b66-f6d4-423c-a3f3-c1352e8d811a' overflow='visible'>\n<path d='M 0.296875 -2.265625 L 2 -2.265625 L 2 -1.640625 L 0.296875 -1.640625 Z M 0.296875 -2.265625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='d78a0a30-b071-45b6-b218-0bf3bba15698' overflow='visible'>\n<path d='M 0.671875 -3.46875 L 0.671875 -3.9375 C 1.117188 -3.976562 1.425781 -4.050781 1.59375 -4.15625 C 1.769531 -4.257812 1.90625 -4.5 2 -4.875 L 2.484375 -4.875 L 2.484375 0 L 1.828125 0 L 1.828125 -3.46875 Z M 0.671875 -3.46875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='708dbc1b-0990-4bb8-b313-630b3685a7b1' overflow='visible'>\n<path d='M 0.21875 0 C 0.238281 -0.414062 0.320312 -0.78125 0.46875 -1.09375 C 0.625 -1.40625 0.925781 -1.691406 1.375 -1.953125 L 2.03125 -2.328125 C 2.320312 -2.503906 2.523438 -2.648438 2.640625 -2.765625 C 2.828125 -2.953125 2.921875 -3.171875 2.921875 -3.421875 C 2.921875 -3.703125 2.832031 -3.925781 2.65625 -4.09375 C 2.488281 -4.257812 2.265625 -4.34375 1.984375 -4.34375 C 1.554688 -4.34375 1.265625 -4.1875 1.109375 -3.875 C 1.023438 -3.695312 0.976562 -3.457031 0.96875 -3.15625 L 0.34375 -3.15625 C 0.34375 -3.582031 0.421875 -3.929688 0.578125 -4.203125 C 0.835938 -4.671875 1.304688 -4.90625 1.984375 -4.90625 C 2.546875 -4.90625 2.957031 -4.753906 3.21875 -4.453125 C 3.476562 -4.148438 3.609375 -3.816406 3.609375 -3.453125 C 3.609375 -3.054688 3.46875 -2.71875 3.1875 -2.4375 C 3.03125 -2.28125 2.742188 -2.085938 2.328125 -1.859375 L 1.859375 -1.59375 C 1.640625 -1.46875 1.46875 -1.347656 1.34375 -1.234375 C 1.113281 -1.035156 0.96875 -0.816406 0.90625 -0.578125 L 3.578125 -0.578125 L 3.578125 0 Z M 0.21875 0 ' style='stroke:none;'/>\n</symbol>\n<symbol id='6314508d-bdd6-4313-a7b2-3fb15b03668b' overflow='visible'>\n<path d='M 1.890625 -4.890625 C 2.523438 -4.890625 2.984375 -4.628906 3.265625 -4.109375 C 3.484375 -3.703125 3.59375 -3.148438 3.59375 -2.453125 C 3.59375 -1.785156 3.492188 -1.234375 3.296875 -0.796875 C 3.015625 -0.171875 2.546875 0.140625 1.890625 0.140625 C 1.304688 0.140625 0.867188 -0.113281 0.578125 -0.625 C 0.335938 -1.050781 0.21875 -1.625 0.21875 -2.34375 C 0.21875 -2.90625 0.289062 -3.382812 0.4375 -3.78125 C 0.707031 -4.519531 1.191406 -4.890625 1.890625 -4.890625 Z M 1.890625 -0.421875 C 2.203125 -0.421875 2.453125 -0.5625 2.640625 -0.84375 C 2.835938 -1.132812 2.9375 -1.664062 2.9375 -2.4375 C 2.9375 -2.988281 2.863281 -3.441406 2.71875 -3.796875 C 2.582031 -4.160156 2.320312 -4.34375 1.9375 -4.34375 C 1.570312 -4.34375 1.304688 -4.171875 1.140625 -3.828125 C 0.972656 -3.484375 0.890625 -2.984375 0.890625 -2.328125 C 0.890625 -1.828125 0.941406 -1.425781 1.046875 -1.125 C 1.210938 -0.65625 1.492188 -0.421875 1.890625 -0.421875 Z M 1.890625 -0.421875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='ca82d3de-5a3f-428f-872d-b2e9cd820bb4' overflow='visible'>\n<path d='M 0.515625 -5.015625 L 1.484375 -5.015625 L 2.9375 -0.78125 L 4.359375 -5.015625 L 5.328125 -5.015625 L 5.328125 0 L 4.6875 0 L 4.6875 -2.96875 C 4.6875 -3.070312 4.6875 -3.238281 4.6875 -3.46875 C 4.695312 -3.707031 4.703125 -3.960938 4.703125 -4.234375 L 3.265625 0 L 2.59375 0 L 1.15625 -4.234375 L 1.15625 -4.078125 C 1.15625 -3.953125 1.15625 -3.765625 1.15625 -3.515625 C 1.164062 -3.265625 1.171875 -3.082031 1.171875 -2.96875 L 1.171875 0 L 0.515625 0 Z M 0.515625 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='99dd145f-47da-4d44-a2e4-f5ff965b4344' overflow='visible'>\n<path d='M 1.90625 -0.390625 C 2.3125 -0.390625 2.585938 -0.546875 2.734375 -0.859375 C 2.890625 -1.171875 2.96875 -1.515625 2.96875 -1.890625 C 2.96875 -2.234375 2.914062 -2.507812 2.8125 -2.71875 C 2.632812 -3.0625 2.332031 -3.234375 1.90625 -3.234375 C 1.53125 -3.234375 1.253906 -3.085938 1.078125 -2.796875 C 0.910156 -2.515625 0.828125 -2.171875 0.828125 -1.765625 C 0.828125 -1.367188 0.910156 -1.039062 1.078125 -0.78125 C 1.253906 -0.519531 1.53125 -0.390625 1.90625 -0.390625 Z M 1.921875 -3.765625 C 2.398438 -3.765625 2.800781 -3.609375 3.125 -3.296875 C 3.445312 -2.984375 3.609375 -2.519531 3.609375 -1.90625 C 3.609375 -1.3125 3.460938 -0.820312 3.171875 -0.4375 C 2.890625 -0.0507812 2.445312 0.140625 1.84375 0.140625 C 1.34375 0.140625 0.941406 -0.03125 0.640625 -0.375 C 0.347656 -0.71875 0.203125 -1.175781 0.203125 -1.75 C 0.203125 -2.363281 0.359375 -2.851562 0.671875 -3.21875 C 0.984375 -3.582031 1.398438 -3.765625 1.921875 -3.765625 Z M 1.90625 -3.75 Z M 1.90625 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='18b51442-86c0-4de3-ba7a-165b19721dc5' overflow='visible'>\n<path d='M 1.0625 -3.65625 L 1.0625 -1.234375 C 1.0625 -1.046875 1.09375 -0.890625 1.15625 -0.765625 C 1.257812 -0.546875 1.460938 -0.4375 1.765625 -0.4375 C 2.191406 -0.4375 2.484375 -0.628906 2.640625 -1.015625 C 2.722656 -1.222656 2.765625 -1.503906 2.765625 -1.859375 L 2.765625 -3.65625 L 3.390625 -3.65625 L 3.390625 0 L 2.796875 0 L 2.8125 -0.546875 C 2.726562 -0.398438 2.628906 -0.28125 2.515625 -0.1875 C 2.273438 0 1.988281 0.09375 1.65625 0.09375 C 1.132812 0.09375 0.78125 -0.078125 0.59375 -0.421875 C 0.488281 -0.609375 0.4375 -0.859375 0.4375 -1.171875 L 0.4375 -3.65625 Z M 1.90625 -3.75 Z M 1.90625 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='04d0175d-fa1a-4a34-bfa1-3845ccb013e0' overflow='visible'>\n<path d='M 0.453125 -3.65625 L 1.03125 -3.65625 L 1.03125 -3.140625 C 1.207031 -3.359375 1.390625 -3.515625 1.578125 -3.609375 C 1.773438 -3.703125 1.992188 -3.75 2.234375 -3.75 C 2.753906 -3.75 3.101562 -3.566406 3.28125 -3.203125 C 3.382812 -3.003906 3.4375 -2.71875 3.4375 -2.34375 L 3.4375 0 L 2.8125 0 L 2.8125 -2.3125 C 2.8125 -2.53125 2.773438 -2.707031 2.703125 -2.84375 C 2.597656 -3.070312 2.398438 -3.1875 2.109375 -3.1875 C 1.960938 -3.1875 1.84375 -3.171875 1.75 -3.140625 C 1.582031 -3.085938 1.4375 -2.988281 1.3125 -2.84375 C 1.207031 -2.71875 1.140625 -2.59375 1.109375 -2.46875 C 1.078125 -2.34375 1.0625 -2.160156 1.0625 -1.921875 L 1.0625 0 L 0.453125 0 Z M 1.890625 -3.75 Z M 1.890625 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='79a882c2-7f7c-43f7-a347-325d14ffacb6' overflow='visible'>\n<path d='M 0.578125 -4.6875 L 1.203125 -4.6875 L 1.203125 -3.65625 L 1.78125 -3.65625 L 1.78125 -3.15625 L 1.203125 -3.15625 L 1.203125 -0.765625 C 1.203125 -0.640625 1.242188 -0.554688 1.328125 -0.515625 C 1.367188 -0.484375 1.445312 -0.46875 1.5625 -0.46875 C 1.59375 -0.46875 1.625 -0.46875 1.65625 -0.46875 C 1.695312 -0.476562 1.738281 -0.484375 1.78125 -0.484375 L 1.78125 0 C 1.707031 0.0195312 1.632812 0.0351562 1.5625 0.046875 C 1.488281 0.0546875 1.40625 0.0625 1.3125 0.0625 C 1.019531 0.0625 0.820312 -0.0078125 0.71875 -0.15625 C 0.625 -0.3125 0.578125 -0.503906 0.578125 -0.734375 L 0.578125 -3.15625 L 0.078125 -3.15625 L 0.078125 -3.65625 L 0.578125 -3.65625 Z M 0.578125 -4.6875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='9d92f1f0-947c-4180-ba6e-3beb1c7c45fc' overflow='visible'>\n<path d='M 0.921875 -0.96875 C 0.921875 -0.789062 0.984375 -0.648438 1.109375 -0.546875 C 1.242188 -0.453125 1.398438 -0.40625 1.578125 -0.40625 C 1.796875 -0.40625 2.003906 -0.453125 2.203125 -0.546875 C 2.546875 -0.710938 2.71875 -0.988281 2.71875 -1.375 L 2.71875 -1.859375 C 2.644531 -1.816406 2.546875 -1.78125 2.421875 -1.75 C 2.304688 -1.71875 2.191406 -1.691406 2.078125 -1.671875 L 1.703125 -1.625 C 1.484375 -1.59375 1.316406 -1.546875 1.203125 -1.484375 C 1.015625 -1.378906 0.921875 -1.207031 0.921875 -0.96875 Z M 2.40625 -2.21875 C 2.550781 -2.238281 2.648438 -2.296875 2.703125 -2.390625 C 2.722656 -2.453125 2.734375 -2.53125 2.734375 -2.625 C 2.734375 -2.84375 2.660156 -3 2.515625 -3.09375 C 2.367188 -3.1875 2.15625 -3.234375 1.875 -3.234375 C 1.550781 -3.234375 1.320312 -3.144531 1.1875 -2.96875 C 1.113281 -2.875 1.066406 -2.734375 1.046875 -2.546875 L 0.46875 -2.546875 C 0.476562 -2.992188 0.625 -3.304688 0.90625 -3.484375 C 1.1875 -3.660156 1.515625 -3.75 1.890625 -3.75 C 2.316406 -3.75 2.664062 -3.664062 2.9375 -3.5 C 3.207031 -3.332031 3.34375 -3.078125 3.34375 -2.734375 L 3.34375 -0.625 C 3.34375 -0.5625 3.351562 -0.507812 3.375 -0.46875 C 3.40625 -0.4375 3.460938 -0.421875 3.546875 -0.421875 C 3.566406 -0.421875 3.59375 -0.421875 3.625 -0.421875 C 3.664062 -0.421875 3.707031 -0.425781 3.75 -0.4375 L 3.75 0.015625 C 3.644531 0.046875 3.566406 0.0625 3.515625 0.0625 C 3.472656 0.0703125 3.40625 0.078125 3.3125 0.078125 C 3.101562 0.078125 2.953125 0.00390625 2.859375 -0.140625 C 2.804688 -0.222656 2.769531 -0.335938 2.75 -0.484375 C 2.625 -0.316406 2.441406 -0.171875 2.203125 -0.046875 C 1.972656 0.0664062 1.71875 0.125 1.4375 0.125 C 1.09375 0.125 0.8125 0.0195312 0.59375 -0.1875 C 0.382812 -0.394531 0.28125 -0.648438 0.28125 -0.953125 C 0.28125 -1.296875 0.382812 -1.5625 0.59375 -1.75 C 0.8125 -1.9375 1.09375 -2.050781 1.4375 -2.09375 Z M 1.90625 -3.75 Z M 1.90625 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='dacc87f4-0b2e-4397-a29d-001caec4525d' overflow='visible'>\n<path d='M 0.453125 -3.640625 L 1.078125 -3.640625 L 1.078125 0 L 0.453125 0 Z M 0.453125 -5.015625 L 1.078125 -5.015625 L 1.078125 -4.328125 L 0.453125 -4.328125 Z M 0.453125 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='f5a4abe7-8816-4a9a-a518-b3cab0acc7c6' overflow='visible'>\n<path d='' style='stroke:none;'/>\n</symbol>\n<symbol id='5b79702e-33ba-417e-ada5-ff6b861c39b4' overflow='visible'>\n<path d='M 0.9375 -5.015625 L 2.375 -0.75 L 3.796875 -5.015625 L 4.5625 -5.015625 L 2.734375 0 L 2 0 L 0.171875 -5.015625 Z M 0.9375 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='6c39d25e-557a-4381-bc22-2c347397ded4' overflow='visible'>\n<path d='M 1.96875 -3.75 C 2.226562 -3.75 2.476562 -3.6875 2.71875 -3.5625 C 2.96875 -3.4375 3.15625 -3.28125 3.28125 -3.09375 C 3.40625 -2.90625 3.488281 -2.6875 3.53125 -2.4375 C 3.570312 -2.269531 3.59375 -2.003906 3.59375 -1.640625 L 0.90625 -1.640625 C 0.914062 -1.265625 1.003906 -0.960938 1.171875 -0.734375 C 1.335938 -0.515625 1.59375 -0.40625 1.9375 -0.40625 C 2.25 -0.40625 2.503906 -0.507812 2.703125 -0.71875 C 2.804688 -0.84375 2.882812 -0.988281 2.9375 -1.15625 L 3.53125 -1.15625 C 3.519531 -1.019531 3.46875 -0.867188 3.375 -0.703125 C 3.289062 -0.535156 3.191406 -0.398438 3.078125 -0.296875 C 2.890625 -0.109375 2.65625 0.015625 2.375 0.078125 C 2.226562 0.109375 2.0625 0.125 1.875 0.125 C 1.425781 0.125 1.039062 -0.0351562 0.71875 -0.359375 C 0.40625 -0.691406 0.25 -1.160156 0.25 -1.765625 C 0.25 -2.347656 0.40625 -2.820312 0.71875 -3.1875 C 1.039062 -3.5625 1.457031 -3.75 1.96875 -3.75 Z M 2.953125 -2.125 C 2.929688 -2.382812 2.875 -2.597656 2.78125 -2.765625 C 2.613281 -3.054688 2.332031 -3.203125 1.9375 -3.203125 C 1.65625 -3.203125 1.414062 -3.097656 1.21875 -2.890625 C 1.03125 -2.691406 0.929688 -2.4375 0.921875 -2.125 Z M 1.921875 -3.75 Z M 1.921875 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='0a2b1e60-e168-4049-a1f8-40d241552549' overflow='visible'>\n<path d='M 0.734375 -3.65625 L 1.4375 -0.78125 L 2.15625 -3.65625 L 2.84375 -3.65625 L 3.5625 -0.796875 L 4.3125 -3.65625 L 4.921875 -3.65625 L 3.859375 0 L 3.21875 0 L 2.484375 -2.828125 L 1.75 0 L 1.125 0 L 0.0625 -3.65625 Z M 0.734375 -3.65625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='35bc8884-c9bc-48af-8391-536167e977d9' overflow='visible'>\n<path d='M 0.59375 -5.015625 L 2.859375 -5.015625 C 3.304688 -5.015625 3.664062 -4.890625 3.9375 -4.640625 C 4.207031 -4.390625 4.34375 -4.035156 4.34375 -3.578125 C 4.34375 -3.191406 4.21875 -2.851562 3.96875 -2.5625 C 3.726562 -2.269531 3.359375 -2.125 2.859375 -2.125 L 1.28125 -2.125 L 1.28125 0 L 0.59375 0 Z M 3.65625 -3.578125 C 3.65625 -3.941406 3.519531 -4.191406 3.25 -4.328125 C 3.101562 -4.398438 2.894531 -4.4375 2.625 -4.4375 L 1.28125 -4.4375 L 1.28125 -2.703125 L 2.625 -2.703125 C 2.9375 -2.703125 3.1875 -2.765625 3.375 -2.890625 C 3.5625 -3.023438 3.65625 -3.253906 3.65625 -3.578125 Z M 3.65625 -3.578125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='eee407df-aae7-4267-a2d8-c6bf27bbb96b' overflow='visible'>\n<path d='M 0.46875 -5.015625 L 1.078125 -5.015625 L 1.078125 0 L 0.46875 0 Z M 0.46875 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='2b371840-a87a-42e6-a793-f8a6f2e91e42' overflow='visible'>\n<path d='M 3.109375 -2.0625 L 2.34375 -4.28125 L 1.53125 -2.0625 Z M 2 -5.015625 L 2.765625 -5.015625 L 4.578125 0 L 3.84375 0 L 3.328125 -1.5 L 1.34375 -1.5 L 0.796875 0 L 0.109375 0 Z M 2.34375 -5.015625 Z M 2.34375 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='cadc9d5b-102b-4f15-b279-1bccc87f045c' overflow='visible'>\n<path d='M 2.875 -2.71875 C 3.1875 -2.71875 3.4375 -2.78125 3.625 -2.90625 C 3.8125 -3.039062 3.90625 -3.273438 3.90625 -3.609375 C 3.90625 -3.960938 3.773438 -4.203125 3.515625 -4.328125 C 3.378906 -4.398438 3.191406 -4.4375 2.953125 -4.4375 L 1.296875 -4.4375 L 1.296875 -2.71875 Z M 0.609375 -5.015625 L 2.9375 -5.015625 C 3.320312 -5.015625 3.640625 -4.960938 3.890625 -4.859375 C 4.359375 -4.640625 4.59375 -4.242188 4.59375 -3.671875 C 4.59375 -3.367188 4.53125 -3.117188 4.40625 -2.921875 C 4.28125 -2.734375 4.109375 -2.582031 3.890625 -2.46875 C 4.085938 -2.382812 4.234375 -2.273438 4.328125 -2.140625 C 4.429688 -2.015625 4.488281 -1.804688 4.5 -1.515625 L 4.515625 -0.84375 C 4.523438 -0.65625 4.539062 -0.515625 4.5625 -0.421875 C 4.601562 -0.265625 4.675781 -0.160156 4.78125 -0.109375 L 4.78125 0 L 3.953125 0 C 3.929688 -0.0390625 3.910156 -0.09375 3.890625 -0.15625 C 3.878906 -0.226562 3.867188 -0.363281 3.859375 -0.5625 L 3.828125 -1.40625 C 3.804688 -1.726562 3.679688 -1.945312 3.453125 -2.0625 C 3.328125 -2.125 3.125 -2.15625 2.84375 -2.15625 L 1.296875 -2.15625 L 1.296875 0 L 0.609375 0 Z M 0.609375 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='253771c0-e953-4187-9dbb-3f4b43811093' overflow='visible'>\n<path d='M 0.84375 -1.78125 C 0.84375 -1.394531 0.925781 -1.066406 1.09375 -0.796875 C 1.257812 -0.535156 1.523438 -0.40625 1.890625 -0.40625 C 2.171875 -0.40625 2.40625 -0.523438 2.59375 -0.765625 C 2.78125 -1.015625 2.875 -1.367188 2.875 -1.828125 C 2.875 -2.285156 2.773438 -2.625 2.578125 -2.84375 C 2.390625 -3.070312 2.160156 -3.1875 1.890625 -3.1875 C 1.578125 -3.1875 1.320312 -3.066406 1.125 -2.828125 C 0.9375 -2.597656 0.84375 -2.25 0.84375 -1.78125 Z M 1.765625 -3.71875 C 2.046875 -3.71875 2.28125 -3.660156 2.46875 -3.546875 C 2.582031 -3.484375 2.707031 -3.363281 2.84375 -3.1875 L 2.84375 -5.03125 L 3.4375 -5.03125 L 3.4375 0 L 2.875 0 L 2.875 -0.515625 C 2.738281 -0.285156 2.570312 -0.117188 2.375 -0.015625 C 2.175781 0.078125 1.953125 0.125 1.703125 0.125 C 1.296875 0.125 0.941406 -0.046875 0.640625 -0.390625 C 0.335938 -0.734375 0.1875 -1.1875 0.1875 -1.75 C 0.1875 -2.28125 0.320312 -2.738281 0.59375 -3.125 C 0.875 -3.519531 1.265625 -3.71875 1.765625 -3.71875 Z M 1.765625 -3.71875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='31ca224b-da2f-4e8c-890a-3a5521249051' overflow='visible'>\n<path d='M 2.65625 -5.15625 C 3.289062 -5.15625 3.78125 -4.988281 4.125 -4.65625 C 4.476562 -4.320312 4.675781 -3.941406 4.71875 -3.515625 L 4.046875 -3.515625 C 3.972656 -3.835938 3.820312 -4.09375 3.59375 -4.28125 C 3.375 -4.46875 3.0625 -4.5625 2.65625 -4.5625 C 2.164062 -4.5625 1.765625 -4.390625 1.453125 -4.046875 C 1.148438 -3.703125 1 -3.171875 1 -2.453125 C 1 -1.859375 1.132812 -1.378906 1.40625 -1.015625 C 1.6875 -0.648438 2.097656 -0.46875 2.640625 -0.46875 C 3.140625 -0.46875 3.519531 -0.660156 3.78125 -1.046875 C 3.925781 -1.242188 4.03125 -1.507812 4.09375 -1.84375 L 4.765625 -1.84375 C 4.703125 -1.3125 4.503906 -0.867188 4.171875 -0.515625 C 3.773438 -0.0859375 3.238281 0.125 2.5625 0.125 C 1.976562 0.125 1.488281 -0.0507812 1.09375 -0.40625 C 0.570312 -0.863281 0.3125 -1.582031 0.3125 -2.5625 C 0.3125 -3.300781 0.503906 -3.910156 0.890625 -4.390625 C 1.316406 -4.898438 1.90625 -5.15625 2.65625 -5.15625 Z M 2.515625 -5.15625 Z M 2.515625 -5.15625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='f0dea103-8c6a-4d6b-b4b5-2b8f0d805944' overflow='visible'>\n<path d='M 2.734375 -3.65625 L 3.421875 -3.65625 C 3.328125 -3.425781 3.132812 -2.890625 2.84375 -2.046875 C 2.613281 -1.421875 2.425781 -0.910156 2.28125 -0.515625 C 1.925781 0.421875 1.675781 0.988281 1.53125 1.1875 C 1.382812 1.394531 1.132812 1.5 0.78125 1.5 C 0.6875 1.5 0.613281 1.492188 0.5625 1.484375 C 0.519531 1.484375 0.46875 1.472656 0.40625 1.453125 L 0.40625 0.890625 C 0.507812 0.921875 0.582031 0.9375 0.625 0.9375 C 0.675781 0.945312 0.722656 0.953125 0.765625 0.953125 C 0.878906 0.953125 0.960938 0.929688 1.015625 0.890625 C 1.066406 0.859375 1.113281 0.816406 1.15625 0.765625 C 1.164062 0.742188 1.203125 0.648438 1.265625 0.484375 C 1.335938 0.316406 1.390625 0.191406 1.421875 0.109375 L 0.078125 -3.65625 L 0.765625 -3.65625 L 1.75 -0.6875 Z M 1.75 -3.75 Z M 1.75 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='2da9ee19-7a59-490e-a444-6b33033d40f1' overflow='visible'>\n<path d='M 0.984375 -1.625 C 0.992188 -1.332031 1.054688 -1.097656 1.171875 -0.921875 C 1.398438 -0.585938 1.800781 -0.421875 2.375 -0.421875 C 2.625 -0.421875 2.851562 -0.457031 3.0625 -0.53125 C 3.46875 -0.675781 3.671875 -0.929688 3.671875 -1.296875 C 3.671875 -1.566406 3.585938 -1.757812 3.421875 -1.875 C 3.242188 -1.988281 2.972656 -2.085938 2.609375 -2.171875 L 1.921875 -2.328125 C 1.484375 -2.429688 1.171875 -2.539062 0.984375 -2.65625 C 0.660156 -2.875 0.5 -3.191406 0.5 -3.609375 C 0.5 -4.054688 0.65625 -4.421875 0.96875 -4.703125 C 1.28125 -4.992188 1.722656 -5.140625 2.296875 -5.140625 C 2.828125 -5.140625 3.273438 -5.015625 3.640625 -4.765625 C 4.003906 -4.515625 4.1875 -4.109375 4.1875 -3.546875 L 3.5625 -3.546875 C 3.519531 -3.816406 3.445312 -4.023438 3.34375 -4.171875 C 3.132812 -4.429688 2.78125 -4.5625 2.28125 -4.5625 C 1.875 -4.5625 1.582031 -4.472656 1.40625 -4.296875 C 1.238281 -4.128906 1.15625 -3.9375 1.15625 -3.71875 C 1.15625 -3.46875 1.253906 -3.285156 1.453125 -3.171875 C 1.585938 -3.097656 1.894531 -3.007812 2.375 -2.90625 L 3.078125 -2.75 C 3.410156 -2.664062 3.671875 -2.554688 3.859375 -2.421875 C 4.179688 -2.191406 4.34375 -1.851562 4.34375 -1.40625 C 4.34375 -0.851562 4.140625 -0.457031 3.734375 -0.21875 C 3.328125 0.0195312 2.859375 0.140625 2.328125 0.140625 C 1.703125 0.140625 1.210938 -0.015625 0.859375 -0.328125 C 0.503906 -0.648438 0.332031 -1.082031 0.34375 -1.625 Z M 2.359375 -5.15625 Z M 2.359375 -5.15625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='4f4d02e8-d8e2-4487-acbe-c632124d6095' overflow='visible'>\n<path d='M 0.59375 -5.015625 L 4.078125 -5.015625 L 4.078125 -4.40625 L 1.28125 -4.40625 L 1.28125 -2.875 L 3.75 -2.875 L 3.75 -2.28125 L 1.28125 -2.28125 L 1.28125 0 L 0.59375 0 Z M 0.59375 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='a7916dd8-f007-437b-9c1f-63d175f1a331' overflow='visible'>\n<path d='M 0.46875 -3.65625 L 1.046875 -3.65625 L 1.046875 -3.03125 C 1.097656 -3.15625 1.21875 -3.304688 1.40625 -3.484375 C 1.59375 -3.660156 1.804688 -3.75 2.046875 -3.75 C 2.054688 -3.75 2.078125 -3.742188 2.109375 -3.734375 C 2.140625 -3.734375 2.1875 -3.726562 2.25 -3.71875 L 2.25 -3.078125 C 2.207031 -3.085938 2.171875 -3.09375 2.140625 -3.09375 C 2.117188 -3.09375 2.085938 -3.09375 2.046875 -3.09375 C 1.734375 -3.09375 1.492188 -2.992188 1.328125 -2.796875 C 1.160156 -2.597656 1.078125 -2.367188 1.078125 -2.109375 L 1.078125 0 L 0.46875 0 Z M 0.46875 -3.65625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='c85ed8e5-20be-43d5-bc2a-cf06e238e45b' overflow='visible'>\n<path d='M 1.859375 -3.765625 C 2.273438 -3.765625 2.609375 -3.664062 2.859375 -3.46875 C 3.117188 -3.269531 3.28125 -2.925781 3.34375 -2.4375 L 2.734375 -2.4375 C 2.703125 -2.65625 2.617188 -2.835938 2.484375 -2.984375 C 2.359375 -3.140625 2.148438 -3.21875 1.859375 -3.21875 C 1.472656 -3.21875 1.191406 -3.023438 1.015625 -2.640625 C 0.910156 -2.390625 0.859375 -2.082031 0.859375 -1.71875 C 0.859375 -1.351562 0.9375 -1.046875 1.09375 -0.796875 C 1.25 -0.546875 1.492188 -0.421875 1.828125 -0.421875 C 2.078125 -0.421875 2.273438 -0.5 2.421875 -0.65625 C 2.578125 -0.8125 2.679688 -1.023438 2.734375 -1.296875 L 3.34375 -1.296875 C 3.269531 -0.804688 3.09375 -0.445312 2.8125 -0.21875 C 2.539062 0 2.195312 0.109375 1.78125 0.109375 C 1.300781 0.109375 0.914062 -0.0625 0.625 -0.40625 C 0.34375 -0.757812 0.203125 -1.203125 0.203125 -1.734375 C 0.203125 -2.367188 0.359375 -2.863281 0.671875 -3.21875 C 0.984375 -3.582031 1.378906 -3.765625 1.859375 -3.765625 Z M 1.765625 -3.75 Z M 1.765625 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='a97ebaf8-88ff-48be-9d64-b9ea4a0d5a58' overflow='visible'>\n<path d='M 0.8125 -1.15625 C 0.832031 -0.945312 0.882812 -0.785156 0.96875 -0.671875 C 1.125 -0.472656 1.394531 -0.375 1.78125 -0.375 C 2.007812 -0.375 2.207031 -0.421875 2.375 -0.515625 C 2.550781 -0.617188 2.640625 -0.773438 2.640625 -0.984375 C 2.640625 -1.140625 2.570312 -1.257812 2.4375 -1.34375 C 2.34375 -1.394531 2.164062 -1.453125 1.90625 -1.515625 L 1.421875 -1.640625 C 1.109375 -1.722656 0.875 -1.8125 0.71875 -1.90625 C 0.457031 -2.070312 0.328125 -2.300781 0.328125 -2.59375 C 0.328125 -2.9375 0.453125 -3.210938 0.703125 -3.421875 C 0.953125 -3.640625 1.285156 -3.75 1.703125 -3.75 C 2.253906 -3.75 2.65625 -3.585938 2.90625 -3.265625 C 3.050781 -3.066406 3.125 -2.847656 3.125 -2.609375 L 2.546875 -2.609375 C 2.535156 -2.742188 2.484375 -2.867188 2.390625 -2.984375 C 2.253906 -3.148438 2.007812 -3.234375 1.65625 -3.234375 C 1.425781 -3.234375 1.25 -3.1875 1.125 -3.09375 C 1.007812 -3.007812 0.953125 -2.894531 0.953125 -2.75 C 0.953125 -2.582031 1.03125 -2.453125 1.1875 -2.359375 C 1.28125 -2.304688 1.414062 -2.257812 1.59375 -2.21875 L 2 -2.109375 C 2.4375 -2.003906 2.734375 -1.898438 2.890625 -1.796875 C 3.128906 -1.640625 3.25 -1.394531 3.25 -1.0625 C 3.25 -0.738281 3.125 -0.457031 2.875 -0.21875 C 2.625 0.0195312 2.25 0.140625 1.75 0.140625 C 1.207031 0.140625 0.820312 0.015625 0.59375 -0.234375 C 0.363281 -0.484375 0.238281 -0.789062 0.21875 -1.15625 Z M 1.71875 -3.75 Z M 1.71875 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='a1ac6047-4ba6-4030-837e-40137af96e57' overflow='visible'>\n<path d='M 3 -1.5 C 3 -1.070312 2.9375 -0.742188 2.8125 -0.515625 C 2.582031 -0.078125 2.140625 0.140625 1.484375 0.140625 C 1.109375 0.140625 0.785156 0.0351562 0.515625 -0.171875 C 0.253906 -0.378906 0.125 -0.742188 0.125 -1.265625 L 0.125 -1.625 L 0.765625 -1.625 L 0.765625 -1.265625 C 0.765625 -0.984375 0.820312 -0.773438 0.9375 -0.640625 C 1.0625 -0.503906 1.253906 -0.4375 1.515625 -0.4375 C 1.867188 -0.4375 2.101562 -0.5625 2.21875 -0.8125 C 2.289062 -0.96875 2.328125 -1.253906 2.328125 -1.671875 L 2.328125 -5.015625 L 3 -5.015625 Z M 3 -1.5 ' style='stroke:none;'/>\n</symbol>\n<symbol id='856ba032-0879-46c9-8057-76dc8f809e4b' overflow='visible'>\n<path d='M 0.265625 0 L 0.265625 -5.734375 L 4.8125 -5.734375 L 4.8125 0 Z M 4.09375 -0.71875 L 4.09375 -5.015625 L 0.984375 -5.015625 L 0.984375 -0.71875 Z M 4.09375 -0.71875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='f6130023-a7bb-4980-a7bd-d30ff963354b' overflow='visible'>\n<path d='M 0.53125 -5.734375 L 1.234375 -5.734375 L 1.234375 0 L 0.53125 0 Z M 0.53125 -5.734375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='20ef76af-4dfb-4765-bddb-4471bba53ec7' overflow='visible'>\n<path d='M 2.171875 -0.453125 C 2.640625 -0.453125 2.957031 -0.628906 3.125 -0.984375 C 3.300781 -1.335938 3.390625 -1.726562 3.390625 -2.15625 C 3.390625 -2.550781 3.328125 -2.867188 3.203125 -3.109375 C 3.003906 -3.492188 2.664062 -3.6875 2.1875 -3.6875 C 1.757812 -3.6875 1.445312 -3.519531 1.25 -3.1875 C 1.050781 -2.863281 0.953125 -2.472656 0.953125 -2.015625 C 0.953125 -1.566406 1.050781 -1.191406 1.25 -0.890625 C 1.445312 -0.597656 1.753906 -0.453125 2.171875 -0.453125 Z M 2.203125 -4.3125 C 2.742188 -4.3125 3.195312 -4.128906 3.5625 -3.765625 C 3.9375 -3.410156 4.125 -2.882812 4.125 -2.1875 C 4.125 -1.507812 3.957031 -0.945312 3.625 -0.5 C 3.300781 -0.0625 2.796875 0.15625 2.109375 0.15625 C 1.535156 0.15625 1.078125 -0.0351562 0.734375 -0.421875 C 0.398438 -0.816406 0.234375 -1.34375 0.234375 -2 C 0.234375 -2.707031 0.410156 -3.269531 0.765625 -3.6875 C 1.117188 -4.101562 1.597656 -4.3125 2.203125 -4.3125 Z M 2.1875 -4.28125 Z M 2.1875 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='dfdeacb8-562a-4221-8a1f-e5cd5fdda78e' overflow='visible'>\n<path d='M 0.515625 -4.1875 L 1.1875 -4.1875 L 1.1875 -3.59375 C 1.382812 -3.832031 1.59375 -4.003906 1.8125 -4.109375 C 2.03125 -4.222656 2.273438 -4.28125 2.546875 -4.28125 C 3.140625 -4.28125 3.539062 -4.070312 3.75 -3.65625 C 3.863281 -3.425781 3.921875 -3.101562 3.921875 -2.6875 L 3.921875 0 L 3.21875 0 L 3.21875 -2.640625 C 3.21875 -2.890625 3.175781 -3.09375 3.09375 -3.25 C 2.96875 -3.507812 2.742188 -3.640625 2.421875 -3.640625 C 2.253906 -3.640625 2.117188 -3.625 2.015625 -3.59375 C 1.816406 -3.539062 1.644531 -3.425781 1.5 -3.25 C 1.382812 -3.113281 1.304688 -2.972656 1.265625 -2.828125 C 1.234375 -2.679688 1.21875 -2.46875 1.21875 -2.1875 L 1.21875 0 L 0.515625 0 Z M 2.171875 -4.28125 Z M 2.171875 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='8099ddb1-0426-4749-8f71-4a63587d238e' overflow='visible'>\n<path d='M 2 -4.265625 C 2.320312 -4.265625 2.609375 -4.179688 2.859375 -4.015625 C 2.984375 -3.921875 3.117188 -3.789062 3.265625 -3.625 L 3.265625 -4.140625 L 3.90625 -4.140625 L 3.90625 -0.34375 C 3.90625 0.1875 3.828125 0.609375 3.671875 0.921875 C 3.378906 1.484375 2.828125 1.765625 2.015625 1.765625 C 1.566406 1.765625 1.1875 1.664062 0.875 1.46875 C 0.570312 1.269531 0.40625 0.953125 0.375 0.515625 L 1.078125 0.515625 C 1.117188 0.703125 1.1875 0.847656 1.28125 0.953125 C 1.445312 1.109375 1.695312 1.1875 2.03125 1.1875 C 2.570312 1.1875 2.925781 1 3.09375 0.625 C 3.195312 0.394531 3.242188 -0.00390625 3.234375 -0.578125 C 3.085938 -0.367188 2.914062 -0.210938 2.71875 -0.109375 C 2.519531 -0.00390625 2.257812 0.046875 1.9375 0.046875 C 1.488281 0.046875 1.09375 -0.113281 0.75 -0.4375 C 0.40625 -0.757812 0.234375 -1.289062 0.234375 -2.03125 C 0.234375 -2.726562 0.40625 -3.273438 0.75 -3.671875 C 1.09375 -4.066406 1.507812 -4.265625 2 -4.265625 Z M 3.265625 -2.109375 C 3.265625 -2.628906 3.15625 -3.015625 2.9375 -3.265625 C 2.726562 -3.515625 2.457031 -3.640625 2.125 -3.640625 C 1.625 -3.640625 1.285156 -3.40625 1.109375 -2.9375 C 1.015625 -2.695312 0.96875 -2.375 0.96875 -1.96875 C 0.96875 -1.5 1.0625 -1.140625 1.25 -0.890625 C 1.4375 -0.648438 1.691406 -0.53125 2.015625 -0.53125 C 2.523438 -0.53125 2.882812 -0.753906 3.09375 -1.203125 C 3.207031 -1.460938 3.265625 -1.765625 3.265625 -2.109375 Z M 2.078125 -4.28125 Z M 2.078125 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='1ceeeb8c-17ed-4456-b333-d1362f8235bb' overflow='visible'>\n<path d='M 0.515625 -4.171875 L 1.234375 -4.171875 L 1.234375 0 L 0.515625 0 Z M 0.515625 -5.734375 L 1.234375 -5.734375 L 1.234375 -4.9375 L 0.515625 -4.9375 Z M 0.515625 -5.734375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='51e88b40-5464-452a-a315-e01461f4b1fc' overflow='visible'>\n<path d='M 0.65625 -5.359375 L 1.375 -5.359375 L 1.375 -4.1875 L 2.03125 -4.1875 L 2.03125 -3.609375 L 1.375 -3.609375 L 1.375 -0.875 C 1.375 -0.726562 1.421875 -0.632812 1.515625 -0.59375 C 1.566406 -0.5625 1.660156 -0.546875 1.796875 -0.546875 C 1.828125 -0.546875 1.859375 -0.546875 1.890625 -0.546875 C 1.929688 -0.546875 1.976562 -0.550781 2.03125 -0.5625 L 2.03125 0 C 1.957031 0.0195312 1.875 0.0351562 1.78125 0.046875 C 1.695312 0.0546875 1.601562 0.0625 1.5 0.0625 C 1.175781 0.0625 0.953125 -0.0195312 0.828125 -0.1875 C 0.710938 -0.351562 0.65625 -0.570312 0.65625 -0.84375 L 0.65625 -3.609375 L 0.09375 -3.609375 L 0.09375 -4.1875 L 0.65625 -4.1875 Z M 0.65625 -5.359375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='1380d38a-a602-4f6c-9e3f-7e037cc7cb71' overflow='visible'>\n<path d='M 1.21875 -4.1875 L 1.21875 -1.40625 C 1.21875 -1.195312 1.253906 -1.023438 1.328125 -0.890625 C 1.453125 -0.640625 1.679688 -0.515625 2.015625 -0.515625 C 2.503906 -0.515625 2.835938 -0.734375 3.015625 -1.171875 C 3.117188 -1.398438 3.171875 -1.71875 3.171875 -2.125 L 3.171875 -4.1875 L 3.875 -4.1875 L 3.875 0 L 3.203125 0 L 3.21875 -0.625 C 3.125 -0.457031 3.007812 -0.320312 2.875 -0.21875 C 2.601562 0 2.273438 0.109375 1.890625 0.109375 C 1.296875 0.109375 0.890625 -0.0859375 0.671875 -0.484375 C 0.554688 -0.703125 0.5 -0.988281 0.5 -1.34375 L 0.5 -4.1875 Z M 2.1875 -4.28125 Z M 2.1875 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='715395ce-8992-4b9e-aeed-8d905de6382e' overflow='visible'>\n<path d='M 0.96875 -2.046875 C 0.96875 -1.597656 1.0625 -1.222656 1.25 -0.921875 C 1.4375 -0.617188 1.738281 -0.46875 2.15625 -0.46875 C 2.488281 -0.46875 2.757812 -0.609375 2.96875 -0.890625 C 3.175781 -1.171875 3.28125 -1.570312 3.28125 -2.09375 C 3.28125 -2.625 3.171875 -3.015625 2.953125 -3.265625 C 2.734375 -3.515625 2.46875 -3.640625 2.15625 -3.640625 C 1.800781 -3.640625 1.515625 -3.503906 1.296875 -3.234375 C 1.078125 -2.960938 0.96875 -2.566406 0.96875 -2.046875 Z M 2.015625 -4.265625 C 2.335938 -4.265625 2.609375 -4.195312 2.828125 -4.0625 C 2.953125 -3.976562 3.09375 -3.835938 3.25 -3.640625 L 3.25 -5.765625 L 3.921875 -5.765625 L 3.921875 0 L 3.296875 0 L 3.296875 -0.578125 C 3.128906 -0.328125 2.929688 -0.140625 2.703125 -0.015625 C 2.484375 0.0976562 2.226562 0.15625 1.9375 0.15625 C 1.476562 0.15625 1.078125 -0.0390625 0.734375 -0.4375 C 0.390625 -0.832031 0.21875 -1.351562 0.21875 -2 C 0.21875 -2.613281 0.375 -3.144531 0.6875 -3.59375 C 1 -4.039062 1.441406 -4.265625 2.015625 -4.265625 Z M 2.015625 -4.265625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='62933a52-0d45-4414-8c01-ed7527dc0fda' overflow='visible'>\n<path d='M 2.265625 -4.28125 C 2.554688 -4.28125 2.84375 -4.207031 3.125 -4.0625 C 3.40625 -3.925781 3.617188 -3.75 3.765625 -3.53125 C 3.898438 -3.320312 3.992188 -3.078125 4.046875 -2.796875 C 4.085938 -2.597656 4.109375 -2.289062 4.109375 -1.875 L 1.03125 -1.875 C 1.050781 -1.445312 1.15625 -1.101562 1.34375 -0.84375 C 1.53125 -0.59375 1.816406 -0.46875 2.203125 -0.46875 C 2.578125 -0.46875 2.875 -0.585938 3.09375 -0.828125 C 3.21875 -0.960938 3.304688 -1.125 3.359375 -1.3125 L 4.046875 -1.3125 C 4.023438 -1.15625 3.960938 -0.984375 3.859375 -0.796875 C 3.753906 -0.609375 3.640625 -0.457031 3.515625 -0.34375 C 3.304688 -0.132812 3.039062 0.0078125 2.71875 0.09375 C 2.550781 0.132812 2.363281 0.15625 2.15625 0.15625 C 1.632812 0.15625 1.191406 -0.03125 0.828125 -0.40625 C 0.460938 -0.789062 0.28125 -1.328125 0.28125 -2.015625 C 0.28125 -2.679688 0.460938 -3.222656 0.828125 -3.640625 C 1.191406 -4.066406 1.671875 -4.28125 2.265625 -4.28125 Z M 3.375 -2.421875 C 3.351562 -2.734375 3.289062 -2.976562 3.1875 -3.15625 C 2.988281 -3.5 2.664062 -3.671875 2.21875 -3.671875 C 1.894531 -3.671875 1.625 -3.550781 1.40625 -3.3125 C 1.1875 -3.082031 1.070312 -2.785156 1.0625 -2.421875 Z M 2.1875 -4.28125 Z M 2.1875 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='97eaf93f-b688-49b5-8bf0-198a0248e67c' overflow='visible'>\n<path d='M 0 -0.265625 L -5.734375 -0.265625 L -5.734375 -4.8125 L 0 -4.8125 Z M -0.71875 -4.09375 L -5.015625 -4.09375 L -5.015625 -0.984375 L -0.71875 -0.984375 Z M -0.71875 -4.09375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='3e42de38-15c4-4d73-bf9c-81716c5e9235' overflow='visible'>\n<path d='M -5.734375 -0.53125 L -5.734375 -1.234375 L 0 -1.234375 L 0 -0.53125 Z M -5.734375 -0.53125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='e0f21094-844d-4906-8c1f-8af3dad0cca4' overflow='visible'>\n<path d='M -1.109375 -1.0625 C -0.910156 -1.0625 -0.75 -1.132812 -0.625 -1.28125 C -0.507812 -1.425781 -0.453125 -1.601562 -0.453125 -1.8125 C -0.453125 -2.050781 -0.507812 -2.289062 -0.625 -2.53125 C -0.820312 -2.914062 -1.132812 -3.109375 -1.5625 -3.109375 L -2.125 -3.109375 C -2.070312 -3.023438 -2.023438 -2.914062 -1.984375 -2.78125 C -1.953125 -2.644531 -1.929688 -2.507812 -1.921875 -2.375 L -1.859375 -1.953125 C -1.828125 -1.703125 -1.773438 -1.507812 -1.703125 -1.375 C -1.578125 -1.164062 -1.378906 -1.0625 -1.109375 -1.0625 Z M -2.53125 -2.765625 C -2.550781 -2.921875 -2.617188 -3.023438 -2.734375 -3.078125 C -2.796875 -3.109375 -2.890625 -3.125 -3.015625 -3.125 C -3.253906 -3.125 -3.425781 -3.039062 -3.53125 -2.875 C -3.632812 -2.707031 -3.6875 -2.460938 -3.6875 -2.140625 C -3.6875 -1.773438 -3.585938 -1.515625 -3.390625 -1.359375 C -3.285156 -1.273438 -3.125 -1.21875 -2.90625 -1.1875 L -2.90625 -0.53125 C -3.425781 -0.550781 -3.785156 -0.71875 -3.984375 -1.03125 C -4.179688 -1.351562 -4.28125 -1.726562 -4.28125 -2.15625 C -4.28125 -2.644531 -4.1875 -3.046875 -4 -3.359375 C -3.8125 -3.660156 -3.519531 -3.8125 -3.125 -3.8125 L -0.71875 -3.8125 C -0.644531 -3.8125 -0.585938 -3.828125 -0.546875 -3.859375 C -0.503906 -3.890625 -0.484375 -3.953125 -0.484375 -4.046875 C -0.484375 -4.078125 -0.484375 -4.113281 -0.484375 -4.15625 C -0.484375 -4.195312 -0.488281 -4.238281 -0.5 -4.28125 L 0.015625 -4.28125 C 0.046875 -4.164062 0.0664062 -4.078125 0.078125 -4.015625 C 0.0859375 -3.960938 0.09375 -3.890625 0.09375 -3.796875 C 0.09375 -3.554688 0.00390625 -3.378906 -0.171875 -3.265625 C -0.265625 -3.210938 -0.394531 -3.171875 -0.5625 -3.140625 C -0.375 -3.003906 -0.207031 -2.800781 -0.0625 -2.53125 C 0.0703125 -2.257812 0.140625 -1.960938 0.140625 -1.640625 C 0.140625 -1.253906 0.0234375 -0.9375 -0.203125 -0.6875 C -0.441406 -0.445312 -0.738281 -0.328125 -1.09375 -0.328125 C -1.488281 -0.328125 -1.789062 -0.445312 -2 -0.6875 C -2.21875 -0.925781 -2.347656 -1.242188 -2.390625 -1.640625 Z M -4.28125 -2.171875 Z M -4.28125 -2.171875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='00146ae9-d469-4aba-b414-88227c7746f3' overflow='visible'>\n<path d='M -5.359375 -0.65625 L -5.359375 -1.375 L -4.1875 -1.375 L -4.1875 -2.03125 L -3.609375 -2.03125 L -3.609375 -1.375 L -0.875 -1.375 C -0.726562 -1.375 -0.632812 -1.421875 -0.59375 -1.515625 C -0.5625 -1.566406 -0.546875 -1.660156 -0.546875 -1.796875 C -0.546875 -1.828125 -0.546875 -1.859375 -0.546875 -1.890625 C -0.546875 -1.929688 -0.550781 -1.976562 -0.5625 -2.03125 L 0 -2.03125 C 0.0195312 -1.957031 0.0351562 -1.875 0.046875 -1.78125 C 0.0546875 -1.695312 0.0625 -1.601562 0.0625 -1.5 C 0.0625 -1.175781 -0.0195312 -0.953125 -0.1875 -0.828125 C -0.351562 -0.710938 -0.570312 -0.65625 -0.84375 -0.65625 L -3.609375 -0.65625 L -3.609375 -0.09375 L -4.1875 -0.09375 L -4.1875 -0.65625 Z M -5.359375 -0.65625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='ec610f47-7f36-46e1-9a98-8601672c4c5e' overflow='visible'>\n<path d='M -4.171875 -0.515625 L -4.171875 -1.234375 L 0 -1.234375 L 0 -0.515625 Z M -5.734375 -0.515625 L -5.734375 -1.234375 L -4.9375 -1.234375 L -4.9375 -0.515625 Z M -5.734375 -0.515625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='ba8d1b0d-bfe6-445a-8866-c1fc98969362' overflow='visible'>\n<path d='M -4.1875 -1.21875 L -1.40625 -1.21875 C -1.195312 -1.21875 -1.023438 -1.253906 -0.890625 -1.328125 C -0.640625 -1.453125 -0.515625 -1.679688 -0.515625 -2.015625 C -0.515625 -2.503906 -0.734375 -2.835938 -1.171875 -3.015625 C -1.398438 -3.117188 -1.71875 -3.171875 -2.125 -3.171875 L -4.1875 -3.171875 L -4.1875 -3.875 L 0 -3.875 L 0 -3.203125 L -0.625 -3.21875 C -0.457031 -3.125 -0.320312 -3.007812 -0.21875 -2.875 C 0 -2.601562 0.109375 -2.273438 0.109375 -1.890625 C 0.109375 -1.296875 -0.0859375 -0.890625 -0.484375 -0.671875 C -0.703125 -0.554688 -0.988281 -0.5 -1.34375 -0.5 L -4.1875 -0.5 Z M -4.28125 -2.1875 Z M -4.28125 -2.1875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='897f70e8-7d32-4e3c-932e-eebb874442c9' overflow='visible'>\n<path d='M -2.046875 -0.96875 C -1.597656 -0.96875 -1.222656 -1.0625 -0.921875 -1.25 C -0.617188 -1.4375 -0.46875 -1.738281 -0.46875 -2.15625 C -0.46875 -2.488281 -0.609375 -2.757812 -0.890625 -2.96875 C -1.171875 -3.175781 -1.570312 -3.28125 -2.09375 -3.28125 C -2.625 -3.28125 -3.015625 -3.171875 -3.265625 -2.953125 C -3.515625 -2.734375 -3.640625 -2.46875 -3.640625 -2.15625 C -3.640625 -1.800781 -3.503906 -1.515625 -3.234375 -1.296875 C -2.960938 -1.078125 -2.566406 -0.96875 -2.046875 -0.96875 Z M -4.265625 -2.015625 C -4.265625 -2.335938 -4.195312 -2.609375 -4.0625 -2.828125 C -3.976562 -2.953125 -3.835938 -3.09375 -3.640625 -3.25 L -5.765625 -3.25 L -5.765625 -3.921875 L 0 -3.921875 L 0 -3.296875 L -0.578125 -3.296875 C -0.328125 -3.128906 -0.140625 -2.929688 -0.015625 -2.703125 C 0.0976562 -2.484375 0.15625 -2.226562 0.15625 -1.9375 C 0.15625 -1.476562 -0.0390625 -1.078125 -0.4375 -0.734375 C -0.832031 -0.390625 -1.351562 -0.21875 -2 -0.21875 C -2.613281 -0.21875 -3.144531 -0.375 -3.59375 -0.6875 C -4.039062 -1 -4.265625 -1.441406 -4.265625 -2.015625 Z M -4.265625 -2.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='6d6aec80-e05a-4449-9e56-90d597c72ccb' overflow='visible'>\n<path d='M -4.28125 -2.265625 C -4.28125 -2.554688 -4.207031 -2.84375 -4.0625 -3.125 C -3.925781 -3.40625 -3.75 -3.617188 -3.53125 -3.765625 C -3.320312 -3.898438 -3.078125 -3.992188 -2.796875 -4.046875 C -2.597656 -4.085938 -2.289062 -4.109375 -1.875 -4.109375 L -1.875 -1.03125 C -1.445312 -1.050781 -1.101562 -1.15625 -0.84375 -1.34375 C -0.59375 -1.53125 -0.46875 -1.816406 -0.46875 -2.203125 C -0.46875 -2.578125 -0.585938 -2.875 -0.828125 -3.09375 C -0.960938 -3.21875 -1.125 -3.304688 -1.3125 -3.359375 L -1.3125 -4.046875 C -1.15625 -4.023438 -0.984375 -3.960938 -0.796875 -3.859375 C -0.609375 -3.753906 -0.457031 -3.640625 -0.34375 -3.515625 C -0.132812 -3.304688 0.0078125 -3.039062 0.09375 -2.71875 C 0.132812 -2.550781 0.15625 -2.363281 0.15625 -2.15625 C 0.15625 -1.632812 -0.03125 -1.191406 -0.40625 -0.828125 C -0.789062 -0.460938 -1.328125 -0.28125 -2.015625 -0.28125 C -2.679688 -0.28125 -3.222656 -0.460938 -3.640625 -0.828125 C -4.066406 -1.191406 -4.28125 -1.671875 -4.28125 -2.265625 Z M -2.421875 -3.375 C -2.734375 -3.351562 -2.976562 -3.289062 -3.15625 -3.1875 C -3.5 -2.988281 -3.671875 -2.664062 -3.671875 -2.21875 C -3.671875 -1.894531 -3.550781 -1.625 -3.3125 -1.40625 C -3.082031 -1.1875 -2.785156 -1.070312 -2.421875 -1.0625 Z M -4.28125 -2.1875 Z M -4.28125 -2.1875 ' style='stroke:none;'/>\n</symbol>\n</g>\n<clipPath id='60f705bd-3db8-44a6-9bb4-c0d4bf4680ef'>\n<path d='M 38.695312 22.472656 L 552 22.472656 L 552 332 L 38.695312 332 Z M 38.695312 22.472656 '/>\n</clipPath>\n<clipPath id='0bfe133e-6850-421d-bc0d-20f4ac626d5a'>\n<path d='M 38.695312 275 L 552 275 L 552 277 L 38.695312 277 Z M 38.695312 275 '/>\n</clipPath>\n<clipPath id='3161951f-19e9-4d6d-b2aa-9d833bf2551a'>\n<path d='M 38.695312 216 L 552 216 L 552 217 L 38.695312 217 Z M 38.695312 216 '/>\n</clipPath>\n<clipPath id='7ca46b22-0e3a-48e3-9c61-ea610c4b93c6'>\n<path d='M 38.695312 157 L 552 157 L 552 158 L 38.695312 158 Z M 38.695312 157 '/>\n</clipPath>\n<clipPath id='f4f7f9c4-8fda-4bf8-8969-3215261ff079'>\n<path d='M 38.695312 98 L 552 98 L 552 99 L 38.695312 99 Z M 38.695312 98 '/>\n</clipPath>\n<clipPath id='38b19675-a27d-496b-86bc-0187acc39b1a'>\n<path d='M 38.695312 39 L 552 39 L 552 40 L 38.695312 40 Z M 38.695312 39 '/>\n</clipPath>\n<clipPath id='22aa0430-e426-4877-ad70-ed67a2363a64'>\n<path d='M 78 22.472656 L 79 22.472656 L 79 332 L 78 332 Z M 78 22.472656 '/>\n</clipPath>\n<clipPath id='0e6d0c2e-a8a5-4eb0-ac29-9c3adbad0989'>\n<path d='M 250 22.472656 L 251 22.472656 L 251 332 L 250 332 Z M 250 22.472656 '/>\n</clipPath>\n<clipPath id='f7e956fe-5a3d-4063-92f9-33316b5c4516'>\n<path d='M 422 22.472656 L 423 22.472656 L 423 332 L 422 332 Z M 422 22.472656 '/>\n</clipPath>\n<clipPath id='cf3eed3d-a2c5-437d-8a78-8e8b89da90fc'>\n<path d='M 38.695312 22.472656 L 552 22.472656 L 552 332 L 38.695312 332 Z M 38.695312 22.472656 '/>\n</clipPath>\n</defs>\n<g id='962d5c84-2f84-43dd-bd91-6170a06705bc'>\n<rect style='fill:rgb(100%,100%,100%);fill-opacity:1;stroke:none;' height='360' width='648' y='0' x='0'/>\n<rect style='fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke:none;' height='360' width='648' y='0' x='0'/>\n<path d='M 0 360 L 648 360 L 648 0 L 0 0 Z M 0 360 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g clip-rule='nonzero' clip-path='url(#60f705bd-3db8-44a6-9bb4-c0d4bf4680ef)'>\n<path d='M 38.695312 331.332031 L 551.464844 331.332031 L 551.464844 22.472656 L 38.695312 22.472656 Z M 38.695312 331.332031 ' style='fill-rule:nonzero;fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#0bfe133e-6850-421d-bc0d-20f4ac626d5a)'>\n<path d='M 38.695312 275.757812 L 551.464844 275.757812 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#3161951f-19e9-4d6d-b2aa-9d833bf2551a)'>\n<path d='M 38.695312 216.652344 L 551.464844 216.652344 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#7ca46b22-0e3a-48e3-9c61-ea610c4b93c6)'>\n<path d='M 38.695312 157.546875 L 551.464844 157.546875 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#f4f7f9c4-8fda-4bf8-8969-3215261ff079)'>\n<path d='M 38.695312 98.4375 L 551.464844 98.4375 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#38b19675-a27d-496b-86bc-0187acc39b1a)'>\n<path d='M 38.695312 39.332031 L 551.464844 39.332031 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#22aa0430-e426-4877-ad70-ed67a2363a64)'>\n<path d='M 78.316406 331.332031 L 78.316406 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#0e6d0c2e-a8a5-4eb0-ac29-9c3adbad0989)'>\n<path d='M 250.453125 331.332031 L 250.453125 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#f7e956fe-5a3d-4063-92f9-33316b5c4516)'>\n<path d='M 422.59375 331.332031 L 422.59375 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<path d='M 509.082031 317.292969 C 509.082031 319.898438 505.171875 319.898438 505.171875 317.292969 C 505.171875 314.6875 509.082031 314.6875 509.082031 317.292969 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(90.588235%,41.960784%,95.294118%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 520.101562 316.722656 C 520.101562 319.328125 516.191406 319.328125 516.191406 316.722656 C 516.191406 314.113281 520.101562 314.113281 520.101562 316.722656 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(90.588235%,41.960784%,95.294118%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 515.003906 314.777344 C 515.003906 317.382812 511.09375 317.382812 511.09375 314.777344 C 511.09375 312.171875 515.003906 312.171875 515.003906 314.777344 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(90.588235%,41.960784%,95.294118%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 516.46875 316.296875 C 516.46875 318.902344 512.558594 318.902344 512.558594 316.296875 C 512.558594 313.691406 516.46875 313.691406 516.46875 316.296875 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(90.588235%,41.960784%,95.294118%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 515.71875 313.160156 C 515.71875 315.769531 511.808594 315.769531 511.808594 313.160156 C 511.808594 310.554688 515.71875 310.554688 515.71875 313.160156 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(90.588235%,41.960784%,95.294118%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 521.855469 314.890625 C 521.855469 317.496094 517.945312 317.496094 517.945312 314.890625 C 517.945312 312.28125 521.855469 312.28125 521.855469 314.890625 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(90.588235%,41.960784%,95.294118%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 522.8125 317.035156 C 522.8125 319.644531 518.902344 319.644531 518.902344 317.035156 C 518.902344 314.429688 522.8125 314.429688 522.8125 317.035156 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(90.588235%,41.960784%,95.294118%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 515.164062 306.054688 C 515.164062 308.664062 511.257812 308.664062 511.257812 306.054688 C 511.257812 303.449219 515.164062 303.449219 515.164062 306.054688 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(90.588235%,41.960784%,95.294118%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 521.808594 312.765625 C 521.808594 315.371094 517.902344 315.371094 517.902344 312.765625 C 517.902344 310.160156 521.808594 310.160156 521.808594 312.765625 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(90.588235%,41.960784%,95.294118%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 522.960938 313.65625 C 522.960938 316.261719 519.050781 316.261719 519.050781 313.65625 C 519.050781 311.050781 522.960938 311.050781 522.960938 313.65625 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(90.588235%,41.960784%,95.294118%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 524.480469 315.472656 C 524.480469 318.082031 520.570312 318.082031 520.570312 315.472656 C 520.570312 312.867188 524.480469 312.867188 524.480469 315.472656 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(90.588235%,41.960784%,95.294118%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 519.277344 311.636719 C 519.277344 314.242188 515.367188 314.242188 515.367188 311.636719 C 515.367188 309.03125 519.277344 309.03125 519.277344 311.636719 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(90.588235%,41.960784%,95.294118%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 510.542969 315.542969 C 510.542969 318.148438 506.636719 318.148438 506.636719 315.542969 C 506.636719 312.9375 510.542969 312.9375 510.542969 315.542969 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(90.588235%,41.960784%,95.294118%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 530.113281 314.796875 C 530.113281 317.402344 526.203125 317.402344 526.203125 314.796875 C 526.203125 312.191406 530.113281 312.191406 530.113281 314.796875 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(90.588235%,41.960784%,95.294118%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 206.140625 228.886719 C 206.140625 231.492188 202.234375 231.492188 202.234375 228.886719 C 202.234375 226.277344 206.140625 226.277344 206.140625 228.886719 ' style='fill-rule:nonzero;fill:rgb(0%,74.901961%,49.019608%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,74.901961%,49.019608%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 224.789062 224.882812 C 224.789062 227.488281 220.878906 227.488281 220.878906 224.882812 C 220.878906 222.273438 224.789062 222.273438 224.789062 224.882812 ' style='fill-rule:nonzero;fill:rgb(0%,74.901961%,49.019608%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,74.901961%,49.019608%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 226.628906 223.972656 C 226.628906 226.578125 222.722656 226.578125 222.722656 223.972656 C 222.722656 221.367188 226.628906 221.367188 226.628906 223.972656 ' style='fill-rule:nonzero;fill:rgb(0%,74.901961%,49.019608%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,74.901961%,49.019608%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 228.804688 225.980469 C 228.804688 228.585938 224.894531 228.585938 224.894531 225.980469 C 224.894531 223.375 228.804688 223.375 228.804688 225.980469 ' style='fill-rule:nonzero;fill:rgb(0%,74.901961%,49.019608%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,74.901961%,49.019608%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 249.578125 225.300781 C 249.578125 227.90625 245.667969 227.90625 245.667969 225.300781 C 245.667969 222.691406 249.578125 222.691406 249.578125 225.300781 ' style='fill-rule:nonzero;fill:rgb(0%,74.901961%,49.019608%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,74.901961%,49.019608%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 231.125 222.738281 C 231.125 225.34375 227.214844 225.34375 227.214844 222.738281 C 227.214844 220.132812 231.125 220.132812 231.125 222.738281 ' style='fill-rule:nonzero;fill:rgb(0%,74.901961%,49.019608%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,74.901961%,49.019608%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 354.058594 282.132812 C 354.058594 284.738281 350.152344 284.738281 350.152344 282.132812 C 350.152344 279.527344 354.058594 279.527344 354.058594 282.132812 ' style='fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(97.254902%,46.27451%,42.745098%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 358.519531 279.09375 C 358.519531 281.699219 354.613281 281.699219 354.613281 279.09375 C 354.613281 276.488281 358.519531 276.488281 358.519531 279.09375 ' style='fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(97.254902%,46.27451%,42.745098%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 332.660156 271.65625 C 332.660156 274.261719 328.753906 274.261719 328.753906 271.65625 C 328.753906 269.050781 332.660156 269.050781 332.660156 271.65625 ' style='fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(97.254902%,46.27451%,42.745098%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 379.382812 278.515625 C 379.382812 281.121094 375.472656 281.121094 375.472656 278.515625 C 375.472656 275.910156 379.382812 275.910156 379.382812 278.515625 ' style='fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(97.254902%,46.27451%,42.745098%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 331.300781 275.496094 C 331.300781 278.105469 327.390625 278.105469 327.390625 275.496094 C 327.390625 272.890625 331.300781 272.890625 331.300781 275.496094 ' style='fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(97.254902%,46.27451%,42.745098%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 352.527344 284.058594 C 352.527344 286.667969 348.617188 286.667969 348.617188 284.058594 C 348.617188 281.453125 352.527344 281.453125 352.527344 284.058594 ' style='fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(97.254902%,46.27451%,42.745098%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 355.15625 263.402344 C 355.15625 266.007812 351.246094 266.007812 351.246094 263.402344 C 351.246094 260.792969 355.15625 260.792969 355.15625 263.402344 ' style='fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(97.254902%,46.27451%,42.745098%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 282.742188 249.757812 C 282.742188 252.367188 278.832031 252.367188 278.832031 249.757812 C 278.832031 247.152344 282.742188 247.152344 282.742188 249.757812 ' style='fill-rule:nonzero;fill:rgb(63.921569%,64.705882%,0%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(63.921569%,64.705882%,0%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 284.175781 249.445312 C 284.175781 252.050781 280.265625 252.050781 280.265625 249.445312 C 280.265625 246.835938 284.175781 246.835938 284.175781 249.445312 ' style='fill-rule:nonzero;fill:rgb(63.921569%,64.705882%,0%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(63.921569%,64.705882%,0%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 301.636719 258.570312 C 301.636719 261.175781 297.726562 261.175781 297.726562 258.570312 C 297.726562 255.964844 301.636719 255.964844 301.636719 258.570312 ' style='fill-rule:nonzero;fill:rgb(63.921569%,64.705882%,0%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(63.921569%,64.705882%,0%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 287.261719 247.035156 C 287.261719 249.640625 283.355469 249.640625 283.355469 247.035156 C 283.355469 244.429688 287.261719 244.429688 287.261719 247.035156 ' style='fill-rule:nonzero;fill:rgb(63.921569%,64.705882%,0%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(63.921569%,64.705882%,0%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 306.101562 260.855469 C 306.101562 263.460938 302.191406 263.460938 302.191406 260.855469 C 302.191406 258.25 306.101562 258.25 306.101562 260.855469 ' style='fill-rule:nonzero;fill:rgb(63.921569%,64.705882%,0%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(63.921569%,64.705882%,0%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 73.011719 48.863281 C 73.011719 51.472656 69.101562 51.472656 69.101562 48.863281 C 69.101562 46.257812 73.011719 46.257812 73.011719 48.863281 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 80.296875 42.285156 C 80.296875 44.890625 76.386719 44.890625 76.386719 42.285156 C 76.386719 39.679688 80.296875 39.679688 80.296875 42.285156 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 81.617188 40.9375 C 81.617188 43.546875 77.707031 43.546875 77.707031 40.9375 C 77.707031 38.332031 81.617188 38.332031 81.617188 40.9375 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 77.753906 42.742188 C 77.753906 45.347656 73.84375 45.347656 73.84375 42.742188 C 73.84375 40.136719 77.753906 40.136719 77.753906 42.742188 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 76.167969 42.035156 C 76.167969 44.640625 72.257812 44.640625 72.257812 42.035156 C 72.257812 39.429688 76.167969 39.429688 76.167969 42.035156 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 77.300781 45.847656 C 77.300781 48.453125 73.390625 48.453125 73.390625 45.847656 C 73.390625 43.242188 77.300781 43.242188 77.300781 45.847656 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 81.539062 39.359375 C 81.539062 41.964844 77.628906 41.964844 77.628906 39.359375 C 77.628906 36.753906 81.539062 36.753906 81.539062 39.359375 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 88.554688 45.445312 C 88.554688 48.050781 84.648438 48.050781 84.648438 45.445312 C 84.648438 42.839844 88.554688 42.839844 88.554688 45.445312 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 85.308594 42.125 C 85.308594 44.730469 81.398438 44.730469 81.398438 42.125 C 81.398438 39.519531 85.308594 39.519531 85.308594 42.125 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 87.988281 44.375 C 87.988281 46.984375 84.078125 46.984375 84.078125 44.375 C 84.078125 41.769531 87.988281 41.769531 87.988281 44.375 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 90.585938 46.925781 C 90.585938 49.53125 86.679688 49.53125 86.679688 46.925781 C 86.679688 44.320312 90.585938 44.320312 90.585938 46.925781 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 84.882812 45.386719 C 84.882812 47.992188 80.972656 47.992188 80.972656 45.386719 C 80.972656 42.78125 84.882812 42.78125 84.882812 45.386719 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 82.777344 43.910156 C 82.777344 46.519531 78.867188 46.519531 78.867188 43.910156 C 78.867188 41.304688 82.777344 41.304688 82.777344 43.910156 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 75.859375 50.117188 C 75.859375 52.722656 71.949219 52.722656 71.949219 50.117188 C 71.949219 47.511719 75.859375 47.511719 75.859375 50.117188 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 65.359375 51.878906 C 65.359375 54.484375 61.449219 54.484375 61.449219 51.878906 C 61.449219 49.273438 65.359375 49.273438 65.359375 51.878906 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 64.257812 50.367188 C 64.257812 52.972656 60.347656 52.972656 60.347656 50.367188 C 60.347656 47.757812 64.257812 47.757812 64.257812 50.367188 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 77.488281 36.511719 C 77.488281 39.117188 73.578125 39.117188 73.578125 36.511719 C 73.578125 33.90625 77.488281 33.90625 77.488281 36.511719 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 88.628906 50.84375 C 88.628906 53.449219 84.71875 53.449219 84.71875 50.84375 C 84.71875 48.234375 88.628906 48.234375 88.628906 50.84375 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 83.335938 47.980469 C 83.335938 50.585938 79.425781 50.585938 79.425781 47.980469 C 79.425781 45.375 83.335938 45.375 83.335938 47.980469 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 81.898438 47.027344 C 81.898438 49.632812 77.988281 49.632812 77.988281 47.027344 C 77.988281 44.421875 81.898438 44.421875 81.898438 47.027344 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 86.519531 49.816406 C 86.519531 52.425781 82.609375 52.425781 82.609375 49.816406 C 82.609375 47.210938 86.519531 47.210938 86.519531 49.816406 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 77.933594 56.4375 C 77.933594 59.042969 74.023438 59.042969 74.023438 56.4375 C 74.023438 53.832031 77.933594 53.832031 77.933594 56.4375 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 63.957031 54.21875 C 63.957031 56.824219 60.046875 56.824219 60.046875 54.21875 C 60.046875 51.613281 63.957031 51.613281 63.957031 54.21875 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 65.308594 53.152344 C 65.308594 55.757812 61.398438 55.757812 61.398438 53.152344 C 61.398438 50.542969 65.308594 50.542969 65.308594 53.152344 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 79.398438 48.269531 C 79.398438 50.875 75.488281 50.875 75.488281 48.269531 C 75.488281 45.664062 79.398438 45.664062 79.398438 48.269531 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 84.167969 53.164062 C 84.167969 55.769531 80.261719 55.769531 80.261719 53.164062 C 80.261719 50.554688 84.167969 50.554688 84.167969 53.164062 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 84.332031 53.164062 C 84.332031 55.769531 80.421875 55.769531 80.421875 53.164062 C 80.421875 50.554688 84.332031 50.554688 84.332031 53.164062 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 72.957031 46.160156 C 72.957031 48.765625 69.046875 48.765625 69.046875 46.160156 C 69.046875 43.554688 72.957031 43.554688 72.957031 46.160156 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 70.160156 50.539062 C 70.160156 53.144531 66.25 53.144531 66.25 50.539062 C 66.25 47.933594 70.160156 47.933594 70.160156 50.539062 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 75.15625 40.574219 C 75.15625 43.179688 71.246094 43.179688 71.246094 40.574219 C 71.246094 37.964844 75.15625 37.964844 75.15625 40.574219 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 84.910156 42.667969 C 84.910156 45.273438 81 45.273438 81 42.667969 C 81 40.0625 84.910156 40.0625 84.910156 42.667969 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 81.085938 44.472656 C 81.085938 47.082031 77.175781 47.082031 77.175781 44.472656 C 77.175781 41.867188 81.085938 41.867188 81.085938 44.472656 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 75.996094 47.425781 C 75.996094 50.03125 72.085938 50.03125 72.085938 47.425781 C 72.085938 44.820312 75.996094 44.820312 75.996094 47.425781 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 79.574219 45.464844 C 79.574219 48.070312 75.664062 48.070312 75.664062 45.464844 C 75.664062 42.855469 79.574219 42.855469 79.574219 45.464844 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 505.683594 303.773438 C 505.683594 306.382812 501.773438 306.382812 501.773438 303.773438 C 501.773438 301.167969 505.683594 301.167969 505.683594 303.773438 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(90.588235%,41.960784%,95.294118%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 79.527344 40.195312 C 79.527344 42.800781 75.617188 42.800781 75.617188 40.195312 C 75.617188 37.585938 79.527344 37.585938 79.527344 40.195312 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 223.960938 221.730469 C 223.960938 224.339844 220.050781 224.339844 220.050781 221.730469 C 220.050781 219.125 223.960938 219.125 223.960938 221.730469 ' style='fill-rule:nonzero;fill:rgb(0%,74.901961%,49.019608%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,74.901961%,49.019608%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 514.390625 309.613281 C 514.390625 312.21875 510.480469 312.21875 510.480469 309.613281 C 510.480469 307.007812 514.390625 307.007812 514.390625 309.613281 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(90.588235%,41.960784%,95.294118%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<g clip-rule='nonzero' clip-path='url(#cf3eed3d-a2c5-437d-8a78-8e8b89da90fc)'>\n<path d='M 38.695312 331.332031 L 551.464844 331.332031 L 551.464844 22.472656 L 38.695312 22.472656 Z M 38.695312 331.332031 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='278.265625' x='21.035156' xlink:href='#76cb1015-cd3a-42f4-9d1e-4fbe417959cc'/>\n<use y='278.265625' x='24.928223' xlink:href='#f8b137c6-b785-4887-819f-6509856f1052'/>\n<use y='278.265625' x='28.821289' xlink:href='#f70967a2-45ae-42ca-9824-939a68c579b6'/>\n<use y='278.265625' x='30.766113' xlink:href='#3fb7d8af-7976-4375-ab12-2f49a25b82a7'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='219.160156' x='21.035156' xlink:href='#76cb1015-cd3a-42f4-9d1e-4fbe417959cc'/>\n<use y='219.160156' x='24.928223' xlink:href='#f8b137c6-b785-4887-819f-6509856f1052'/>\n<use y='219.160156' x='28.821289' xlink:href='#f70967a2-45ae-42ca-9824-939a68c579b6'/>\n<use y='219.160156' x='30.766113' xlink:href='#70c90257-a8ce-476e-8592-83a8154478a6'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='160.054688' x='21.035156' xlink:href='#76cb1015-cd3a-42f4-9d1e-4fbe417959cc'/>\n<use y='160.054688' x='24.928223' xlink:href='#f8b137c6-b785-4887-819f-6509856f1052'/>\n<use y='160.054688' x='28.821289' xlink:href='#f70967a2-45ae-42ca-9824-939a68c579b6'/>\n<use y='160.054688' x='30.766113' xlink:href='#dffc1239-9538-4cf6-8a9d-20cfdf089427'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='100.945312' x='21.035156' xlink:href='#76cb1015-cd3a-42f4-9d1e-4fbe417959cc'/>\n<use y='100.945312' x='24.928223' xlink:href='#f8b137c6-b785-4887-819f-6509856f1052'/>\n<use y='100.945312' x='28.821289' xlink:href='#f70967a2-45ae-42ca-9824-939a68c579b6'/>\n<use y='100.945312' x='30.766113' xlink:href='#f8b137c6-b785-4887-819f-6509856f1052'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='41.839844' x='21.035156' xlink:href='#76cb1015-cd3a-42f4-9d1e-4fbe417959cc'/>\n<use y='41.839844' x='24.928223' xlink:href='#f8b137c6-b785-4887-819f-6509856f1052'/>\n<use y='41.839844' x='28.821289' xlink:href='#f70967a2-45ae-42ca-9824-939a68c579b6'/>\n<use y='41.839844' x='30.766113' xlink:href='#b20091d9-4f1f-47e9-beda-eef096c91571'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='68.390625' xlink:href='#cf906b66-f6d4-423c-a3f3-c1352e8d811a'/>\n<use y='340.382812' x='70.72168' xlink:href='#d78a0a30-b071-45b6-b218-0bf3bba15698'/>\n<use y='340.382812' x='74.614746' xlink:href='#708dbc1b-0990-4bb8-b313-630b3685a7b1'/>\n<use y='340.382812' x='78.507812' xlink:href='#708dbc1b-0990-4bb8-b313-630b3685a7b1'/>\n<use y='340.382812' x='82.400879' xlink:href='#f70967a2-45ae-42ca-9824-939a68c579b6'/>\n<use y='340.382812' x='84.345703' xlink:href='#3fb7d8af-7976-4375-ab12-2f49a25b82a7'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='240.527344' xlink:href='#cf906b66-f6d4-423c-a3f3-c1352e8d811a'/>\n<use y='340.382812' x='242.858398' xlink:href='#d78a0a30-b071-45b6-b218-0bf3bba15698'/>\n<use y='340.382812' x='246.751465' xlink:href='#708dbc1b-0990-4bb8-b313-630b3685a7b1'/>\n<use y='340.382812' x='250.644531' xlink:href='#708dbc1b-0990-4bb8-b313-630b3685a7b1'/>\n<use y='340.382812' x='254.537598' xlink:href='#f70967a2-45ae-42ca-9824-939a68c579b6'/>\n<use y='340.382812' x='256.482422' xlink:href='#708dbc1b-0990-4bb8-b313-630b3685a7b1'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='412.667969' xlink:href='#cf906b66-f6d4-423c-a3f3-c1352e8d811a'/>\n<use y='340.382812' x='414.999023' xlink:href='#d78a0a30-b071-45b6-b218-0bf3bba15698'/>\n<use y='340.382812' x='418.89209' xlink:href='#708dbc1b-0990-4bb8-b313-630b3685a7b1'/>\n<use y='340.382812' x='422.785156' xlink:href='#708dbc1b-0990-4bb8-b313-630b3685a7b1'/>\n<use y='340.382812' x='426.678223' xlink:href='#f70967a2-45ae-42ca-9824-939a68c579b6'/>\n<use y='340.382812' x='428.623047' xlink:href='#6314508d-bdd6-4313-a7b2-3fb15b03668b'/>\n</g>\n<g style='fill:rgb(32.156863%,32.156863%,32.156863%);fill-opacity:1;'>\n<use y='349.703125' x='278.84375' xlink:href='#f6130023-a7bb-4980-a7bd-d30ff963354b'/>\n<use y='349.703125' x='280.621094' xlink:href='#20ef76af-4dfb-4765-bddb-4471bba53ec7'/>\n<use y='349.703125' x='285.070312' xlink:href='#dfdeacb8-562a-4221-8a1f-e5cd5fdda78e'/>\n<use y='349.703125' x='289.519531' xlink:href='#8099ddb1-0426-4749-8f71-4a63587d238e'/>\n<use y='349.703125' x='293.96875' xlink:href='#1ceeeb8c-17ed-4456-b333-d1362f8235bb'/>\n<use y='349.703125' x='295.746094' xlink:href='#51e88b40-5464-452a-a315-e01461f4b1fc'/>\n<use y='349.703125' x='297.96875' xlink:href='#1380d38a-a602-4f6c-9e3f-7e037cc7cb71'/>\n<use y='349.703125' x='302.417969' xlink:href='#715395ce-8992-4b9e-aeed-8d905de6382e'/>\n<use y='349.703125' x='306.867188' xlink:href='#62933a52-0d45-4414-8c01-ed7527dc0fda'/>\n</g>\n<g style='fill:rgb(32.156863%,32.156863%,32.156863%);fill-opacity:1;'>\n<use y='189.800781' x='17.449219' xlink:href='#3e42de38-15c4-4d73-bf9c-81716c5e9235'/>\n<use y='188.023438' x='17.449219' xlink:href='#e0f21094-844d-4906-8c1f-8af3dad0cca4'/>\n<use y='183.574219' x='17.449219' xlink:href='#00146ae9-d469-4aba-b414-88227c7746f3'/>\n<use y='181.351562' x='17.449219' xlink:href='#ec610f47-7f36-46e1-9a98-8601672c4c5e'/>\n<use y='179.574219' x='17.449219' xlink:href='#00146ae9-d469-4aba-b414-88227c7746f3'/>\n<use y='177.351562' x='17.449219' xlink:href='#ba8d1b0d-bfe6-445a-8866-c1fc98969362'/>\n<use y='172.902344' x='17.449219' xlink:href='#897f70e8-7d32-4e3c-932e-eebb874442c9'/>\n<use y='168.453125' x='17.449219' xlink:href='#6d6aec80-e05a-4449-9e56-90d597c72ccb'/>\n</g>\n<path d='M 559.96875 226.515625 L 633.828125 226.515625 L 633.828125 127.292969 L 559.96875 127.292969 Z M 559.96875 226.515625 ' style=' stroke:none;fill-rule:nonzero;fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;'/>\n<path d='M 564.222656 153.140625 L 581.503906 153.140625 L 581.503906 135.859375 L 564.222656 135.859375 Z M 564.222656 153.140625 ' style='fill-rule:nonzero;fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 574.816406 144.503906 C 574.816406 147.109375 570.90625 147.109375 570.90625 144.503906 C 570.90625 141.894531 574.816406 141.894531 574.816406 144.503906 ' style='fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(97.254902%,46.27451%,42.745098%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 564.222656 170.421875 L 581.503906 170.421875 L 581.503906 153.140625 L 564.222656 153.140625 Z M 564.222656 170.421875 ' style='fill-rule:nonzero;fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 574.816406 161.78125 C 574.816406 164.386719 570.90625 164.386719 570.90625 161.78125 C 570.90625 159.175781 574.816406 159.175781 574.816406 161.78125 ' style='fill-rule:nonzero;fill:rgb(63.921569%,64.705882%,0%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(63.921569%,64.705882%,0%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 564.222656 187.703125 L 581.503906 187.703125 L 581.503906 170.421875 L 564.222656 170.421875 Z M 564.222656 187.703125 ' style='fill-rule:nonzero;fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 574.816406 179.0625 C 574.816406 181.667969 570.90625 181.667969 570.90625 179.0625 C 570.90625 176.457031 574.816406 176.457031 574.816406 179.0625 ' style='fill-rule:nonzero;fill:rgb(0%,74.901961%,49.019608%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,74.901961%,49.019608%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 564.222656 204.980469 L 581.503906 204.980469 L 581.503906 187.699219 L 564.222656 187.699219 Z M 564.222656 204.980469 ' style='fill-rule:nonzero;fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 574.816406 196.34375 C 574.816406 198.949219 570.90625 198.949219 570.90625 196.34375 C 570.90625 193.734375 574.816406 193.734375 574.816406 196.34375 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,69.019608%,96.470588%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 564.222656 222.261719 L 581.503906 222.261719 L 581.503906 204.980469 L 564.222656 204.980469 Z M 564.222656 222.261719 ' style='fill-rule:nonzero;fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 574.816406 213.621094 C 574.816406 216.230469 570.90625 216.230469 570.90625 213.621094 C 570.90625 211.015625 574.816406 211.015625 574.816406 213.621094 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:0.708661;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(90.588235%,41.960784%,95.294118%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<g style='fill:rgb(32.156863%,32.156863%,32.156863%);fill-opacity:1;'>\n<use y='147.011719' x='583.660156' xlink:href='#ca82d3de-5a3f-428f-872d-b2e9cd820bb4'/>\n<use y='147.011719' x='589.491211' xlink:href='#99dd145f-47da-4d44-a2e4-f5ff965b4344'/>\n<use y='147.011719' x='593.384277' xlink:href='#18b51442-86c0-4de3-ba7a-165b19721dc5'/>\n<use y='147.011719' x='597.277344' xlink:href='#04d0175d-fa1a-4a34-bfa1-3845ccb013e0'/>\n<use y='147.011719' x='601.17041' xlink:href='#79a882c2-7f7c-43f7-a347-325d14ffacb6'/>\n<use y='147.011719' x='603.115234' xlink:href='#9d92f1f0-947c-4180-ba6e-3beb1c7c45fc'/>\n<use y='147.011719' x='607.008301' xlink:href='#dacc87f4-0b2e-4397-a29d-001caec4525d'/>\n<use y='147.011719' x='608.563477' xlink:href='#04d0175d-fa1a-4a34-bfa1-3845ccb013e0'/>\n<use y='147.011719' x='612.456543' xlink:href='#f5a4abe7-8816-4a9a-a518-b3cab0acc7c6'/>\n<use y='147.011719' x='614.401367' xlink:href='#5b79702e-33ba-417e-ada5-ff6b861c39b4'/>\n<use y='147.011719' x='619.070312' xlink:href='#dacc87f4-0b2e-4397-a29d-001caec4525d'/>\n<use y='147.011719' x='620.625488' xlink:href='#6c39d25e-557a-4381-bc22-2c347397ded4'/>\n<use y='147.011719' x='624.518555' xlink:href='#0a2b1e60-e168-4049-a1f8-40d241552549'/>\n</g>\n<g style='fill:rgb(32.156863%,32.156863%,32.156863%);fill-opacity:1;'>\n<use y='164.289062' x='583.660156' xlink:href='#35bc8884-c9bc-48af-8391-536167e977d9'/>\n<use y='164.289062' x='588.329102' xlink:href='#9d92f1f0-947c-4180-ba6e-3beb1c7c45fc'/>\n<use y='164.289062' x='592.222168' xlink:href='#eee407df-aae7-4267-a2d8-c6bf27bbb96b'/>\n<use y='164.289062' x='593.777344' xlink:href='#99dd145f-47da-4d44-a2e4-f5ff965b4344'/>\n<use y='164.289062' x='597.67041' xlink:href='#f5a4abe7-8816-4a9a-a518-b3cab0acc7c6'/>\n<use y='164.289062' x='599.615234' xlink:href='#2b371840-a87a-42e6-a793-f8a6f2e91e42'/>\n<use y='164.289062' x='604.28418' xlink:href='#eee407df-aae7-4267-a2d8-c6bf27bbb96b'/>\n<use y='164.289062' x='605.839355' xlink:href='#79a882c2-7f7c-43f7-a347-325d14ffacb6'/>\n<use y='164.289062' x='607.78418' xlink:href='#99dd145f-47da-4d44-a2e4-f5ff965b4344'/>\n</g>\n<g style='fill:rgb(32.156863%,32.156863%,32.156863%);fill-opacity:1;'>\n<use y='181.570312' x='583.660156' xlink:href='#cadc9d5b-102b-4f15-b279-1bccc87f045c'/>\n<use y='181.570312' x='588.715332' xlink:href='#6c39d25e-557a-4381-bc22-2c347397ded4'/>\n<use y='181.570312' x='592.608398' xlink:href='#253771c0-e953-4187-9dbb-3f4b43811093'/>\n<use y='181.570312' x='596.501465' xlink:href='#0a2b1e60-e168-4049-a1f8-40d241552549'/>\n<use y='181.570312' x='601.556641' xlink:href='#99dd145f-47da-4d44-a2e4-f5ff965b4344'/>\n<use y='181.570312' x='605.449707' xlink:href='#99dd145f-47da-4d44-a2e4-f5ff965b4344'/>\n<use y='181.570312' x='609.342773' xlink:href='#253771c0-e953-4187-9dbb-3f4b43811093'/>\n<use y='181.570312' x='613.23584' xlink:href='#f5a4abe7-8816-4a9a-a518-b3cab0acc7c6'/>\n<use y='181.570312' x='615.180664' xlink:href='#31ca224b-da2f-4e8c-890a-3a5521249051'/>\n<use y='181.570312' x='620.23584' xlink:href='#dacc87f4-0b2e-4397-a29d-001caec4525d'/>\n<use y='181.570312' x='621.791016' xlink:href='#79a882c2-7f7c-43f7-a347-325d14ffacb6'/>\n<use y='181.570312' x='623.73584' xlink:href='#f0dea103-8c6a-4d6b-b4b5-2b8f0d805944'/>\n</g>\n<g style='fill:rgb(32.156863%,32.156863%,32.156863%);fill-opacity:1;'>\n<use y='198.851562' x='583.660156' xlink:href='#2da9ee19-7a59-490e-a444-6b33033d40f1'/>\n<use y='198.851562' x='588.329102' xlink:href='#9d92f1f0-947c-4180-ba6e-3beb1c7c45fc'/>\n<use y='198.851562' x='592.222168' xlink:href='#04d0175d-fa1a-4a34-bfa1-3845ccb013e0'/>\n<use y='198.851562' x='596.115234' xlink:href='#f5a4abe7-8816-4a9a-a518-b3cab0acc7c6'/>\n<use y='198.851562' x='598.060059' xlink:href='#4f4d02e8-d8e2-4487-acbe-c632124d6095'/>\n<use y='198.851562' x='602.335938' xlink:href='#a7916dd8-f007-437b-9c1f-63d175f1a331'/>\n<use y='198.851562' x='604.666992' xlink:href='#9d92f1f0-947c-4180-ba6e-3beb1c7c45fc'/>\n<use y='198.851562' x='608.560059' xlink:href='#04d0175d-fa1a-4a34-bfa1-3845ccb013e0'/>\n<use y='198.851562' x='612.453125' xlink:href='#c85ed8e5-20be-43d5-bc2a-cf06e238e45b'/>\n<use y='198.851562' x='615.953125' xlink:href='#dacc87f4-0b2e-4397-a29d-001caec4525d'/>\n<use y='198.851562' x='617.508301' xlink:href='#a97ebaf8-88ff-48be-9d64-b9ea4a0d5a58'/>\n<use y='198.851562' x='621.008301' xlink:href='#c85ed8e5-20be-43d5-bc2a-cf06e238e45b'/>\n<use y='198.851562' x='624.508301' xlink:href='#99dd145f-47da-4d44-a2e4-f5ff965b4344'/>\n</g>\n<g style='fill:rgb(32.156863%,32.156863%,32.156863%);fill-opacity:1;'>\n<use y='216.128906' x='583.660156' xlink:href='#2da9ee19-7a59-490e-a444-6b33033d40f1'/>\n<use y='216.128906' x='588.329102' xlink:href='#9d92f1f0-947c-4180-ba6e-3beb1c7c45fc'/>\n<use y='216.128906' x='592.222168' xlink:href='#04d0175d-fa1a-4a34-bfa1-3845ccb013e0'/>\n<use y='216.128906' x='596.115234' xlink:href='#f5a4abe7-8816-4a9a-a518-b3cab0acc7c6'/>\n<use y='216.128906' x='598.060059' xlink:href='#a1ac6047-4ba6-4030-837e-40137af96e57'/>\n<use y='216.128906' x='601.560059' xlink:href='#99dd145f-47da-4d44-a2e4-f5ff965b4344'/>\n<use y='216.128906' x='605.453125' xlink:href='#a97ebaf8-88ff-48be-9d64-b9ea4a0d5a58'/>\n<use y='216.128906' x='608.953125' xlink:href='#6c39d25e-557a-4381-bc22-2c347397ded4'/>\n</g>\n</g>\n</svg>\n","value":"#huri.plot.GGView{:plot-command [[:<- :g [:data.frame {:longitude [:c -121.901782 -121.888979 -121.894902 -121.8932 -121.894074 -121.886943 -121.885831 -121.894715 -121.886995 -121.88566 -121.883891 -121.889937 -121.900084 -121.877349 -122.253755 -122.232089 -122.229951 -122.227424 -122.203288 -122.224728 -122.081896 -122.076713 -122.106758 -122.052476 -122.108338 -122.083678 -122.080623 -122.164759 -122.163093 -122.142805 -122.159504 -122.137619 -122.408433 -122.39997 -122.398436 -122.402923 -122.404767 -122.403452 -122.398525 -122.390373 -122.394148 -122.391034 -122.388013 -122.394643 -122.397086 -122.405127 -122.417324 -122.418603 -122.403234 -122.390288 -122.39644 -122.398108 -122.392738 -122.402717 -122.418954 -122.417385 -122.401014 -122.39547 -122.39528 -122.408499 -122.411748 -122.405942 -122.39461 -122.399051 -122.404966 -122.400811 -121.905733 -122.400862 -122.233051 -121.895617], :latitude [:c 37.329732 37.330698 37.333988 37.331415 37.336721 37.333798 37.330165 37.348742 37.337391 37.335885 37.332808 37.339301 37.332692 37.333955 37.479303 37.486078 37.487616 37.484219 37.48537 37.489704 37.389218 37.394358 37.40694 37.395337 37.400443 37.385956 37.420909 37.443988 37.444521 37.429082 37.448598 37.425216 37.783871 37.795001 37.79728 37.794231 37.795425 37.788975 37.799953 37.789656 37.795275 37.791464 37.787152 37.789756 37.792251 37.781752 37.778772 37.781332 37.80477 37.780526 37.785366 37.786978 37.782259 37.771058 37.774814 37.776619 37.784878 37.7766 37.776599 37.788446 37.781039 37.7979 37.794355 37.7913 37.786305 37.789625 37.352601 37.798541 37.491405 37.342725], :city [:c \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"Redwood City\" \"Redwood City\" \"Redwood City\" \"Redwood City\" \"Redwood City\" \"Redwood City\" \"Mountain View\" \"Mountain View\" \"Mountain View\" \"Mountain View\" \"Mountain View\" \"Mountain View\" \"Mountain View\" \"Palo Alto\" \"Palo Alto\" \"Palo Alto\" \"Palo Alto\" \"Palo Alto\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Jose\" \"San Francisco\" \"Redwood City\" \"San Jose\"]}]] [[:library :ggplot2] [:library :scales] [:library :grid] [:library :RColorBrewer] [:library :ggrepel] [:<- :palette [:brewer.pal \"Greys\" {:n 9}]] {:color.background :palette[2]} {:color.grid.major :palette[3]} {:color.axis.text :palette[6]} {:color.axis.title :palette[7]} {:color.title :palette[9]}] [:+ [:+ [:+ [:+ [:+ [:ggplot :g [:aes {:x :longitude, :y :latitude, :colour :city}]] [:geom_point [:aes {}] {:alpha 0.5}]] [:scale_x_continuous {:labels :comma}]] [:scale_y_continuous {:labels :comma}]] [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:theme_bw {:base_size 9}] [:theme {:panel.background [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:plot.background [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:panel.border [:element_rect {:color :color.background}]}]] [:theme {:panel.grid.major [:element_line {:color :color.grid.major, :size 0.25}]}]] [:theme {:panel.grid.minor [:element_blank]}]] [:theme {:axis.ticks [:element_blank]}]] [:theme {:legend.background [:element_rect {:fill :color.background}]}]] [:theme {:legend.key [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:legend.text [:element_text {:size 7, :color :color.axis.title}]}]] [:theme {:legend.title [:element_blank]}]] [:theme {:plot.title [:element_text {:size 10, :color :color.title, :vjust 1.25}]}]] [:theme {:axis.text.x [:element_text {:size 7, :color :color.axis.text}]}]] [:theme {:axis.text.y [:element_text {:size 7, :color :color.axis.text}]}]] [:theme {:axis.title.x [:element_text {:size 8, :color :color.axis.title, :vjust 0}]}]] [:theme {:axis.title.y [:element_text {:size 8, :color :color.axis.title, :vjust 1.25}]}]] [:theme {:plot.margin [:unit [:c 0.35 0.2 0.3 0.35] \"cm\"]}]]] [:labs {:y \"latitude\", :title \"\", :x \"longitude\"}]]], :options {:width 9, :height 5}}"}
;; <=

;; @@
(violin-plot :city :availableBikes df)
;; @@
;; =>
;;; {"type":"html","content":"<?xml version='1.0' encoding='UTF-8'?>\n<svg version='1.1' viewBox='0 0 648 360' height='360pt' width='648pt' xmlns:xlink='http://www.w3.org/1999/xlink' xmlns='http://www.w3.org/2000/svg'>\n<defs>\n<g>\n<symbol id='1fd006cf-d85d-4ffc-9003-ffce1f55523c' overflow='visible'>\n<path d='M 0.21875 0 L 0.21875 -5.015625 L 4.21875 -5.015625 L 4.21875 0 Z M 3.578125 -0.625 L 3.578125 -4.390625 L 0.859375 -4.390625 L 0.859375 -0.625 Z M 3.578125 -0.625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='ee5e8d0f-d720-472b-ab4c-55f82924b4d5' overflow='visible'>\n<path d='M 1.890625 -4.890625 C 2.523438 -4.890625 2.984375 -4.628906 3.265625 -4.109375 C 3.484375 -3.703125 3.59375 -3.148438 3.59375 -2.453125 C 3.59375 -1.785156 3.492188 -1.234375 3.296875 -0.796875 C 3.015625 -0.171875 2.546875 0.140625 1.890625 0.140625 C 1.304688 0.140625 0.867188 -0.113281 0.578125 -0.625 C 0.335938 -1.050781 0.21875 -1.625 0.21875 -2.34375 C 0.21875 -2.90625 0.289062 -3.382812 0.4375 -3.78125 C 0.707031 -4.519531 1.191406 -4.890625 1.890625 -4.890625 Z M 1.890625 -0.421875 C 2.203125 -0.421875 2.453125 -0.5625 2.640625 -0.84375 C 2.835938 -1.132812 2.9375 -1.664062 2.9375 -2.4375 C 2.9375 -2.988281 2.863281 -3.441406 2.71875 -3.796875 C 2.582031 -4.160156 2.320312 -4.34375 1.9375 -4.34375 C 1.570312 -4.34375 1.304688 -4.171875 1.140625 -3.828125 C 0.972656 -3.484375 0.890625 -2.984375 0.890625 -2.328125 C 0.890625 -1.828125 0.941406 -1.425781 1.046875 -1.125 C 1.210938 -0.65625 1.492188 -0.421875 1.890625 -0.421875 Z M 1.890625 -0.421875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='1cb5964f-4b40-4e4a-b820-373522f98df8' overflow='visible'>\n<path d='M 0.859375 -1.25 C 0.898438 -0.894531 1.066406 -0.648438 1.359375 -0.515625 C 1.503906 -0.453125 1.671875 -0.421875 1.859375 -0.421875 C 2.222656 -0.421875 2.492188 -0.535156 2.671875 -0.765625 C 2.847656 -0.992188 2.9375 -1.25 2.9375 -1.53125 C 2.9375 -1.875 2.832031 -2.140625 2.625 -2.328125 C 2.414062 -2.515625 2.164062 -2.609375 1.875 -2.609375 C 1.65625 -2.609375 1.46875 -2.566406 1.3125 -2.484375 C 1.164062 -2.398438 1.039062 -2.285156 0.9375 -2.140625 L 0.40625 -2.171875 L 0.78125 -4.8125 L 3.3125 -4.8125 L 3.3125 -4.21875 L 1.234375 -4.21875 L 1.03125 -2.859375 C 1.144531 -2.941406 1.253906 -3.003906 1.359375 -3.046875 C 1.535156 -3.128906 1.742188 -3.171875 1.984375 -3.171875 C 2.429688 -3.171875 2.8125 -3.023438 3.125 -2.734375 C 3.4375 -2.441406 3.59375 -2.070312 3.59375 -1.625 C 3.59375 -1.164062 3.445312 -0.757812 3.15625 -0.40625 C 2.875 -0.0507812 2.421875 0.125 1.796875 0.125 C 1.398438 0.125 1.046875 0.015625 0.734375 -0.203125 C 0.429688 -0.429688 0.257812 -0.78125 0.21875 -1.25 Z M 0.859375 -1.25 ' style='stroke:none;'/>\n</symbol>\n<symbol id='3e625fa4-404b-413d-adb1-a377fe35f6a0' overflow='visible'>\n<path d='M 0.671875 -3.46875 L 0.671875 -3.9375 C 1.117188 -3.976562 1.425781 -4.050781 1.59375 -4.15625 C 1.769531 -4.257812 1.90625 -4.5 2 -4.875 L 2.484375 -4.875 L 2.484375 0 L 1.828125 0 L 1.828125 -3.46875 Z M 0.671875 -3.46875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='90daeba2-e225-470b-a3d9-e31d3a4992da' overflow='visible'>\n<path d='M 0.515625 -5.015625 L 1.484375 -5.015625 L 2.9375 -0.78125 L 4.359375 -5.015625 L 5.328125 -5.015625 L 5.328125 0 L 4.6875 0 L 4.6875 -2.96875 C 4.6875 -3.070312 4.6875 -3.238281 4.6875 -3.46875 C 4.695312 -3.707031 4.703125 -3.960938 4.703125 -4.234375 L 3.265625 0 L 2.59375 0 L 1.15625 -4.234375 L 1.15625 -4.078125 C 1.15625 -3.953125 1.15625 -3.765625 1.15625 -3.515625 C 1.164062 -3.265625 1.171875 -3.082031 1.171875 -2.96875 L 1.171875 0 L 0.515625 0 Z M 0.515625 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='f8d5fa39-c492-44ea-9d7d-68a38e61c951' overflow='visible'>\n<path d='M 1.90625 -0.390625 C 2.3125 -0.390625 2.585938 -0.546875 2.734375 -0.859375 C 2.890625 -1.171875 2.96875 -1.515625 2.96875 -1.890625 C 2.96875 -2.234375 2.914062 -2.507812 2.8125 -2.71875 C 2.632812 -3.0625 2.332031 -3.234375 1.90625 -3.234375 C 1.53125 -3.234375 1.253906 -3.085938 1.078125 -2.796875 C 0.910156 -2.515625 0.828125 -2.171875 0.828125 -1.765625 C 0.828125 -1.367188 0.910156 -1.039062 1.078125 -0.78125 C 1.253906 -0.519531 1.53125 -0.390625 1.90625 -0.390625 Z M 1.921875 -3.765625 C 2.398438 -3.765625 2.800781 -3.609375 3.125 -3.296875 C 3.445312 -2.984375 3.609375 -2.519531 3.609375 -1.90625 C 3.609375 -1.3125 3.460938 -0.820312 3.171875 -0.4375 C 2.890625 -0.0507812 2.445312 0.140625 1.84375 0.140625 C 1.34375 0.140625 0.941406 -0.03125 0.640625 -0.375 C 0.347656 -0.71875 0.203125 -1.175781 0.203125 -1.75 C 0.203125 -2.363281 0.359375 -2.851562 0.671875 -3.21875 C 0.984375 -3.582031 1.398438 -3.765625 1.921875 -3.765625 Z M 1.90625 -3.75 Z M 1.90625 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='1b32b2f8-625d-46b4-9f39-870ef8cdec9e' overflow='visible'>\n<path d='M 1.0625 -3.65625 L 1.0625 -1.234375 C 1.0625 -1.046875 1.09375 -0.890625 1.15625 -0.765625 C 1.257812 -0.546875 1.460938 -0.4375 1.765625 -0.4375 C 2.191406 -0.4375 2.484375 -0.628906 2.640625 -1.015625 C 2.722656 -1.222656 2.765625 -1.503906 2.765625 -1.859375 L 2.765625 -3.65625 L 3.390625 -3.65625 L 3.390625 0 L 2.796875 0 L 2.8125 -0.546875 C 2.726562 -0.398438 2.628906 -0.28125 2.515625 -0.1875 C 2.273438 0 1.988281 0.09375 1.65625 0.09375 C 1.132812 0.09375 0.78125 -0.078125 0.59375 -0.421875 C 0.488281 -0.609375 0.4375 -0.859375 0.4375 -1.171875 L 0.4375 -3.65625 Z M 1.90625 -3.75 Z M 1.90625 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='c70ea6ec-e340-4d96-904e-44ec0f8e0741' overflow='visible'>\n<path d='M 0.453125 -3.65625 L 1.03125 -3.65625 L 1.03125 -3.140625 C 1.207031 -3.359375 1.390625 -3.515625 1.578125 -3.609375 C 1.773438 -3.703125 1.992188 -3.75 2.234375 -3.75 C 2.753906 -3.75 3.101562 -3.566406 3.28125 -3.203125 C 3.382812 -3.003906 3.4375 -2.71875 3.4375 -2.34375 L 3.4375 0 L 2.8125 0 L 2.8125 -2.3125 C 2.8125 -2.53125 2.773438 -2.707031 2.703125 -2.84375 C 2.597656 -3.070312 2.398438 -3.1875 2.109375 -3.1875 C 1.960938 -3.1875 1.84375 -3.171875 1.75 -3.140625 C 1.582031 -3.085938 1.4375 -2.988281 1.3125 -2.84375 C 1.207031 -2.71875 1.140625 -2.59375 1.109375 -2.46875 C 1.078125 -2.34375 1.0625 -2.160156 1.0625 -1.921875 L 1.0625 0 L 0.453125 0 Z M 1.890625 -3.75 Z M 1.890625 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='dba415b5-c0df-446f-a7f2-56388b34a9c6' overflow='visible'>\n<path d='M 0.578125 -4.6875 L 1.203125 -4.6875 L 1.203125 -3.65625 L 1.78125 -3.65625 L 1.78125 -3.15625 L 1.203125 -3.15625 L 1.203125 -0.765625 C 1.203125 -0.640625 1.242188 -0.554688 1.328125 -0.515625 C 1.367188 -0.484375 1.445312 -0.46875 1.5625 -0.46875 C 1.59375 -0.46875 1.625 -0.46875 1.65625 -0.46875 C 1.695312 -0.476562 1.738281 -0.484375 1.78125 -0.484375 L 1.78125 0 C 1.707031 0.0195312 1.632812 0.0351562 1.5625 0.046875 C 1.488281 0.0546875 1.40625 0.0625 1.3125 0.0625 C 1.019531 0.0625 0.820312 -0.0078125 0.71875 -0.15625 C 0.625 -0.3125 0.578125 -0.503906 0.578125 -0.734375 L 0.578125 -3.15625 L 0.078125 -3.15625 L 0.078125 -3.65625 L 0.578125 -3.65625 Z M 0.578125 -4.6875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='d3f65531-fea4-4828-a135-dc4bfbb6e530' overflow='visible'>\n<path d='M 0.921875 -0.96875 C 0.921875 -0.789062 0.984375 -0.648438 1.109375 -0.546875 C 1.242188 -0.453125 1.398438 -0.40625 1.578125 -0.40625 C 1.796875 -0.40625 2.003906 -0.453125 2.203125 -0.546875 C 2.546875 -0.710938 2.71875 -0.988281 2.71875 -1.375 L 2.71875 -1.859375 C 2.644531 -1.816406 2.546875 -1.78125 2.421875 -1.75 C 2.304688 -1.71875 2.191406 -1.691406 2.078125 -1.671875 L 1.703125 -1.625 C 1.484375 -1.59375 1.316406 -1.546875 1.203125 -1.484375 C 1.015625 -1.378906 0.921875 -1.207031 0.921875 -0.96875 Z M 2.40625 -2.21875 C 2.550781 -2.238281 2.648438 -2.296875 2.703125 -2.390625 C 2.722656 -2.453125 2.734375 -2.53125 2.734375 -2.625 C 2.734375 -2.84375 2.660156 -3 2.515625 -3.09375 C 2.367188 -3.1875 2.15625 -3.234375 1.875 -3.234375 C 1.550781 -3.234375 1.320312 -3.144531 1.1875 -2.96875 C 1.113281 -2.875 1.066406 -2.734375 1.046875 -2.546875 L 0.46875 -2.546875 C 0.476562 -2.992188 0.625 -3.304688 0.90625 -3.484375 C 1.1875 -3.660156 1.515625 -3.75 1.890625 -3.75 C 2.316406 -3.75 2.664062 -3.664062 2.9375 -3.5 C 3.207031 -3.332031 3.34375 -3.078125 3.34375 -2.734375 L 3.34375 -0.625 C 3.34375 -0.5625 3.351562 -0.507812 3.375 -0.46875 C 3.40625 -0.4375 3.460938 -0.421875 3.546875 -0.421875 C 3.566406 -0.421875 3.59375 -0.421875 3.625 -0.421875 C 3.664062 -0.421875 3.707031 -0.425781 3.75 -0.4375 L 3.75 0.015625 C 3.644531 0.046875 3.566406 0.0625 3.515625 0.0625 C 3.472656 0.0703125 3.40625 0.078125 3.3125 0.078125 C 3.101562 0.078125 2.953125 0.00390625 2.859375 -0.140625 C 2.804688 -0.222656 2.769531 -0.335938 2.75 -0.484375 C 2.625 -0.316406 2.441406 -0.171875 2.203125 -0.046875 C 1.972656 0.0664062 1.71875 0.125 1.4375 0.125 C 1.09375 0.125 0.8125 0.0195312 0.59375 -0.1875 C 0.382812 -0.394531 0.28125 -0.648438 0.28125 -0.953125 C 0.28125 -1.296875 0.382812 -1.5625 0.59375 -1.75 C 0.8125 -1.9375 1.09375 -2.050781 1.4375 -2.09375 Z M 1.90625 -3.75 Z M 1.90625 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='35cf7a23-9662-4e0c-a07d-31d81d3837b5' overflow='visible'>\n<path d='M 0.453125 -3.640625 L 1.078125 -3.640625 L 1.078125 0 L 0.453125 0 Z M 0.453125 -5.015625 L 1.078125 -5.015625 L 1.078125 -4.328125 L 0.453125 -4.328125 Z M 0.453125 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='b17ed1ec-94d3-47cf-b22b-5ab046c81c9d' overflow='visible'>\n<path d='' style='stroke:none;'/>\n</symbol>\n<symbol id='28fdeedd-93fd-450e-85f9-5b8ed888377e' overflow='visible'>\n<path d='M 0.9375 -5.015625 L 2.375 -0.75 L 3.796875 -5.015625 L 4.5625 -5.015625 L 2.734375 0 L 2 0 L 0.171875 -5.015625 Z M 0.9375 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='7377efc0-f320-4a81-b375-7dbb2d1722b7' overflow='visible'>\n<path d='M 1.96875 -3.75 C 2.226562 -3.75 2.476562 -3.6875 2.71875 -3.5625 C 2.96875 -3.4375 3.15625 -3.28125 3.28125 -3.09375 C 3.40625 -2.90625 3.488281 -2.6875 3.53125 -2.4375 C 3.570312 -2.269531 3.59375 -2.003906 3.59375 -1.640625 L 0.90625 -1.640625 C 0.914062 -1.265625 1.003906 -0.960938 1.171875 -0.734375 C 1.335938 -0.515625 1.59375 -0.40625 1.9375 -0.40625 C 2.25 -0.40625 2.503906 -0.507812 2.703125 -0.71875 C 2.804688 -0.84375 2.882812 -0.988281 2.9375 -1.15625 L 3.53125 -1.15625 C 3.519531 -1.019531 3.46875 -0.867188 3.375 -0.703125 C 3.289062 -0.535156 3.191406 -0.398438 3.078125 -0.296875 C 2.890625 -0.109375 2.65625 0.015625 2.375 0.078125 C 2.226562 0.109375 2.0625 0.125 1.875 0.125 C 1.425781 0.125 1.039062 -0.0351562 0.71875 -0.359375 C 0.40625 -0.691406 0.25 -1.160156 0.25 -1.765625 C 0.25 -2.347656 0.40625 -2.820312 0.71875 -3.1875 C 1.039062 -3.5625 1.457031 -3.75 1.96875 -3.75 Z M 2.953125 -2.125 C 2.929688 -2.382812 2.875 -2.597656 2.78125 -2.765625 C 2.613281 -3.054688 2.332031 -3.203125 1.9375 -3.203125 C 1.65625 -3.203125 1.414062 -3.097656 1.21875 -2.890625 C 1.03125 -2.691406 0.929688 -2.4375 0.921875 -2.125 Z M 1.921875 -3.75 Z M 1.921875 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='6e83c428-661c-4ed7-b5c2-dc49c74df737' overflow='visible'>\n<path d='M 0.734375 -3.65625 L 1.4375 -0.78125 L 2.15625 -3.65625 L 2.84375 -3.65625 L 3.5625 -0.796875 L 4.3125 -3.65625 L 4.921875 -3.65625 L 3.859375 0 L 3.21875 0 L 2.484375 -2.828125 L 1.75 0 L 1.125 0 L 0.0625 -3.65625 Z M 0.734375 -3.65625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='d57c37e9-68d9-42fe-b75d-1690fb90ed20' overflow='visible'>\n<path d='M 0.59375 -5.015625 L 2.859375 -5.015625 C 3.304688 -5.015625 3.664062 -4.890625 3.9375 -4.640625 C 4.207031 -4.390625 4.34375 -4.035156 4.34375 -3.578125 C 4.34375 -3.191406 4.21875 -2.851562 3.96875 -2.5625 C 3.726562 -2.269531 3.359375 -2.125 2.859375 -2.125 L 1.28125 -2.125 L 1.28125 0 L 0.59375 0 Z M 3.65625 -3.578125 C 3.65625 -3.941406 3.519531 -4.191406 3.25 -4.328125 C 3.101562 -4.398438 2.894531 -4.4375 2.625 -4.4375 L 1.28125 -4.4375 L 1.28125 -2.703125 L 2.625 -2.703125 C 2.9375 -2.703125 3.1875 -2.765625 3.375 -2.890625 C 3.5625 -3.023438 3.65625 -3.253906 3.65625 -3.578125 Z M 3.65625 -3.578125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='bf4e80c0-b751-4261-b946-62b857b4b92b' overflow='visible'>\n<path d='M 0.46875 -5.015625 L 1.078125 -5.015625 L 1.078125 0 L 0.46875 0 Z M 0.46875 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='ab2210c4-ff5e-4f46-b159-cd56ac6cfcb9' overflow='visible'>\n<path d='M 3.109375 -2.0625 L 2.34375 -4.28125 L 1.53125 -2.0625 Z M 2 -5.015625 L 2.765625 -5.015625 L 4.578125 0 L 3.84375 0 L 3.328125 -1.5 L 1.34375 -1.5 L 0.796875 0 L 0.109375 0 Z M 2.34375 -5.015625 Z M 2.34375 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='57c98abf-9ca3-4d51-b779-25ac929d5a61' overflow='visible'>\n<path d='M 2.875 -2.71875 C 3.1875 -2.71875 3.4375 -2.78125 3.625 -2.90625 C 3.8125 -3.039062 3.90625 -3.273438 3.90625 -3.609375 C 3.90625 -3.960938 3.773438 -4.203125 3.515625 -4.328125 C 3.378906 -4.398438 3.191406 -4.4375 2.953125 -4.4375 L 1.296875 -4.4375 L 1.296875 -2.71875 Z M 0.609375 -5.015625 L 2.9375 -5.015625 C 3.320312 -5.015625 3.640625 -4.960938 3.890625 -4.859375 C 4.359375 -4.640625 4.59375 -4.242188 4.59375 -3.671875 C 4.59375 -3.367188 4.53125 -3.117188 4.40625 -2.921875 C 4.28125 -2.734375 4.109375 -2.582031 3.890625 -2.46875 C 4.085938 -2.382812 4.234375 -2.273438 4.328125 -2.140625 C 4.429688 -2.015625 4.488281 -1.804688 4.5 -1.515625 L 4.515625 -0.84375 C 4.523438 -0.65625 4.539062 -0.515625 4.5625 -0.421875 C 4.601562 -0.265625 4.675781 -0.160156 4.78125 -0.109375 L 4.78125 0 L 3.953125 0 C 3.929688 -0.0390625 3.910156 -0.09375 3.890625 -0.15625 C 3.878906 -0.226562 3.867188 -0.363281 3.859375 -0.5625 L 3.828125 -1.40625 C 3.804688 -1.726562 3.679688 -1.945312 3.453125 -2.0625 C 3.328125 -2.125 3.125 -2.15625 2.84375 -2.15625 L 1.296875 -2.15625 L 1.296875 0 L 0.609375 0 Z M 0.609375 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='962ed40d-5823-47b8-9118-8e3a00d75f24' overflow='visible'>\n<path d='M 0.84375 -1.78125 C 0.84375 -1.394531 0.925781 -1.066406 1.09375 -0.796875 C 1.257812 -0.535156 1.523438 -0.40625 1.890625 -0.40625 C 2.171875 -0.40625 2.40625 -0.523438 2.59375 -0.765625 C 2.78125 -1.015625 2.875 -1.367188 2.875 -1.828125 C 2.875 -2.285156 2.773438 -2.625 2.578125 -2.84375 C 2.390625 -3.070312 2.160156 -3.1875 1.890625 -3.1875 C 1.578125 -3.1875 1.320312 -3.066406 1.125 -2.828125 C 0.9375 -2.597656 0.84375 -2.25 0.84375 -1.78125 Z M 1.765625 -3.71875 C 2.046875 -3.71875 2.28125 -3.660156 2.46875 -3.546875 C 2.582031 -3.484375 2.707031 -3.363281 2.84375 -3.1875 L 2.84375 -5.03125 L 3.4375 -5.03125 L 3.4375 0 L 2.875 0 L 2.875 -0.515625 C 2.738281 -0.285156 2.570312 -0.117188 2.375 -0.015625 C 2.175781 0.078125 1.953125 0.125 1.703125 0.125 C 1.296875 0.125 0.941406 -0.046875 0.640625 -0.390625 C 0.335938 -0.734375 0.1875 -1.1875 0.1875 -1.75 C 0.1875 -2.28125 0.320312 -2.738281 0.59375 -3.125 C 0.875 -3.519531 1.265625 -3.71875 1.765625 -3.71875 Z M 1.765625 -3.71875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='333fb815-388c-49b8-a335-bf92e586e41c' overflow='visible'>\n<path d='M 2.65625 -5.15625 C 3.289062 -5.15625 3.78125 -4.988281 4.125 -4.65625 C 4.476562 -4.320312 4.675781 -3.941406 4.71875 -3.515625 L 4.046875 -3.515625 C 3.972656 -3.835938 3.820312 -4.09375 3.59375 -4.28125 C 3.375 -4.46875 3.0625 -4.5625 2.65625 -4.5625 C 2.164062 -4.5625 1.765625 -4.390625 1.453125 -4.046875 C 1.148438 -3.703125 1 -3.171875 1 -2.453125 C 1 -1.859375 1.132812 -1.378906 1.40625 -1.015625 C 1.6875 -0.648438 2.097656 -0.46875 2.640625 -0.46875 C 3.140625 -0.46875 3.519531 -0.660156 3.78125 -1.046875 C 3.925781 -1.242188 4.03125 -1.507812 4.09375 -1.84375 L 4.765625 -1.84375 C 4.703125 -1.3125 4.503906 -0.867188 4.171875 -0.515625 C 3.773438 -0.0859375 3.238281 0.125 2.5625 0.125 C 1.976562 0.125 1.488281 -0.0507812 1.09375 -0.40625 C 0.570312 -0.863281 0.3125 -1.582031 0.3125 -2.5625 C 0.3125 -3.300781 0.503906 -3.910156 0.890625 -4.390625 C 1.316406 -4.898438 1.90625 -5.15625 2.65625 -5.15625 Z M 2.515625 -5.15625 Z M 2.515625 -5.15625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='13f515cf-a0e2-4a99-b590-6eeef27a8bdb' overflow='visible'>\n<path d='M 2.734375 -3.65625 L 3.421875 -3.65625 C 3.328125 -3.425781 3.132812 -2.890625 2.84375 -2.046875 C 2.613281 -1.421875 2.425781 -0.910156 2.28125 -0.515625 C 1.925781 0.421875 1.675781 0.988281 1.53125 1.1875 C 1.382812 1.394531 1.132812 1.5 0.78125 1.5 C 0.6875 1.5 0.613281 1.492188 0.5625 1.484375 C 0.519531 1.484375 0.46875 1.472656 0.40625 1.453125 L 0.40625 0.890625 C 0.507812 0.921875 0.582031 0.9375 0.625 0.9375 C 0.675781 0.945312 0.722656 0.953125 0.765625 0.953125 C 0.878906 0.953125 0.960938 0.929688 1.015625 0.890625 C 1.066406 0.859375 1.113281 0.816406 1.15625 0.765625 C 1.164062 0.742188 1.203125 0.648438 1.265625 0.484375 C 1.335938 0.316406 1.390625 0.191406 1.421875 0.109375 L 0.078125 -3.65625 L 0.765625 -3.65625 L 1.75 -0.6875 Z M 1.75 -3.75 Z M 1.75 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='468bcb4c-1b02-4b33-97a0-4afc199daf21' overflow='visible'>\n<path d='M 0.984375 -1.625 C 0.992188 -1.332031 1.054688 -1.097656 1.171875 -0.921875 C 1.398438 -0.585938 1.800781 -0.421875 2.375 -0.421875 C 2.625 -0.421875 2.851562 -0.457031 3.0625 -0.53125 C 3.46875 -0.675781 3.671875 -0.929688 3.671875 -1.296875 C 3.671875 -1.566406 3.585938 -1.757812 3.421875 -1.875 C 3.242188 -1.988281 2.972656 -2.085938 2.609375 -2.171875 L 1.921875 -2.328125 C 1.484375 -2.429688 1.171875 -2.539062 0.984375 -2.65625 C 0.660156 -2.875 0.5 -3.191406 0.5 -3.609375 C 0.5 -4.054688 0.65625 -4.421875 0.96875 -4.703125 C 1.28125 -4.992188 1.722656 -5.140625 2.296875 -5.140625 C 2.828125 -5.140625 3.273438 -5.015625 3.640625 -4.765625 C 4.003906 -4.515625 4.1875 -4.109375 4.1875 -3.546875 L 3.5625 -3.546875 C 3.519531 -3.816406 3.445312 -4.023438 3.34375 -4.171875 C 3.132812 -4.429688 2.78125 -4.5625 2.28125 -4.5625 C 1.875 -4.5625 1.582031 -4.472656 1.40625 -4.296875 C 1.238281 -4.128906 1.15625 -3.9375 1.15625 -3.71875 C 1.15625 -3.46875 1.253906 -3.285156 1.453125 -3.171875 C 1.585938 -3.097656 1.894531 -3.007812 2.375 -2.90625 L 3.078125 -2.75 C 3.410156 -2.664062 3.671875 -2.554688 3.859375 -2.421875 C 4.179688 -2.191406 4.34375 -1.851562 4.34375 -1.40625 C 4.34375 -0.851562 4.140625 -0.457031 3.734375 -0.21875 C 3.328125 0.0195312 2.859375 0.140625 2.328125 0.140625 C 1.703125 0.140625 1.210938 -0.015625 0.859375 -0.328125 C 0.503906 -0.648438 0.332031 -1.082031 0.34375 -1.625 Z M 2.359375 -5.15625 Z M 2.359375 -5.15625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='107514fc-f8f1-41ad-9dd0-179805ece07a' overflow='visible'>\n<path d='M 0.59375 -5.015625 L 4.078125 -5.015625 L 4.078125 -4.40625 L 1.28125 -4.40625 L 1.28125 -2.875 L 3.75 -2.875 L 3.75 -2.28125 L 1.28125 -2.28125 L 1.28125 0 L 0.59375 0 Z M 0.59375 -5.015625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='973d2e25-aae7-4f81-944b-308b40c177cd' overflow='visible'>\n<path d='M 0.46875 -3.65625 L 1.046875 -3.65625 L 1.046875 -3.03125 C 1.097656 -3.15625 1.21875 -3.304688 1.40625 -3.484375 C 1.59375 -3.660156 1.804688 -3.75 2.046875 -3.75 C 2.054688 -3.75 2.078125 -3.742188 2.109375 -3.734375 C 2.140625 -3.734375 2.1875 -3.726562 2.25 -3.71875 L 2.25 -3.078125 C 2.207031 -3.085938 2.171875 -3.09375 2.140625 -3.09375 C 2.117188 -3.09375 2.085938 -3.09375 2.046875 -3.09375 C 1.734375 -3.09375 1.492188 -2.992188 1.328125 -2.796875 C 1.160156 -2.597656 1.078125 -2.367188 1.078125 -2.109375 L 1.078125 0 L 0.46875 0 Z M 0.46875 -3.65625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='ef51ada5-2d4c-462c-b6e5-5afc24314459' overflow='visible'>\n<path d='M 1.859375 -3.765625 C 2.273438 -3.765625 2.609375 -3.664062 2.859375 -3.46875 C 3.117188 -3.269531 3.28125 -2.925781 3.34375 -2.4375 L 2.734375 -2.4375 C 2.703125 -2.65625 2.617188 -2.835938 2.484375 -2.984375 C 2.359375 -3.140625 2.148438 -3.21875 1.859375 -3.21875 C 1.472656 -3.21875 1.191406 -3.023438 1.015625 -2.640625 C 0.910156 -2.390625 0.859375 -2.082031 0.859375 -1.71875 C 0.859375 -1.351562 0.9375 -1.046875 1.09375 -0.796875 C 1.25 -0.546875 1.492188 -0.421875 1.828125 -0.421875 C 2.078125 -0.421875 2.273438 -0.5 2.421875 -0.65625 C 2.578125 -0.8125 2.679688 -1.023438 2.734375 -1.296875 L 3.34375 -1.296875 C 3.269531 -0.804688 3.09375 -0.445312 2.8125 -0.21875 C 2.539062 0 2.195312 0.109375 1.78125 0.109375 C 1.300781 0.109375 0.914062 -0.0625 0.625 -0.40625 C 0.34375 -0.757812 0.203125 -1.203125 0.203125 -1.734375 C 0.203125 -2.367188 0.359375 -2.863281 0.671875 -3.21875 C 0.984375 -3.582031 1.378906 -3.765625 1.859375 -3.765625 Z M 1.765625 -3.75 Z M 1.765625 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='a45cb9db-1d84-4d5b-8eda-6b0b6811481c' overflow='visible'>\n<path d='M 0.8125 -1.15625 C 0.832031 -0.945312 0.882812 -0.785156 0.96875 -0.671875 C 1.125 -0.472656 1.394531 -0.375 1.78125 -0.375 C 2.007812 -0.375 2.207031 -0.421875 2.375 -0.515625 C 2.550781 -0.617188 2.640625 -0.773438 2.640625 -0.984375 C 2.640625 -1.140625 2.570312 -1.257812 2.4375 -1.34375 C 2.34375 -1.394531 2.164062 -1.453125 1.90625 -1.515625 L 1.421875 -1.640625 C 1.109375 -1.722656 0.875 -1.8125 0.71875 -1.90625 C 0.457031 -2.070312 0.328125 -2.300781 0.328125 -2.59375 C 0.328125 -2.9375 0.453125 -3.210938 0.703125 -3.421875 C 0.953125 -3.640625 1.285156 -3.75 1.703125 -3.75 C 2.253906 -3.75 2.65625 -3.585938 2.90625 -3.265625 C 3.050781 -3.066406 3.125 -2.847656 3.125 -2.609375 L 2.546875 -2.609375 C 2.535156 -2.742188 2.484375 -2.867188 2.390625 -2.984375 C 2.253906 -3.148438 2.007812 -3.234375 1.65625 -3.234375 C 1.425781 -3.234375 1.25 -3.1875 1.125 -3.09375 C 1.007812 -3.007812 0.953125 -2.894531 0.953125 -2.75 C 0.953125 -2.582031 1.03125 -2.453125 1.1875 -2.359375 C 1.28125 -2.304688 1.414062 -2.257812 1.59375 -2.21875 L 2 -2.109375 C 2.4375 -2.003906 2.734375 -1.898438 2.890625 -1.796875 C 3.128906 -1.640625 3.25 -1.394531 3.25 -1.0625 C 3.25 -0.738281 3.125 -0.457031 2.875 -0.21875 C 2.625 0.0195312 2.25 0.140625 1.75 0.140625 C 1.207031 0.140625 0.820312 0.015625 0.59375 -0.234375 C 0.363281 -0.484375 0.238281 -0.789062 0.21875 -1.15625 Z M 1.71875 -3.75 Z M 1.71875 -3.75 ' style='stroke:none;'/>\n</symbol>\n<symbol id='4d758035-8258-47ea-9963-021105405aaa' overflow='visible'>\n<path d='M 3 -1.5 C 3 -1.070312 2.9375 -0.742188 2.8125 -0.515625 C 2.582031 -0.078125 2.140625 0.140625 1.484375 0.140625 C 1.109375 0.140625 0.785156 0.0351562 0.515625 -0.171875 C 0.253906 -0.378906 0.125 -0.742188 0.125 -1.265625 L 0.125 -1.625 L 0.765625 -1.625 L 0.765625 -1.265625 C 0.765625 -0.984375 0.820312 -0.773438 0.9375 -0.640625 C 1.0625 -0.503906 1.253906 -0.4375 1.515625 -0.4375 C 1.867188 -0.4375 2.101562 -0.5625 2.21875 -0.8125 C 2.289062 -0.96875 2.328125 -1.253906 2.328125 -1.671875 L 2.328125 -5.015625 L 3 -5.015625 Z M 3 -1.5 ' style='stroke:none;'/>\n</symbol>\n<symbol id='6434afa0-0184-408e-a62b-8987c6602361' overflow='visible'>\n<path d='M 0.265625 0 L 0.265625 -5.734375 L 4.8125 -5.734375 L 4.8125 0 Z M 4.09375 -0.71875 L 4.09375 -5.015625 L 0.984375 -5.015625 L 0.984375 -0.71875 Z M 4.09375 -0.71875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='7516e1e3-0773-4027-9cbf-8840ca81fcca' overflow='visible'>\n<path d='M 2.125 -4.3125 C 2.601562 -4.3125 2.988281 -4.195312 3.28125 -3.96875 C 3.570312 -3.738281 3.75 -3.34375 3.8125 -2.78125 L 3.125 -2.78125 C 3.082031 -3.039062 2.988281 -3.253906 2.84375 -3.421875 C 2.695312 -3.585938 2.457031 -3.671875 2.125 -3.671875 C 1.675781 -3.671875 1.359375 -3.453125 1.171875 -3.015625 C 1.046875 -2.734375 0.984375 -2.382812 0.984375 -1.96875 C 0.984375 -1.550781 1.070312 -1.195312 1.25 -0.90625 C 1.425781 -0.625 1.703125 -0.484375 2.078125 -0.484375 C 2.367188 -0.484375 2.597656 -0.570312 2.765625 -0.75 C 2.941406 -0.925781 3.0625 -1.171875 3.125 -1.484375 L 3.8125 -1.484375 C 3.738281 -0.929688 3.546875 -0.523438 3.234375 -0.265625 C 2.921875 -0.00390625 2.519531 0.125 2.03125 0.125 C 1.476562 0.125 1.039062 -0.0703125 0.71875 -0.46875 C 0.394531 -0.875 0.234375 -1.378906 0.234375 -1.984375 C 0.234375 -2.710938 0.410156 -3.28125 0.765625 -3.6875 C 1.117188 -4.101562 1.570312 -4.3125 2.125 -4.3125 Z M 2.015625 -4.28125 Z M 2.015625 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='f79db7a0-3bc8-4275-ac50-7e0487c76514' overflow='visible'>\n<path d='M 0.515625 -4.171875 L 1.234375 -4.171875 L 1.234375 0 L 0.515625 0 Z M 0.515625 -5.734375 L 1.234375 -5.734375 L 1.234375 -4.9375 L 0.515625 -4.9375 Z M 0.515625 -5.734375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='5d54b0e8-a19f-4938-be0d-0bb89bde688f' overflow='visible'>\n<path d='M 0.65625 -5.359375 L 1.375 -5.359375 L 1.375 -4.1875 L 2.03125 -4.1875 L 2.03125 -3.609375 L 1.375 -3.609375 L 1.375 -0.875 C 1.375 -0.726562 1.421875 -0.632812 1.515625 -0.59375 C 1.566406 -0.5625 1.660156 -0.546875 1.796875 -0.546875 C 1.828125 -0.546875 1.859375 -0.546875 1.890625 -0.546875 C 1.929688 -0.546875 1.976562 -0.550781 2.03125 -0.5625 L 2.03125 0 C 1.957031 0.0195312 1.875 0.0351562 1.78125 0.046875 C 1.695312 0.0546875 1.601562 0.0625 1.5 0.0625 C 1.175781 0.0625 0.953125 -0.0195312 0.828125 -0.1875 C 0.710938 -0.351562 0.65625 -0.570312 0.65625 -0.84375 L 0.65625 -3.609375 L 0.09375 -3.609375 L 0.09375 -4.1875 L 0.65625 -4.1875 Z M 0.65625 -5.359375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='d50cb9c8-d65b-4198-9590-3eb99a5be5ad' overflow='visible'>\n<path d='M 3.125 -4.1875 L 3.90625 -4.1875 C 3.8125 -3.914062 3.59375 -3.300781 3.25 -2.34375 C 2.988281 -1.625 2.773438 -1.039062 2.609375 -0.59375 C 2.203125 0.476562 1.914062 1.128906 1.75 1.359375 C 1.582031 1.597656 1.296875 1.71875 0.890625 1.71875 C 0.785156 1.71875 0.707031 1.710938 0.65625 1.703125 C 0.601562 1.691406 0.539062 1.675781 0.46875 1.65625 L 0.46875 1.015625 C 0.582031 1.054688 0.664062 1.078125 0.71875 1.078125 C 0.78125 1.085938 0.832031 1.09375 0.875 1.09375 C 1 1.09375 1.09375 1.070312 1.15625 1.03125 C 1.21875 0.988281 1.269531 0.9375 1.3125 0.875 C 1.320312 0.851562 1.367188 0.742188 1.453125 0.546875 C 1.535156 0.359375 1.59375 0.21875 1.625 0.125 L 0.078125 -4.1875 L 0.875 -4.1875 L 2 -0.78125 Z M 2 -4.28125 Z M 2 -4.28125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='689e9531-c18d-4c51-bc35-c0aabc979b3a' overflow='visible'>\n<path d='M 0 -0.265625 L -5.734375 -0.265625 L -5.734375 -4.8125 L 0 -4.8125 Z M -0.71875 -4.09375 L -5.015625 -4.09375 L -5.015625 -0.984375 L -0.71875 -0.984375 Z M -0.71875 -4.09375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='ff1f68c4-39bc-444f-a396-333178de5876' overflow='visible'>\n<path d='M -1.109375 -1.0625 C -0.910156 -1.0625 -0.75 -1.132812 -0.625 -1.28125 C -0.507812 -1.425781 -0.453125 -1.601562 -0.453125 -1.8125 C -0.453125 -2.050781 -0.507812 -2.289062 -0.625 -2.53125 C -0.820312 -2.914062 -1.132812 -3.109375 -1.5625 -3.109375 L -2.125 -3.109375 C -2.070312 -3.023438 -2.023438 -2.914062 -1.984375 -2.78125 C -1.953125 -2.644531 -1.929688 -2.507812 -1.921875 -2.375 L -1.859375 -1.953125 C -1.828125 -1.703125 -1.773438 -1.507812 -1.703125 -1.375 C -1.578125 -1.164062 -1.378906 -1.0625 -1.109375 -1.0625 Z M -2.53125 -2.765625 C -2.550781 -2.921875 -2.617188 -3.023438 -2.734375 -3.078125 C -2.796875 -3.109375 -2.890625 -3.125 -3.015625 -3.125 C -3.253906 -3.125 -3.425781 -3.039062 -3.53125 -2.875 C -3.632812 -2.707031 -3.6875 -2.460938 -3.6875 -2.140625 C -3.6875 -1.773438 -3.585938 -1.515625 -3.390625 -1.359375 C -3.285156 -1.273438 -3.125 -1.21875 -2.90625 -1.1875 L -2.90625 -0.53125 C -3.425781 -0.550781 -3.785156 -0.71875 -3.984375 -1.03125 C -4.179688 -1.351562 -4.28125 -1.726562 -4.28125 -2.15625 C -4.28125 -2.644531 -4.1875 -3.046875 -4 -3.359375 C -3.8125 -3.660156 -3.519531 -3.8125 -3.125 -3.8125 L -0.71875 -3.8125 C -0.644531 -3.8125 -0.585938 -3.828125 -0.546875 -3.859375 C -0.503906 -3.890625 -0.484375 -3.953125 -0.484375 -4.046875 C -0.484375 -4.078125 -0.484375 -4.113281 -0.484375 -4.15625 C -0.484375 -4.195312 -0.488281 -4.238281 -0.5 -4.28125 L 0.015625 -4.28125 C 0.046875 -4.164062 0.0664062 -4.078125 0.078125 -4.015625 C 0.0859375 -3.960938 0.09375 -3.890625 0.09375 -3.796875 C 0.09375 -3.554688 0.00390625 -3.378906 -0.171875 -3.265625 C -0.265625 -3.210938 -0.394531 -3.171875 -0.5625 -3.140625 C -0.375 -3.003906 -0.207031 -2.800781 -0.0625 -2.53125 C 0.0703125 -2.257812 0.140625 -1.960938 0.140625 -1.640625 C 0.140625 -1.253906 0.0234375 -0.9375 -0.203125 -0.6875 C -0.441406 -0.445312 -0.738281 -0.328125 -1.09375 -0.328125 C -1.488281 -0.328125 -1.789062 -0.445312 -2 -0.6875 C -2.21875 -0.925781 -2.347656 -1.242188 -2.390625 -1.640625 Z M -4.28125 -2.171875 Z M -4.28125 -2.171875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='88021c7e-29be-4aeb-88d5-648e75d69dfc' overflow='visible'>\n<path d='M -4.1875 -0.859375 L -0.78125 -1.984375 L -4.1875 -3.140625 L -4.1875 -3.921875 L 0 -2.34375 L 0 -1.59375 L -4.1875 -0.046875 Z M -4.1875 -0.859375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='b350b481-9749-4761-878f-584ccbab6926' overflow='visible'>\n<path d='M -4.171875 -0.515625 L -4.171875 -1.234375 L 0 -1.234375 L 0 -0.515625 Z M -5.734375 -0.515625 L -5.734375 -1.234375 L -4.9375 -1.234375 L -4.9375 -0.515625 Z M -5.734375 -0.515625 ' style='stroke:none;'/>\n</symbol>\n<symbol id='f2766d9b-3744-4107-b37d-adc38ed36a27' overflow='visible'>\n<path d='M -5.734375 -0.53125 L -5.734375 -1.234375 L 0 -1.234375 L 0 -0.53125 Z M -5.734375 -0.53125 ' style='stroke:none;'/>\n</symbol>\n<symbol id='de80a8aa-b936-4dee-98db-9b49a47fd1ac' overflow='visible'>\n<path d='M -5.765625 -0.46875 L -5.765625 -1.140625 L -3.671875 -1.140625 C -3.878906 -1.296875 -4.035156 -1.476562 -4.140625 -1.6875 C -4.242188 -1.90625 -4.296875 -2.140625 -4.296875 -2.390625 C -4.296875 -2.910156 -4.113281 -3.332031 -3.75 -3.65625 C -3.394531 -3.976562 -2.867188 -4.140625 -2.171875 -4.140625 C -1.515625 -4.140625 -0.960938 -3.976562 -0.515625 -3.65625 C -0.078125 -3.34375 0.140625 -2.898438 0.140625 -2.328125 C 0.140625 -2.015625 0.0625 -1.75 -0.09375 -1.53125 C -0.1875 -1.394531 -0.332031 -1.253906 -0.53125 -1.109375 L 0 -1.109375 L 0 -0.46875 Z M -0.484375 -2.296875 C -0.484375 -2.671875 -0.632812 -2.953125 -0.9375 -3.140625 C -1.238281 -3.328125 -1.640625 -3.421875 -2.140625 -3.421875 C -2.578125 -3.421875 -2.941406 -3.328125 -3.234375 -3.140625 C -3.523438 -2.953125 -3.671875 -2.675781 -3.671875 -2.3125 C -3.671875 -1.988281 -3.550781 -1.707031 -3.3125 -1.46875 C -3.070312 -1.226562 -2.679688 -1.109375 -2.140625 -1.109375 C -1.742188 -1.109375 -1.425781 -1.15625 -1.1875 -1.25 C -0.71875 -1.4375 -0.484375 -1.785156 -0.484375 -2.296875 Z M -0.484375 -2.296875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='52bf7b8f-8b0b-4b9e-88c1-946e4d038a23' overflow='visible'>\n<path d='M -4.28125 -2.265625 C -4.28125 -2.554688 -4.207031 -2.84375 -4.0625 -3.125 C -3.925781 -3.40625 -3.75 -3.617188 -3.53125 -3.765625 C -3.320312 -3.898438 -3.078125 -3.992188 -2.796875 -4.046875 C -2.597656 -4.085938 -2.289062 -4.109375 -1.875 -4.109375 L -1.875 -1.03125 C -1.445312 -1.050781 -1.101562 -1.15625 -0.84375 -1.34375 C -0.59375 -1.53125 -0.46875 -1.816406 -0.46875 -2.203125 C -0.46875 -2.578125 -0.585938 -2.875 -0.828125 -3.09375 C -0.960938 -3.21875 -1.125 -3.304688 -1.3125 -3.359375 L -1.3125 -4.046875 C -1.15625 -4.023438 -0.984375 -3.960938 -0.796875 -3.859375 C -0.609375 -3.753906 -0.457031 -3.640625 -0.34375 -3.515625 C -0.132812 -3.304688 0.0078125 -3.039062 0.09375 -2.71875 C 0.132812 -2.550781 0.15625 -2.363281 0.15625 -2.15625 C 0.15625 -1.632812 -0.03125 -1.191406 -0.40625 -0.828125 C -0.789062 -0.460938 -1.328125 -0.28125 -2.015625 -0.28125 C -2.679688 -0.28125 -3.222656 -0.460938 -3.640625 -0.828125 C -4.066406 -1.191406 -4.28125 -1.671875 -4.28125 -2.265625 Z M -2.421875 -3.375 C -2.734375 -3.351562 -2.976562 -3.289062 -3.15625 -3.1875 C -3.5 -2.988281 -3.671875 -2.664062 -3.671875 -2.21875 C -3.671875 -1.894531 -3.550781 -1.625 -3.3125 -1.40625 C -3.082031 -1.1875 -2.785156 -1.070312 -2.421875 -1.0625 Z M -4.28125 -2.1875 Z M -4.28125 -2.1875 ' style='stroke:none;'/>\n</symbol>\n<symbol id='b11abd74-2bcb-43a4-b32c-a0060ba6f7c2' overflow='visible'>\n<path d='M -3.3125 -2.765625 C -3.3125 -3.097656 -3.359375 -3.351562 -3.453125 -3.53125 C -3.597656 -3.820312 -3.851562 -3.96875 -4.21875 -3.96875 C -4.59375 -3.96875 -4.847656 -3.8125 -4.984375 -3.5 C -5.054688 -3.332031 -5.09375 -3.078125 -5.09375 -2.734375 L -5.09375 -1.359375 L -3.3125 -1.359375 Z M -0.671875 -3.03125 C -0.671875 -3.507812 -0.804688 -3.847656 -1.078125 -4.046875 C -1.253906 -4.171875 -1.46875 -4.234375 -1.71875 -4.234375 C -2.132812 -4.234375 -2.414062 -4.050781 -2.5625 -3.6875 C -2.644531 -3.488281 -2.6875 -3.222656 -2.6875 -2.890625 L -2.6875 -1.359375 L -0.671875 -1.359375 Z M -5.734375 -0.59375 L -5.734375 -3.0625 C -5.734375 -3.726562 -5.535156 -4.203125 -5.140625 -4.484375 C -4.898438 -4.648438 -4.625 -4.734375 -4.3125 -4.734375 C -3.957031 -4.734375 -3.660156 -4.632812 -3.421875 -4.4375 C -3.304688 -4.320312 -3.195312 -4.164062 -3.09375 -3.96875 C -2.976562 -4.257812 -2.851562 -4.476562 -2.71875 -4.625 C -2.46875 -4.882812 -2.117188 -5.015625 -1.671875 -5.015625 C -1.304688 -5.015625 -0.972656 -4.898438 -0.671875 -4.671875 C -0.222656 -4.328125 0 -3.773438 0 -3.015625 L 0 -0.59375 Z M -5.734375 -0.59375 ' style='stroke:none;'/>\n</symbol>\n<symbol id='100d437d-b9c4-402d-aa6e-40eedb4ed9b1' overflow='visible'>\n<path d='M -5.734375 -0.5 L -5.734375 -1.171875 L -2.40625 -1.171875 L -4.1875 -2.984375 L -4.1875 -3.875 L -2.625 -2.28125 L 0 -3.96875 L 0 -3.078125 L -2.109375 -1.765625 L -1.578125 -1.171875 L 0 -1.171875 L 0 -0.5 Z M -5.734375 -0.5 ' style='stroke:none;'/>\n</symbol>\n<symbol id='ff8feb97-63d5-4dd4-a6ab-5f55725821be' overflow='visible'>\n<path d='M -1.3125 -0.9375 C -1.082031 -0.957031 -0.90625 -1.015625 -0.78125 -1.109375 C -0.550781 -1.285156 -0.4375 -1.59375 -0.4375 -2.03125 C -0.4375 -2.289062 -0.492188 -2.519531 -0.609375 -2.71875 C -0.722656 -2.914062 -0.894531 -3.015625 -1.125 -3.015625 C -1.3125 -3.015625 -1.453125 -2.9375 -1.546875 -2.78125 C -1.597656 -2.675781 -1.660156 -2.472656 -1.734375 -2.171875 L -1.875 -1.625 C -1.96875 -1.257812 -2.066406 -0.992188 -2.171875 -0.828125 C -2.367188 -0.523438 -2.632812 -0.375 -2.96875 -0.375 C -3.363281 -0.375 -3.679688 -0.515625 -3.921875 -0.796875 C -4.171875 -1.085938 -4.296875 -1.472656 -4.296875 -1.953125 C -4.296875 -2.585938 -4.109375 -3.039062 -3.734375 -3.3125 C -3.503906 -3.488281 -3.253906 -3.578125 -2.984375 -3.578125 L -2.984375 -2.90625 C -3.140625 -2.894531 -3.285156 -2.835938 -3.421875 -2.734375 C -3.597656 -2.578125 -3.6875 -2.300781 -3.6875 -1.90625 C -3.6875 -1.632812 -3.632812 -1.429688 -3.53125 -1.296875 C -3.4375 -1.160156 -3.304688 -1.09375 -3.140625 -1.09375 C -2.953125 -1.09375 -2.804688 -1.179688 -2.703125 -1.359375 C -2.640625 -1.460938 -2.582031 -1.617188 -2.53125 -1.828125 L -2.421875 -2.28125 C -2.296875 -2.789062 -2.175781 -3.128906 -2.0625 -3.296875 C -1.882812 -3.578125 -1.601562 -3.71875 -1.21875 -3.71875 C -0.84375 -3.71875 -0.519531 -3.570312 -0.25 -3.28125 C 0.0195312 -3 0.15625 -2.570312 0.15625 -2 C 0.15625 -1.375 0.015625 -0.929688 -0.265625 -0.671875 C -0.546875 -0.410156 -0.894531 -0.273438 -1.3125 -0.265625 Z M -4.28125 -1.96875 Z M -4.28125 -1.96875 ' style='stroke:none;'/>\n</symbol>\n</g>\n<clipPath id='4a0916ed-40cf-48cb-96fd-e0dfb42e54bf'>\n<path d='M 32.855469 22.472656 L 643 22.472656 L 643 332 L 32.855469 332 Z M 32.855469 22.472656 '/>\n</clipPath>\n<clipPath id='bef663d4-e3a6-46d0-9a7d-9c5ae7fa2bd9'>\n<path d='M 32.855469 317 L 643 317 L 643 318 L 32.855469 318 Z M 32.855469 317 '/>\n</clipPath>\n<clipPath id='e3af9aa4-05bd-4832-b7f4-58bc2f28376d'>\n<path d='M 32.855469 243 L 643 243 L 643 244 L 32.855469 244 Z M 32.855469 243 '/>\n</clipPath>\n<clipPath id='837c7cb2-028f-40fa-b1f2-86129d7188ca'>\n<path d='M 32.855469 169 L 643 169 L 643 170 L 32.855469 170 Z M 32.855469 169 '/>\n</clipPath>\n<clipPath id='e6676b8d-213d-4374-a29d-5a04cda5bcdd'>\n<path d='M 32.855469 95 L 643 95 L 643 96 L 32.855469 96 Z M 32.855469 95 '/>\n</clipPath>\n<clipPath id='9e3d55c1-ec51-485c-ba8c-5f28e3ec9c26'>\n<path d='M 102 22.472656 L 104 22.472656 L 104 332 L 102 332 Z M 102 22.472656 '/>\n</clipPath>\n<clipPath id='1817ae4d-3ee9-491b-9a45-555abc516496'>\n<path d='M 220 22.472656 L 221 22.472656 L 221 332 L 220 332 Z M 220 22.472656 '/>\n</clipPath>\n<clipPath id='0fe2b30e-916a-4818-8fee-7dad9927137e'>\n<path d='M 337 22.472656 L 338 22.472656 L 338 332 L 337 332 Z M 337 22.472656 '/>\n</clipPath>\n<clipPath id='4eb4e9ef-d13a-4cd8-a97c-5a45aa1b5fef'>\n<path d='M 454 22.472656 L 456 22.472656 L 456 332 L 454 332 Z M 454 22.472656 '/>\n</clipPath>\n<clipPath id='b8535d9e-0a0d-49b3-8094-db5dd98a8fd9'>\n<path d='M 571 22.472656 L 573 22.472656 L 573 332 L 571 332 Z M 571 22.472656 '/>\n</clipPath>\n<clipPath id='c7d6984c-89b9-4f16-84c8-3e4bceaebdff'>\n<path d='M 32.855469 22.472656 L 643 22.472656 L 643 332 L 32.855469 332 Z M 32.855469 22.472656 '/>\n</clipPath>\n</defs>\n<g id='7c19f147-e861-4f2f-9e60-17d5d2a72513'>\n<rect style='fill:rgb(100%,100%,100%);fill-opacity:1;stroke:none;' height='360' width='648' y='0' x='0'/>\n<rect style='fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke:none;' height='360' width='648' y='0' x='0'/>\n<path d='M 0 360 L 648 360 L 648 0 L 0 0 Z M 0 360 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g clip-rule='nonzero' clip-path='url(#4a0916ed-40cf-48cb-96fd-e0dfb42e54bf)'>\n<path d='M 32.855469 331.332031 L 642.328125 331.332031 L 642.328125 22.472656 L 32.855469 22.472656 Z M 32.855469 331.332031 ' style='fill-rule:nonzero;fill:rgb(94.117647%,94.117647%,94.117647%);fill-opacity:1;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#bef663d4-e3a6-46d0-9a7d-9c5ae7fa2bd9)'>\n<path d='M 32.855469 317.292969 L 642.332031 317.292969 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#e3af9aa4-05bd-4832-b7f4-58bc2f28376d)'>\n<path d='M 32.855469 243.402344 L 642.332031 243.402344 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#837c7cb2-028f-40fa-b1f2-86129d7188ca)'>\n<path d='M 32.855469 169.511719 L 642.332031 169.511719 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#e6676b8d-213d-4374-a29d-5a04cda5bcdd)'>\n<path d='M 32.855469 95.625 L 642.332031 95.625 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#9e3d55c1-ec51-485c-ba8c-5f28e3ec9c26)'>\n<path d='M 103.179688 331.332031 L 103.179688 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#1817ae4d-3ee9-491b-9a45-555abc516496)'>\n<path d='M 220.386719 331.332031 L 220.386719 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#0fe2b30e-916a-4818-8fee-7dad9927137e)'>\n<path d='M 337.59375 331.332031 L 337.59375 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#4eb4e9ef-d13a-4cd8-a97c-5a45aa1b5fef)'>\n<path d='M 454.800781 331.332031 L 454.800781 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g clip-rule='nonzero' clip-path='url(#b8535d9e-0a0d-49b3-8094-db5dd98a8fd9)'>\n<path d='M 572.007812 331.332031 L 572.007812 22.472656 ' style='fill:none;stroke-width:0.533489;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(85.098039%,85.098039%,85.098039%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<path d='M 101.277344 287.738281 L 101.265625 287.417969 L 101.253906 287.101562 L 101.242188 286.78125 L 101.230469 286.464844 L 101.21875 286.144531 L 101.210938 285.828125 L 101.199219 285.507812 L 101.175781 284.875 L 101.164062 284.554688 L 101.15625 284.238281 L 101.144531 283.917969 L 101.132812 283.601562 L 101.121094 283.28125 L 101.109375 282.964844 L 101.101562 282.648438 L 101.089844 282.328125 L 101.078125 282.011719 L 101.066406 281.691406 L 101.054688 281.375 L 101.046875 281.054688 L 101.035156 280.738281 L 101.023438 280.417969 L 101 279.785156 L 100.992188 279.464844 L 100.980469 279.148438 L 100.96875 278.828125 L 100.957031 278.511719 L 100.949219 278.191406 L 100.925781 277.558594 L 100.914062 277.238281 L 100.90625 276.921875 L 100.894531 276.601562 L 100.882812 276.285156 L 100.871094 275.964844 L 100.863281 275.648438 L 100.851562 275.328125 L 100.839844 275.011719 L 100.832031 274.695312 L 100.820312 274.375 L 100.808594 274.058594 L 100.796875 273.738281 L 100.789062 273.421875 L 100.777344 273.101562 L 100.765625 272.785156 L 100.757812 272.46875 L 100.746094 272.148438 L 100.734375 271.832031 L 100.726562 271.511719 L 100.714844 271.195312 L 100.707031 270.875 L 100.695312 270.558594 L 100.683594 270.238281 L 100.675781 269.921875 L 100.664062 269.605469 L 100.65625 269.285156 L 100.644531 268.96875 L 100.632812 268.648438 L 100.625 268.332031 L 100.613281 268.011719 L 100.605469 267.695312 L 100.59375 267.378906 L 100.585938 267.058594 L 100.574219 266.742188 L 100.566406 266.421875 L 100.554688 266.105469 L 100.546875 265.785156 L 100.535156 265.46875 L 100.527344 265.148438 L 100.515625 264.832031 L 100.507812 264.515625 L 100.5 264.195312 L 100.488281 263.878906 L 100.480469 263.558594 L 100.46875 263.242188 L 100.460938 262.921875 L 100.453125 262.605469 L 100.441406 262.289062 L 100.433594 261.96875 L 100.425781 261.652344 L 100.417969 261.332031 L 100.40625 261.015625 L 100.398438 260.695312 L 100.390625 260.378906 L 100.382812 260.058594 L 100.371094 259.742188 L 100.363281 259.425781 L 100.355469 259.105469 L 100.347656 258.789062 L 100.339844 258.46875 L 100.332031 258.152344 L 100.320312 257.832031 L 100.304688 257.199219 L 100.296875 256.878906 L 100.289062 256.5625 L 100.28125 256.242188 L 100.273438 255.925781 L 100.265625 255.605469 L 100.257812 255.289062 L 100.25 254.96875 L 100.234375 254.335938 L 100.230469 254.015625 L 100.222656 253.699219 L 100.214844 253.378906 L 100.207031 253.0625 L 100.199219 252.742188 L 100.191406 252.425781 L 100.1875 252.109375 L 100.179688 251.789062 L 100.171875 251.472656 L 100.167969 251.152344 L 100.160156 250.835938 L 100.152344 250.515625 L 100.148438 250.199219 L 100.140625 249.878906 L 100.132812 249.5625 L 100.128906 249.246094 L 100.121094 248.925781 L 100.117188 248.609375 L 100.109375 248.289062 L 100.105469 247.972656 L 100.097656 247.652344 L 100.09375 247.335938 L 100.085938 247.019531 L 100.082031 246.699219 L 100.078125 246.382812 L 100.070312 246.0625 L 100.066406 245.746094 L 100.0625 245.425781 L 100.054688 245.109375 L 100.050781 244.789062 L 100.042969 244.15625 L 100.039062 243.835938 L 100.03125 243.519531 L 100.027344 243.199219 L 100.023438 242.882812 L 100.019531 242.5625 L 100.011719 241.929688 L 100.007812 241.609375 L 100.003906 241.292969 L 100 240.972656 L 99.996094 240.65625 L 99.992188 240.335938 L 99.988281 240.019531 L 99.988281 239.699219 L 99.980469 239.066406 L 99.976562 238.746094 L 99.976562 238.429688 L 99.972656 238.109375 L 99.96875 237.792969 L 99.964844 237.472656 L 99.964844 237.15625 L 99.960938 236.839844 L 99.960938 236.519531 L 99.957031 236.203125 L 99.957031 235.882812 L 99.953125 235.566406 L 99.953125 235.246094 L 99.949219 234.929688 L 99.949219 234.609375 L 99.945312 234.292969 L 99.945312 233.65625 L 99.941406 233.339844 L 99.941406 232.703125 L 99.9375 232.382812 L 99.9375 230.792969 L 99.933594 230.476562 L 99.933594 227.929688 L 99.9375 227.613281 L 99.9375 226.023438 L 99.941406 225.703125 L 99.941406 225.066406 L 99.945312 224.75 L 99.945312 224.113281 L 99.949219 223.796875 L 99.949219 223.476562 L 99.953125 223.160156 L 99.953125 222.523438 L 99.957031 222.203125 L 99.957031 221.886719 L 99.960938 221.570312 L 99.960938 221.25 L 99.964844 220.933594 L 99.96875 220.613281 L 99.96875 220.296875 L 99.972656 219.976562 L 99.972656 219.660156 L 99.976562 219.339844 L 99.980469 219.023438 L 99.980469 218.707031 L 99.984375 218.386719 L 99.988281 218.070312 L 99.992188 217.75 L 99.992188 217.433594 L 99.996094 217.113281 L 100.003906 216.480469 L 100.003906 216.160156 L 100.007812 215.84375 L 100.011719 215.523438 L 100.015625 215.207031 L 100.019531 214.886719 L 100.023438 214.570312 L 100.023438 214.25 L 100.03125 213.617188 L 100.035156 213.296875 L 100.039062 212.980469 L 100.042969 212.660156 L 100.046875 212.34375 L 100.050781 212.023438 L 100.058594 211.390625 L 100.0625 211.070312 L 100.066406 210.753906 L 100.070312 210.433594 L 100.074219 210.117188 L 100.078125 209.796875 L 100.082031 209.480469 L 100.085938 209.160156 L 100.09375 208.527344 L 100.097656 208.207031 L 100.101562 207.890625 L 100.105469 207.570312 L 100.109375 207.253906 L 100.113281 206.933594 L 100.121094 206.300781 L 100.125 205.980469 L 100.128906 205.664062 L 100.132812 205.34375 L 100.136719 205.027344 L 100.140625 204.707031 L 100.144531 204.390625 L 100.148438 204.070312 L 100.15625 203.4375 L 100.160156 203.117188 L 100.167969 202.800781 L 100.171875 202.480469 L 100.175781 202.164062 L 100.179688 201.84375 L 100.1875 201.210938 L 100.191406 200.890625 L 100.195312 200.574219 L 100.199219 200.253906 L 100.203125 199.9375 L 100.207031 199.617188 L 100.210938 199.300781 L 100.214844 198.980469 L 100.222656 198.347656 L 100.226562 198.027344 L 100.230469 197.710938 L 100.234375 197.390625 L 100.238281 197.074219 L 100.238281 196.753906 L 100.246094 196.121094 L 100.25 195.800781 L 100.253906 195.484375 L 100.257812 195.164062 L 100.261719 194.847656 L 100.265625 194.527344 L 100.269531 194.210938 L 100.269531 193.890625 L 100.277344 193.257812 L 100.28125 192.9375 L 100.285156 192.621094 L 100.285156 192.300781 L 100.289062 191.984375 L 100.292969 191.664062 L 100.296875 191.347656 L 100.296875 191.03125 L 100.300781 190.710938 L 100.304688 190.394531 L 100.308594 190.074219 L 100.308594 189.757812 L 100.3125 189.4375 L 100.316406 189.121094 L 100.316406 188.800781 L 100.320312 188.484375 L 100.320312 188.167969 L 100.324219 187.847656 L 100.328125 187.53125 L 100.328125 187.210938 L 100.332031 186.894531 L 100.332031 186.574219 L 100.335938 186.257812 L 100.335938 185.941406 L 100.339844 185.621094 L 100.339844 185.304688 L 100.34375 184.984375 L 100.34375 184.347656 L 100.347656 184.03125 L 100.347656 183.394531 L 100.351562 183.078125 L 100.351562 182.441406 L 100.355469 182.121094 L 100.355469 181.484375 L 100.359375 181.167969 L 100.359375 179.578125 L 100.363281 179.257812 L 100.363281 174.167969 L 100.359375 173.851562 L 100.359375 172.261719 L 100.355469 171.941406 L 100.355469 170.988281 L 100.351562 170.671875 L 100.351562 169.714844 L 100.347656 169.398438 L 100.347656 168.761719 L 100.34375 168.441406 L 100.34375 167.808594 L 100.339844 167.488281 L 100.339844 166.851562 L 100.335938 166.535156 L 100.335938 166.214844 L 100.332031 165.898438 L 100.332031 165.261719 L 100.328125 164.945312 L 100.328125 164.625 L 100.324219 164.308594 L 100.324219 163.988281 L 100.320312 163.671875 L 100.320312 163.351562 L 100.316406 163.035156 L 100.316406 162.398438 L 100.3125 162.082031 L 100.3125 161.761719 L 100.308594 161.445312 L 100.308594 161.125 L 100.304688 160.808594 L 100.304688 160.492188 L 100.300781 160.171875 L 100.300781 159.855469 L 100.296875 159.535156 L 100.296875 159.21875 L 100.292969 158.898438 L 100.292969 158.582031 L 100.289062 158.261719 L 100.289062 157.628906 L 100.285156 157.308594 L 100.285156 156.992188 L 100.28125 156.671875 L 100.28125 156.355469 L 100.277344 156.035156 L 100.277344 155.71875 L 100.273438 155.402344 L 100.273438 154.765625 L 100.269531 154.445312 L 100.269531 154.128906 L 100.265625 153.808594 L 100.265625 153.171875 L 100.261719 152.855469 L 100.261719 152.21875 L 100.257812 151.902344 L 100.257812 151.265625 L 100.253906 150.945312 L 100.253906 149.992188 L 100.25 149.675781 L 100.25 148.71875 L 100.246094 148.402344 L 100.246094 142.359375 L 100.25 142.039062 L 100.25 141.085938 L 100.253906 140.765625 L 100.253906 140.132812 L 100.257812 139.8125 L 100.257812 139.175781 L 100.261719 138.859375 L 100.261719 138.539062 L 100.265625 138.222656 L 100.265625 137.902344 L 100.273438 137.269531 L 100.273438 136.949219 L 100.277344 136.632812 L 100.277344 136.3125 L 100.28125 135.996094 L 100.285156 135.675781 L 100.289062 135.359375 L 100.289062 135.042969 L 100.292969 134.722656 L 100.296875 134.40625 L 100.300781 134.085938 L 100.304688 133.769531 L 100.308594 133.449219 L 100.308594 133.132812 L 100.3125 132.8125 L 100.320312 132.179688 L 100.328125 131.859375 L 100.332031 131.542969 L 100.335938 131.222656 L 100.339844 130.90625 L 100.34375 130.585938 L 100.351562 129.953125 L 100.359375 129.632812 L 100.363281 129.316406 L 100.367188 128.996094 L 100.375 128.679688 L 100.378906 128.359375 L 100.386719 128.042969 L 100.390625 127.722656 L 100.394531 127.40625 L 100.402344 127.089844 L 100.40625 126.769531 L 100.414062 126.453125 L 100.421875 126.132812 L 100.425781 125.816406 L 100.433594 125.496094 L 100.441406 125.179688 L 105.921875 125.179688 L 105.925781 125.496094 L 105.933594 125.816406 L 105.941406 126.132812 L 105.945312 126.453125 L 105.953125 126.769531 L 105.957031 127.089844 L 105.964844 127.40625 L 105.96875 127.722656 L 105.976562 128.042969 L 105.980469 128.359375 L 105.988281 128.679688 L 105.992188 128.996094 L 105.996094 129.316406 L 106.003906 129.632812 L 106.007812 129.953125 L 106.015625 130.585938 L 106.019531 130.90625 L 106.027344 131.222656 L 106.03125 131.542969 L 106.035156 131.859375 L 106.039062 132.179688 L 106.046875 132.8125 L 106.050781 133.132812 L 106.054688 133.449219 L 106.058594 133.769531 L 106.0625 134.085938 L 106.0625 134.40625 L 106.066406 134.722656 L 106.070312 135.042969 L 106.078125 135.675781 L 106.078125 135.996094 L 106.082031 136.3125 L 106.085938 136.632812 L 106.085938 136.949219 L 106.089844 137.269531 L 106.089844 137.585938 L 106.09375 137.902344 L 106.097656 138.222656 L 106.097656 138.859375 L 106.101562 139.175781 L 106.101562 139.496094 L 106.105469 139.8125 L 106.105469 140.449219 L 106.109375 140.765625 L 106.109375 141.402344 L 106.113281 141.722656 L 106.113281 143.3125 L 106.117188 143.628906 L 106.117188 146.8125 L 106.113281 147.128906 L 106.113281 149.039062 L 106.109375 149.355469 L 106.109375 150.3125 L 106.105469 150.628906 L 106.105469 151.265625 L 106.101562 151.582031 L 106.101562 152.21875 L 106.097656 152.539062 L 106.097656 153.171875 L 106.09375 153.492188 L 106.09375 154.128906 L 106.089844 154.445312 L 106.089844 154.765625 L 106.085938 155.082031 L 106.085938 155.402344 L 106.082031 155.71875 L 106.082031 156.355469 L 106.078125 156.671875 L 106.078125 156.992188 L 106.074219 157.308594 L 106.074219 157.628906 L 106.070312 157.945312 L 106.070312 158.261719 L 106.066406 158.582031 L 106.066406 158.898438 L 106.0625 159.21875 L 106.0625 159.855469 L 106.058594 160.171875 L 106.058594 160.492188 L 106.054688 160.808594 L 106.054688 161.125 L 106.050781 161.445312 L 106.050781 161.761719 L 106.046875 162.082031 L 106.046875 162.398438 L 106.042969 162.71875 L 106.042969 163.035156 L 106.039062 163.351562 L 106.039062 163.988281 L 106.035156 164.308594 L 106.035156 164.625 L 106.03125 164.945312 L 106.03125 165.261719 L 106.027344 165.582031 L 106.027344 166.214844 L 106.023438 166.535156 L 106.023438 166.851562 L 106.019531 167.171875 L 106.019531 167.808594 L 106.015625 168.125 L 106.015625 168.761719 L 106.011719 169.078125 L 106.011719 169.714844 L 106.007812 170.035156 L 106.007812 170.988281 L 106.003906 171.304688 L 106.003906 172.578125 L 106 172.898438 L 106 174.804688 L 105.996094 175.125 L 105.996094 178.304688 L 106 178.621094 L 106 180.53125 L 106.003906 180.851562 L 106.003906 181.804688 L 106.007812 182.121094 L 106.007812 182.757812 L 106.011719 183.078125 L 106.011719 183.710938 L 106.015625 184.03125 L 106.015625 184.667969 L 106.019531 184.984375 L 106.019531 185.304688 L 106.023438 185.621094 L 106.023438 185.941406 L 106.027344 186.257812 L 106.027344 186.574219 L 106.03125 186.894531 L 106.03125 187.210938 L 106.035156 187.53125 L 106.035156 187.847656 L 106.039062 188.167969 L 106.042969 188.484375 L 106.042969 188.800781 L 106.046875 189.121094 L 106.046875 189.4375 L 106.050781 189.757812 L 106.054688 190.074219 L 106.054688 190.394531 L 106.058594 190.710938 L 106.0625 191.03125 L 106.066406 191.347656 L 106.066406 191.664062 L 106.070312 191.984375 L 106.074219 192.300781 L 106.078125 192.621094 L 106.078125 192.9375 L 106.082031 193.257812 L 106.089844 193.890625 L 106.09375 194.210938 L 106.097656 194.527344 L 106.097656 194.847656 L 106.101562 195.164062 L 106.105469 195.484375 L 106.109375 195.800781 L 106.113281 196.121094 L 106.121094 196.753906 L 106.125 197.074219 L 106.128906 197.390625 L 106.132812 197.710938 L 106.136719 198.027344 L 106.140625 198.347656 L 106.148438 198.980469 L 106.152344 199.300781 L 106.15625 199.617188 L 106.160156 199.9375 L 106.164062 200.253906 L 106.167969 200.574219 L 106.171875 200.890625 L 106.175781 201.210938 L 106.183594 201.84375 L 106.1875 202.164062 L 106.191406 202.480469 L 106.195312 202.800781 L 106.199219 203.117188 L 106.203125 203.4375 L 106.210938 204.070312 L 106.214844 204.390625 L 106.21875 204.707031 L 106.222656 205.027344 L 106.226562 205.34375 L 106.230469 205.664062 L 106.234375 205.980469 L 106.238281 206.300781 L 106.246094 206.933594 L 106.25 207.253906 L 106.257812 207.570312 L 106.261719 207.890625 L 106.265625 208.207031 L 106.269531 208.527344 L 106.277344 209.160156 L 106.28125 209.480469 L 106.285156 209.796875 L 106.289062 210.117188 L 106.292969 210.433594 L 106.296875 210.753906 L 106.300781 211.070312 L 106.304688 211.390625 L 106.3125 212.023438 L 106.3125 212.34375 L 106.316406 212.660156 L 106.320312 212.980469 L 106.324219 213.296875 L 106.328125 213.617188 L 106.335938 214.25 L 106.339844 214.570312 L 106.34375 214.886719 L 106.347656 215.207031 L 106.347656 215.523438 L 106.351562 215.84375 L 106.355469 216.160156 L 106.359375 216.480469 L 106.363281 216.796875 L 106.363281 217.113281 L 106.367188 217.433594 L 106.371094 217.75 L 106.375 218.070312 L 106.375 218.386719 L 106.378906 218.707031 L 106.382812 219.023438 L 106.382812 219.339844 L 106.386719 219.660156 L 106.390625 219.976562 L 106.390625 220.296875 L 106.394531 220.613281 L 106.394531 220.933594 L 106.398438 221.25 L 106.398438 221.570312 L 106.402344 221.886719 L 106.402344 222.203125 L 106.40625 222.523438 L 106.40625 222.839844 L 106.410156 223.160156 L 106.410156 223.476562 L 106.414062 223.796875 L 106.414062 224.429688 L 106.417969 224.75 L 106.417969 225.386719 L 106.421875 225.703125 L 106.421875 226.976562 L 106.425781 227.292969 L 106.425781 231.429688 L 106.421875 231.75 L 106.421875 232.703125 L 106.417969 233.019531 L 106.417969 233.65625 L 106.414062 233.976562 L 106.414062 234.609375 L 106.410156 234.929688 L 106.410156 235.246094 L 106.40625 235.566406 L 106.40625 235.882812 L 106.402344 236.203125 L 106.402344 236.519531 L 106.398438 236.839844 L 106.398438 237.15625 L 106.394531 237.472656 L 106.390625 237.792969 L 106.390625 238.109375 L 106.386719 238.429688 L 106.382812 238.746094 L 106.378906 239.066406 L 106.378906 239.382812 L 106.375 239.699219 L 106.371094 240.019531 L 106.367188 240.335938 L 106.363281 240.65625 L 106.359375 240.972656 L 106.355469 241.292969 L 106.351562 241.609375 L 106.347656 241.929688 L 106.339844 242.5625 L 106.335938 242.882812 L 106.332031 243.199219 L 106.328125 243.519531 L 106.324219 243.835938 L 106.320312 244.15625 L 106.3125 244.472656 L 106.308594 244.789062 L 106.304688 245.109375 L 106.300781 245.425781 L 106.292969 245.746094 L 106.289062 246.0625 L 106.285156 246.382812 L 106.277344 246.699219 L 106.273438 247.019531 L 106.269531 247.335938 L 106.261719 247.652344 L 106.257812 247.972656 L 106.25 248.289062 L 106.246094 248.609375 L 106.238281 248.925781 L 106.234375 249.246094 L 106.21875 249.878906 L 106.214844 250.199219 L 106.207031 250.515625 L 106.203125 250.835938 L 106.195312 251.152344 L 106.1875 251.472656 L 106.179688 251.789062 L 106.175781 252.109375 L 106.160156 252.742188 L 106.152344 253.0625 L 106.144531 253.378906 L 106.140625 253.699219 L 106.132812 254.015625 L 106.125 254.335938 L 106.109375 254.96875 L 106.101562 255.289062 L 106.09375 255.605469 L 106.085938 255.925781 L 106.078125 256.242188 L 106.070312 256.5625 L 106.0625 256.878906 L 106.054688 257.199219 L 106.039062 257.832031 L 106.03125 258.152344 L 106.023438 258.46875 L 106.011719 258.789062 L 106.003906 259.105469 L 105.996094 259.425781 L 105.980469 260.058594 L 105.972656 260.378906 L 105.960938 260.695312 L 105.953125 261.015625 L 105.945312 261.332031 L 105.9375 261.652344 L 105.925781 261.96875 L 105.917969 262.289062 L 105.910156 262.605469 L 105.898438 262.921875 L 105.890625 263.242188 L 105.882812 263.558594 L 105.871094 263.878906 L 105.863281 264.195312 L 105.851562 264.515625 L 105.84375 264.832031 L 105.832031 265.148438 L 105.824219 265.46875 L 105.816406 265.785156 L 105.804688 266.105469 L 105.796875 266.421875 L 105.785156 266.742188 L 105.777344 267.058594 L 105.765625 267.378906 L 105.757812 267.695312 L 105.746094 268.011719 L 105.738281 268.332031 L 105.726562 268.648438 L 105.714844 268.96875 L 105.707031 269.285156 L 105.695312 269.605469 L 105.6875 269.921875 L 105.675781 270.238281 L 105.664062 270.558594 L 105.65625 270.875 L 105.644531 271.195312 L 105.636719 271.511719 L 105.625 271.832031 L 105.613281 272.148438 L 105.605469 272.46875 L 105.582031 273.101562 L 105.574219 273.421875 L 105.5625 273.738281 L 105.550781 274.058594 L 105.542969 274.375 L 105.53125 274.695312 L 105.507812 275.328125 L 105.5 275.648438 L 105.488281 275.964844 L 105.476562 276.285156 L 105.464844 276.601562 L 105.457031 276.921875 L 105.445312 277.238281 L 105.433594 277.558594 L 105.421875 277.875 L 105.414062 278.191406 L 105.402344 278.511719 L 105.390625 278.828125 L 105.378906 279.148438 L 105.371094 279.464844 L 105.359375 279.785156 L 105.335938 280.417969 L 105.328125 280.738281 L 105.316406 281.054688 L 105.304688 281.375 L 105.292969 281.691406 L 105.28125 282.011719 L 105.273438 282.328125 L 105.261719 282.648438 L 105.238281 283.28125 L 105.226562 283.601562 L 105.21875 283.917969 L 105.207031 284.238281 L 105.195312 284.554688 L 105.183594 284.875 L 105.171875 285.191406 L 105.164062 285.507812 L 105.152344 285.828125 L 105.140625 286.144531 L 105.128906 286.464844 L 105.117188 286.78125 L 105.105469 287.101562 L 105.097656 287.417969 L 105.085938 287.738281 Z M 101.277344 287.738281 ' style='fill-rule:nonzero;fill:rgb(97.254902%,46.27451%,42.745098%);fill-opacity:0.501961;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(74.117647%,74.117647%,74.117647%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 213.425781 243.402344 L 213.40625 243.257812 L 213.382812 243.113281 L 213.363281 242.96875 L 213.339844 242.824219 L 213.261719 242.246094 L 213.246094 242.101562 L 213.207031 241.8125 L 213.082031 240.65625 L 213.070312 240.511719 L 213.054688 240.367188 L 212.972656 239.355469 L 212.957031 239.066406 L 212.945312 238.921875 L 212.914062 238.34375 L 212.90625 238.195312 L 212.902344 238.050781 L 212.894531 237.90625 L 212.890625 237.761719 L 212.882812 237.617188 L 212.859375 236.75 L 212.859375 236.605469 L 212.855469 236.460938 L 212.855469 236.171875 L 212.851562 236.027344 L 212.851562 235.449219 L 212.855469 235.304688 L 212.855469 235.160156 L 212.859375 235.015625 L 212.859375 234.871094 L 212.890625 233.714844 L 212.898438 233.570312 L 212.902344 233.425781 L 212.917969 233.136719 L 212.921875 232.992188 L 212.960938 232.269531 L 212.972656 232.125 L 212.988281 231.835938 L 213 231.691406 L 213.007812 231.546875 L 213.066406 230.824219 L 213.074219 230.679688 L 213.089844 230.535156 L 213.125 230.101562 L 213.140625 229.957031 L 213.152344 229.8125 L 213.167969 229.664062 L 213.179688 229.519531 L 213.210938 229.230469 L 213.222656 229.085938 L 213.332031 228.074219 L 213.351562 227.929688 L 213.382812 227.640625 L 213.402344 227.496094 L 213.417969 227.351562 L 213.4375 227.207031 L 213.453125 227.0625 L 213.492188 226.773438 L 213.507812 226.628906 L 213.722656 225.039062 L 213.746094 224.894531 L 213.785156 224.605469 L 213.808594 224.460938 L 213.847656 224.171875 L 213.894531 223.882812 L 213.914062 223.738281 L 213.9375 223.59375 L 213.957031 223.449219 L 214.027344 223.015625 L 214.046875 222.871094 L 214.28125 221.425781 L 214.304688 221.277344 L 214.332031 221.132812 L 214.402344 220.699219 L 214.429688 220.554688 L 214.476562 220.265625 L 214.503906 220.121094 L 214.550781 219.832031 L 214.578125 219.6875 L 214.601562 219.542969 L 214.628906 219.398438 L 214.652344 219.253906 L 214.679688 219.109375 L 214.703125 218.964844 L 214.757812 218.675781 L 214.78125 218.53125 L 214.835938 218.242188 L 214.859375 218.097656 L 214.914062 217.808594 L 214.9375 217.664062 L 215.046875 217.085938 L 215.070312 216.941406 L 215.644531 213.90625 L 215.675781 213.761719 L 215.839844 212.894531 L 215.871094 212.746094 L 216.007812 212.023438 L 216.039062 211.878906 L 216.175781 211.15625 L 216.207031 211.011719 L 216.371094 210.144531 L 216.402344 210 L 216.894531 207.398438 L 216.917969 207.253906 L 217.027344 206.675781 L 217.050781 206.53125 L 217.105469 206.242188 L 217.128906 206.097656 L 217.183594 205.808594 L 217.207031 205.664062 L 217.234375 205.519531 L 217.257812 205.375 L 217.285156 205.230469 L 217.332031 204.941406 L 217.359375 204.796875 L 217.40625 204.507812 L 217.433594 204.359375 L 217.714844 202.625 L 217.734375 202.480469 L 217.78125 202.191406 L 217.800781 202.046875 L 217.824219 201.902344 L 217.863281 201.613281 L 217.886719 201.46875 L 217.925781 201.179688 L 217.949219 201.035156 L 218.066406 200.167969 L 218.082031 200.023438 L 218.121094 199.734375 L 218.136719 199.589844 L 218.15625 199.445312 L 218.171875 199.300781 L 218.191406 199.15625 L 218.207031 199.011719 L 218.226562 198.867188 L 218.335938 197.855469 L 218.347656 197.710938 L 218.378906 197.421875 L 218.390625 197.277344 L 218.40625 197.132812 L 218.417969 196.988281 L 218.433594 196.84375 L 218.503906 195.976562 L 218.515625 195.828125 L 218.539062 195.539062 L 218.546875 195.394531 L 218.558594 195.25 L 218.566406 195.105469 L 218.578125 194.960938 L 218.601562 194.527344 L 218.613281 194.382812 L 218.628906 194.09375 L 218.632812 193.949219 L 218.65625 193.515625 L 218.660156 193.371094 L 218.667969 193.226562 L 218.671875 193.082031 L 218.679688 192.9375 L 218.714844 191.636719 L 218.714844 191.492188 L 218.71875 191.347656 L 218.71875 191.203125 L 218.722656 191.058594 L 218.722656 190.914062 L 218.726562 190.769531 L 218.726562 189.757812 L 218.722656 189.613281 L 218.722656 189.324219 L 218.71875 189.179688 L 218.71875 189.035156 L 218.714844 188.890625 L 218.714844 188.746094 L 218.703125 188.3125 L 218.703125 188.167969 L 218.691406 187.734375 L 218.683594 187.589844 L 218.679688 187.441406 L 218.671875 187.152344 L 218.664062 187.007812 L 218.65625 186.71875 L 218.648438 186.574219 L 218.644531 186.429688 L 218.628906 186.140625 L 218.625 185.996094 L 218.609375 185.707031 L 218.605469 185.5625 L 218.542969 184.40625 L 218.53125 184.261719 L 218.5 183.683594 L 218.488281 183.539062 L 218.472656 183.25 L 218.460938 183.105469 L 218.453125 182.960938 L 218.441406 182.816406 L 218.433594 182.671875 L 218.421875 182.527344 L 218.414062 182.382812 L 218.402344 182.238281 L 218.394531 182.09375 L 218.382812 181.949219 L 218.375 181.804688 L 218.363281 181.660156 L 218.355469 181.515625 L 218.34375 181.371094 L 218.335938 181.226562 L 218.3125 180.9375 L 218.304688 180.792969 L 218.28125 180.503906 L 218.273438 180.359375 L 218.261719 180.214844 L 218.253906 180.070312 L 218.230469 179.78125 L 218.222656 179.636719 L 218.199219 179.347656 L 218.191406 179.203125 L 218.179688 179.058594 L 218.171875 178.910156 L 218.160156 178.765625 L 218.152344 178.621094 L 218.140625 178.476562 L 218.132812 178.332031 L 218.121094 178.1875 L 218.113281 178.042969 L 218.101562 177.898438 L 218.09375 177.753906 L 218.082031 177.609375 L 218.074219 177.464844 L 218.0625 177.320312 L 218.046875 177.03125 L 218.035156 176.886719 L 218.011719 176.453125 L 218 176.308594 L 217.921875 174.863281 L 217.917969 174.71875 L 217.902344 174.429688 L 217.898438 174.285156 L 217.882812 173.996094 L 217.878906 173.851562 L 217.871094 173.707031 L 217.863281 173.417969 L 217.855469 173.273438 L 217.839844 172.695312 L 217.832031 172.550781 L 217.824219 172.261719 L 217.824219 172.117188 L 217.8125 171.683594 L 217.8125 171.539062 L 217.804688 171.25 L 217.804688 171.105469 L 217.800781 170.960938 L 217.800781 170.671875 L 217.796875 170.527344 L 217.796875 169.511719 L 222.976562 169.511719 L 222.976562 170.527344 L 222.972656 170.671875 L 222.972656 170.960938 L 222.96875 171.105469 L 222.96875 171.25 L 222.964844 171.394531 L 222.964844 171.539062 L 222.953125 171.972656 L 222.953125 172.117188 L 222.9375 172.695312 L 222.929688 172.839844 L 222.914062 173.417969 L 222.90625 173.5625 L 222.902344 173.707031 L 222.894531 173.851562 L 222.890625 173.996094 L 222.882812 174.140625 L 222.878906 174.285156 L 222.855469 174.71875 L 222.851562 174.863281 L 222.765625 176.453125 L 222.753906 176.597656 L 222.738281 176.886719 L 222.726562 177.03125 L 222.710938 177.320312 L 222.699219 177.464844 L 222.683594 177.753906 L 222.671875 177.898438 L 222.664062 178.042969 L 222.652344 178.1875 L 222.644531 178.332031 L 222.632812 178.476562 L 222.625 178.621094 L 222.601562 178.910156 L 222.59375 179.058594 L 222.582031 179.203125 L 222.574219 179.347656 L 222.550781 179.636719 L 222.542969 179.78125 L 222.53125 179.925781 L 222.523438 180.070312 L 222.5 180.359375 L 222.492188 180.503906 L 222.46875 180.792969 L 222.460938 180.9375 L 222.449219 181.082031 L 222.441406 181.226562 L 222.417969 181.515625 L 222.410156 181.660156 L 222.398438 181.804688 L 222.390625 181.949219 L 222.378906 182.09375 L 222.371094 182.238281 L 222.359375 182.382812 L 222.351562 182.527344 L 222.339844 182.671875 L 222.332031 182.816406 L 222.320312 182.960938 L 222.304688 183.25 L 222.292969 183.394531 L 222.277344 183.683594 L 222.265625 183.828125 L 222.242188 184.261719 L 222.230469 184.40625 L 222.175781 185.417969 L 222.171875 185.5625 L 222.148438 185.996094 L 222.144531 186.140625 L 222.136719 186.285156 L 222.132812 186.429688 L 222.117188 186.71875 L 222.109375 187.007812 L 222.101562 187.152344 L 222.09375 187.441406 L 222.089844 187.589844 L 222.0625 188.601562 L 222.0625 188.746094 L 222.054688 189.035156 L 222.054688 189.179688 L 222.050781 189.324219 L 222.050781 189.613281 L 222.046875 189.757812 L 222.046875 190.625 L 222.050781 190.769531 L 222.050781 191.058594 L 222.054688 191.203125 L 222.054688 191.347656 L 222.058594 191.492188 L 222.058594 191.636719 L 222.09375 192.9375 L 222.101562 193.082031 L 222.105469 193.226562 L 222.113281 193.371094 L 222.117188 193.515625 L 222.1875 194.816406 L 222.199219 194.960938 L 222.214844 195.25 L 222.238281 195.539062 L 222.246094 195.683594 L 222.257812 195.828125 L 222.269531 195.976562 L 222.328125 196.699219 L 222.34375 196.84375 L 222.367188 197.132812 L 222.382812 197.277344 L 222.394531 197.421875 L 222.425781 197.710938 L 222.4375 197.855469 L 222.53125 198.722656 L 222.550781 198.867188 L 222.582031 199.15625 L 222.601562 199.300781 L 222.617188 199.445312 L 222.636719 199.589844 L 222.652344 199.734375 L 222.710938 200.167969 L 222.726562 200.3125 L 222.804688 200.890625 L 222.828125 201.035156 L 222.886719 201.46875 L 222.910156 201.613281 L 222.949219 201.902344 L 222.996094 202.191406 L 223.015625 202.335938 L 223.0625 202.625 L 223.082031 202.769531 L 223.339844 204.359375 L 223.367188 204.507812 L 223.414062 204.796875 L 223.441406 204.941406 L 223.488281 205.230469 L 223.515625 205.375 L 223.539062 205.519531 L 223.59375 205.808594 L 223.617188 205.953125 L 223.644531 206.097656 L 223.667969 206.242188 L 223.75 206.675781 L 223.773438 206.820312 L 223.882812 207.398438 L 223.90625 207.542969 L 224.316406 209.710938 L 224.347656 209.855469 L 224.539062 210.867188 L 224.570312 211.011719 L 224.707031 211.734375 L 224.738281 211.878906 L 224.875 212.601562 L 224.90625 212.746094 L 224.933594 212.894531 L 225.070312 213.617188 L 225.101562 213.761719 L 225.730469 217.085938 L 225.753906 217.230469 L 225.835938 217.664062 L 225.859375 217.808594 L 225.941406 218.242188 L 225.964844 218.386719 L 226.019531 218.675781 L 226.042969 218.820312 L 226.070312 218.964844 L 226.09375 219.109375 L 226.121094 219.253906 L 226.144531 219.398438 L 226.171875 219.542969 L 226.195312 219.6875 L 226.222656 219.832031 L 226.246094 219.976562 L 226.273438 220.121094 L 226.320312 220.410156 L 226.347656 220.554688 L 226.417969 220.988281 L 226.445312 221.132812 L 226.46875 221.277344 L 226.492188 221.425781 L 226.75 223.015625 L 226.769531 223.160156 L 226.816406 223.449219 L 226.835938 223.59375 L 226.882812 223.882812 L 226.902344 224.027344 L 226.925781 224.171875 L 226.945312 224.316406 L 226.96875 224.460938 L 227.007812 224.75 L 227.03125 224.894531 L 227.109375 225.472656 L 227.132812 225.617188 L 227.191406 226.050781 L 227.207031 226.195312 L 227.285156 226.773438 L 227.300781 226.917969 L 227.320312 227.0625 L 227.335938 227.207031 L 227.355469 227.351562 L 227.371094 227.496094 L 227.390625 227.640625 L 227.40625 227.785156 L 227.425781 227.929688 L 227.566406 229.230469 L 227.578125 229.375 L 227.609375 229.664062 L 227.621094 229.8125 L 227.632812 229.957031 L 227.648438 230.101562 L 227.671875 230.390625 L 227.6875 230.535156 L 227.722656 230.96875 L 227.730469 231.113281 L 227.765625 231.546875 L 227.773438 231.691406 L 227.785156 231.835938 L 227.792969 231.980469 L 227.804688 232.125 L 227.859375 233.136719 L 227.863281 233.28125 L 227.871094 233.425781 L 227.875 233.570312 L 227.882812 233.714844 L 227.894531 234.148438 L 227.902344 234.292969 L 227.90625 234.4375 L 227.90625 234.582031 L 227.914062 234.871094 L 227.914062 235.015625 L 227.917969 235.160156 L 227.917969 235.304688 L 227.921875 235.449219 L 227.921875 236.171875 L 227.917969 236.316406 L 227.917969 236.460938 L 227.914062 236.605469 L 227.914062 236.75 L 227.886719 237.761719 L 227.878906 237.90625 L 227.875 238.050781 L 227.867188 238.195312 L 227.859375 238.34375 L 227.820312 239.066406 L 227.808594 239.210938 L 227.800781 239.355469 L 227.71875 240.367188 L 227.703125 240.511719 L 227.691406 240.65625 L 227.660156 240.945312 L 227.648438 241.089844 L 227.617188 241.378906 L 227.597656 241.523438 L 227.566406 241.8125 L 227.546875 241.957031 L 227.53125 242.101562 L 227.433594 242.824219 L 227.410156 242.96875 L 227.371094 243.257812 L 227.347656 243.402344 Z M 213.425781 243.402344 ' style='fill-rule:nonzero;fill:rgb(63.921569%,64.705882%,0%);fill-opacity:0.501961;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(74.117647%,74.117647%,74.117647%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 332.765625 243.402344 L 332.734375 242.941406 L 332.71875 242.707031 L 332.671875 242.015625 L 332.660156 241.785156 L 332.644531 241.550781 L 332.628906 241.320312 L 332.617188 241.089844 L 332.601562 240.859375 L 332.589844 240.625 L 332.574219 240.394531 L 332.550781 239.933594 L 332.539062 239.699219 L 332.492188 238.777344 L 332.480469 238.542969 L 332.457031 238.082031 L 332.449219 237.851562 L 332.4375 237.617188 L 332.425781 237.386719 L 332.410156 236.925781 L 332.398438 236.695312 L 332.390625 236.460938 L 332.382812 236.230469 L 332.371094 236 L 332.363281 235.769531 L 332.355469 235.535156 L 332.332031 234.84375 L 332.324219 234.609375 L 332.316406 234.378906 L 332.3125 234.148438 L 332.296875 233.6875 L 332.292969 233.453125 L 332.285156 233.222656 L 332.28125 232.992188 L 332.273438 232.761719 L 332.269531 232.527344 L 332.261719 232.296875 L 332.253906 231.835938 L 332.246094 231.605469 L 332.242188 231.371094 L 332.230469 230.679688 L 332.226562 230.445312 L 332.214844 229.753906 L 332.210938 229.519531 L 332.203125 229.058594 L 332.203125 228.828125 L 332.199219 228.597656 L 332.195312 228.363281 L 332.195312 228.132812 L 332.1875 227.671875 L 332.1875 227.4375 L 332.183594 227.207031 L 332.183594 226.976562 L 332.179688 226.746094 L 332.179688 226.515625 L 332.175781 226.28125 L 332.175781 225.820312 L 332.171875 225.589844 L 332.171875 225.125 L 332.167969 224.894531 L 332.167969 224.429688 L 332.164062 224.199219 L 332.164062 223.042969 L 332.160156 222.8125 L 332.160156 220.035156 L 332.15625 219.804688 L 332.15625 217.027344 L 332.160156 216.796875 L 332.160156 214.019531 L 332.15625 213.789062 L 332.15625 204.304688 L 332.160156 204.070312 L 332.160156 202.914062 L 332.164062 202.683594 L 332.164062 201.988281 L 332.167969 201.757812 L 332.167969 201.296875 L 332.171875 201.066406 L 332.171875 200.832031 L 332.175781 200.601562 L 332.175781 200.371094 L 332.179688 200.140625 L 332.179688 199.90625 L 332.183594 199.675781 L 332.183594 199.445312 L 332.1875 199.214844 L 332.1875 198.980469 L 332.195312 198.519531 L 332.195312 198.289062 L 332.199219 198.058594 L 332.203125 197.824219 L 332.214844 197.132812 L 332.21875 196.898438 L 332.21875 196.667969 L 332.222656 196.4375 L 332.230469 196.207031 L 332.234375 195.976562 L 332.238281 195.742188 L 332.25 195.050781 L 332.257812 194.816406 L 332.265625 194.355469 L 332.273438 194.125 L 332.277344 193.890625 L 332.285156 193.660156 L 332.289062 193.429688 L 332.304688 192.96875 L 332.3125 192.734375 L 332.316406 192.503906 L 332.332031 192.042969 L 332.339844 191.808594 L 332.371094 190.886719 L 332.378906 190.652344 L 332.390625 190.421875 L 332.40625 189.960938 L 332.417969 189.726562 L 332.425781 189.496094 L 332.4375 189.265625 L 332.445312 189.035156 L 332.457031 188.800781 L 332.464844 188.570312 L 332.5 187.878906 L 332.511719 187.644531 L 332.546875 186.953125 L 332.558594 186.71875 L 332.582031 186.257812 L 332.597656 186.027344 L 332.609375 185.796875 L 332.621094 185.5625 L 332.636719 185.332031 L 332.648438 185.101562 L 332.664062 184.871094 L 332.675781 184.636719 L 332.707031 184.175781 L 332.71875 183.945312 L 332.734375 183.710938 L 332.796875 182.789062 L 332.8125 182.554688 L 332.828125 182.324219 L 332.847656 182.09375 L 332.863281 181.863281 L 332.878906 181.628906 L 332.894531 181.398438 L 332.914062 181.167969 L 332.929688 180.9375 L 332.949219 180.707031 L 332.964844 180.472656 L 332.984375 180.242188 L 333 180.011719 L 333.019531 179.78125 L 333.039062 179.546875 L 333.058594 179.316406 L 333.074219 179.085938 L 333.09375 178.855469 L 333.113281 178.621094 L 333.191406 177.699219 L 333.210938 177.464844 L 333.269531 176.773438 L 333.289062 176.539062 L 333.3125 176.308594 L 333.371094 175.617188 L 333.394531 175.382812 L 333.414062 175.152344 L 333.4375 174.921875 L 333.457031 174.691406 L 333.476562 174.457031 L 333.5 174.226562 L 333.519531 173.996094 L 333.542969 173.765625 L 333.5625 173.53125 L 333.585938 173.300781 L 333.605469 173.070312 L 333.628906 172.839844 L 333.648438 172.609375 L 333.671875 172.375 L 333.695312 172.144531 L 333.714844 171.914062 L 333.738281 171.683594 L 333.761719 171.449219 L 333.78125 171.21875 L 333.804688 170.988281 L 333.824219 170.757812 L 333.847656 170.527344 L 333.871094 170.292969 L 333.890625 170.0625 L 333.9375 169.601562 L 333.957031 169.367188 L 334.003906 168.90625 L 334.023438 168.675781 L 334.046875 168.441406 L 334.070312 168.210938 L 334.089844 167.980469 L 334.113281 167.75 L 334.132812 167.519531 L 334.15625 167.285156 L 334.179688 167.054688 L 334.199219 166.824219 L 334.222656 166.59375 L 334.242188 166.359375 L 334.265625 166.128906 L 334.285156 165.898438 L 334.308594 165.667969 L 334.328125 165.433594 L 334.351562 165.203125 L 334.390625 164.742188 L 334.414062 164.511719 L 334.433594 164.277344 L 334.453125 164.046875 L 334.476562 163.816406 L 334.496094 163.585938 L 334.515625 163.351562 L 334.535156 163.121094 L 334.558594 162.890625 L 334.597656 162.429688 L 334.617188 162.195312 L 334.675781 161.503906 L 334.695312 161.269531 L 334.734375 160.808594 L 334.75 160.578125 L 334.769531 160.34375 L 334.808594 159.882812 L 334.824219 159.652344 L 334.84375 159.421875 L 334.859375 159.1875 L 334.878906 158.957031 L 334.894531 158.726562 L 334.914062 158.496094 L 334.929688 158.261719 L 334.949219 158.03125 L 334.996094 157.339844 L 335.015625 157.105469 L 335.0625 156.414062 L 335.078125 156.179688 L 335.09375 155.949219 L 335.105469 155.71875 L 335.121094 155.488281 L 335.136719 155.253906 L 335.152344 155.023438 L 335.164062 154.792969 L 335.195312 154.332031 L 335.207031 154.097656 L 335.222656 153.867188 L 335.246094 153.40625 L 335.261719 153.171875 L 335.308594 152.25 L 335.320312 152.015625 L 335.335938 151.785156 L 335.34375 151.554688 L 335.355469 151.324219 L 335.367188 151.089844 L 335.402344 150.398438 L 335.410156 150.164062 L 335.421875 149.933594 L 335.429688 149.703125 L 335.441406 149.472656 L 335.449219 149.242188 L 335.460938 149.007812 L 335.476562 148.546875 L 335.488281 148.316406 L 335.496094 148.082031 L 335.527344 147.160156 L 335.535156 146.925781 L 335.558594 146.234375 L 335.566406 146 L 335.574219 145.769531 L 335.578125 145.539062 L 335.585938 145.308594 L 335.59375 145.074219 L 335.597656 144.84375 L 335.613281 144.382812 L 335.617188 144.152344 L 335.621094 143.917969 L 335.628906 143.6875 L 335.632812 143.457031 L 335.640625 143.226562 L 335.644531 142.992188 L 335.648438 142.761719 L 335.65625 142.53125 L 335.664062 142.070312 L 335.667969 141.835938 L 335.679688 141.144531 L 335.6875 140.910156 L 335.699219 140.21875 L 335.699219 139.984375 L 335.714844 139.0625 L 335.71875 138.828125 L 335.726562 138.367188 L 335.726562 138.136719 L 335.730469 137.902344 L 335.738281 137.441406 L 335.738281 137.210938 L 335.742188 136.980469 L 335.746094 136.746094 L 335.746094 136.515625 L 335.753906 136.054688 L 335.753906 135.820312 L 335.761719 135.359375 L 335.761719 135.128906 L 335.765625 134.894531 L 335.765625 134.664062 L 335.773438 134.203125 L 335.773438 133.972656 L 335.777344 133.738281 L 335.777344 133.507812 L 335.785156 133.046875 L 335.785156 132.8125 L 335.789062 132.582031 L 335.789062 132.351562 L 335.796875 131.890625 L 335.796875 131.65625 L 335.800781 131.425781 L 335.800781 131.195312 L 335.804688 130.964844 L 335.808594 130.730469 L 335.808594 130.5 L 335.816406 130.039062 L 335.816406 129.804688 L 335.824219 129.34375 L 335.824219 129.113281 L 335.828125 128.882812 L 335.832031 128.648438 L 335.832031 128.417969 L 335.839844 127.957031 L 335.84375 127.722656 L 335.84375 127.492188 L 335.855469 126.800781 L 335.859375 126.566406 L 335.859375 126.335938 L 335.867188 125.875 L 335.871094 125.640625 L 335.878906 125.179688 L 339.308594 125.179688 L 339.316406 125.640625 L 339.320312 125.875 L 339.328125 126.335938 L 339.328125 126.566406 L 339.332031 126.800781 L 339.34375 127.492188 L 339.34375 127.722656 L 339.347656 127.957031 L 339.355469 128.417969 L 339.355469 128.648438 L 339.359375 128.882812 L 339.363281 129.113281 L 339.363281 129.34375 L 339.371094 129.804688 L 339.371094 130.039062 L 339.378906 130.5 L 339.378906 130.730469 L 339.382812 130.964844 L 339.382812 131.195312 L 339.390625 131.65625 L 339.390625 131.890625 L 339.394531 132.121094 L 339.394531 132.351562 L 339.402344 132.8125 L 339.402344 133.046875 L 339.40625 133.277344 L 339.40625 133.507812 L 339.410156 133.738281 L 339.414062 133.972656 L 339.414062 134.203125 L 339.417969 134.433594 L 339.417969 134.664062 L 339.421875 134.894531 L 339.425781 135.128906 L 339.425781 135.359375 L 339.433594 135.820312 L 339.433594 136.054688 L 339.441406 136.515625 L 339.441406 136.746094 L 339.445312 136.980469 L 339.449219 137.210938 L 339.449219 137.441406 L 339.457031 137.902344 L 339.460938 138.136719 L 339.460938 138.367188 L 339.46875 138.828125 L 339.472656 139.0625 L 339.480469 139.523438 L 339.480469 139.753906 L 339.484375 139.984375 L 339.488281 140.21875 L 339.5 140.910156 L 339.503906 141.144531 L 339.511719 141.375 L 339.519531 141.835938 L 339.523438 142.070312 L 339.53125 142.53125 L 339.539062 142.761719 L 339.542969 142.992188 L 339.546875 143.226562 L 339.554688 143.457031 L 339.5625 143.917969 L 339.570312 144.152344 L 339.574219 144.382812 L 339.589844 144.84375 L 339.59375 145.074219 L 339.601562 145.308594 L 339.609375 145.539062 L 339.613281 145.769531 L 339.621094 146 L 339.628906 146.234375 L 339.652344 146.925781 L 339.660156 147.160156 L 339.691406 148.082031 L 339.699219 148.316406 L 339.710938 148.546875 L 339.726562 149.007812 L 339.738281 149.242188 L 339.746094 149.472656 L 339.757812 149.703125 L 339.765625 149.933594 L 339.777344 150.164062 L 339.785156 150.398438 L 339.820312 151.089844 L 339.832031 151.324219 L 339.839844 151.554688 L 339.863281 152.015625 L 339.875 152.25 L 339.890625 152.480469 L 339.925781 153.171875 L 339.941406 153.40625 L 339.964844 153.867188 L 339.980469 154.097656 L 339.992188 154.332031 L 340.007812 154.5625 L 340.019531 154.792969 L 340.050781 155.253906 L 340.066406 155.488281 L 340.078125 155.71875 L 340.109375 156.179688 L 340.125 156.414062 L 340.171875 157.105469 L 340.191406 157.339844 L 340.238281 158.03125 L 340.257812 158.261719 L 340.273438 158.496094 L 340.289062 158.726562 L 340.308594 158.957031 L 340.324219 159.1875 L 340.34375 159.421875 L 340.363281 159.652344 L 340.378906 159.882812 L 340.417969 160.34375 L 340.433594 160.578125 L 340.492188 161.269531 L 340.511719 161.503906 L 340.570312 162.195312 L 340.589844 162.429688 L 340.648438 163.121094 L 340.671875 163.351562 L 340.691406 163.585938 L 340.730469 164.046875 L 340.753906 164.277344 L 340.773438 164.511719 L 340.792969 164.742188 L 340.816406 164.972656 L 340.835938 165.203125 L 340.859375 165.433594 L 340.878906 165.667969 L 340.902344 165.898438 L 340.921875 166.128906 L 340.945312 166.359375 L 340.964844 166.59375 L 340.988281 166.824219 L 341.007812 167.054688 L 341.03125 167.285156 L 341.050781 167.519531 L 341.097656 167.980469 L 341.117188 168.210938 L 341.140625 168.441406 L 341.164062 168.675781 L 341.183594 168.90625 L 341.230469 169.367188 L 341.25 169.601562 L 341.273438 169.832031 L 341.292969 170.0625 L 341.316406 170.292969 L 341.339844 170.527344 L 341.359375 170.757812 L 341.40625 171.21875 L 341.425781 171.449219 L 341.449219 171.683594 L 341.472656 171.914062 L 341.492188 172.144531 L 341.515625 172.375 L 341.535156 172.609375 L 341.582031 173.070312 L 341.601562 173.300781 L 341.625 173.53125 L 341.644531 173.765625 L 341.667969 173.996094 L 341.6875 174.226562 L 341.710938 174.457031 L 341.730469 174.691406 L 341.75 174.921875 L 341.773438 175.152344 L 341.792969 175.382812 L 341.8125 175.617188 L 341.835938 175.847656 L 341.875 176.308594 L 341.898438 176.539062 L 341.917969 176.773438 L 341.976562 177.464844 L 341.996094 177.699219 L 342.074219 178.621094 L 342.09375 178.855469 L 342.113281 179.085938 L 342.128906 179.316406 L 342.148438 179.546875 L 342.167969 179.78125 L 342.183594 180.011719 L 342.222656 180.472656 L 342.238281 180.707031 L 342.257812 180.9375 L 342.289062 181.398438 L 342.308594 181.628906 L 342.324219 181.863281 L 342.339844 182.09375 L 342.359375 182.324219 L 342.375 182.554688 L 342.390625 182.789062 L 342.453125 183.710938 L 342.464844 183.945312 L 342.511719 184.636719 L 342.523438 184.871094 L 342.539062 185.101562 L 342.550781 185.332031 L 342.566406 185.5625 L 342.578125 185.796875 L 342.589844 186.027344 L 342.605469 186.257812 L 342.628906 186.71875 L 342.640625 186.953125 L 342.675781 187.644531 L 342.6875 187.878906 L 342.710938 188.339844 L 342.71875 188.570312 L 342.730469 188.800781 L 342.742188 189.035156 L 342.75 189.265625 L 342.761719 189.496094 L 342.769531 189.726562 L 342.78125 189.960938 L 342.796875 190.421875 L 342.808594 190.652344 L 342.816406 190.886719 L 342.847656 191.808594 L 342.855469 192.042969 L 342.871094 192.503906 L 342.875 192.734375 L 342.882812 192.96875 L 342.890625 193.199219 L 342.894531 193.429688 L 342.910156 193.890625 L 342.914062 194.125 L 342.917969 194.355469 L 342.925781 194.585938 L 342.929688 194.816406 L 342.933594 195.050781 L 342.941406 195.28125 L 342.949219 195.742188 L 342.953125 195.976562 L 342.96875 196.898438 L 342.972656 197.132812 L 342.984375 197.824219 L 342.988281 198.058594 L 342.988281 198.289062 L 343 198.980469 L 343 199.214844 L 343.003906 199.445312 L 343.003906 199.675781 L 343.007812 199.90625 L 343.007812 200.140625 L 343.011719 200.371094 L 343.011719 200.601562 L 343.015625 200.832031 L 343.015625 201.296875 L 343.019531 201.527344 L 343.019531 201.988281 L 343.023438 202.222656 L 343.023438 202.914062 L 343.027344 203.148438 L 343.027344 204.304688 L 343.03125 204.535156 L 343.03125 212.632812 L 343.027344 212.863281 L 343.027344 222.8125 L 343.023438 223.042969 L 343.023438 223.96875 L 343.019531 224.199219 L 343.019531 224.894531 L 343.015625 225.125 L 343.015625 225.589844 L 343.011719 225.820312 L 343.011719 226.28125 L 343.007812 226.515625 L 343.007812 226.746094 L 343.003906 226.976562 L 343.003906 227.207031 L 343 227.4375 L 343 227.671875 L 342.992188 228.132812 L 342.992188 228.363281 L 342.988281 228.597656 L 342.980469 229.058594 L 342.980469 229.289062 L 342.976562 229.519531 L 342.972656 229.753906 L 342.960938 230.445312 L 342.957031 230.679688 L 342.945312 231.371094 L 342.941406 231.605469 L 342.933594 231.835938 L 342.925781 232.296875 L 342.917969 232.527344 L 342.914062 232.761719 L 342.90625 232.992188 L 342.902344 233.222656 L 342.894531 233.453125 L 342.890625 233.6875 L 342.875 234.148438 L 342.871094 234.378906 L 342.863281 234.609375 L 342.855469 234.84375 L 342.832031 235.535156 L 342.824219 235.769531 L 342.816406 236 L 342.804688 236.230469 L 342.796875 236.460938 L 342.789062 236.695312 L 342.777344 236.925781 L 342.769531 237.15625 L 342.757812 237.386719 L 342.75 237.617188 L 342.738281 237.851562 L 342.730469 238.082031 L 342.707031 238.542969 L 342.695312 238.777344 L 342.648438 239.699219 L 342.636719 239.933594 L 342.625 240.164062 L 342.609375 240.394531 L 342.597656 240.625 L 342.585938 240.859375 L 342.570312 241.089844 L 342.558594 241.320312 L 342.542969 241.550781 L 342.527344 241.785156 L 342.515625 242.015625 L 342.46875 242.707031 L 342.453125 242.941406 L 342.421875 243.402344 Z M 332.765625 243.402344 ' style='fill-rule:nonzero;fill:rgb(0%,74.901961%,49.019608%);fill-opacity:0.501961;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(74.117647%,74.117647%,74.117647%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 439.21875 272.957031 L 439.136719 272.496094 L 439.054688 272.03125 L 438.980469 271.570312 L 438.902344 271.109375 L 438.828125 270.644531 L 438.757812 270.183594 L 438.6875 269.71875 L 438.621094 269.257812 L 438.554688 268.792969 L 438.492188 268.332031 L 438.429688 267.867188 L 438.371094 267.40625 L 438.3125 266.941406 L 438.203125 266.019531 L 438.152344 265.554688 L 438.105469 265.09375 L 438.058594 264.628906 L 438.011719 264.167969 L 437.96875 263.703125 L 437.929688 263.242188 L 437.886719 262.777344 L 437.851562 262.316406 L 437.816406 261.851562 L 437.753906 260.929688 L 437.722656 260.464844 L 437.695312 260.003906 L 437.671875 259.539062 L 437.648438 259.078125 L 437.625 258.613281 L 437.605469 258.152344 L 437.585938 257.6875 L 437.570312 257.226562 L 437.558594 256.761719 L 437.535156 255.839844 L 437.523438 255.375 L 437.515625 254.914062 L 437.511719 254.449219 L 437.507812 253.988281 L 437.503906 253.523438 L 437.5 253.0625 L 437.503906 252.597656 L 437.503906 252.136719 L 437.507812 251.671875 L 437.515625 250.75 L 437.523438 250.285156 L 437.53125 249.824219 L 437.542969 249.359375 L 437.550781 248.898438 L 437.5625 248.433594 L 437.578125 247.972656 L 437.589844 247.507812 L 437.605469 247.046875 L 437.621094 246.582031 L 437.636719 246.121094 L 437.65625 245.660156 L 437.671875 245.195312 L 437.691406 244.734375 L 437.710938 244.269531 L 437.734375 243.808594 L 437.753906 243.34375 L 437.777344 242.882812 L 437.800781 242.417969 L 437.824219 241.957031 L 437.847656 241.492188 L 437.894531 240.570312 L 437.921875 240.105469 L 437.945312 239.644531 L 437.972656 239.179688 L 437.996094 238.71875 L 438.023438 238.253906 L 438.050781 237.792969 L 438.074219 237.328125 L 438.101562 236.867188 L 438.128906 236.402344 L 438.183594 235.480469 L 438.210938 235.015625 L 438.234375 234.554688 L 438.261719 234.089844 L 438.289062 233.628906 L 438.316406 233.164062 L 438.34375 232.703125 L 438.367188 232.238281 L 438.394531 231.777344 L 438.417969 231.3125 L 438.445312 230.851562 L 438.46875 230.390625 L 438.496094 229.925781 L 438.519531 229.464844 L 438.542969 229 L 438.566406 228.539062 L 438.589844 228.074219 L 438.613281 227.613281 L 438.636719 227.148438 L 438.65625 226.6875 L 438.679688 226.222656 L 438.699219 225.761719 L 438.722656 225.300781 L 438.742188 224.835938 L 438.761719 224.375 L 438.78125 223.910156 L 438.800781 223.449219 L 438.816406 222.984375 L 438.835938 222.523438 L 438.851562 222.058594 L 438.871094 221.597656 L 438.886719 221.132812 L 438.917969 220.210938 L 438.933594 219.746094 L 438.945312 219.285156 L 438.960938 218.820312 L 438.972656 218.359375 L 438.984375 217.894531 L 439 217.433594 L 439.011719 216.96875 L 439.023438 216.507812 L 439.035156 216.042969 L 439.042969 215.582031 L 439.054688 215.121094 L 439.066406 214.65625 L 439.074219 214.195312 L 439.082031 213.730469 L 439.09375 213.269531 L 439.101562 212.804688 L 439.109375 212.34375 L 439.117188 211.878906 L 439.125 211.417969 L 439.132812 210.953125 L 439.148438 210.03125 L 439.15625 209.566406 L 439.160156 209.105469 L 439.167969 208.640625 L 439.175781 208.179688 L 439.183594 207.714844 L 439.1875 207.253906 L 439.195312 206.789062 L 439.203125 206.328125 L 439.210938 205.863281 L 439.214844 205.402344 L 439.222656 204.941406 L 439.230469 204.476562 L 439.238281 204.015625 L 439.246094 203.550781 L 439.253906 203.089844 L 439.261719 202.625 L 439.269531 202.164062 L 439.277344 201.699219 L 439.285156 201.238281 L 439.296875 200.773438 L 439.304688 200.3125 L 439.316406 199.851562 L 439.324219 199.386719 L 439.335938 198.925781 L 439.347656 198.460938 L 439.359375 198 L 439.371094 197.535156 L 439.386719 197.074219 L 439.398438 196.609375 L 439.414062 196.148438 L 439.425781 195.683594 L 439.441406 195.222656 L 439.460938 194.761719 L 439.476562 194.296875 L 439.496094 193.835938 L 439.511719 193.371094 L 439.53125 192.910156 L 439.550781 192.445312 L 439.574219 191.984375 L 439.59375 191.519531 L 439.617188 191.058594 L 439.640625 190.59375 L 439.664062 190.132812 L 439.691406 189.671875 L 439.71875 189.207031 L 439.746094 188.746094 L 439.773438 188.28125 L 439.800781 187.820312 L 439.832031 187.355469 L 439.863281 186.894531 L 439.894531 186.429688 L 439.929688 185.96875 L 439.964844 185.503906 L 440.035156 184.582031 L 440.070312 184.117188 L 440.109375 183.65625 L 440.148438 183.191406 L 440.191406 182.730469 L 440.230469 182.265625 L 440.273438 181.804688 L 440.320312 181.339844 L 440.363281 180.878906 L 440.410156 180.414062 L 440.503906 179.492188 L 440.554688 179.027344 L 440.605469 178.566406 L 440.65625 178.101562 L 440.707031 177.640625 L 440.757812 177.175781 L 440.8125 176.714844 L 440.871094 176.25 L 440.925781 175.789062 L 440.984375 175.324219 L 441.101562 174.402344 L 441.160156 173.9375 L 441.222656 173.476562 L 441.285156 173.011719 L 441.347656 172.550781 L 441.410156 172.085938 L 441.476562 171.625 L 441.539062 171.160156 L 441.605469 170.699219 L 441.675781 170.234375 L 441.742188 169.773438 L 441.8125 169.3125 L 441.878906 168.847656 L 441.949219 168.386719 L 442.019531 167.921875 L 442.09375 167.460938 L 442.164062 166.996094 L 442.238281 166.535156 L 442.3125 166.070312 L 442.386719 165.609375 L 442.460938 165.144531 L 442.609375 164.222656 L 442.6875 163.757812 L 442.765625 163.296875 L 442.839844 162.832031 L 442.917969 162.371094 L 442.996094 161.90625 L 443.074219 161.445312 L 443.152344 160.980469 L 443.230469 160.519531 L 443.3125 160.054688 L 443.46875 159.132812 L 443.550781 158.667969 L 443.628906 158.207031 L 443.710938 157.742188 L 443.789062 157.28125 L 443.871094 156.816406 L 443.949219 156.355469 L 444.03125 155.890625 L 444.109375 155.429688 L 444.191406 154.964844 L 444.269531 154.503906 L 444.351562 154.042969 L 444.429688 153.578125 L 444.511719 153.117188 L 444.589844 152.652344 L 444.667969 152.191406 L 444.75 151.726562 L 444.828125 151.265625 L 444.90625 150.800781 L 444.984375 150.339844 L 445.0625 149.875 L 445.140625 149.414062 L 445.214844 148.953125 L 445.292969 148.488281 L 445.371094 148.027344 L 445.445312 147.5625 L 445.519531 147.101562 L 445.59375 146.636719 L 445.667969 146.175781 L 445.742188 145.710938 L 445.816406 145.25 L 445.890625 144.785156 L 446.03125 143.863281 L 446.101562 143.398438 L 446.171875 142.9375 L 446.242188 142.472656 L 446.308594 142.011719 L 446.378906 141.546875 L 446.445312 141.085938 L 446.511719 140.621094 L 446.578125 140.160156 L 446.640625 139.695312 L 446.703125 139.234375 L 446.769531 138.773438 L 446.828125 138.308594 L 446.890625 137.847656 L 446.953125 137.382812 L 447.011719 136.921875 L 447.070312 136.457031 L 447.128906 135.996094 L 447.183594 135.53125 L 447.242188 135.070312 L 447.296875 134.605469 L 447.351562 134.144531 L 447.402344 133.683594 L 447.457031 133.21875 L 447.507812 132.757812 L 447.554688 132.292969 L 447.605469 131.832031 L 447.652344 131.367188 L 447.699219 130.90625 L 447.746094 130.441406 L 447.792969 129.980469 L 447.835938 129.515625 L 447.921875 128.59375 L 447.964844 128.128906 L 448.003906 127.667969 L 448.042969 127.203125 L 448.082031 126.742188 L 448.121094 126.277344 L 448.15625 125.816406 L 448.191406 125.351562 L 448.226562 124.890625 L 448.261719 124.425781 L 448.324219 123.503906 L 448.355469 123.039062 L 448.382812 122.578125 L 448.414062 122.113281 L 448.441406 121.652344 L 448.464844 121.1875 L 448.492188 120.726562 L 448.515625 120.261719 L 448.542969 119.800781 L 448.5625 119.335938 L 448.585938 118.875 L 448.605469 118.414062 L 448.628906 117.949219 L 448.648438 117.488281 L 448.664062 117.023438 L 448.683594 116.5625 L 448.699219 116.097656 L 448.714844 115.636719 L 448.730469 115.171875 L 448.746094 114.710938 L 448.757812 114.246094 L 448.78125 113.324219 L 448.792969 112.859375 L 448.804688 112.398438 L 448.8125 111.933594 L 448.820312 111.472656 L 448.828125 111.007812 L 448.835938 110.546875 L 448.84375 110.082031 L 448.847656 109.621094 L 448.855469 109.15625 L 448.863281 108.234375 L 448.867188 107.769531 L 448.867188 107.308594 L 448.871094 106.84375 L 448.871094 105.917969 L 448.875 105.457031 L 448.871094 104.992188 L 448.871094 104.066406 L 448.867188 103.605469 L 448.867188 103.144531 L 448.863281 102.679688 L 448.859375 102.21875 L 448.855469 101.753906 L 448.851562 101.292969 L 448.847656 100.828125 L 448.84375 100.367188 L 448.835938 99.902344 L 448.832031 99.441406 L 448.824219 98.976562 L 448.820312 98.515625 L 448.8125 98.054688 L 448.804688 97.589844 L 448.796875 97.128906 L 448.789062 96.664062 L 448.78125 96.203125 L 448.773438 95.738281 L 448.765625 95.277344 L 448.757812 94.8125 L 448.75 94.351562 L 448.738281 93.886719 L 448.722656 92.964844 L 448.710938 92.5 L 448.703125 92.039062 L 448.691406 91.574219 L 448.683594 91.113281 L 448.671875 90.648438 L 448.664062 90.1875 L 448.652344 89.722656 L 448.644531 89.261719 L 448.632812 88.796875 L 448.625 88.335938 L 448.613281 87.875 L 448.605469 87.410156 L 448.59375 86.949219 L 448.585938 86.484375 L 448.578125 86.023438 L 448.566406 85.558594 L 448.558594 85.097656 L 448.546875 84.632812 L 448.539062 84.171875 L 448.53125 83.707031 L 448.519531 83.246094 L 448.511719 82.785156 L 448.503906 82.320312 L 448.496094 81.859375 L 448.488281 81.394531 L 448.480469 80.933594 L 448.472656 80.46875 L 448.464844 80.007812 L 448.460938 79.542969 L 448.453125 79.082031 L 448.445312 78.617188 L 448.441406 78.15625 L 448.433594 77.695312 L 448.429688 77.230469 L 448.425781 76.769531 L 448.417969 76.304688 L 448.414062 75.84375 L 448.410156 75.378906 L 448.40625 74.917969 L 448.40625 74.453125 L 448.402344 73.992188 L 448.398438 73.527344 L 448.398438 72.605469 L 448.394531 72.140625 L 448.394531 71.214844 L 448.398438 70.753906 L 448.398438 69.828125 L 448.402344 69.363281 L 448.402344 68.902344 L 448.40625 68.4375 L 448.414062 67.515625 L 448.417969 67.050781 L 448.425781 66.589844 L 448.429688 66.125 L 448.4375 65.664062 L 448.445312 65.199219 L 448.453125 64.738281 L 448.460938 64.273438 L 448.46875 63.8125 L 448.480469 63.347656 L 448.488281 62.886719 L 448.5 62.425781 L 448.511719 61.960938 L 448.523438 61.5 L 448.535156 61.035156 L 448.550781 60.574219 L 448.5625 60.109375 L 448.578125 59.648438 L 448.59375 59.183594 L 448.609375 58.722656 L 448.625 58.257812 L 448.640625 57.796875 L 448.660156 57.335938 L 448.675781 56.871094 L 448.695312 56.410156 L 448.714844 55.945312 L 448.734375 55.484375 L 448.757812 55.019531 L 448.777344 54.558594 L 448.800781 54.09375 L 448.824219 53.632812 L 448.847656 53.167969 L 448.894531 52.246094 L 448.921875 51.78125 L 448.945312 51.320312 L 448.972656 50.855469 L 449 50.394531 L 449.027344 49.929688 L 449.054688 49.46875 L 449.085938 49.003906 L 449.113281 48.542969 L 449.144531 48.078125 L 449.207031 47.15625 L 449.238281 46.691406 L 449.269531 46.230469 L 449.304688 45.765625 L 449.335938 45.304688 L 449.371094 44.839844 L 449.40625 44.378906 L 449.4375 43.914062 L 449.476562 43.453125 L 449.511719 42.988281 L 449.546875 42.527344 L 449.585938 42.066406 L 449.621094 41.601562 L 449.660156 41.140625 L 449.699219 40.675781 L 449.738281 40.214844 L 449.777344 39.75 L 449.816406 39.289062 L 449.855469 38.824219 L 449.894531 38.363281 L 449.9375 37.898438 L 449.976562 37.4375 L 450.019531 36.976562 L 450.0625 36.511719 L 459.539062 36.511719 L 459.582031 36.976562 L 459.621094 37.4375 L 459.664062 37.898438 L 459.703125 38.363281 L 459.746094 38.824219 L 459.785156 39.289062 L 459.824219 39.75 L 459.863281 40.214844 L 459.902344 40.675781 L 459.941406 41.140625 L 459.980469 41.601562 L 460.015625 42.066406 L 460.054688 42.527344 L 460.089844 42.988281 L 460.125 43.453125 L 460.160156 43.914062 L 460.195312 44.378906 L 460.230469 44.839844 L 460.265625 45.304688 L 460.296875 45.765625 L 460.332031 46.230469 L 460.363281 46.691406 L 460.394531 47.15625 L 460.457031 48.078125 L 460.484375 48.542969 L 460.515625 49.003906 L 460.542969 49.46875 L 460.574219 49.929688 L 460.601562 50.394531 L 460.628906 50.855469 L 460.652344 51.320312 L 460.679688 51.78125 L 460.703125 52.246094 L 460.730469 52.707031 L 460.753906 53.167969 L 460.777344 53.632812 L 460.800781 54.09375 L 460.820312 54.558594 L 460.84375 55.019531 L 460.863281 55.484375 L 460.882812 55.945312 L 460.902344 56.410156 L 460.921875 56.871094 L 460.941406 57.335938 L 460.957031 57.796875 L 460.976562 58.257812 L 460.992188 58.722656 L 461.007812 59.183594 L 461.023438 59.648438 L 461.039062 60.109375 L 461.050781 60.574219 L 461.0625 61.035156 L 461.078125 61.5 L 461.089844 61.960938 L 461.101562 62.425781 L 461.109375 62.886719 L 461.121094 63.347656 L 461.128906 63.8125 L 461.140625 64.273438 L 461.148438 64.738281 L 461.15625 65.199219 L 461.164062 65.664062 L 461.167969 66.125 L 461.175781 66.589844 L 461.179688 67.050781 L 461.183594 67.515625 L 461.191406 67.976562 L 461.191406 68.4375 L 461.195312 68.902344 L 461.199219 69.363281 L 461.203125 69.828125 L 461.203125 73.066406 L 461.199219 73.527344 L 461.199219 73.992188 L 461.195312 74.453125 L 461.191406 74.917969 L 461.1875 75.378906 L 461.183594 75.84375 L 461.179688 76.304688 L 461.175781 76.769531 L 461.171875 77.230469 L 461.164062 77.695312 L 461.160156 78.15625 L 461.152344 78.617188 L 461.148438 79.082031 L 461.140625 79.542969 L 461.132812 80.007812 L 461.128906 80.46875 L 461.121094 80.933594 L 461.113281 81.394531 L 461.105469 81.859375 L 461.097656 82.320312 L 461.085938 82.785156 L 461.070312 83.707031 L 461.0625 84.171875 L 461.050781 84.632812 L 461.042969 85.097656 L 461.035156 85.558594 L 461.023438 86.023438 L 461.015625 86.484375 L 461.003906 86.949219 L 460.996094 87.410156 L 460.984375 87.875 L 460.976562 88.335938 L 460.964844 88.796875 L 460.957031 89.261719 L 460.945312 89.722656 L 460.9375 90.1875 L 460.925781 90.648438 L 460.917969 91.113281 L 460.90625 91.574219 L 460.898438 92.039062 L 460.886719 92.5 L 460.878906 92.964844 L 460.871094 93.425781 L 460.859375 93.886719 L 460.851562 94.351562 L 460.84375 94.8125 L 460.835938 95.277344 L 460.824219 95.738281 L 460.816406 96.203125 L 460.808594 96.664062 L 460.800781 97.128906 L 460.796875 97.589844 L 460.789062 98.054688 L 460.773438 98.976562 L 460.769531 99.441406 L 460.761719 99.902344 L 460.757812 100.367188 L 460.753906 100.828125 L 460.75 101.292969 L 460.742188 101.753906 L 460.742188 102.21875 L 460.738281 102.679688 L 460.734375 103.144531 L 460.730469 103.605469 L 460.730469 104.066406 L 460.726562 104.53125 L 460.726562 106.382812 L 460.730469 106.84375 L 460.730469 107.308594 L 460.734375 107.769531 L 460.738281 108.234375 L 460.746094 109.15625 L 460.75 109.621094 L 460.757812 110.082031 L 460.761719 110.546875 L 460.769531 111.007812 L 460.777344 111.472656 L 460.789062 111.933594 L 460.796875 112.398438 L 460.808594 112.859375 L 460.816406 113.324219 L 460.828125 113.785156 L 460.84375 114.246094 L 460.855469 114.710938 L 460.871094 115.171875 L 460.886719 115.636719 L 460.902344 116.097656 L 460.917969 116.5625 L 460.933594 117.023438 L 460.953125 117.488281 L 460.972656 117.949219 L 460.992188 118.414062 L 461.015625 118.875 L 461.035156 119.335938 L 461.058594 119.800781 L 461.082031 120.261719 L 461.109375 120.726562 L 461.132812 121.1875 L 461.160156 121.652344 L 461.1875 122.113281 L 461.214844 122.578125 L 461.246094 123.039062 L 461.277344 123.503906 L 461.339844 124.425781 L 461.375 124.890625 L 461.410156 125.351562 L 461.445312 125.816406 L 461.480469 126.277344 L 461.519531 126.742188 L 461.554688 127.203125 L 461.59375 127.667969 L 461.636719 128.128906 L 461.675781 128.59375 L 461.761719 129.515625 L 461.808594 129.980469 L 461.851562 130.441406 L 461.898438 130.90625 L 461.945312 131.367188 L 461.996094 131.832031 L 462.042969 132.292969 L 462.09375 132.757812 L 462.144531 133.21875 L 462.199219 133.683594 L 462.25 134.144531 L 462.304688 134.605469 L 462.359375 135.070312 L 462.414062 135.53125 L 462.472656 135.996094 L 462.53125 136.457031 L 462.589844 136.921875 L 462.648438 137.382812 L 462.707031 137.847656 L 462.769531 138.308594 L 462.832031 138.773438 L 462.894531 139.234375 L 462.960938 139.695312 L 463.023438 140.160156 L 463.089844 140.621094 L 463.15625 141.085938 L 463.222656 141.546875 L 463.289062 142.011719 L 463.359375 142.472656 L 463.429688 142.9375 L 463.496094 143.398438 L 463.570312 143.863281 L 463.710938 144.785156 L 463.785156 145.25 L 463.855469 145.710938 L 463.929688 146.175781 L 464.003906 146.636719 L 464.078125 147.101562 L 464.15625 147.5625 L 464.230469 148.027344 L 464.308594 148.488281 L 464.382812 148.953125 L 464.539062 149.875 L 464.617188 150.339844 L 464.695312 150.800781 L 464.773438 151.265625 L 464.851562 151.726562 L 464.929688 152.191406 L 465.011719 152.652344 L 465.089844 153.117188 L 465.167969 153.578125 L 465.25 154.042969 L 465.328125 154.503906 L 465.410156 154.964844 L 465.488281 155.429688 L 465.570312 155.890625 L 465.648438 156.355469 L 465.730469 156.816406 L 465.808594 157.28125 L 465.890625 157.742188 L 465.96875 158.207031 L 466.050781 158.667969 L 466.128906 159.132812 L 466.210938 159.59375 L 466.289062 160.054688 L 466.367188 160.519531 L 466.445312 160.980469 L 466.527344 161.445312 L 466.605469 161.90625 L 466.679688 162.371094 L 466.757812 162.832031 L 466.835938 163.296875 L 466.914062 163.757812 L 466.988281 164.222656 L 467.066406 164.683594 L 467.140625 165.144531 L 467.214844 165.609375 L 467.289062 166.070312 L 467.363281 166.535156 L 467.433594 166.996094 L 467.507812 167.460938 L 467.578125 167.921875 L 467.648438 168.386719 L 467.71875 168.847656 L 467.789062 169.3125 L 467.859375 169.773438 L 467.925781 170.234375 L 467.992188 170.699219 L 468.058594 171.160156 L 468.125 171.625 L 468.191406 172.085938 L 468.253906 172.550781 L 468.316406 173.011719 L 468.378906 173.476562 L 468.441406 173.9375 L 468.5 174.402344 L 468.617188 175.324219 L 468.675781 175.789062 L 468.730469 176.25 L 468.785156 176.714844 L 468.839844 177.175781 L 468.894531 177.640625 L 468.945312 178.101562 L 468.996094 178.566406 L 469.046875 179.027344 L 469.097656 179.492188 L 469.191406 180.414062 L 469.238281 180.878906 L 469.28125 181.339844 L 469.324219 181.804688 L 469.367188 182.265625 L 469.410156 182.730469 L 469.449219 183.191406 L 469.488281 183.65625 L 469.527344 184.117188 L 469.566406 184.582031 L 469.636719 185.503906 L 469.671875 185.96875 L 469.703125 186.429688 L 469.738281 186.894531 L 469.769531 187.355469 L 469.796875 187.820312 L 469.828125 188.28125 L 469.855469 188.746094 L 469.882812 189.207031 L 469.910156 189.671875 L 469.933594 190.132812 L 469.960938 190.59375 L 469.984375 191.058594 L 470.003906 191.519531 L 470.027344 191.984375 L 470.046875 192.445312 L 470.066406 192.910156 L 470.085938 193.371094 L 470.105469 193.835938 L 470.125 194.296875 L 470.140625 194.761719 L 470.171875 195.683594 L 470.1875 196.148438 L 470.203125 196.609375 L 470.214844 197.074219 L 470.230469 197.535156 L 470.242188 198 L 470.253906 198.460938 L 470.265625 198.925781 L 470.277344 199.386719 L 470.285156 199.851562 L 470.296875 200.3125 L 470.304688 200.773438 L 470.316406 201.238281 L 470.324219 201.699219 L 470.332031 202.164062 L 470.339844 202.625 L 470.347656 203.089844 L 470.355469 203.550781 L 470.363281 204.015625 L 470.371094 204.476562 L 470.378906 204.941406 L 470.382812 205.402344 L 470.390625 205.863281 L 470.398438 206.328125 L 470.40625 206.789062 L 470.410156 207.253906 L 470.417969 207.714844 L 470.425781 208.179688 L 470.429688 208.640625 L 470.4375 209.105469 L 470.445312 209.566406 L 470.453125 210.03125 L 470.46875 210.953125 L 470.476562 211.417969 L 470.484375 211.878906 L 470.492188 212.34375 L 470.5 212.804688 L 470.507812 213.269531 L 470.515625 213.730469 L 470.527344 214.195312 L 470.535156 214.65625 L 470.546875 215.121094 L 470.554688 215.582031 L 470.566406 216.042969 L 470.578125 216.507812 L 470.589844 216.96875 L 470.601562 217.433594 L 470.613281 217.894531 L 470.625 218.359375 L 470.640625 218.820312 L 470.652344 219.285156 L 470.667969 219.746094 L 470.683594 220.210938 L 470.714844 221.132812 L 470.730469 221.597656 L 470.746094 222.058594 L 470.765625 222.523438 L 470.78125 222.984375 L 470.800781 223.449219 L 470.820312 223.910156 L 470.839844 224.375 L 470.859375 224.835938 L 470.878906 225.300781 L 470.898438 225.761719 L 470.921875 226.222656 L 470.941406 226.6875 L 470.964844 227.148438 L 470.988281 227.613281 L 471.011719 228.074219 L 471.035156 228.539062 L 471.058594 229 L 471.082031 229.464844 L 471.105469 229.925781 L 471.128906 230.390625 L 471.15625 230.851562 L 471.179688 231.3125 L 471.207031 231.777344 L 471.230469 232.238281 L 471.257812 232.703125 L 471.285156 233.164062 L 471.3125 233.628906 L 471.335938 234.089844 L 471.363281 234.554688 L 471.390625 235.015625 L 471.417969 235.480469 L 471.472656 236.402344 L 471.496094 236.867188 L 471.523438 237.328125 L 471.550781 237.792969 L 471.578125 238.253906 L 471.605469 238.71875 L 471.628906 239.179688 L 471.65625 239.644531 L 471.679688 240.105469 L 471.707031 240.570312 L 471.753906 241.492188 L 471.777344 241.957031 L 471.800781 242.417969 L 471.824219 242.882812 L 471.84375 243.34375 L 471.867188 243.808594 L 471.886719 244.269531 L 471.90625 244.734375 L 471.925781 245.195312 L 471.945312 245.660156 L 471.964844 246.121094 L 471.980469 246.582031 L 471.996094 247.046875 L 472.011719 247.507812 L 472.023438 247.972656 L 472.035156 248.433594 L 472.046875 248.898438 L 472.058594 249.359375 L 472.070312 249.824219 L 472.078125 250.285156 L 472.082031 250.75 L 472.089844 251.210938 L 472.09375 251.671875 L 472.097656 252.136719 L 472.097656 253.523438 L 472.09375 253.988281 L 472.089844 254.449219 L 472.085938 254.914062 L 472.074219 255.375 L 472.066406 255.839844 L 472.042969 256.761719 L 472.027344 257.226562 L 472.011719 257.6875 L 471.996094 258.152344 L 471.976562 258.613281 L 471.953125 259.078125 L 471.929688 259.539062 L 471.902344 260.003906 L 471.878906 260.464844 L 471.847656 260.929688 L 471.785156 261.851562 L 471.75 262.316406 L 471.710938 262.777344 L 471.671875 263.242188 L 471.632812 263.703125 L 471.589844 264.167969 L 471.542969 264.628906 L 471.496094 265.09375 L 471.449219 265.554688 L 471.394531 266.019531 L 471.34375 266.480469 L 471.285156 266.941406 L 471.230469 267.40625 L 471.171875 267.867188 L 471.109375 268.332031 L 471.042969 268.792969 L 470.980469 269.257812 L 470.910156 269.71875 L 470.84375 270.183594 L 470.769531 270.644531 L 470.699219 271.109375 L 470.542969 272.03125 L 470.464844 272.496094 L 470.382812 272.957031 Z M 439.21875 272.957031 ' style='fill-rule:nonzero;fill:rgb(0%,69.019608%,96.470588%);fill-opacity:0.501961;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(74.117647%,74.117647%,74.117647%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 569.96875 317.292969 L 569.960938 316.972656 L 569.957031 316.65625 L 569.949219 316.339844 L 569.945312 316.019531 L 569.9375 315.703125 L 569.933594 315.382812 L 569.925781 315.066406 L 569.921875 314.746094 L 569.914062 314.429688 L 569.910156 314.109375 L 569.902344 313.476562 L 569.898438 313.15625 L 569.890625 312.839844 L 569.886719 312.519531 L 569.882812 312.203125 L 569.878906 311.882812 L 569.871094 311.25 L 569.867188 310.929688 L 569.863281 310.613281 L 569.859375 310.292969 L 569.859375 309.976562 L 569.855469 309.65625 L 569.851562 309.339844 L 569.847656 309.019531 L 569.839844 308.386719 L 569.835938 308.066406 L 569.832031 307.75 L 569.828125 307.429688 L 569.824219 307.113281 L 569.820312 306.792969 L 569.8125 306.160156 L 569.808594 305.839844 L 569.804688 305.523438 L 569.796875 305.203125 L 569.792969 304.886719 L 569.789062 304.566406 L 569.78125 304.25 L 569.777344 303.929688 L 569.773438 303.613281 L 569.765625 303.296875 L 569.757812 302.976562 L 569.753906 302.660156 L 569.746094 302.339844 L 569.738281 302.023438 L 569.730469 301.703125 L 569.714844 301.070312 L 569.703125 300.75 L 569.695312 300.433594 L 569.683594 300.113281 L 569.675781 299.796875 L 569.664062 299.476562 L 569.652344 299.160156 L 569.640625 298.839844 L 569.617188 298.207031 L 569.601562 297.886719 L 569.589844 297.570312 L 569.574219 297.25 L 569.558594 296.933594 L 569.542969 296.613281 L 569.511719 295.980469 L 569.496094 295.660156 L 569.476562 295.34375 L 569.457031 295.023438 L 569.4375 294.707031 L 569.417969 294.386719 L 569.398438 294.070312 L 569.378906 293.75 L 569.332031 293.117188 L 569.308594 292.796875 L 569.285156 292.480469 L 569.261719 292.160156 L 569.234375 291.84375 L 569.207031 291.523438 L 569.183594 291.207031 L 569.152344 290.890625 L 569.125 290.570312 L 569.097656 290.253906 L 569.066406 289.933594 L 569.035156 289.617188 L 569.003906 289.296875 L 568.972656 288.980469 L 568.941406 288.660156 L 568.871094 288.027344 L 568.835938 287.707031 L 568.800781 287.390625 L 568.761719 287.070312 L 568.726562 286.753906 L 568.6875 286.433594 L 568.609375 285.800781 L 568.566406 285.480469 L 568.527344 285.164062 L 568.484375 284.84375 L 568.441406 284.527344 L 568.394531 284.207031 L 568.351562 283.890625 L 568.304688 283.570312 L 568.257812 283.253906 L 568.214844 282.9375 L 568.164062 282.617188 L 568.117188 282.300781 L 568.066406 281.980469 L 568.015625 281.664062 L 567.964844 281.34375 L 567.863281 280.710938 L 567.808594 280.390625 L 567.757812 280.074219 L 567.703125 279.753906 L 567.648438 279.4375 L 567.589844 279.117188 L 567.535156 278.800781 L 567.476562 278.480469 L 567.421875 278.164062 L 567.363281 277.847656 L 567.300781 277.527344 L 567.242188 277.210938 L 567.183594 276.890625 L 567.121094 276.574219 L 567.058594 276.253906 L 566.933594 275.621094 L 566.871094 275.300781 L 566.808594 274.984375 L 566.742188 274.664062 L 566.679688 274.347656 L 566.613281 274.027344 L 566.546875 273.710938 L 566.480469 273.390625 L 566.347656 272.757812 L 566.277344 272.4375 L 566.210938 272.121094 L 566.140625 271.800781 L 566.074219 271.484375 L 566.003906 271.164062 L 565.863281 270.53125 L 565.792969 270.210938 L 565.722656 269.894531 L 565.648438 269.574219 L 565.578125 269.257812 L 565.507812 268.9375 L 565.433594 268.621094 L 565.363281 268.300781 L 565.214844 267.667969 L 565.144531 267.347656 L 565.070312 267.03125 L 564.996094 266.710938 L 564.921875 266.394531 L 564.847656 266.074219 L 564.773438 265.757812 L 564.703125 265.441406 L 564.628906 265.121094 L 564.554688 264.804688 L 564.480469 264.484375 L 564.40625 264.167969 L 564.332031 263.847656 L 564.257812 263.53125 L 564.183594 263.210938 L 564.035156 262.578125 L 563.960938 262.257812 L 563.890625 261.941406 L 563.816406 261.621094 L 563.742188 261.304688 L 563.667969 260.984375 L 563.597656 260.667969 L 563.523438 260.351562 L 563.453125 260.03125 L 563.378906 259.714844 L 563.308594 259.394531 L 563.238281 259.078125 L 563.167969 258.757812 L 563.09375 258.441406 L 563.023438 258.121094 L 562.957031 257.804688 L 562.886719 257.488281 L 562.816406 257.167969 L 562.75 256.851562 L 562.679688 256.53125 L 562.613281 256.214844 L 562.546875 255.894531 L 562.414062 255.261719 L 562.347656 254.941406 L 562.285156 254.625 L 562.21875 254.304688 L 562.15625 253.988281 L 562.09375 253.667969 L 562.03125 253.351562 L 561.96875 253.03125 L 561.910156 252.714844 L 561.847656 252.398438 L 561.789062 252.078125 L 561.730469 251.761719 L 561.675781 251.441406 L 561.617188 251.125 L 561.5625 250.804688 L 561.453125 250.171875 L 561.398438 249.851562 L 561.34375 249.535156 L 561.292969 249.214844 L 561.242188 248.898438 L 561.191406 248.578125 L 561.144531 248.261719 L 561.097656 247.941406 L 561.046875 247.625 L 561.003906 247.308594 L 560.957031 246.988281 L 560.914062 246.671875 L 560.867188 246.351562 L 560.828125 246.035156 L 560.785156 245.714844 L 560.707031 245.082031 L 560.667969 244.761719 L 560.628906 244.445312 L 560.59375 244.125 L 560.558594 243.808594 L 560.523438 243.488281 L 560.492188 243.171875 L 560.457031 242.851562 L 560.425781 242.535156 L 560.398438 242.21875 L 560.367188 241.898438 L 560.339844 241.582031 L 560.3125 241.261719 L 560.285156 240.945312 L 560.261719 240.625 L 560.214844 239.992188 L 560.191406 239.671875 L 560.171875 239.355469 L 560.152344 239.035156 L 560.132812 238.71875 L 560.113281 238.398438 L 560.097656 238.082031 L 560.078125 237.761719 L 560.0625 237.445312 L 560.050781 237.128906 L 560.035156 236.808594 L 560.023438 236.492188 L 560.011719 236.171875 L 560 235.855469 L 559.992188 235.535156 L 559.976562 234.902344 L 559.96875 234.582031 L 559.960938 234.265625 L 559.957031 233.945312 L 559.953125 233.628906 L 559.949219 233.308594 L 559.945312 232.992188 L 559.941406 232.671875 L 559.941406 232.355469 L 559.9375 232.039062 L 559.9375 231.402344 L 559.941406 231.082031 L 559.941406 230.765625 L 559.945312 230.445312 L 559.953125 229.8125 L 559.957031 229.492188 L 559.960938 229.175781 L 559.964844 228.855469 L 559.972656 228.539062 L 559.980469 228.21875 L 559.984375 227.902344 L 559.992188 227.582031 L 560 227.265625 L 560.011719 226.949219 L 560.019531 226.628906 L 560.027344 226.3125 L 560.039062 225.992188 L 560.046875 225.675781 L 560.058594 225.355469 L 560.082031 224.722656 L 560.09375 224.402344 L 560.105469 224.085938 L 560.117188 223.765625 L 560.128906 223.449219 L 560.144531 223.128906 L 560.15625 222.8125 L 560.167969 222.492188 L 560.183594 222.175781 L 560.195312 221.859375 L 560.210938 221.539062 L 560.226562 221.222656 L 560.238281 220.902344 L 560.253906 220.585938 L 560.269531 220.265625 L 560.285156 219.949219 L 560.296875 219.632812 L 560.3125 219.3125 L 560.328125 218.996094 L 560.34375 218.675781 L 560.359375 218.359375 L 560.375 218.039062 L 560.390625 217.722656 L 560.40625 217.402344 L 560.4375 216.769531 L 560.453125 216.449219 L 560.46875 216.132812 L 560.488281 215.8125 L 560.503906 215.496094 L 560.519531 215.175781 L 560.550781 214.542969 L 560.566406 214.222656 L 560.582031 213.90625 L 560.597656 213.585938 L 560.613281 213.269531 L 560.628906 212.949219 L 560.648438 212.632812 L 560.664062 212.3125 L 560.695312 211.679688 L 560.710938 211.359375 L 560.726562 211.042969 L 560.742188 210.722656 L 560.757812 210.40625 L 560.773438 210.085938 L 560.789062 209.769531 L 560.804688 209.449219 L 560.835938 208.816406 L 560.851562 208.496094 L 560.867188 208.179688 L 560.882812 207.859375 L 560.898438 207.542969 L 560.914062 207.222656 L 560.945312 206.589844 L 560.960938 206.269531 L 560.976562 205.953125 L 560.992188 205.632812 L 561.007812 205.316406 L 561.023438 204.996094 L 561.035156 204.679688 L 561.050781 204.359375 L 561.082031 203.726562 L 561.097656 203.40625 L 561.113281 203.089844 L 561.125 202.769531 L 561.140625 202.453125 L 561.15625 202.132812 L 561.1875 201.5 L 561.199219 201.179688 L 561.214844 200.863281 L 561.230469 200.542969 L 561.246094 200.226562 L 561.257812 199.90625 L 561.273438 199.589844 L 561.289062 199.269531 L 561.300781 198.953125 L 561.316406 198.636719 L 561.332031 198.316406 L 561.34375 198 L 561.359375 197.679688 L 561.371094 197.363281 L 561.386719 197.042969 L 561.402344 196.726562 L 561.414062 196.410156 L 561.429688 196.089844 L 561.441406 195.773438 L 561.457031 195.453125 L 561.46875 195.136719 L 561.484375 194.816406 L 561.5 194.5 L 561.511719 194.179688 L 561.527344 193.863281 L 561.539062 193.546875 L 561.554688 193.226562 L 561.566406 192.910156 L 561.578125 192.589844 L 561.59375 192.273438 L 561.605469 191.953125 L 561.621094 191.636719 L 561.632812 191.320312 L 561.648438 191 L 561.660156 190.683594 L 561.675781 190.363281 L 561.6875 190.046875 L 561.703125 189.726562 L 561.714844 189.410156 L 561.726562 189.089844 L 561.742188 188.773438 L 561.753906 188.457031 L 561.769531 188.136719 L 561.78125 187.820312 L 561.796875 187.5 L 561.808594 187.183594 L 561.824219 186.863281 L 561.835938 186.546875 L 561.851562 186.230469 L 561.863281 185.910156 L 561.878906 185.59375 L 561.894531 185.273438 L 561.90625 184.957031 L 561.921875 184.636719 L 561.9375 184.320312 L 561.949219 184 L 561.980469 183.367188 L 561.996094 183.046875 L 562.011719 182.730469 L 562.027344 182.410156 L 562.042969 182.09375 L 562.058594 181.773438 L 562.089844 181.140625 L 562.105469 180.820312 L 562.125 180.503906 L 562.140625 180.183594 L 562.15625 179.867188 L 562.175781 179.546875 L 562.195312 179.230469 L 562.210938 178.910156 L 562.25 178.277344 L 562.269531 177.957031 L 562.289062 177.640625 L 562.308594 177.320312 L 562.332031 177.003906 L 562.351562 176.683594 L 562.375 176.367188 L 562.394531 176.050781 L 562.417969 175.730469 L 562.441406 175.414062 L 562.464844 175.09375 L 562.488281 174.777344 L 562.515625 174.457031 L 562.539062 174.140625 L 562.566406 173.820312 L 562.621094 173.1875 L 562.648438 172.867188 L 562.675781 172.550781 L 562.703125 172.230469 L 562.734375 171.914062 L 562.765625 171.59375 L 562.828125 170.960938 L 562.859375 170.640625 L 562.894531 170.324219 L 562.925781 170.003906 L 562.960938 169.6875 L 562.996094 169.367188 L 563.03125 169.050781 L 563.070312 168.730469 L 563.105469 168.414062 L 563.144531 168.097656 L 563.183594 167.777344 L 563.222656 167.460938 L 563.265625 167.140625 L 563.304688 166.824219 L 563.347656 166.503906 L 563.433594 165.871094 L 563.480469 165.550781 L 563.523438 165.234375 L 563.570312 164.914062 L 563.617188 164.597656 L 563.667969 164.277344 L 563.714844 163.960938 L 563.765625 163.640625 L 563.8125 163.324219 L 563.867188 163.007812 L 563.917969 162.6875 L 563.96875 162.371094 L 564.023438 162.050781 L 564.078125 161.734375 L 564.132812 161.414062 L 564.242188 160.78125 L 564.300781 160.460938 L 564.359375 160.144531 L 564.417969 159.824219 L 564.476562 159.507812 L 564.535156 159.1875 L 564.59375 158.871094 L 564.65625 158.550781 L 564.78125 157.917969 L 564.84375 157.597656 L 564.90625 157.28125 L 564.972656 156.960938 L 565.035156 156.644531 L 565.101562 156.324219 L 565.234375 155.691406 L 565.300781 155.371094 L 565.367188 155.054688 L 565.433594 154.734375 L 578.578125 154.734375 L 578.648438 155.054688 L 578.714844 155.371094 L 578.78125 155.691406 L 578.914062 156.324219 L 578.976562 156.644531 L 579.042969 156.960938 L 579.105469 157.28125 L 579.171875 157.597656 L 579.234375 157.917969 L 579.296875 158.234375 L 579.355469 158.550781 L 579.417969 158.871094 L 579.476562 159.1875 L 579.539062 159.507812 L 579.597656 159.824219 L 579.65625 160.144531 L 579.714844 160.460938 L 579.769531 160.78125 L 579.828125 161.097656 L 579.882812 161.414062 L 579.9375 161.734375 L 579.992188 162.050781 L 580.042969 162.371094 L 580.097656 162.6875 L 580.148438 163.007812 L 580.25 163.640625 L 580.296875 163.960938 L 580.347656 164.277344 L 580.394531 164.597656 L 580.441406 164.914062 L 580.488281 165.234375 L 580.535156 165.550781 L 580.578125 165.871094 L 580.664062 166.503906 L 580.707031 166.824219 L 580.75 167.140625 L 580.789062 167.460938 L 580.828125 167.777344 L 580.867188 168.097656 L 580.945312 168.730469 L 580.980469 169.050781 L 581.015625 169.367188 L 581.050781 169.6875 L 581.085938 170.003906 L 581.121094 170.324219 L 581.152344 170.640625 L 581.1875 170.960938 L 581.25 171.59375 L 581.277344 171.914062 L 581.308594 172.230469 L 581.335938 172.550781 L 581.367188 172.867188 L 581.394531 173.1875 L 581.449219 173.820312 L 581.472656 174.140625 L 581.5 174.457031 L 581.523438 174.777344 L 581.546875 175.09375 L 581.570312 175.414062 L 581.59375 175.730469 L 581.617188 176.050781 L 581.640625 176.367188 L 581.660156 176.683594 L 581.683594 177.003906 L 581.703125 177.320312 L 581.722656 177.640625 L 581.742188 177.957031 L 581.761719 178.277344 L 581.800781 178.910156 L 581.820312 179.230469 L 581.835938 179.546875 L 581.855469 179.867188 L 581.871094 180.183594 L 581.890625 180.503906 L 581.90625 180.820312 L 581.921875 181.140625 L 581.941406 181.457031 L 581.957031 181.773438 L 581.972656 182.09375 L 581.988281 182.410156 L 582.003906 182.730469 L 582.019531 183.046875 L 582.035156 183.367188 L 582.046875 183.683594 L 582.0625 184 L 582.078125 184.320312 L 582.09375 184.636719 L 582.105469 184.957031 L 582.121094 185.273438 L 582.132812 185.59375 L 582.148438 185.910156 L 582.164062 186.230469 L 582.175781 186.546875 L 582.191406 186.863281 L 582.203125 187.183594 L 582.21875 187.5 L 582.230469 187.820312 L 582.246094 188.136719 L 582.257812 188.457031 L 582.273438 188.773438 L 582.285156 189.089844 L 582.300781 189.410156 L 582.3125 189.726562 L 582.324219 190.046875 L 582.339844 190.363281 L 582.351562 190.683594 L 582.367188 191 L 582.378906 191.320312 L 582.394531 191.636719 L 582.40625 191.953125 L 582.421875 192.273438 L 582.433594 192.589844 L 582.445312 192.910156 L 582.460938 193.226562 L 582.472656 193.546875 L 582.488281 193.863281 L 582.5 194.179688 L 582.515625 194.5 L 582.527344 194.816406 L 582.542969 195.136719 L 582.558594 195.453125 L 582.570312 195.773438 L 582.585938 196.089844 L 582.597656 196.410156 L 582.613281 196.726562 L 582.625 197.042969 L 582.640625 197.363281 L 582.65625 197.679688 L 582.667969 198 L 582.683594 198.316406 L 582.699219 198.636719 L 582.710938 198.953125 L 582.726562 199.269531 L 582.742188 199.589844 L 582.753906 199.90625 L 582.769531 200.226562 L 582.785156 200.542969 L 582.796875 200.863281 L 582.8125 201.179688 L 582.828125 201.5 L 582.84375 201.816406 L 582.855469 202.132812 L 582.871094 202.453125 L 582.886719 202.769531 L 582.902344 203.089844 L 582.917969 203.40625 L 582.929688 203.726562 L 582.960938 204.359375 L 582.976562 204.679688 L 582.992188 204.996094 L 583.007812 205.316406 L 583.023438 205.632812 L 583.039062 205.953125 L 583.050781 206.269531 L 583.066406 206.589844 L 583.097656 207.222656 L 583.113281 207.542969 L 583.128906 207.859375 L 583.144531 208.179688 L 583.160156 208.496094 L 583.175781 208.816406 L 583.207031 209.449219 L 583.222656 209.769531 L 583.238281 210.085938 L 583.253906 210.40625 L 583.269531 210.722656 L 583.285156 211.042969 L 583.304688 211.359375 L 583.320312 211.679688 L 583.351562 212.3125 L 583.367188 212.632812 L 583.382812 212.949219 L 583.398438 213.269531 L 583.414062 213.585938 L 583.429688 213.90625 L 583.445312 214.222656 L 583.464844 214.542969 L 583.496094 215.175781 L 583.511719 215.496094 L 583.527344 215.8125 L 583.542969 216.132812 L 583.558594 216.449219 L 583.574219 216.769531 L 583.605469 217.402344 L 583.621094 217.722656 L 583.636719 218.039062 L 583.652344 218.359375 L 583.667969 218.675781 L 583.683594 218.996094 L 583.699219 219.3125 L 583.714844 219.632812 L 583.746094 220.265625 L 583.757812 220.585938 L 583.773438 220.902344 L 583.789062 221.222656 L 583.800781 221.539062 L 583.816406 221.859375 L 583.832031 222.175781 L 583.84375 222.492188 L 583.855469 222.8125 L 583.871094 223.128906 L 583.882812 223.449219 L 583.894531 223.765625 L 583.90625 224.085938 L 583.921875 224.402344 L 583.933594 224.722656 L 583.941406 225.039062 L 583.953125 225.355469 L 583.964844 225.675781 L 583.976562 225.992188 L 583.984375 226.3125 L 583.992188 226.628906 L 584.003906 226.949219 L 584.019531 227.582031 L 584.027344 227.902344 L 584.035156 228.21875 L 584.039062 228.539062 L 584.046875 228.855469 L 584.050781 229.175781 L 584.058594 229.492188 L 584.0625 229.8125 L 584.070312 230.445312 L 584.070312 230.765625 L 584.074219 231.082031 L 584.074219 232.355469 L 584.070312 232.671875 L 584.070312 232.992188 L 584.066406 233.308594 L 584.0625 233.628906 L 584.058594 233.945312 L 584.050781 234.265625 L 584.046875 234.582031 L 584.039062 234.902344 L 584.03125 235.21875 L 584.019531 235.535156 L 584.011719 235.855469 L 584 236.171875 L 583.988281 236.492188 L 583.976562 236.808594 L 583.964844 237.128906 L 583.933594 237.761719 L 583.917969 238.082031 L 583.898438 238.398438 L 583.882812 238.71875 L 583.863281 239.035156 L 583.84375 239.355469 L 583.820312 239.671875 L 583.800781 239.992188 L 583.753906 240.625 L 583.726562 240.945312 L 583.703125 241.261719 L 583.675781 241.582031 L 583.644531 241.898438 L 583.617188 242.21875 L 583.554688 242.851562 L 583.523438 243.171875 L 583.488281 243.488281 L 583.457031 243.808594 L 583.417969 244.125 L 583.382812 244.445312 L 583.347656 244.761719 L 583.308594 245.082031 L 583.269531 245.398438 L 583.226562 245.714844 L 583.1875 246.035156 L 583.144531 246.351562 L 583.101562 246.671875 L 583.058594 246.988281 L 583.011719 247.308594 L 582.917969 247.941406 L 582.871094 248.261719 L 582.820312 248.578125 L 582.769531 248.898438 L 582.71875 249.214844 L 582.667969 249.535156 L 582.617188 249.851562 L 582.5625 250.171875 L 582.453125 250.804688 L 582.394531 251.125 L 582.339844 251.441406 L 582.28125 251.761719 L 582.222656 252.078125 L 582.164062 252.398438 L 582.105469 252.714844 L 582.042969 253.03125 L 581.980469 253.351562 L 581.917969 253.667969 L 581.859375 253.988281 L 581.792969 254.304688 L 581.730469 254.625 L 581.664062 254.941406 L 581.601562 255.261719 L 581.46875 255.894531 L 581.402344 256.214844 L 581.332031 256.53125 L 581.265625 256.851562 L 581.195312 257.167969 L 581.128906 257.488281 L 580.988281 258.121094 L 580.917969 258.441406 L 580.847656 258.757812 L 580.777344 259.078125 L 580.703125 259.394531 L 580.632812 259.714844 L 580.5625 260.03125 L 580.488281 260.351562 L 580.417969 260.667969 L 580.34375 260.984375 L 580.269531 261.304688 L 580.199219 261.621094 L 580.125 261.941406 L 580.050781 262.257812 L 579.976562 262.578125 L 579.828125 263.210938 L 579.757812 263.53125 L 579.683594 263.847656 L 579.609375 264.167969 L 579.535156 264.484375 L 579.460938 264.804688 L 579.386719 265.121094 L 579.3125 265.441406 L 579.164062 266.074219 L 579.089844 266.394531 L 579.015625 266.710938 L 578.945312 267.03125 L 578.871094 267.347656 L 578.796875 267.667969 L 578.726562 267.984375 L 578.652344 268.300781 L 578.578125 268.621094 L 578.507812 268.9375 L 578.4375 269.257812 L 578.363281 269.574219 L 578.292969 269.894531 L 578.222656 270.210938 L 578.152344 270.53125 L 578.011719 271.164062 L 577.941406 271.484375 L 577.871094 271.800781 L 577.804688 272.121094 L 577.734375 272.4375 L 577.667969 272.757812 L 577.601562 273.074219 L 577.53125 273.390625 L 577.464844 273.710938 L 577.398438 274.027344 L 577.335938 274.347656 L 577.269531 274.664062 L 577.207031 274.984375 L 577.140625 275.300781 L 577.078125 275.621094 L 576.953125 276.253906 L 576.890625 276.574219 L 576.832031 276.890625 L 576.769531 277.210938 L 576.710938 277.527344 L 576.652344 277.847656 L 576.535156 278.480469 L 576.480469 278.800781 L 576.421875 279.117188 L 576.367188 279.4375 L 576.3125 279.753906 L 576.257812 280.074219 L 576.203125 280.390625 L 576.152344 280.710938 L 576.097656 281.027344 L 576.046875 281.34375 L 575.996094 281.664062 L 575.945312 281.980469 L 575.898438 282.300781 L 575.847656 282.617188 L 575.800781 282.9375 L 575.707031 283.570312 L 575.664062 283.890625 L 575.617188 284.207031 L 575.574219 284.527344 L 575.53125 284.84375 L 575.488281 285.164062 L 575.445312 285.480469 L 575.40625 285.800781 L 575.328125 286.433594 L 575.289062 286.753906 L 575.25 287.070312 L 575.214844 287.390625 L 575.175781 287.707031 L 575.140625 288.027344 L 575.109375 288.34375 L 575.074219 288.660156 L 575.039062 288.980469 L 575.007812 289.296875 L 574.976562 289.617188 L 574.945312 289.933594 L 574.917969 290.253906 L 574.886719 290.570312 L 574.859375 290.890625 L 574.804688 291.523438 L 574.777344 291.84375 L 574.753906 292.160156 L 574.726562 292.480469 L 574.703125 292.796875 L 574.679688 293.117188 L 574.660156 293.433594 L 574.636719 293.75 L 574.617188 294.070312 L 574.59375 294.386719 L 574.574219 294.707031 L 574.554688 295.023438 L 574.539062 295.34375 L 574.519531 295.660156 L 574.503906 295.980469 L 574.484375 296.296875 L 574.46875 296.613281 L 574.453125 296.933594 L 574.4375 297.25 L 574.425781 297.570312 L 574.410156 297.886719 L 574.398438 298.207031 L 574.386719 298.523438 L 574.371094 298.839844 L 574.359375 299.160156 L 574.351562 299.476562 L 574.339844 299.796875 L 574.328125 300.113281 L 574.320312 300.433594 L 574.308594 300.75 L 574.300781 301.070312 L 574.285156 301.703125 L 574.277344 302.023438 L 574.269531 302.339844 L 574.261719 302.660156 L 574.253906 302.976562 L 574.25 303.296875 L 574.234375 303.929688 L 574.230469 304.25 L 574.226562 304.566406 L 574.21875 304.886719 L 574.214844 305.203125 L 574.210938 305.523438 L 574.207031 305.839844 L 574.199219 306.160156 L 574.191406 306.792969 L 574.1875 307.113281 L 574.183594 307.429688 L 574.179688 307.75 L 574.175781 308.066406 L 574.171875 308.386719 L 574.171875 308.703125 L 574.167969 309.019531 L 574.164062 309.339844 L 574.160156 309.65625 L 574.15625 309.976562 L 574.152344 310.292969 L 574.148438 310.613281 L 574.144531 310.929688 L 574.140625 311.25 L 574.132812 311.882812 L 574.128906 312.203125 L 574.125 312.519531 L 574.121094 312.839844 L 574.117188 313.15625 L 574.113281 313.476562 L 574.109375 313.792969 L 574.101562 314.109375 L 574.097656 314.429688 L 574.09375 314.746094 L 574.085938 315.066406 L 574.082031 315.382812 L 574.074219 315.703125 L 574.070312 316.019531 L 574.0625 316.339844 L 574.058594 316.65625 L 574.050781 316.972656 L 574.042969 317.292969 Z M 569.96875 317.292969 ' style='fill-rule:nonzero;fill:rgb(90.588235%,41.960784%,95.294118%);fill-opacity:0.501961;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(74.117647%,74.117647%,74.117647%);stroke-opacity:0.501961;stroke-miterlimit:10;'/>\n<path d='M 103.179688 220.75 L 103.179688 173.167969 ' style='fill:none;stroke-width:1.066978;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 220.386719 233.460938 L 220.386719 206.054688 ' style='fill:none;stroke-width:1.066978;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 337.59375 217.117188 L 337.59375 185.242188 ' style='fill:none;stroke-width:1.066978;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 454.800781 205.695312 L 454.800781 180.621094 ' style='fill:none;stroke-width:1.066978;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 572.007812 230.335938 L 572.007812 208.441406 ' style='fill:none;stroke-width:1.066978;stroke-linecap:butt;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 106.023438 196.957031 C 106.023438 200.75 100.335938 200.75 100.335938 196.957031 C 100.335938 193.167969 106.023438 193.167969 106.023438 196.957031 ' style='fill-rule:nonzero;fill:rgb(0%,0%,0%);fill-opacity:1;stroke-width:1.417323;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 223.230469 219.757812 C 223.230469 223.546875 217.542969 223.546875 217.542969 219.757812 C 217.542969 215.96875 223.230469 215.96875 223.230469 219.757812 ' style='fill-rule:nonzero;fill:rgb(0%,0%,0%);fill-opacity:1;stroke-width:1.417323;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 340.4375 201.179688 C 340.4375 204.96875 334.75 204.96875 334.75 201.179688 C 334.75 197.390625 340.4375 197.390625 340.4375 201.179688 ' style='fill-rule:nonzero;fill:rgb(0%,0%,0%);fill-opacity:1;stroke-width:1.417323;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 457.644531 193.15625 C 457.644531 196.949219 451.957031 196.949219 451.957031 193.15625 C 451.957031 189.367188 457.644531 189.367188 457.644531 193.15625 ' style='fill-rule:nonzero;fill:rgb(0%,0%,0%);fill-opacity:1;stroke-width:1.417323;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<path d='M 574.847656 219.386719 C 574.847656 223.179688 569.164062 223.179688 569.164062 219.386719 C 569.164062 215.597656 574.847656 215.597656 574.847656 219.386719 ' style='fill-rule:nonzero;fill:rgb(0%,0%,0%);fill-opacity:1;stroke-width:1.417323;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(0%,0%,0%);stroke-opacity:1;stroke-miterlimit:10;'/>\n<g clip-rule='nonzero' clip-path='url(#c7d6984c-89b9-4f16-84c8-3e4bceaebdff)'>\n<path d='M 32.855469 331.332031 L 642.328125 331.332031 L 642.328125 22.472656 L 32.855469 22.472656 Z M 32.855469 331.332031 ' style='fill:none;stroke-width:1.066978;stroke-linecap:round;stroke-linejoin:round;stroke:rgb(94.117647%,94.117647%,94.117647%);stroke-opacity:1;stroke-miterlimit:10;'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='319.800781' x='24.925781' xlink:href='#ee5e8d0f-d720-472b-ab4c-55f82924b4d5'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='245.910156' x='24.925781' xlink:href='#1cb5964f-4b40-4e4a-b820-373522f98df8'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='172.019531' x='21.035156' xlink:href='#3e625fa4-404b-413d-adb1-a377fe35f6a0'/>\n<use y='172.019531' x='24.928223' xlink:href='#ee5e8d0f-d720-472b-ab4c-55f82924b4d5'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='98.132812' x='21.035156' xlink:href='#3e625fa4-404b-413d-adb1-a377fe35f6a0'/>\n<use y='98.132812' x='24.928223' xlink:href='#1cb5964f-4b40-4e4a-b820-373522f98df8'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='80.222656' xlink:href='#90daeba2-e225-470b-a3d9-e31d3a4992da'/>\n<use y='340.382812' x='86.053711' xlink:href='#f8d5fa39-c492-44ea-9d7d-68a38e61c951'/>\n<use y='340.382812' x='89.946777' xlink:href='#1b32b2f8-625d-46b4-9f39-870ef8cdec9e'/>\n<use y='340.382812' x='93.839844' xlink:href='#c70ea6ec-e340-4d96-904e-44ec0f8e0741'/>\n<use y='340.382812' x='97.73291' xlink:href='#dba415b5-c0df-446f-a7f2-56388b34a9c6'/>\n<use y='340.382812' x='99.677734' xlink:href='#d3f65531-fea4-4828-a135-dc4bfbb6e530'/>\n<use y='340.382812' x='103.570801' xlink:href='#35cf7a23-9662-4e0c-a07d-31d81d3837b5'/>\n<use y='340.382812' x='105.125977' xlink:href='#c70ea6ec-e340-4d96-904e-44ec0f8e0741'/>\n<use y='340.382812' x='109.019043' xlink:href='#b17ed1ec-94d3-47cf-b22b-5ab046c81c9d'/>\n<use y='340.382812' x='110.963867' xlink:href='#28fdeedd-93fd-450e-85f9-5b8ed888377e'/>\n<use y='340.382812' x='115.632812' xlink:href='#35cf7a23-9662-4e0c-a07d-31d81d3837b5'/>\n<use y='340.382812' x='117.187988' xlink:href='#7377efc0-f320-4a81-b375-7dbb2d1722b7'/>\n<use y='340.382812' x='121.081055' xlink:href='#6e83c428-661c-4ed7-b5c2-dc49c74df737'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='206.378906' xlink:href='#d57c37e9-68d9-42fe-b75d-1690fb90ed20'/>\n<use y='340.382812' x='211.047852' xlink:href='#d3f65531-fea4-4828-a135-dc4bfbb6e530'/>\n<use y='340.382812' x='214.940918' xlink:href='#bf4e80c0-b751-4261-b946-62b857b4b92b'/>\n<use y='340.382812' x='216.496094' xlink:href='#f8d5fa39-c492-44ea-9d7d-68a38e61c951'/>\n<use y='340.382812' x='220.38916' xlink:href='#b17ed1ec-94d3-47cf-b22b-5ab046c81c9d'/>\n<use y='340.382812' x='222.333984' xlink:href='#ab2210c4-ff5e-4f46-b159-cd56ac6cfcb9'/>\n<use y='340.382812' x='227.00293' xlink:href='#bf4e80c0-b751-4261-b946-62b857b4b92b'/>\n<use y='340.382812' x='228.558105' xlink:href='#dba415b5-c0df-446f-a7f2-56388b34a9c6'/>\n<use y='340.382812' x='230.50293' xlink:href='#f8d5fa39-c492-44ea-9d7d-68a38e61c951'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='315.804688' xlink:href='#57c98abf-9ca3-4d51-b779-25ac929d5a61'/>\n<use y='340.382812' x='320.859863' xlink:href='#7377efc0-f320-4a81-b375-7dbb2d1722b7'/>\n<use y='340.382812' x='324.75293' xlink:href='#962ed40d-5823-47b8-9118-8e3a00d75f24'/>\n<use y='340.382812' x='328.645996' xlink:href='#6e83c428-661c-4ed7-b5c2-dc49c74df737'/>\n<use y='340.382812' x='333.701172' xlink:href='#f8d5fa39-c492-44ea-9d7d-68a38e61c951'/>\n<use y='340.382812' x='337.594238' xlink:href='#f8d5fa39-c492-44ea-9d7d-68a38e61c951'/>\n<use y='340.382812' x='341.487305' xlink:href='#962ed40d-5823-47b8-9118-8e3a00d75f24'/>\n<use y='340.382812' x='345.380371' xlink:href='#b17ed1ec-94d3-47cf-b22b-5ab046c81c9d'/>\n<use y='340.382812' x='347.325195' xlink:href='#333fb815-388c-49b8-a335-bf92e586e41c'/>\n<use y='340.382812' x='352.380371' xlink:href='#35cf7a23-9662-4e0c-a07d-31d81d3837b5'/>\n<use y='340.382812' x='353.935547' xlink:href='#dba415b5-c0df-446f-a7f2-56388b34a9c6'/>\n<use y='340.382812' x='355.880371' xlink:href='#13f515cf-a0e2-4a99-b590-6eeef27a8bdb'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='432.429688' xlink:href='#468bcb4c-1b02-4b33-97a0-4afc199daf21'/>\n<use y='340.382812' x='437.098633' xlink:href='#d3f65531-fea4-4828-a135-dc4bfbb6e530'/>\n<use y='340.382812' x='440.991699' xlink:href='#c70ea6ec-e340-4d96-904e-44ec0f8e0741'/>\n<use y='340.382812' x='444.884766' xlink:href='#b17ed1ec-94d3-47cf-b22b-5ab046c81c9d'/>\n<use y='340.382812' x='446.82959' xlink:href='#107514fc-f8f1-41ad-9dd0-179805ece07a'/>\n<use y='340.382812' x='451.105469' xlink:href='#973d2e25-aae7-4f81-944b-308b40c177cd'/>\n<use y='340.382812' x='453.436523' xlink:href='#d3f65531-fea4-4828-a135-dc4bfbb6e530'/>\n<use y='340.382812' x='457.32959' xlink:href='#c70ea6ec-e340-4d96-904e-44ec0f8e0741'/>\n<use y='340.382812' x='461.222656' xlink:href='#ef51ada5-2d4c-462c-b6e5-5afc24314459'/>\n<use y='340.382812' x='464.722656' xlink:href='#35cf7a23-9662-4e0c-a07d-31d81d3837b5'/>\n<use y='340.382812' x='466.277832' xlink:href='#a45cb9db-1d84-4d5b-8eda-6b0b6811481c'/>\n<use y='340.382812' x='469.777832' xlink:href='#ef51ada5-2d4c-462c-b6e5-5afc24314459'/>\n<use y='340.382812' x='473.277832' xlink:href='#f8d5fa39-c492-44ea-9d7d-68a38e61c951'/>\n</g>\n<g style='fill:rgb(45.098039%,45.098039%,45.098039%);fill-opacity:1;'>\n<use y='340.382812' x='557.414062' xlink:href='#468bcb4c-1b02-4b33-97a0-4afc199daf21'/>\n<use y='340.382812' x='562.083008' xlink:href='#d3f65531-fea4-4828-a135-dc4bfbb6e530'/>\n<use y='340.382812' x='565.976074' xlink:href='#c70ea6ec-e340-4d96-904e-44ec0f8e0741'/>\n<use y='340.382812' x='569.869141' xlink:href='#b17ed1ec-94d3-47cf-b22b-5ab046c81c9d'/>\n<use y='340.382812' x='571.813965' xlink:href='#4d758035-8258-47ea-9963-021105405aaa'/>\n<use y='340.382812' x='575.313965' xlink:href='#f8d5fa39-c492-44ea-9d7d-68a38e61c951'/>\n<use y='340.382812' x='579.207031' xlink:href='#a45cb9db-1d84-4d5b-8eda-6b0b6811481c'/>\n<use y='340.382812' x='582.707031' xlink:href='#7377efc0-f320-4a81-b375-7dbb2d1722b7'/>\n</g>\n<g style='fill:rgb(32.156863%,32.156863%,32.156863%);fill-opacity:1;'>\n<use y='349.703125' x='331.59375' xlink:href='#7516e1e3-0773-4027-9cbf-8840ca81fcca'/>\n<use y='349.703125' x='335.59375' xlink:href='#f79db7a0-3bc8-4275-ac50-7e0487c76514'/>\n<use y='349.703125' x='337.371094' xlink:href='#5d54b0e8-a19f-4938-be0d-0bb89bde688f'/>\n<use y='349.703125' x='339.59375' xlink:href='#d50cb9c8-d65b-4198-9590-3eb99a5be5ad'/>\n</g>\n<g style='fill:rgb(32.156863%,32.156863%,32.156863%);fill-opacity:1;'>\n<use y='202.472656' x='17.449219' xlink:href='#ff1f68c4-39bc-444f-a396-333178de5876'/>\n<use y='198.023438' x='17.449219' xlink:href='#88021c7e-29be-4aeb-88d5-648e75d69dfc'/>\n<use y='194.023438' x='17.449219' xlink:href='#ff1f68c4-39bc-444f-a396-333178de5876'/>\n<use y='189.574219' x='17.449219' xlink:href='#b350b481-9749-4761-878f-584ccbab6926'/>\n<use y='187.796875' x='17.449219' xlink:href='#f2766d9b-3744-4107-b37d-adc38ed36a27'/>\n<use y='186.019531' x='17.449219' xlink:href='#ff1f68c4-39bc-444f-a396-333178de5876'/>\n<use y='181.570312' x='17.449219' xlink:href='#de80a8aa-b936-4dee-98db-9b49a47fd1ac'/>\n<use y='177.121094' x='17.449219' xlink:href='#f2766d9b-3744-4107-b37d-adc38ed36a27'/>\n<use y='175.34375' x='17.449219' xlink:href='#52bf7b8f-8b0b-4b9e-88c1-946e4d038a23'/>\n<use y='170.894531' x='17.449219' xlink:href='#b11abd74-2bcb-43a4-b32c-a0060ba6f7c2'/>\n<use y='165.558594' x='17.449219' xlink:href='#b350b481-9749-4761-878f-584ccbab6926'/>\n<use y='163.78125' x='17.449219' xlink:href='#100d437d-b9c4-402d-aa6e-40eedb4ed9b1'/>\n<use y='159.78125' x='17.449219' xlink:href='#52bf7b8f-8b0b-4b9e-88c1-946e4d038a23'/>\n<use y='155.332031' x='17.449219' xlink:href='#ff8feb97-63d5-4dd4-a6ab-5f55725821be'/>\n</g>\n</g>\n</svg>\n","value":"#huri.plot.GGView{:plot-command [[:<- :g [:data.frame {:city [:c \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"San Jose\" \"Redwood City\" \"Redwood City\" \"Redwood City\" \"Redwood City\" \"Redwood City\" \"Redwood City\" \"Mountain View\" \"Mountain View\" \"Mountain View\" \"Mountain View\" \"Mountain View\" \"Mountain View\" \"Mountain View\" \"Palo Alto\" \"Palo Alto\" \"Palo Alto\" \"Palo Alto\" \"Palo Alto\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Francisco\" \"San Jose\" \"San Francisco\" \"Redwood City\" \"San Jose\"], :availableBikes [:c 8 7 10 11 8 0 5 8 10 10 6 5 6 4 5 13 8 9 9 5 5 13 7 6 11 2 13 5 7 10 5 6 10 3 3 3 3 12 4 15 18 6 11 10 3 9 9 8 3 17 6 4 5 8 7 4 4 18 16 9 9 14 19 3 8 9 3 4 6 5]}]] [[:library :ggplot2] [:library :scales] [:library :grid] [:library :RColorBrewer] [:library :ggrepel] [:<- :palette [:brewer.pal \"Greys\" {:n 9}]] {:color.background :palette[2]} {:color.grid.major :palette[3]} {:color.axis.text :palette[6]} {:color.axis.title :palette[7]} {:color.title :palette[9]}] [:+ [:+ [:+ [:+ [:+ [:+ [:ggplot :g [:aes {:x :city, :y :availableBikes, :fill :city}]] [:geom_violin {:alpha 0.5, :colour :palette[4], :trim true, :scale \"count\"}]] [:stat_summary {:fun.data \"mean_se\", :geom \"pointrange\"}]] [:scale_y_continuous {:labels :comma}]] [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:+ [:theme_bw {:base_size 9}] [:theme {:panel.background [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:plot.background [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:panel.border [:element_rect {:color :color.background}]}]] [:theme {:panel.grid.major [:element_line {:color :color.grid.major, :size 0.25}]}]] [:theme {:panel.grid.minor [:element_blank]}]] [:theme {:axis.ticks [:element_blank]}]] [:theme {:legend.background [:element_rect {:fill :color.background}]}]] [:theme {:legend.key [:element_rect {:fill :color.background, :color :color.background}]}]] [:theme {:legend.text [:element_text {:size 7, :color :color.axis.title}]}]] [:theme {:legend.title [:element_blank]}]] [:theme {:plot.title [:element_text {:size 10, :color :color.title, :vjust 1.25}]}]] [:theme {:axis.text.x [:element_text {:size 7, :color :color.axis.text}]}]] [:theme {:axis.text.y [:element_text {:size 7, :color :color.axis.text}]}]] [:theme {:axis.title.x [:element_text {:size 8, :color :color.axis.title, :vjust 0}]}]] [:theme {:axis.title.y [:element_text {:size 8, :color :color.axis.title, :vjust 1.25}]}]] [:theme {:plot.margin [:unit [:c 0.35 0.2 0.3 0.35] \"cm\"]}]]] [:theme {:legend.position \"none\"}]] [:labs {:y \"availableBikes\", :title \"\", :x \"city\"}]]], :options {:width 9, :height 5}}"}
;; <=

;; @@

;; @@
